/*

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.


    Author: Sebastian Zagrodzki <s.zagrodzki@net.icm.edu.pl>, 2007
    URL: http://sokrates.mimuw.edu.pl/~sebek/gvrpcd/

*/

#include <libnet.h>
#include <unistd.h>
#include <stdio.h>
#include <ctype.h>
#include <string.h>
#include <stdlib.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <syslog.h>
#include <errno.h>
#include <signal.h>

#define VERSION "1.3"

#define log_err(fstr, args...) { syslog(LOG_ERR, fstr, ## args); }
#define log(fstr, args...) { syslog(LOG_NOTICE, fstr, ## args); }
#define log_info(fstr, args...) { if (verb > 0) { syslog(LOG_INFO, fstr, ## args); } }
#define log_debug(fstr, args...) { if (verb > 1) { syslog(LOG_DEBUG, fstr, ## args); } }

#define init_packet() { \
       	memcpy(packet, packet_header, 20); \
       	pktlen = 20; \
       	if (!leavesent) { \
	       	memcpy(packet + 20, packet_leaveall, 2); \
	       	pktlen += 2; \
	       	leavesent = 1; \
       	} \
}

#define send_packet() { \
	if (verb) log_info("Sending packet..."); \
	memcpy(packet + pktlen, packet_tail, 2); \
	pktlen += 2; \
	pktlen_n = htons(pktlen); \
	memcpy(packet + 12, &pktlen_n, 2); \
	if (libnet_write_link(l, packet, pktlen) < 0) { \
		log_err("libnet_write_link: %s", libnet_geterror(l)); \
		exit(1); \
	} \
}

int do_shutdown = 0;

void shutdown_handler(int signo) {
	do_shutdown = signo;
}

int main(int argc, char *argv[]) {
    libnet_t *l;
    char errbuf[LIBNET_ERRBUF_SIZE];
    struct libnet_ether_addr *eth_saddr_s;
    u_int8_t packet_header[20] = {
	  0x01, 0x80, 0xc2, 0x00, 0x00, 0x21, //dst
	  0xff, 0xff, 0xff, 0xff, 0xff, 0xff, //src
	  0xff, 0xff, //pkt lenght
	  0x42, 0x42, 0x03, //LLC
	  0x00, 0x01, //GVRP
	  0x01, // VID
    };
    u_int8_t packet_leaveall[2] = {
	  0x02, // attribute length
	  0x00, // leave all
    };
    u_int8_t packet_joinin[4] = {
	  0x04, // attribute length
	  0x02, //join in (1 == join empty)
	  0xff, 0xff // vlan
    };
    u_int8_t packet_tail[2] = { 0x00, 0x00 };
    u_int8_t packet[1500];
    
    u_int16_t pktlen;
    u_int16_t pktlen_n;
    int leavesent;
    FILE *fd;
    char *fname;
    char *devname;
    char *pidfile;
    
    u_int16_t vlan_id;
    u_int16_t packet_vlan[2];
    char ifname[1024];
    char tmp[1024];
    int i, res;
    int interval;
    char c;
    int verb = 0;
    int daemonize = 0;
    int logtoconsole = 0;

    devname = "eth0";
    pidfile = "/var/run/gvrpcd.pid";
    fname = "/proc/net/vlan/config";
    interval = 3;

    while ((c = getopt (argc, argv, "f:i:t:p:cvhd")) != EOF) {
	    switch (c) {
		    case 'f':
			    fname = optarg; break;
		    case 't':
			    interval = atoi(optarg); break;
		    case 'i':
			    devname = optarg; break;
		    case 'v':
			    verb++; break;
		    case 'd':
			    daemonize = 1; break;
		    case 'p':
			    pidfile = optarg; break;
		    case 'c':
			    logtoconsole = LOG_CONS | LOG_PERROR; break;
		    default:
		    case 'h':
			    printf(
"GVRPCD version " VERSION "    http://sokrates.mimuw.edu.pl/~sebek/gvrpcd/\n"
"Usage: gvrpcd [-cdhv] [-f configfile] [-i iface] [-p pid] [-t time]\n"
"	-f configfile - VLAN config file, defaults to /proc/net/vlan/config\n"
"	-i iface - network interface, default eth0\n"
"	-t time - time between packets, default 3s\n"
"	-p pidfile - pid file (daemon mode), default /var/run/gvrpcd.pid\n"
"	-c - log to console in addition to syslog, default is syslog only\n"
"	-v - increases verbosity\n"
"	-d - become a daemon\n"
"	-h - this help\n");
			    exit(c == 'h' ? 0 : 1);
			    break;
	    }
    }

    openlog("gvrpcd", LOG_PID | logtoconsole, LOG_DAEMON);

    log("iface: %s", devname);
    log("file: %s", fname);
    log("interval: %ds", interval);
    log("verbosity: %d", verb);
    log("starting...");

    l = libnet_init(LIBNET_LINK, devname, errbuf);

    if (l == NULL) {
        log_err("libnet_init() failed: %s", errbuf);
        exit(1); 
    }

    if (daemonize) {
	    pid_t pid;
	    int maxd;
	    
	    log("daemonizing...");
	    umask(0);
	    if ((pid = fork()) < 0) {
		    log_err("fork: %s", strerror(errno));
		    exit (1); 
	    } else if (pid != 0) {
		    _exit(0);
	    }
	    setsid(); 
	    if ((pid = fork()) < 0) {
		    log_err("fork: %s", strerror(errno));
		    exit (1);
	    } else if (pid != 0) {
		    _exit(0); 
	    }
	    if (chdir("/") < 0) {
		    log_err("chdir: %s", strerror(errno));
		    exit(1);
	    }
	    pid = getpid();
	    fd = fopen(pidfile, "wx");
	    if (fd == NULL) {
		    int oldpid;
		    log("fopen(%s): %s", pidfile, strerror(errno));
		    log("stale lockfile?");
		    fd = fopen(pidfile, "r");
		    if (fscanf(fd, "%d", &oldpid)) {
			    log("old PID: %d", oldpid);
			    if (kill(oldpid, 0) == 0) {
				    log_err("Another instance with same pid file running");
				    exit(1);
			    }
		    	    log("looks like one. Unlinking");
		    } else {
			    log("this doesn't look like PID file. Unlinking");
		    }
		    fclose(fd);
		    if (unlink(pidfile) < 0) {
			    log_err("Can't unlink stale lockfile: %s", pidfile);
			    exit(1);
		    }
		    fd = fopen(pidfile, "wx");
	    }
	    if (fd == NULL) {
		    log_err("fopen(%s, final attempt): %s", pidfile,
				    strerror(errno));
		    exit(1);
	    }
	    fprintf(fd, "%d", pid);
	    fclose(fd);

#ifdef HAVE_UNISTD_H
	    maxd = getdtablesize();
#else
	    maxd = 1024;
#endif
	    for (i = 0; i < maxd; i++) {
		if (i != l->fd)
		    close(i);
	    }
	    open("/dev/null", O_NOCTTY);
	    open("/dev/null", O_NOCTTY);
	    open("/dev/null", O_NOCTTY);

	    log("Now running as daemon, pid %d", (int)pid);
    }

    eth_saddr_s = libnet_get_hwaddr(l);
    memcpy(packet_header + 6, eth_saddr_s->ether_addr_octet, 6);

    if (daemonize)
	    signal(SIGHUP, SIG_IGN);
    else
	    signal(SIGHUP, shutdown_handler);
    signal(SIGINT, shutdown_handler);
    signal(SIGTERM, shutdown_handler);
    signal(SIGUSR1, SIG_IGN);
    signal(SIGUSR2, SIG_IGN);

    while(do_shutdown == 0) {
	    fd = fopen(fname, "r");
	    if (!fd) { log_err("fopen: %s: %s", fname, strerror(errno)); exit(1); }
	    log_debug("Config file opened");
	    for (i = 0; i < 2; i++) {
		    if (fgets(tmp, 1000, fd) == NULL) {
			    log_err("fgets: %s", strerror(errno)); exit(1);
		    }
	    }
	    log_debug("Read config file header (2 lines)");
	    leavesent = 0;
	    init_packet();
	    while ((res = fscanf(fd, "%*[^|]| %hd  | %s", &vlan_id, ifname)) == 2) {
		    log_debug("vlan: %d", vlan_id);
		    log_debug("dev: %s", ifname);
		    if (strcmp(devname, ifname) != 0) continue;
		    log_debug("device ok, processing vlan");
		    if (pktlen + 6 > 1500) {
			    send_packet();
			    init_packet();
		    }
		    *packet_vlan = htons(vlan_id);
		    memcpy(packet + pktlen, packet_joinin, 4);
		    memcpy(packet + pktlen + 2, packet_vlan, 2);
		    pktlen += 4;
	    }
	    if (res == EOF) {
		    fclose(fd);
		    send_packet();
	    } else {
		    log_debug("fscanf: %d", res);
		    log_debug("vlan_id: %hd", vlan_id);
		    log_debug("dev: %s", ifname);
		    log_err("fscanf: %s", strerror(errno)); exit(1);
	    }
	    log_debug("Sleeping...");
	    sleep(interval);
    }
    log("Terminating on signal %d", do_shutdown);
    if (daemonize) {
	    unlink(pidfile);
    }
    return 0;
}
