/******************************************************************************
*******************************************************************************
**
**    Copyright 1998-1999 Grant M. Erickson <gerickson@brocade.com>
**
**    Brocade Communications Systems, Inc.
** 
**    Module name: fwdl-sun.C
**    Date:        1998/09/30
**
**    Description:
**      This modules implements code which downloads firmware to a Seagate
**      drive attached to a Sun Microsystem's workstation.
**
*******************************************************************************
******************************************************************************/

/*
 *    This program is free software; you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation; either version 2 of the License, or
 *    (at your option) any later version.
 *  
 *    This program is distributed in the hope that it will be useful,  
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the   
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with this program; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 */

#include <errno.h>
#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <strings.h>
#include <unistd.h>

#include <sys/types.h>
#include <sys/fcntl.h>
#include <sys/stat.h>

#include <sys/scsi/scsi.h>
#include <sys/scsi/generic/status.h>
#include <sys/scsi/impl/uscsi.h>

#include "fwdl.h"
#include "fwdl-sun.h"


// Preprocessor Defines

#ifndef STATUS_ACA_ACTIVE
#define STATUS_ACA_ACTIVE	0x30
#endif


// Type Definitions

// 16-byte SCSI command descriptor block

typedef struct scsi_cdb_s {
  unsigned int	 w1;
  unsigned int	 w2;
  unsigned int	 w3;
  unsigned int	 w4;
} scsi_cdb_t;


// Global Variables

static struct uscsi_cmd	 req;		// Sun user SCSI command structure
static scsi_cdb_t	 cdb;		// SCSI command block buffer


// Function Prototypes

static int	 openDevice(const char *device, int flags);
static int	 closeDevice(int fd);
static int	 scsiTestUnitReady(int fd);
static int	 scsiInquiry(int fd, void *buffer);
static int	 scsiWriteFW(int fd, void *buffer, unsigned int size);
#ifdef DEBUG
static void	 dumpReqStruct(void);
#endif
static int	 checkStatus(void *data);

scsi_ops_t sunOps = {
	openDevice,
	closeDevice,
	scsiTestUnitReady,
	scsiInquiry,
	scsiWriteFW,
};


/******************************************************************************
*******************************************************************************
**
** static int openDevice()
**
** Description:
**   This routine opens a SunOS disk device.
**
** Input(s):
**  *device - Name of the target device to open.
**   flags  - Flags passed to the open system call.
**
** Returns:
**   A non-negative integer representing the lowest numbered unused file
**   descriptor. Otherwise, -1 on error.
**
*******************************************************************************
******************************************************************************/

static int
openDevice(const char *device, int flags)
{
  struct stat statbuf;
  int fd = -1;

  // Check to see if the device actually exists first
  
  if (stat(device, &statbuf) < 0)
    {
      fprintf(stderr, "Could not open \"%s\": %s\n", device, strerror(errno));
      exit(EXIT_FAILURE);
    }
  else if (!S_ISCHR(statbuf.st_mode) && !S_ISBLK(statbuf.st_mode))
    {
      fprintf(stderr, "The device file \"%s\" is not a character or block "
	      "device.\n", device);
      exit(EXIT_FAILURE);
    }

  // We need to have exclusive, non-blocking access to the device in order to
  // download the firmware.

  if ((fd = open(device, O_RDWR | O_EXCL | O_NDELAY)) < 0)
    {
      fprintf(stderr,
	      "Could not open \"%s\" for read/write/exclusive access.\n", 
	      device);
      perror("");
      exit(EXIT_FAILURE);
    }

  return (fd);
}


/******************************************************************************
*******************************************************************************
**
** static int closeDevice()
**
** Description:
**   This routine closes the device associated with the file descriptor 'fd'.
**
** Input(s):
**   fd - A file descriptor for the device to be closed.
**
** Returns:
**   0 if OK, -1 on error.
**
*******************************************************************************
******************************************************************************/

static int
closeDevice(int fd)
{
  return (close(fd));
}


/******************************************************************************
*******************************************************************************
**
** static int scsiTestUnitReady()
**
** Description:
**   This routine closes the device associated with the file descriptor.
**
** Input(s):
**   fd - File descriptor of the disk target.
**
** Returns:
**   SCSI command status.
**
*******************************************************************************
******************************************************************************/

static int
scsiTestUnitReady(int fd)
{
  int status;

  // Clear command block and data request structure
  bzero(&req, sizeof(struct uscsi_cmd));

  // Clear global sense data buffer
  bzero(&global_sdata, sizeof(senseData_t));

  cdb.w1 = 0x00000000;
  cdb.w2 = 0x00000000;
  cdb.w3 = 0x00000000;
  cdb.w4 = 0x00000000;
  req.uscsi_cdb = (caddr_t)&cdb;
  req.uscsi_cdblen = 6;
  req.uscsi_flags = (USCSI_RQENABLE);
  req.uscsi_timeout = (short)SCSI_CMD_TIMEOUT;
  req.uscsi_rqbuf = (caddr_t)&global_sdata;
  req.uscsi_rqlen = sizeof(senseData_t);

  status = ioctl(fd, USCSICMD, &req);

#ifdef DEBUG
  if (debug) {
    dumpReqStruct();
  }
#endif

  checkStatus(&req);

  return(status);
}


/******************************************************************************
*******************************************************************************
**
** static int scsiInquiry()
**
** Description:
**   This routine...
**
** Input(s):
**
**
** Output(s):
**
**
** Returns:
**
**
*******************************************************************************
******************************************************************************/

static int
scsiInquiry(int fd, void *buffer)
{
  int   status;

  // Clear command block and data request structure area
  bzero(&req, sizeof(struct uscsi_cmd));

  // Clear global sense data buffer
  bzero(&global_sdata, sizeof(senseData_t));
  
  cdb.w1 = 0x12000000;
  cdb.w2 = 0x30000000; /* 0xff000000 ? */
  cdb.w3 = 0x00000000;
  cdb.w4 = 0x00000000;
  req.uscsi_cdb = (caddr_t)&cdb;
  req.uscsi_cdblen = 6;
  req.uscsi_flags = (USCSI_RQENABLE | USCSI_READ);
  req.uscsi_timeout = (short)SCSI_CMD_TIMEOUT;
  req.uscsi_bufaddr = (caddr_t)buffer;
  req.uscsi_buflen = sizeof(inquiryData_t);
  req.uscsi_rqbuf = (caddr_t)&global_sdata;
  req.uscsi_rqlen = sizeof(senseData_t);

  status = ioctl(fd, USCSICMD, &req);

#ifdef DEBUG
  if (debug) {
    dumpReqStruct();
  }
#endif

  checkStatus(&req);

  return(status);
}


/******************************************************************************
*******************************************************************************
**
** static int scsiWriteFW()
**
** Description:
**   This routine...
**
** Input(s):
**   fd     - 
**  *buffer - 
**   size   - 
**
** Returns:
**   0 if OK, -1 on error.
**
*******************************************************************************
******************************************************************************/

static int
scsiWriteFW(int fd, void *buffer, unsigned int size)
{
  int   status;

  // Clear command block and data request structure area
  bzero(&req, sizeof(struct uscsi_cmd));

  // Clear global sense data buffer
  bzero(&global_sdata, sizeof(senseData_t));

  cdb.w1 = 0x3B050000;
  cdb.w2 = (size & 0x00FFFF00) >>  8;
  cdb.w3 = (size & 0x000000FF) << 24;
  cdb.w4 = 0x00000000;
  req.uscsi_cdb = (caddr_t)&cdb;
  req.uscsi_cdblen = 12;
  req.uscsi_flags = (USCSI_RQENABLE | USCSI_WRITE);
  req.uscsi_timeout = (short)SCSI_CMD_TIMEOUT;
  req.uscsi_bufaddr = (caddr_t)buffer;
  req.uscsi_buflen = size;
  req.uscsi_rqbuf = (caddr_t)&global_sdata;
  req.uscsi_rqlen = sizeof(senseData_t);

  status = ioctl(fd, USCSICMD, &req);
      
#ifdef DEBUG
  if (debug) {
    dumpReqStruct();
  }
#endif
  
  if (checkStatus(&req) == STATUS_CHECK)
    {
      if (global_sdata.asc == 0x26)
	{
	  if (global_sdata.ascq == 0x98)
	    {
	      fprintf(stderr, "Firmware checksum failed!\n");
	    }
	  else if (global_sdata.ascq == 0x99)
	    {
	      fprintf(stderr, "The firmware is incompatible with this product!\n");
	    }
	}
    }
  
  return(status);
}

#ifdef DEBUG
/******************************************************************************
*******************************************************************************
**
** static void dumpReqStruct()
**
** Description:
**   This prints out the contents of the global user SCSI command request
**   structure.
**
*******************************************************************************
******************************************************************************/

static void
dumpReqStruct(void)
{
  int		 size;
  scsi_cdb_t	*cdb;


  fprintf(stderr, "\nSCSI initiator request structure:\n");
  fprintf(stderr, "  Request structure address:  0x%08x\n", &req);
  fprintf(stderr, 
	  "--------------------------------------------------------\n");
  fprintf(stderr, "Timeout (sec):        %01.1f\t", 
	  (float)(req.uscsi_timeout) / 100.0);
  fprintf(stderr, "Transfer Type:     %s\n", 
	  (req.uscsi_flags & USCSI_READ ? "READ" : 
	   ((req.uscsi_buflen) ? "WRITE" : "NONE")));
  fprintf(stderr, "Flags:          0x%08x\n", req.uscsi_flags);
  fprintf(stderr, "Data Xfer Length:  0x%03x\t", req.uscsi_buflen);
  fprintf(stderr, "Sense Buffer Size: 0x%03x\n",req.uscsi_rqlen);
  fprintf(stderr, "SCSI Command Block:\t");
  cdb = (scsi_cdb_t *)req.uscsi_cdb;
  fprintf(stderr, "0x%08x\n", cdb->w1);
  fprintf(stderr, "\t\t\t0x%08x\n", cdb->w2);
  fprintf(stderr, "\t\t\t0x%08x\n", cdb->w3);
  fprintf(stderr, "\t\t\t0x%08x\n", cdb->w4);
  if (req.uscsi_flags & (USCSI_READ | USCSI_WRITE))
    {
      fprintf(stderr, 
	      "--------------------------------------------------------\n");
      size = req.uscsi_buflen - req.uscsi_resid;
      fprintf(stderr, "Data Buffer: (%d bytes)\n\t", size);
      for (int i = 0; i < size; i++)
	{
	  fprintf(stderr, "%02x ", (unsigned char)req.uscsi_bufaddr[i]);
	  if ( (i + 1) % 16 == 0)
	    fprintf(stderr, "\n\t");
	}
      fprintf(stderr,"\n");
    }
  fprintf(stderr, 
	  "--------------------------------------------------------\n");
  fprintf(stderr, "SCSI Status:  0x%02x\n", req.uscsi_status);
  fprintf(stderr, 
	  "--------------------------------------------------------\n");
  if (req.uscsi_status != STATUS_GOOD)
    {
      size = req.uscsi_rqlen - req.uscsi_rqresid;
      fprintf(stderr, "Sense data: (%d bytes)\n\t", size);
      for (int i = 0; i < sizeof(senseData_t); i++)
	{
	  fprintf(stderr, "%02x ", (unsigned char)req.uscsi_rqbuf[i]);
	  if ( (i + 1) % 16 == 0)
	    fprintf(stderr, "\n\t");
	}
      fprintf(stderr, 
	      "\n--------------------------------------------------------\n");
    }
  fflush(stderr);    
}
#endif // DEBUG


/******************************************************************************
*******************************************************************************
**
** static int checkStatus()
**
** Description:
**   This routine...
**
** Input(s):
**  *data - Pointer to a SunOS user-SCSI command request structure.
**
** Returns:
**   SCSI status condition.
**
*******************************************************************************
******************************************************************************/

static int
checkStatus(void *data)
{
  int			 size;
  struct uscsi_cmd	*req = (struct uscsi_cmd *)data;

  if (req->uscsi_status == STATUS_GOOD)
    return (STATUS_GOOD);
  else
    {
      size = req->uscsi_rqlen - req->uscsi_rqresid;
      fprintf(stderr, "Sense data: (%d bytes)\n\t", size);
      for (int i = 0; i < sizeof(senseData_t); i++)
	{
	  fprintf(stderr, "%02x ", (unsigned char)req->uscsi_rqbuf[i]);
	  if ( (i + 1) % 16 == 0)
	    fprintf(stderr, "\n\t");
	}
      fprintf(stderr, 
	      "\n--------------------------------------------------------\n");

      switch(req->uscsi_status)
	{
	case STATUS_RESERVATION_CONFLICT:
	  fprintf(stderr, "Reservation Conflict\n");
	  break;
	case STATUS_CHECK:
	  fprintf(stderr, "Check Condition\n");
	  break;
	case STATUS_MET:
	case STATUS_BUSY:
	case STATUS_INTERMEDIATE:
	case STATUS_INTERMEDIATE_MET:
	case STATUS_TERMINATED:
	case STATUS_QFULL:
	case STATUS_ACA_ACTIVE:
	  fprintf(stderr, "Bad Condition: 0x%x\n", req->uscsi_status);
	  break;
	}
      return(req->uscsi_status);
    }
}
