//**************************************************************************************************
//                                          FrmMain.cpp                                            *
//                                         -------------                                           *
// Started     : 2003-08-18                                                                        *
// Last Update : 2024-03-21                                                                        *
// Copyright   : (C) 2003-2023 MSWaters                                                            *
//**************************************************************************************************

//**************************************************************************************************
//                                                                                                 *
//      This program is free software; you can redistribute it and/or modify it under the          *
//      terms of the GNU General Public License as published by the Free Software Foundation;      *
//      either version 3 of the License, or (at your option) any later version.                    *
//                                                                                                 *
//**************************************************************************************************

#include "FrmMain.hpp"

// The application icons
#include "icons/gspiceui-32x32.xpm"
#include "icons/file-open.xpm"
#include "icons/file-import.xpm"
#include "icons/file-reload.xpm"
#include "icons/file-close.xpm"
#include "icons/sim-create.xpm"
#include "icons/sim-run.xpm"
#include "icons/sim-stop.xpm"
#include "icons/sim-edit.xpm"
#include "icons/sim-plot.xpm"
#include "icons/calculator.xpm"
#include "icons/preferences.xpm"
#include "icons/help.xpm"

//**************************************************************************************************
// Implement an event table in which the events are routed to their respective handler functions in
// the class. If -1 is given as the ID, the given handler will be invoked for any event of the
// specified type.

wxBEGIN_EVENT_TABLE( FrmMain, wxFrame )

  EVT_MENU( FrmMain::ID_MNU_OPEN    , FrmMain::OnOpen       )
  EVT_MENU( FrmMain::ID_MNU_IMPORT  , FrmMain::OnImport     )
  EVT_MENU( FrmMain::ID_MNU_RELOAD  , FrmMain::OnReload     )
  EVT_MENU( FrmMain::ID_MNU_CLOSE   , FrmMain::OnClose      )
  EVT_MENU( FrmMain::ID_MNU_QUIT    , FrmMain::OnQuit       )

  EVT_MENU( FrmMain::ID_MNU_CREATE  , FrmMain::OnSimCreate  )
  EVT_MENU( FrmMain::ID_MNU_RUN     , FrmMain::OnSimRun     )
  EVT_MENU( FrmMain::ID_MNU_STOP    , FrmMain::OnSimStop    )
  EVT_MENU( FrmMain::ID_MNU_SCHEM   , FrmMain::OnSchematic  )
  EVT_MENU( FrmMain::ID_MNU_VIEWDATA, FrmMain::OnViewData   )
  EVT_MENU( FrmMain::ID_MNU_CALC    , FrmMain::OnCalculator )

  EVT_MENU( FrmMain::ID_MNU_NGSPICE , FrmMain::OnSelSimEng  )
  EVT_MENU( FrmMain::ID_MNU_GNUCAP  , FrmMain::OnSelSimEng  )
  EVT_MENU( FrmMain::ID_MNU_LEPTON  , FrmMain::OnSelEDA_TS  )
  EVT_MENU( FrmMain::ID_MNU_GEDAGAF , FrmMain::OnSelEDA_TS  )
  EVT_MENU( FrmMain::ID_MNU_GAW     , FrmMain::OnSelDataVwr )
  EVT_MENU( FrmMain::ID_MNU_GWAVE   , FrmMain::OnSelDataVwr )
  EVT_MENU( FrmMain::ID_MNU_KST     , FrmMain::OnSelDataVwr )
  EVT_MENU( FrmMain::ID_MNU_PREFS   , FrmMain::OnPrefs      )

  EVT_MENU( FrmMain::ID_MNU_MAN_USER, FrmMain::OnManUser    )
  EVT_MENU( FrmMain::ID_MNU_MAN_NGSP, FrmMain::OnManNgSpice )
  EVT_MENU( FrmMain::ID_MNU_MAN_GCAP, FrmMain::OnManGnuCap  )
  EVT_MENU( FrmMain::ID_MNU_ABOUT   , FrmMain::OnAbout      )

  EVT_TOOL( FrmMain::ID_TBR_OPEN    , FrmMain::OnOpen       )
  EVT_TOOL( FrmMain::ID_TBR_IMPORT  , FrmMain::OnImport     )
  EVT_TOOL( FrmMain::ID_TBR_RELOAD  , FrmMain::OnReload     )
  EVT_TOOL( FrmMain::ID_TBR_CLOSE   , FrmMain::OnClose      )
  EVT_TOOL( FrmMain::ID_TBR_CREATE  , FrmMain::OnSimCreate  )
  EVT_TOOL( FrmMain::ID_TBR_RUN     , FrmMain::OnSimRun     )
  EVT_TOOL( FrmMain::ID_TBR_STOP    , FrmMain::OnSimStop    )
  EVT_TOOL( FrmMain::ID_TBR_SCHEM   , FrmMain::OnSchematic  )
  EVT_TOOL( FrmMain::ID_TBR_VIEWER  , FrmMain::OnViewData   )
  EVT_TOOL( FrmMain::ID_TBR_CALC    , FrmMain::OnCalculator )
  EVT_TOOL( FrmMain::ID_TBR_PREFS   , FrmMain::OnPrefs      )
  EVT_TOOL( FrmMain::ID_TBR_HELP    , FrmMain::OnManUser    )

  EVT_TOOL_ENTER( -1                , FrmMain::OnToolEnter  )

  EVT_CLOSE(                          FrmMain::OnSysExit    )

wxEND_EVENT_TABLE( )

//**************************************************************************************************
// Constructor.
//
// Argument List :
//   poApp - A pointer to the class that created this object

FrmMain::FrmMain( const wxApp * poApp ) :
                  wxFrame( (wxFrame *) NULL, -1, "", wxDefaultPosition, wxDefaultSize,
                           wxDEFAULT_FRAME_STYLE & ~wxMAXIMIZE_BOX ),
                  m_oFileTasks( this ), m_oSimnNgSpice( ), m_oSimnGnuCap( ), m_oPrcSchem( ),
                  m_oPrcDataVwr( )
{
  UNUSED( poApp );  // Suppresses a compiler warning about unused function parameters

  // Indicate that the main frame is open
  m_bIsOpen = true;

  // Initialize pointers
  m_poSimn      = NULL;
  m_poPrcSimEng = NULL;
  m_poNbkSimEng = NULL;

  // Create everything
  Initialize( );
}

//**************************************************************************************************
// Destructor.

FrmMain::~FrmMain( )
{
  if( m_poPrcSimEng != NULL ) { delete m_poPrcSimEng; m_poPrcSimEng = NULL; }
  if( m_poNbkSimEng != NULL ) { delete m_poNbkSimEng; m_poNbkSimEng = NULL; }
}

//**************************************************************************************************
// Initialize the main frame.

void  FrmMain::Initialize( void )
{
  wxString  os1;

  // Set the frame icon, font and title
#ifndef __WXMSW__
  SetIcon( wxICON( gspiceui32x32 ) );
#else
  SetIcon( wxICON( gspiceui_ico ) );
#endif
  SetFont( FONT_NORM );
  m_oFileTasks.bSetTitle( );

  // Automatically update the frame layout when it is resized
  SetAutoLayout( true );

  // Set the main frame layout
  m_uiFrmLayout = g_oConfig.uiGetFrmLayout( );

  // Call the initialization functions
  InitMenuBar   ( );
  InitToolBar   ( );
  InitLstBoxs   ( );
  InitNbkTCtls  ( );
  InitStatBar   ( );
  InitToolTips  ( );
  InitSimEng    ( );
  DoLayout      ( );
  InitPosnSize  ( );
  InitCalculator( );
  InitPdfViewer ( );

  // Display desired analysis page
  m_poNbkSimEng->bSetPage( g_oConfig.eGetAnalysis( ) );

  // Set status bar text
  os1 << APP_NAME << " version " << APP_VERSION << " (" << APP_DATE << ")";
  SetStatusText( " Welcome to " + os1, ePANE_MESAGE );

  // Initialize the FileTasks object (eg. load schematic or netlist file/s) and set log file name
  m_oFileTasks.Initialize( );
  InitLogFiles( ); // The log files can't be configured until m_oFileTsks.Initialize( ) is called

  // If necessary load the simulation or netlist
  if( ! m_oNetLst.bIsEmpty( ) ) bSimnLoad( );

  // Display the appropriate text control page
  if(      ! m_oNbkTxtCtls.poGetPage( NbkTxtCtls::ePAGE_SIMULTN )->bIsEmpty( ) )
    m_oNbkTxtCtls.bSetPage( NbkTxtCtls::ePAGE_SIMULTN );
  else if( ! m_oNbkTxtCtls.poGetPage( NbkTxtCtls::ePAGE_NETLIST )->bIsEmpty( ) )
    m_oNbkTxtCtls.bSetPage( NbkTxtCtls::ePAGE_NETLIST );
  else
    m_oNbkTxtCtls.bSetPage( NbkTxtCtls::ePAGE_CONSOLE );
}

//**************************************************************************************************
// Initialize the menu bar.

void  FrmMain::InitMenuBar( void )
{
  SysScan  oSysScan;

  // Create the menu bar and menus
  wxMenuBar * poMenuBar  = new wxMenuBar;
  wxMenu    * poMenuFile = new wxMenu;
  wxMenu    * poMenuSimu = new wxMenu;
  wxMenu    * poMenuSets = new wxMenu;
  wxMenu    * poMenuHelp = new wxMenu;

  // Set the tool bar font
  poMenuBar->SetFont( FONT_NORM );

  // Load the menus with items
  poMenuFile->Append(          ID_MNU_OPEN    ,  " Open Netlist ... "       );
  poMenuFile->Append(          ID_MNU_IMPORT  ,  " Import Schematic/s ... " );
  poMenuFile->Append(          ID_MNU_RELOAD  ,  " Reload "                 );
  poMenuFile->Append(          ID_MNU_CLOSE   ,  " Close "                  );
  poMenuFile->AppendSeparator( );
  poMenuFile->Append(          ID_MNU_QUIT    ,  " Quit "                   );

  poMenuSimu->Append(          ID_MNU_CREATE  ,  " Create Simulation "      );
  poMenuSimu->Append(          ID_MNU_RUN     ,  " Run Simulation "         );
  poMenuSimu->Append(          ID_MNU_STOP    ,  " Stop Simulation "        );
  poMenuSimu->AppendSeparator( );
  poMenuSimu->Append(          ID_MNU_SCHEM   , " Schematic Capture ... "   );
  poMenuSimu->Append(          ID_MNU_VIEWDATA, " View Results ... "        );
  poMenuSimu->Append(          ID_MNU_CALC    , " Calculator ... "          );

  poMenuSets->AppendRadioItem( ID_MNU_NGSPICE , " NG-SPICE "                );
  poMenuSets->AppendRadioItem( ID_MNU_GNUCAP  , " GNU-CAP "                 );
  poMenuSets->AppendSeparator( );
  poMenuSets->AppendRadioItem( ID_MNU_LEPTON  , " Lepton EDA "              );
  poMenuSets->AppendRadioItem( ID_MNU_GEDAGAF , " gEDA/gaf "                );
  poMenuSets->AppendSeparator( );
  poMenuSets->AppendRadioItem( ID_MNU_GAW     , " Gaw "                     );
  poMenuSets->AppendRadioItem( ID_MNU_GWAVE   , " GWave "                   );
  poMenuSets->AppendRadioItem( ID_MNU_KST     , " Kst "                     );
  poMenuSets->AppendSeparator( );
  poMenuSets->Append(          ID_MNU_PREFS   , " Preferences ... "         );

  poMenuHelp->Append(          ID_MNU_MAN_USER, " gSpiceUI Manual ... "     );
  poMenuHelp->Append(          ID_MNU_MAN_NGSP, " NG-SPICE Manual ... "     );
  poMenuHelp->Append(          ID_MNU_MAN_GCAP, " GNU-CAP Manual ... "      );
  poMenuHelp->AppendSeparator( );
  poMenuHelp->Append(          ID_MNU_ABOUT   , " About ... "               );

  // Load the menubar with menus
  poMenuBar->Append( poMenuFile, " File "     );
  poMenuBar->Append( poMenuSimu, " Simulate " );
  poMenuBar->Append( poMenuSets, " Settings " );
  poMenuBar->Append( poMenuHelp, " Help "     );

  // Disable (grey out) menu items associated with binaries which aren't currently installed
  if( ! oSysScan.bIsInstalled( BIN_NGSPICE  ) ) poMenuSets->Enable( ID_MNU_NGSPICE, false );
  if( ! oSysScan.bIsInstalled( BIN_GNUCAP   ) ) poMenuSets->Enable( ID_MNU_GNUCAP , false );
  if( ! oSysScan.bIsInstalled( BIN_LNETLIST ) ) poMenuSets->Enable( ID_MNU_LEPTON , false );
  if( ! oSysScan.bIsInstalled( BIN_GNETLIST ) ) poMenuSets->Enable( ID_MNU_GEDAGAF, false );
  if( ! oSysScan.bIsInstalled( BIN_GAW      ) ) poMenuSets->Enable( ID_MNU_GAW    , false );
  if( ! oSysScan.bIsInstalled( BIN_GWAVE    ) ) poMenuSets->Enable( ID_MNU_GWAVE  , false );
  if( ! oSysScan.bIsInstalled( BIN_KST      ) ) poMenuSets->Enable( ID_MNU_KST    , false );

  // Check the appropriate simulation engine menu item
  if( g_oConfig.eGetSimEngine( )    == eSIMR_NGSPICE ) poMenuSets->Check( ID_MNU_NGSPICE, true );
  else                                                 poMenuSets->Check( ID_MNU_GNUCAP , true );

  // Check the appropriate EDA tool suite menu item
  if( g_oConfig.eGetEdaToolSuite( ) == eEDA_LEPTON   ) poMenuSets->Check( ID_MNU_LEPTON , true );
  else                                                 poMenuSets->Check( ID_MNU_GEDAGAF, true );

  // Check the appropriate data viewer menu item
  switch( g_oConfig.eGetDataViewer( ) )
  {
    case eDVWR_GAW   : poMenuSets->Check( ID_MNU_GAW  , true ); break;
    case eDVWR_GWAVE : poMenuSets->Check( ID_MNU_GWAVE, true ); break;
    case eDVWR_KST   : poMenuSets->Check( ID_MNU_KST  , true ); break;
    default          : break;
  }

  // Attach the menu bar to the frame
  SetMenuBar( poMenuBar );
}

//**************************************************************************************************
// Initialize the tool bar.

void  FrmMain::InitToolBar( void )
{
  wxToolBar * poToolBar;
  wxBitmap  * poPixMap[ 12 ];

  // Create the tool bar
  poToolBar = CreateToolBar( wxHORIZONTAL | wxTB_FLAT );
  poToolBar->SetToolSeparation( 10 );

  // Create the bitmaps for the tools
  poPixMap[  0 ] = new wxBitmap( file_open_xpm   );
  poPixMap[  1 ] = new wxBitmap( file_import_xpm );
  poPixMap[  2 ] = new wxBitmap( file_reload_xpm );
  poPixMap[  3 ] = new wxBitmap( file_close_xpm  );
  poPixMap[  4 ] = new wxBitmap( sim_create_xpm  );
  poPixMap[  5 ] = new wxBitmap( sim_run_xpm     );
  poPixMap[  6 ] = new wxBitmap( sim_stop_xpm    );
  poPixMap[  7 ] = new wxBitmap( sim_edit_xpm    );
  poPixMap[  8 ] = new wxBitmap( sim_plot_xpm    );
  poPixMap[  9 ] = new wxBitmap( calculator_xpm  );
  poPixMap[ 10 ] = new wxBitmap( preferences_xpm );
  poPixMap[ 11 ] = new wxBitmap( help_xpm        );

  // Add the tools to the toolbar
  poToolBar->AddTool( ID_TBR_OPEN  , "", *(poPixMap[  0 ]), " Open netlist file "                 );
  poToolBar->AddTool( ID_TBR_IMPORT, "", *(poPixMap[  1 ]), " Import schematic file/s "           );
  poToolBar->AddTool( ID_TBR_RELOAD, "", *(poPixMap[  2 ]), " Reload netlist / schematic file/s " );
  poToolBar->AddTool( ID_TBR_CLOSE , "", *(poPixMap[  3 ]), " Close netlist / schematic file/s "  );

  poToolBar->AddSeparator( );

  poToolBar->AddTool( ID_TBR_CREATE, "", *(poPixMap[  4 ]), " Create simulation file " );
  poToolBar->AddTool( ID_TBR_RUN   , "", *(poPixMap[  5 ]), " Run simulation "         );
  poToolBar->AddTool( ID_TBR_STOP  , "", *(poPixMap[  6 ]), " Stop simulation "        );

  poToolBar->AddSeparator( );

  poToolBar->AddTool( ID_TBR_SCHEM , "", *(poPixMap[  7 ]), " Schematic Capture "       );
  poToolBar->AddTool( ID_TBR_VIEWER, "", *(poPixMap[  8 ]), " View simulation results " );
  poToolBar->AddTool( ID_TBR_CALC  , "", *(poPixMap[  9 ]), " Calculator "              );

  poToolBar->AddSeparator( );

  poToolBar->AddTool( ID_TBR_PREFS , "", *(poPixMap[ 10 ]), " Preferences " );

  poToolBar->AddSeparator( );

  poToolBar->AddTool( ID_TBR_HELP  , "", *(poPixMap[ 11 ]), " View gSpiceUI user manual " );

  // Realize the toolbar
  poToolBar->Realize( );

  // Delete the bitmaps
  for( int i1=0; i1<10; i1++ ) delete poPixMap[ i1 ];
}

//**************************************************************************************************
// Initialize the test node and component list boxes.

void  FrmMain::InitLstBoxs( void )
{
  long  lStyle;

  // Create the node list label and component list label
  lStyle = wxALIGN_CENTER;
  m_oLblNodes.Create( this, ID_UNUSED, "Nodes",      wxDefaultPosition, wxDefaultSize, lStyle );
  m_oLblCpnts.Create( this, ID_UNUSED, "Components", wxDefaultPosition, wxDefaultSize, lStyle );

  // Create the node and component list boxes
  lStyle = wxLB_EXTENDED | wxLB_NEEDED_SB;
  m_oLbxNodes.Create( this, ID_LBX_NODES, wxDefaultPosition, wxSize( 100, -1 ), 0, NULL, lStyle );
  m_oLbxCpnts.Create( this, ID_LBX_CPNTS, wxDefaultPosition, wxSize( 100, -1 ), 0, NULL, lStyle );
}

//**************************************************************************************************
// Initialize the console notebook.

void  FrmMain::InitNbkTCtls( void )
{
  uint  ui1;

  // Create the text control notebook
  m_oNbkTxtCtls.bCreate( this, ID_NBK_TXTCTRLS );

  // Set the text control maximum lines
  ui1 = g_oConfig.uiGetNbkMaxLns( );
  m_oNbkTxtCtls.bSetLinesMax( ui1 );

  // Set the precision of the results data
  ui1 = g_oConfig.uiGetPrecision( );
  CnvtType::bSetFltRes( ui1 );
}

//**************************************************************************************************
// Initialize the status bar.
//
// Note : The first field in the status bar has benn effectively disable by setting it's width to
//        near zero. It is not used since the frame insists on writing text into this field as the
//        user moves the mouse over display controls.

void  FrmMain::InitStatBar( void )
{
  wxStatusBar * poStatusBar = new wxStatusBar;
  long          liStyle;
  int           iaPaneWidths[ ePANE_LAST+1 ];
  int           iaPaneStyles[ ePANE_LAST+1 ];

  // Set the style bits
  liStyle = wxFULL_REPAINT_ON_RESIZE;

  // Create the status bar
  poStatusBar->Create( this, wxID_ANY, liStyle );
  poStatusBar->SetFieldsCount( ePANE_LAST+1 );
  poStatusBar->SetMinHeight( 25 );

  // Set the status bar pane widths
  iaPaneWidths[ ePANE_MESAGE ] = -1;  // Variable width
  iaPaneWidths[ ePANE_SIMENG ] = 75;  // Fixed width
  iaPaneWidths[ ePANE_EDATLS ] = 90;  // Fixed width
  iaPaneWidths[ ePANE_DATVWR ] = 60;  // Fixed width
  poStatusBar->SetStatusWidths( ePANE_LAST+1, iaPaneWidths );

  // Set the status bar pane styles
  iaPaneStyles[ ePANE_MESAGE ] = wxSB_SUNKEN;
  iaPaneStyles[ ePANE_SIMENG ] = wxSB_SUNKEN;
  iaPaneStyles[ ePANE_EDATLS ] = wxSB_SUNKEN;
  iaPaneStyles[ ePANE_DATVWR ] = wxSB_SUNKEN;
  poStatusBar->SetStatusStyles( ePANE_LAST+1, iaPaneStyles );

  // Set status bar text
  poStatusBar->SetStatusText( " Welcome message"                   , ePANE_MESAGE );
  poStatusBar->SetStatusText( " " + g_oConfig.rosGetSimEngine( )   , ePANE_SIMENG );
  poStatusBar->SetStatusText( " " + g_oConfig.rosGetEdaToolSuite( ), ePANE_EDATLS );
  poStatusBar->SetStatusText( " " + rosEnumVwrToStr( g_oConfig.eGetDataViewer( ) ), ePANE_DATVWR );

  // Set the main frame status bar
  SetStatusBar( poStatusBar );

  // Disable the status bar pane used to display menu and toolbar help
  SetStatusBarPane( -1 );
}

//**************************************************************************************************
// Initialize the tool tips.

void  FrmMain::InitToolTips( void )
{
  wxString  os1;

  // Define tool tips for the probe list controls
  os1 = " List of all test nodes ";
  m_oLblNodes.SetToolTip( os1 );
  m_oLbxNodes.SetToolTip( os1 );
  os1 = " List of all test components ";
  m_oLblCpnts.SetToolTip( os1 );
  m_oLbxCpnts.SetToolTip( os1 );

  // Define a tool tip for the status bar
  os1.Empty( );
  os1 << " Status bar pane :\n"
      << "    1 - status messages \n"
      << "    2 - simulation engine \n"
      << "    3 - EDA tool suite \n"
      << "    4 - waveform data viewer ";
  GetStatusBar( )->SetToolTip( os1 );

  // Set global tool tip attributes
  wxToolTip::Enable( g_oConfig.bGetToolTips( ) );
  wxToolTip::SetDelay( g_oConfig.uiGetToolTipDly( ) );
}

//**************************************************************************************************
// Set the electronic circuit simulator engine to use.

void  FrmMain::InitSimEng( void )
{
  // Delete any existing simulation engine notebook object
  if( m_poNbkSimEng != NULL )
  {
    GetSizer( )->Detach( m_poNbkSimEng );
    delete m_poNbkSimEng;
    m_poNbkSimEng = NULL;
  }

  // Delete any existing simulation engine process object
  if( m_poPrcSimEng != NULL )
  {
    delete m_poPrcSimEng;
    m_poPrcSimEng = NULL;
  }

  // Create the new simulation engine objects and update the status text
  if( GetMenuBar( )->IsChecked( ID_MNU_NGSPICE ) )
  {
    m_poNbkSimEng = new NbkNgSpice( this, ID_NBK_ANALYSIS );
    m_poPrcSimEng = new PrcNgSpice( );
    m_poSimn      = &m_oSimnNgSpice;

    SetStatusText( " " + rosEnumEngToStr( eSIMR_NGSPICE ), ePANE_SIMENG );
  }
  else
  {
    m_poNbkSimEng = new NbkGnuCap( this, ID_NBK_ANALYSIS );
    m_poPrcSimEng = new PrcGnuCap( );
    m_poSimn      = &m_oSimnGnuCap;

    SetStatusText( " " + rosEnumEngToStr( eSIMR_GNUCAP ), ePANE_SIMENG );
  }
}

//**************************************************************************************************
// Initialize the log file names in the process objects.

void  FrmMain::InitLogFiles( void )
{
  wxFileName  ofn1;
  wxString    os1;

  // Check that a netlist file has been defined
  ofn1 = m_oFileTasks.rosGetNetLstFile( );
  if( ! ofn1.IsOk( )       ) return;
  if( ! ofn1.FileExists( ) ) return;

  // Get the path to the schematic or netlist file
#ifndef __WXMSW__
  os1 = ofn1.GetPath( ) + "/gspiceui.log";
#else
  os1 = ofn1.GetPath( ) + "\\gspiceui.log";
#endif

  // Set the log file path for the gNetList process object
  m_oFileTasks.bSetLogFile( os1 );

  // Set the log file path for the simulation process object
  if( m_poPrcSimEng != NULL ) m_poPrcSimEng->bSetLogFile( os1 );
}

//**************************************************************************************************
// Initialize the frames position amd size.

void  FrmMain::InitPosnSize( void )
{
  int  iPosnX, iPosnY, iSizeW, iSizeH;

  // Get the position and size from the configuration object
  iPosnX = g_oConfig.iGetMainPosnX( );
  iPosnY = g_oConfig.iGetMainPosnY( );
  iSizeW = g_oConfig.iGetMainSizeW( );
  iSizeH = g_oConfig.iGetMainSizeH( );

  // Set the position and size
  if( iPosnX>=0 && iPosnY>=0 ) Move( iPosnX, iPosnY );
  if( iSizeW> 0 && iSizeH> 0 ) SetClientSize( iSizeW, iSizeH );
}

//**************************************************************************************************
// Initialize the calculator application class.

void  FrmMain::InitCalculator( void )
{
  // Set the calculator application binary according to the configuration file setting
  m_oPrcCalculator.bSetBinFile( g_oConfig.rosGetCalculator( ) );

  // Load the calculator binary list with applications which are currently installed on the system
  PrcCalc::InitCalcBinLst( );

  // If the calculator binary isn't installed pick the first one on the list of available binaries
  if( ! m_oPrcCalculator.bBinExists( ) )
  {
    if( ! PrcCalc::m_osaCalcBinLst.IsEmpty( ) )
    {
      m_oPrcCalculator.bSetBinFile( PrcCalc::m_osaCalcBinLst.Item( 0 ) );
      g_oConfig.bSetPdfViewer( PrcCalc::m_osaCalcBinLst.Item( 0 ) );
      g_oConfig.bFlush( );
    }
  }
}

//**************************************************************************************************
// Initialize the PDF documentation viewer application class.

void  FrmMain::InitPdfViewer( void )
{
  // Set the PDF viewer application binary according to the configuration file setting
  m_oPrcManUser   .bSetBinFile( g_oConfig.rosGetPdfViewer( ) );
  m_oPrcManNgSpice.bSetBinFile( g_oConfig.rosGetPdfViewer( ) );
  m_oPrcManGnuCap .bSetBinFile( g_oConfig.rosGetPdfViewer( ) );

  // Load the PDF viewer binary list with applications which are currently installed on the system
  PrcPdfVwr::InitPdfVwrBinLst( );

  // If the PDF viewer binary isn't installed pick the first one on the list of available binaries
  if( ! m_oPrcManNgSpice.bBinExists( ) )
  {
    m_oPrcManUser   .bSetBinFile( PrcPdfVwr::m_osaPdfVwrBinLst.Item( 0 ) );
    m_oPrcManNgSpice.bSetBinFile( PrcPdfVwr::m_osaPdfVwrBinLst.Item( 0 ) );
    m_oPrcManGnuCap .bSetBinFile( PrcPdfVwr::m_osaPdfVwrBinLst.Item( 0 ) );
    g_oConfig.bSetPdfViewer( PrcPdfVwr::m_osaPdfVwrBinLst.Item( 0 ) );
    g_oConfig.bFlush( );
  }
}

//**************************************************************************************************
// Layout the main frame display objects.

void  FrmMain::DoLayout( void )
{
  wxGridBagSizer * poSzr;
  wxGBPosition     oPosn;
  wxGBSpan         oSpan;
  int              iFlags;
  int              iBorder;
  size_t           sz1;

  if( GetSizer( ) == NULL )
  {
    // Create and set the frame's sizer (at startup)
    poSzr = new wxGridBagSizer( );
    SetSizer( poSzr );

    // Specify how the sizer will grow when resized
    poSzr->SetNonFlexibleGrowMode( wxFLEX_GROWMODE_SPECIFIED );
    poSzr->SetCols( 7 );
    poSzr->AddGrowableCol( 0 );
    poSzr->AddGrowableCol( 1 );
    poSzr->SetRows( 15 );
    for( sz1=8; sz1<15; sz1++ ) poSzr->AddGrowableRow( sz1 );
  }
  else
  {
    // Detach the display objects from the sizer
    poSzr = (wxGridBagSizer *) GetSizer( );
    poSzr->Detach( &m_oLblNodes   );
    poSzr->Detach( &m_oLbxNodes   );
    poSzr->Detach( &m_oLblCpnts   );
    poSzr->Detach( &m_oLbxCpnts   );
    poSzr->Detach(  m_poNbkSimEng );
    poSzr->Detach( &m_oNbkTxtCtls );
    poSzr->Clear( );
  }

  // Set horizontal & vertical gaps between cells and the border on the cell out edges
  poSzr->SetHGap( 7 );  // Horizontal gap between cells
  poSzr->SetVGap( 7 );  // Vertical   gap between cells
  iBorder = 7;          // Outer border between cells and panel

  // Add the nodes and components list box labels
  iFlags = wxTOP | wxALIGN_CENTER_HORIZONTAL | wxALIGN_BOTTOM;
  oPosn.SetCol( 0 ); oSpan.SetColspan( 1 );
  oPosn.SetRow( 0 ); oSpan.SetRowspan( 1 );
  poSzr->Add( &m_oLblNodes, oPosn, oSpan, iFlags, iBorder+8 );
  oPosn.SetCol( 1 );
  poSzr->Add( &m_oLblCpnts, oPosn, oSpan, iFlags, iBorder+8 );

  // Add the nodes and components list boxes
  oPosn.SetCol( 0 ); oSpan.SetColspan( 1 );
  oPosn.SetRow( 1 );
  if( m_uiFrmLayout == 0 ) oSpan.SetRowspan( 14 );
  else                     oSpan.SetRowspan(  6 );
  iFlags = wxTOP | wxLEFT | wxALIGN_CENTER_HORIZONTAL | wxEXPAND;
  if( m_uiFrmLayout == 0 ) iFlags |= wxBOTTOM;
  poSzr->Add( &m_oLbxNodes, oPosn, oSpan, iFlags, iBorder );
  iFlags = wxTOP |          wxALIGN_CENTER_HORIZONTAL | wxEXPAND;
  if( m_uiFrmLayout == 0 ) iFlags |= wxBOTTOM;
  oPosn.SetCol( 1 );
  poSzr->Add( &m_oLbxCpnts, oPosn, oSpan, iFlags, iBorder );

  // Add the simulator notebook
  iFlags = wxTOP | wxRIGHT | wxALIGN_CENTER | wxEXPAND;
  oPosn.SetCol( 2 ); oSpan.SetColspan( 5 );
  oPosn.SetRow( 0 ); oSpan.SetRowspan( 7 );
  poSzr->Add( m_poNbkSimEng, oPosn, oSpan, iFlags, iBorder );

  // Add the console notebook
  iFlags = wxBOTTOM | wxRIGHT | wxALIGN_TOP | wxEXPAND;
  oPosn.SetRow( 7 ); oSpan.SetRowspan( 8 );
  if( m_uiFrmLayout == 0 ) { oPosn.SetCol( 2 ); oSpan.SetColspan( 5 ); }
  else                     { oPosn.SetCol( 0 ); oSpan.SetColspan( 7 ); iFlags |= wxLEFT; }
  poSzr->Add( &m_oNbkTxtCtls, oPosn, oSpan, iFlags, iBorder );

  // Set minimum and initial sizes as calculated by the frame's sizer
  poSzr->SetSizeHints( this );
}

//**************************************************************************************************
// Load information from the Simulation object (FrmMain::m_poSimn) into the various display
// controls.
//
// Return Values :
//   true  - Success
//   false - Failure

bool  FrmMain::bSimnLoad( void )
{
  TextCtrl     * poTxtCtl;
  wxArrayString  osa1;
  size_t         sz1;

  // Load all the nodes (except ground) into the "Nodes" list box
  for( sz1=0; sz1<m_poSimn->m_osaNodeLbls.GetCount( ); sz1++ )
    m_oLbxNodes.Append( m_poSimn->m_osaNodeLbls.Item( sz1 ) );
  // Select the test nodes (if any have been specified)
  for( sz1=0; sz1<m_poSimn->rosaGetTstNodes( ).GetCount( ); sz1++ )
    m_oLbxNodes.SetStringSelection( m_poSimn->rosaGetTstNodes( ).Item( sz1 ) );

  // Load all 2-port components into the "Components" list box
  for( sz1=0; sz1<m_poSimn->m_oaCpnts.GetCount( ); sz1++ )
  {
    Component & roCpnt = m_poSimn->m_oaCpnts.Item( sz1 );
    switch( roCpnt.eGetType( ) )  // Only collect two terminal components
    {
      case eCPNT_CAP   :  // Capacitor
      case eCPNT_RES   :  // Resistor
      case eCPNT_IND   :  // Inductor
      case eCPNT_CIND  :  // Coupled (Mutual) Inductors
      case eCPNT_DIODE :  // Diode
      case eCPNT_VCVS  :  // Voltage Controlled Voltage Source
      case eCPNT_CCCS  :  // Current Controlled Current Source
      case eCPNT_VCCS  :  // Voltage Controlled Current Source
      case eCPNT_CCVS  :  // Current Controlled Voltage Source
      case eCPNT_IVS   :  // Independent Voltage Source
      case eCPNT_ICS   :  // Independent Current Source
      case eCPNT_NLDS  :  // Non-Linear Dependent Source
      case eCPNT_NLDCS :  // Non-Linear Dependent Current Source
      case eCPNT_NLDVS :  // Non-Linear Dependent Voltage Source
      case eCPNT_STJ   :  // Super-conducting Tunnel Junction
        osa1.Add( roCpnt.rosGetName( ) );
        break;

      default          :  // Do nothing
        break;
    }
  }
  osa1.Sort( &iStrCmpCpnt );  // Function iStrCmpCpnt( ) is defined in utility/StrUtils.hpp
  if( ! osa1.IsEmpty( ) ) m_oLbxCpnts.InsertItems( osa1, 0 );
  // Select the test components (if any have been specified)
  const wxArrayString & rosa1 = m_poSimn->rosaGetTstCpnts( );
  for( sz1=0; sz1<rosa1.GetCount( ); sz1++ )
    m_oLbxCpnts.SetStringSelection( rosa1.Item( sz1 ) );

  // Load the circuit description into the NetList text control
  if( ! m_oNetLst.m_osaNetLst.IsEmpty( ) )
  {
    // Load the circuit description into the NetList text control
    m_oNbkTxtCtls.bSetPage( NbkTxtCtls::ePAGE_NETLIST );
    poTxtCtl = m_oNbkTxtCtls.poGetPage( );
    poTxtCtl->bClear( );
    for( sz1=0; sz1<m_oNetLst.m_osaNetLst.GetCount( ); sz1++ )
      poTxtCtl->bAppendLine( m_oNetLst.m_osaNetLst.Item( sz1 ) );
    m_oNbkTxtCtls.bSetPosn( 0 );
  }

  // Load the analysis notebook with simulation information
  if( ! m_poNbkSimEng->bLoad( *m_poSimn ) ) return( false );

  // Load the circuit description into the Simulation text control
  m_oNbkTxtCtls.bSetPage( NbkTxtCtls::ePAGE_SIMULTN );
  poTxtCtl = m_oNbkTxtCtls.poGetPage( );
  poTxtCtl->bClear( );
  for( sz1=0; sz1<m_oNetLst.m_osaNetLst.GetCount( ); sz1++ )
    poTxtCtl->bAppendLine( m_oNetLst.m_osaNetLst.Item( sz1 ) );
  m_oNbkTxtCtls.bSetPosn( 0 );
  poTxtCtl->SetEditable( true );

  return( true );
}

//**************************************************************************************************
// Reload information from the Simulation object (FrmMain::m_poSimn) into the various display
// controls.
//
// Return Values :
//   true  - Success
//   false - Failure

bool  FrmMain::bSimnReload( void )
{
  wxArrayString  osaTstNodes;
  wxArrayString  osaTstCpnts;
  size_t         sz1;
  int            i1;

  // Record the nodes that are currently selected
  for( sz1=0; sz1<m_oLbxNodes.GetCount( ); sz1++ )
    if( m_oLbxNodes.IsSelected( sz1 ) )
      osaTstNodes.Add( m_oLbxNodes.GetString( sz1 ) );

  // Record the components that are currently selected
  for( sz1=0; sz1<m_oLbxCpnts.GetCount( ); sz1++ )
    if( m_oLbxCpnts.IsSelected( sz1 ) )
      osaTstCpnts.Add( m_oLbxCpnts.GetString( sz1 ) );

  // Clear the lists of nodes and components
  m_oLbxNodes.Clear( );
  m_oLbxCpnts.Clear( );

  bSimnLoad( );  // Load the Simulation object

  // Select nodes that where previously selected
  if( ! osaTstNodes.IsEmpty( ) )
  {
    for( sz1=0; sz1<osaTstNodes.GetCount( ); sz1++ )
    {
      i1 = m_oLbxNodes.FindString( osaTstNodes.Item( sz1 ) );
      if( i1 != wxNOT_FOUND ) m_oLbxNodes.Select( i1 );
    }
  }

  // Select components that where previously selected
  if( ! osaTstCpnts.IsEmpty( ) )
  {
    for( sz1=0; sz1<osaTstCpnts.GetCount( ); sz1++ )
    {
      i1 = m_oLbxCpnts.FindString( osaTstCpnts.Item( sz1 ) );
      if( i1 != wxNOT_FOUND ) m_oLbxCpnts.Select( i1 );
    }
  }

  return( true );
}

//**************************************************************************************************
// Save information in the various display controls to the Simulation object (FrmMain::m_poSimn).
//
// Return Values :
//   true  - Success
//   false - Failure

bool  FrmMain::bSimnSave( void )
{
  wxArrayInt  oaiSelNodes, oaiSelCpnts;
  size_t      sz1;
  int         iSelNo;
  wxString    osSel;

  // Clear all previously test points and command objects
  m_poSimn->bClrCmds( );

  // Get an array of indicies to the currently selected test points
  m_oLbxNodes.GetSelections( oaiSelNodes );
  m_oLbxCpnts.GetSelections( oaiSelCpnts );

  // Load the test nodes into the simulation object
  for( sz1=0; sz1<oaiSelNodes.GetCount( ); sz1++ )
  {
    iSelNo = oaiSelNodes.Item( sz1 );
    osSel = m_oLbxNodes.GetString( iSelNo );
    m_poSimn->bAddTstNode( osSel );
  }

  // Load the test components into the simulation object
  for( sz1=0; sz1<oaiSelCpnts.GetCount( ); sz1++ )
  {
    iSelNo = oaiSelCpnts.Item( sz1 );
    osSel = m_oLbxCpnts.GetString( iSelNo );
    m_poSimn->bAddTstCpnt( osSel );
  }

  // Transfer the simulation parameters into the simulation object
  if( ! m_poNbkSimEng->bSave( *m_poSimn ) ) return( false );

  // Create the simulation information and save it to file
  if( ! m_poSimn->bSave( ) )                return( false );
  if( ! m_poSimn->bSaveFile( ) )            return( false );

  return( true );
}

//**************************************************************************************************
// Lock GUI controls.
//
// Argument List :
//   bEnable - Enable or disable lock GUI mode

void  FrmMain::LockGUI( bool bEnable )
{
  size_t  sz1;

  GetMenuBar( )->Enable( ! bEnable );    // Set the menu bar state
  m_poNbkSimEng->Enable( ! bEnable );    // Set the simulator notebook state
  m_oLbxNodes   .Enable( ! bEnable );    // Set the node label list box state
  m_oLbxCpnts   .Enable( ! bEnable );    // Set the component label list box state

  // Set the individual state of each tool in the tool bar
  for( sz1=ID_TBR_FST; sz1<=ID_TBR_LST; sz1++ ) GetToolBar( )->EnableTool( sz1, ! bEnable );

  if( bEnable ) ::wxBeginBusyCursor( );  // Change the cursor to the wait symbol
  else          ::wxEndBusyCursor( );    // Change the cursor to the default
}

//**************************************************************************************************
// Clear the object attributes.

bool  FrmMain::bClear( void )
{
  bool  bRtn = true;

  // Clear all previously selected test points
  m_oLbxNodes.Clear( );
  m_oLbxCpnts.Clear( );

  // Clear simulation object attributes
  m_oSimnGnuCap .bClear( );
  m_oSimnNgSpice.bClear( );

  // Clear the simulation object and the analysis panels
  if( m_poNbkSimEng != NULL )
    if( ! m_poNbkSimEng->bClear( ) )  bRtn = false;

  // Clear the text controls
  if( ! m_oNbkTxtCtls.bClear( ) )     bRtn = false;
  m_oNbkTxtCtls.bInitialize( );

  // Terminate any simulation process
  if( m_poPrcSimEng != NULL )
    if( m_poPrcSimEng->bIsExec( ) )
      if( ! m_poPrcSimEng->bKill( ) ) bRtn = false;

  // Terminate any schematic editor process
  if( m_oPrcSchem.bIsExec( ) )
    if( ! m_oPrcSchem.bKill( ) )      bRtn = false;

  // Terminate any waveform data viewer processes
  if( m_oPrcDataVwr.bIsExec( ) )
    if( ! m_oPrcDataVwr.bKill( ) )    bRtn = false;

  // Terminate any calculator processes
  if( m_oPrcCalculator.bIsExec( ) )
    if( ! m_oPrcCalculator.bKill( ) ) bRtn = false;

  // Terminate any PDF viewer processes
  if( m_oPrcManNgSpice.bIsExec( ) )
    if( ! m_oPrcManNgSpice.bKill( ) ) bRtn = false;
  if( m_oPrcManGnuCap.bIsExec( ) )
    if( ! m_oPrcManGnuCap.bKill( ) )  bRtn = false;

  return( bRtn );
}

//**************************************************************************************************
// Display a dialog box containing an error message.
//
// Note : At start-up the first error message is held over until the the main GUI is created and
//        displayed; to display this message call this function with no arguments.
//
// Argument List :
//   rosMsg   - The error message
//   rosTitle - The dialogue box title (if omitted derived from message type)

void  FrmMain::DlgMsg( eTypeDlgMsg eMsgType, const wxString & rosMsg, const wxString & rosTitle )
{
  static  wxMessageDialog  * poDlgMsg=NULL;
          wxString           osTitle;
          wxString           osStatus;
          wxStringTokenizer  ostk1;
          wxString           os1;
          long               lStyle=0;

  // Use the first complete sentence as the status bar message
  ostk1.SetString( rosMsg, wxDEFAULT_DELIMITERS, wxTOKEN_RET_EMPTY );
  while( ostk1.HasMoreTokens( ) )
  {
    // Get the next token
    os1 = ostk1.GetNextToken( );

    // Look for a file name (ie. it's preceeded by empty tokens)
    if( os1.IsEmpty( ) )
    {
      while( ostk1.HasMoreTokens( ) )
      {
        os1 = ostk1.GetNextToken( );
        if( ! os1.IsEmpty( ) ) break;
      }

      // Remove the path from file names
      if( os1.Freq( '/' ) > 0 )
      {
        osStatus << ' ' << os1.AfterLast( '/' );
        break;
      }
    }

    // Append the token to the message
    osStatus << ' ' << os1;

    // Look for the end of the sentence
    if( os1.Last( ) == '.' ) break;
  }

  // If debug mode is enabled send the error message to the console
  if( g_bDebug ) std::cout << "DEBUG : FrmMain::DlgMsg( ) :" << osStatus << '\n';

  // Set the status line message
  SetStatusText( osStatus, ePANE_MESAGE );

  // If FrmMain isn't displayed yet don't proceed any further
  if( ! IsShown( ) ) return;

  // Create message dialogue title and determine the icon to use
  osTitle = rosTitle;
  switch( eMsgType )
  {
    case eDMSG_INFO  :
      if( osTitle.IsEmpty( ) ) osTitle = "Information";
      osStatus = " Info :";
      lStyle = wxICON_INFORMATION;
      break;

    case eDMSG_WARN  :
      if( osTitle.IsEmpty( ) ) osTitle = "Warning";
      osStatus = " Warning :";
      lStyle = wxICON_EXCLAMATION;
      break;

    case eDMSG_ERROR :
      if( osTitle.IsEmpty( ) ) osTitle = "ERROR";
      osStatus = " ERROR :";
      lStyle = wxICON_ERROR;
      break;
  }

  // Create the error message dialog
  os1 = '\n' + rosMsg;
  poDlgMsg = new wxMessageDialog( this, os1, osTitle, wxOK | lStyle );


  // Display the error message dialog
  poDlgMsg->ShowModal( );

  // Delete the error message dialog
  delete poDlgMsg;
  poDlgMsg = NULL;
}

//**************************************************************************************************
//                                         Event Handlers                                          *
//**************************************************************************************************
// Open a circuit description file.
//
// Argument List :
//   roEvtCmd - The event to be processed

void  FrmMain::OnOpen( wxCommandEvent & roEvtCmd )
{
  UNUSED( roEvtCmd );  // Suppresses a compiler warning about unused function parameters

  if( ! m_oFileTasks.bDlgOpen( ) ) return;   // Attempt to get a netlist file name from the user

  LockGUI( true );                           // Lock the GUI controls

  bClear( );                                 // Clear the main frame object attributes
  if( m_oFileTasks.bOpen( ) ) bSimnLoad( );  // Attempt to open and read the netlist file
  InitLogFiles( );                           // Initialize the process log files

  LockGUI( false );                          // Unlock the GUI controls
}

//**************************************************************************************************
// Import a schematic file using gnetlist to convert the schematic to a circuit description.
//
// Argument List :
//   roEvtCmd - The event to be processed

void  FrmMain::OnImport( wxCommandEvent & roEvtCmd )
{
  UNUSED( roEvtCmd );  // Suppresses a compiler warning about unused function parameters

  if( ! m_oFileTasks.bDlgImport( ) ) return;  // Attempt to get the schematic file name/s from user

  LockGUI( true );                            // Lock the GUI controls

  bClear( );                                  // Clear the main frame object attributes
  if( m_oFileTasks.bImport( ) ) bSimnLoad( ); // Attempt to import schematic/s and read netlist file
  InitLogFiles( );                            // Initialize the process log files

  LockGUI( false );                           // Unlock the GUI controls
}

//**************************************************************************************************
// Reload the schematic or netlist file.
//
// Argument List :
//   roEvtCmd - The event to be processed

void  FrmMain::OnReload( wxCommandEvent & roEvtCmd )
{
  UNUSED( roEvtCmd );  // Suppresses a compiler warning about unused function parameters

  LockGUI( true );          // Lock the GUI controls

  m_oFileTasks.bReload( );  // Reload schematic file/s and/or the netlist file
  bSimnReload( );           // Reload the simulation information

  LockGUI( false );         // Unlock the GUI controls
}

//**************************************************************************************************
// Close the circuit description file.
//
// Argument List :
//   roEvtCmd - The event to be processed

void  FrmMain::OnClose( wxCommandEvent & roEvtCmd )
{
  UNUSED( roEvtCmd );  // Suppresses a compiler warning about unused function parameters

  m_oFileTasks.bClose( );                   // Delete temporary files
  bClear( );                                // Clear the object attributes
  m_oFileTasks.bSetTitle( );                // Set the frame title
  m_poNbkSimEng->bSetPage( (eTypeCmd) 0 );  // Display first analysis panel
}

//**************************************************************************************************
// Exit the application.
//
// Argument List :
//   roEvtCmd - The event to be processed

void  FrmMain::OnQuit( wxCommandEvent & roEvtCmd )
{
  UNUSED( roEvtCmd );  // Suppresses a compiler warning about unused function parameters

  Close( true );  // Generates a wxCloseEvent which is handled by OnSysExit( )
}

//**************************************************************************************************
// Create the simulation.
//
// Argument List :
//   roEvtCmd - The event to be processed

void  FrmMain::OnSimCreate( wxCommandEvent & roEvtCmd )
{
  TextCtrl * poTxtCtl;
  wxString   osErrMsg;

  UNUSED( roEvtCmd );  // Suppresses a compiler warning about unused function parameters

  // If a simulation is currently running do nothing
  if( m_poPrcSimEng->bIsExec( ) ) return;

  LockGUI( true );   // Lock the GUI controls

  if( ! m_oNetLst.m_osaNetLst.IsEmpty( ) )
  { // Save the simulation info. in the simulation object
    if( bSimnSave( ) )
    {
      // Print the simulation to the simulation text control
      m_oNbkTxtCtls.bSetPage( NbkTxtCtls::ePAGE_SIMULTN );
      poTxtCtl = m_oNbkTxtCtls.poGetPage( NbkTxtCtls::ePAGE_SIMULTN );
      poTxtCtl->bLoadFile( m_poSimn->rofnGetSaveFile( ).GetFullPath( ) );
      poTxtCtl->SetEditable( true );
    }
    else
    {
      if(      ! m_poNbkSimEng->bIsOk( ) ) osErrMsg = m_poNbkSimEng->rosGetErrMsg( );
      else if( ! m_poSimn->bIsValid( )   ) osErrMsg = m_poSimn     ->rosGetErrMsg( );
      else                                 osErrMsg = "Unknown error.";
    }
  }
  else osErrMsg = "There's no schematic or netlist file loaded.";

  // Display exit status message
  if( ! osErrMsg.IsEmpty( ) )
       DlgMsg( eDMSG_ERROR, osErrMsg, "Create Simulation Error" );
  else SetStatusText( " Simulation ran successfully", ePANE_MESAGE );

  LockGUI( false );  // Unlock the GUI controls
}

//**************************************************************************************************
// Run the simulation.
//
// Argument List :
//   roEvtCmd - The event to be processed

void  FrmMain::OnSimRun( wxCommandEvent & roEvtCmd )
{
  TextCtrl *             poTxtCtl;
  NbkTxtCtls::ePageType  ePage;
  wxString               osErrMsg, os1;

  UNUSED( roEvtCmd );  // Suppresses a compiler warning about unused function parameters

  //------------------------------------------------------------------------------------------------
  // Check if the schematic file/s have been updated and whether to re-generate the netlist before
  // running simulation run (if this feature enabled)

  if( g_oConfig.bGetAutoRegen( ) )
  {
    if( m_oFileTasks.bReloadReqd( ) )
    {
      wxCommandEvent  oEvtCmd;
      OnReload( oEvtCmd );
    }
  }

  //------------------------------------------------------------------------------------------------
  // Prepare the GUI to run the simulation

  LockGUI( true );                                 // Lock the GUI controls,
  GetToolBar( )->EnableTool( ID_TBR_STOP, true );  // except for the stop simulation button
  SetStatusText( " Running the simulation ...", ePANE_MESAGE );

  //------------------------------------------------------------------------------------------------
  // Check if a schematic or netlist file is currently open

  if( ! wxFileName( m_oFileTasks.rosGetNetLstFile( ) ).IsFileReadable( ) )
    { osErrMsg = "There's no schematic or netlist file loaded.";                goto OnSimRun_Rtn; }

  //------------------------------------------------------------------------------------------------
  // Update the Simulation text page before running the simulation

  poTxtCtl = m_oNbkTxtCtls.poGetPage( NbkTxtCtls::ePAGE_SIMULTN );
  if( poTxtCtl->IsModified( ) )
  { // The user has modified the simulation, save the Simulation text page contents to file
    poTxtCtl->SaveFile( m_poSimn->rofnGetSaveFile( ).GetFullPath( ) );
  }
  else if( bSimnSave( ) )
  { // Load the simulation on file into the Simulation text page
    poTxtCtl->bLoadFile( m_poSimn->rofnGetSaveFile( ).GetFullPath( ) );
    poTxtCtl->DiscardEdits( );           // Reset modified flag as if changes had been saved
    poTxtCtl->SetEditable( true );       // Make the text control editable
  }
  else
  { // There's been an error, report it to the user and return
    if(      ! m_poNbkSimEng->bIsOk( ) ) osErrMsg = m_poNbkSimEng->rosGetErrMsg( );
    else if( ! m_poSimn->bIsValid( )   ) osErrMsg = m_poSimn     ->rosGetErrMsg( );
    else                                 osErrMsg = "Unknown error.";
                                                                                goto OnSimRun_Rtn;
  }
  m_oNbkTxtCtls.bSetPage( NbkTxtCtls::ePAGE_SIMULTN );

  //------------------------------------------------------------------------------------------------
  // Run the simulation and update the Console text page

  if( ! m_poPrcSimEng->bMakeArgLst( *m_poSimn ) )
                                   { osErrMsg = m_poPrcSimEng->rosGetErrMsg( ); goto OnSimRun_Rtn; }
  m_poPrcSimEng->bExec( );
  m_oNbkTxtCtls.bSetPage( NbkTxtCtls::ePAGE_CONSOLE );
  poTxtCtl = m_oNbkTxtCtls.poGetPage( NbkTxtCtls::ePAGE_CONSOLE );
  poTxtCtl->bClear( );                   // Clear the Console text page
  m_poPrcSimEng->PrintCmd( *poTxtCtl );  // Print simulator cmd to Console page

  // Return immediately if the user aborted the simulation
  if( GetStatusBar( )->GetStatusText( 1 ).Contains( "aborted" ) )               goto OnSimRun_Rtn;

  // Print simulator output to Console page
  m_poPrcSimEng->PrintRsp( *poTxtCtl );

  // Check for errors during the simulation run
  if( ! m_poPrcSimEng->bIsOk( ) )  { osErrMsg = m_poPrcSimEng->rosGetErrMsg( ); goto OnSimRun_Rtn; }

  //------------------------------------------------------------------------------------------------
  // Format the simulation results and update the Results text page

  SetStatusText( " Formatting the simulation results ...", ePANE_MESAGE );
  GetToolBar( )->EnableTool( ID_TBR_STOP, false );  // Disable the stop simulation button
  wxTheApp->Yield( true );                          // Allow the display to update
  if( ! m_poPrcSimEng->bFmtResults( ) )
                                  { osErrMsg = m_poPrcSimEng->rosGetErrMsg( );  goto OnSimRun_Rtn; }
  if( m_poNbkSimEng->eGetSimEngine( ) == eSIMR_GNUCAP ) ePage = NbkTxtCtls::ePAGE_GNUCAP;
  else                                                  ePage = NbkTxtCtls::ePAGE_NGSPICE;
  poTxtCtl = m_oNbkTxtCtls.poGetPage( ePage );
  os1 = m_poPrcSimEng->rofnGetResultsFile( ).GetFullPath( );

  // Load the simulation output into the results text page
  if( ! poTxtCtl->bLoadFile( os1 ) )
              { osErrMsg = "Couldn't load the results file : \n\n" + os1;       goto OnSimRun_Rtn; }
  m_oNbkTxtCtls.bSetPage( ePage );       // Change to the appropriate results page

  //------------------------------------------------------------------------------------------------
  // Display status and any errors messages and return

OnSimRun_Rtn:

  // Display exit status message
  if( ! osErrMsg.IsEmpty( ) )
       DlgMsg( eDMSG_ERROR, osErrMsg, "Run Simulation Error" );
  else SetStatusText( " Simulation ran successfully", ePANE_MESAGE );

  LockGUI( false );                      // Unlock the GUI controls
}

//**************************************************************************************************
// Stop the simulation.
//
// Argument List :
//   roEvtCmd - The event to be processed

void  FrmMain::OnSimStop( wxCommandEvent & roEvtCmd )
{
  UNUSED( roEvtCmd );  // Suppresses a compiler warning about unused function parameters

  if( m_poPrcSimEng->bIsExec( ) )
  { // A simulation is currently running, abort it
    m_poPrcSimEng->bKill( );
    SetStatusText( " Simulation aborted by the user", ePANE_MESAGE );
  }
  else
  { // No simulation is currently running, display an error message dialogue
    DlgMsg( eDMSG_ERROR, "   There's no simulation currently running.   ", "Stop Simulation Error" );
  }
}

//**************************************************************************************************
// Edit / view a schematic file.
//
// Argument List :
//   roEvtCmd - The event to be processed

void  FrmMain::OnSchematic( wxCommandEvent & roEvtCmd )
{
  wxString    osErrHdr = "Schematic Editor Error";
  wxString    osErrMsg;

  UNUSED( roEvtCmd );  // Suppresses a compiler warning about unused function parameters

  // Check if the schematic editor process is already running
  if( m_oPrcSchem.bIsExec( ) )
  {
    osErrMsg = "The schematic editor process is already running.";
    DlgMsg( eDMSG_INFO, osErrMsg );
    return;
  }

  // Check if schematic file/s currently set
  m_oPrcSchem.bSetSchemFiles( m_oFileTasks.rosaGetSchemFiles( ) );
  if( m_oPrcSchem.rosGetSchemFiles( ).IsEmpty( ) )
  {
    osErrMsg = "There's no schematic loaded or the netlist has no associated schematic file.";
    DlgMsg( eDMSG_ERROR, osErrMsg, osErrHdr );
    return;
  }

  // Execute the schematic editor process
  if( ! m_oPrcSchem.bExec( ) )
  { // Something went wrong, display an error message
    osErrMsg = m_oPrcSchem.rosGetErrMsg( );
    DlgMsg( eDMSG_ERROR, osErrMsg, osErrHdr );
  }
}

//**************************************************************************************************
// View the simulation results.
//
// Argument List :
//   roEvtCmd - The event to be processed

void  FrmMain::OnViewData( wxCommandEvent & roEvtCmd )
{
  wxFileName  ofn1;
  wxString    osTitle = "Waveform Data Viewer Error";
  wxString    os1;

  UNUSED( roEvtCmd );  // Suppresses a compiler warning about unused function parameters

  // Check if the waveform viewer process is already running
  if( m_oPrcDataVwr.bIsExec( ) )
  {
    DlgMsg( eDMSG_INFO, "The waveform data viewer process is already running." );
    return;
  }

  // Check if a schematic or netlist file is currently open
  ofn1 = m_oFileTasks.rosGetNetLstFile( );
  if( ofn1.GetFullPath( ).IsEmpty( ) )
  {
    DlgMsg( eDMSG_ERROR, "There's no schematic or netlist file loaded.", osTitle );
    return;
  }

  // Create the appropriate name for the results file
  os1 = ofn1.GetName( );
  switch( m_poNbkSimEng->eGetSimEngine( ) )
  {
    case eSIMR_GNUCAP :
      if( m_oNbkTxtCtls.eGetPage( ) == NbkTxtCtls::ePAGE_NGSPICE )
           os1 << ".ngspice";
      else os1 << ".gnucap";
      break;

    case eSIMR_NGSPICE :
      if( m_oNbkTxtCtls.eGetPage( ) == NbkTxtCtls::ePAGE_GNUCAP )
           os1 << ".gnucap";
      else os1 << ".ngspice";
      break;

    default :
      return;
  }
  ofn1.SetName( os1 );
  ofn1.SetExt( m_poNbkSimEng->rosGetPage( ) );

  // Set the results file in the waveform viewer process
  if( ! ofn1.FileExists( ) )
  {
    os1.Empty( );
    os1 << "The simulation results file :\n\n" << ofn1.GetFullPath( )
        << "\n\nhasn't yet been generated.";
    DlgMsg( eDMSG_WARN, os1, osTitle );
    return;
  }
  m_oPrcDataVwr.bSetResultsFile( ofn1.GetFullPath( ) );

  // Execute the waveform viewer process
  os1.Empty( );
  if( ! m_oPrcDataVwr.bExec( ) ) os1 = m_oPrcDataVwr.rosGetErrMsg( );

  // If an error occurs then display a message to the user
  if( ! os1.IsEmpty( ) ) DlgMsg( eDMSG_ERROR, os1, osTitle );
}

//**************************************************************************************************
// Envoke the calculator application.
//
// Argument List :
//   roEvtCmd - The event to be processed

void  FrmMain::OnCalculator( wxCommandEvent & roEvtCmd )
{
  wxString    osTitle = "Calculator Error";
  wxString    os1;

  UNUSED( roEvtCmd );  // Suppresses a compiler warning about unused function parameters

  // Check if the calculator process is already running
  if( m_oPrcCalculator.bIsExec( ) )
  {
    DlgMsg( eDMSG_INFO, "The calculator process is already running." );
    return;
  }

  // Execute the calculator process
  os1.Empty( );
  if( ! m_oPrcCalculator.bExec( ) ) os1 = m_oPrcCalculator.rosGetErrMsg( );

  // If an error occurs then display a message to the user
  if( ! os1.IsEmpty( ) ) DlgMsg( eDMSG_ERROR, os1, osTitle );
}

//**************************************************************************************************
// Select which simulation engine to use.
//
// Argument List :
//   roEvtCmd - The event to be processed

void  FrmMain::OnSelSimEng( wxCommandEvent & roEvtCmd )
{
  int  iw, ih;

  UNUSED( roEvtCmd );  // Suppresses a compiler warning about unused function parameters

  // Temporarily store the current main frame size
  GetClientSize( &iw, &ih );

  // Change cursor to the wait symbol and hide the GUI while re-constructing it
  ::wxBeginBusyCursor( );
  Show( false );

  // Create and display a progress dialog
  wxProgressDialog  oDlgProgress( " GNU Spice GUI", "Changing simulation engines ...", 100, this,
                             wxPD_AUTO_HIDE | wxPD_APP_MODAL | wxPD_SMOOTH );
  oDlgProgress.CenterOnParent( );
  oDlgProgress.Update( 5 ); ::wxMilliSleep( 250 );

  // Record the simulation engine change in the configuration file
  if( GetMenuBar( )->IsChecked( ID_MNU_NGSPICE ) ) g_oConfig.bSetSimEngine( eSIMR_NGSPICE );
  else                                             g_oConfig.bSetSimEngine( eSIMR_GNUCAP  );
  g_oConfig.bFlush( );  // Write changes to the configuration file

  // Update the progress dialog
  oDlgProgress.Update( 20 );

  // Change the simulator
  InitSimEng( );

  // Update the progress dialog
  oDlgProgress.Update( 50 ); ::wxMilliSleep( 150 );

  // Set the simulator process log file name
  InitLogFiles( );

  // Update the progress dialog
  oDlgProgress.Update( 60 ); ::wxMilliSleep( 150 );

  // Update the GUI components
  DoLayout( );

  // Update the progress dialog
  oDlgProgress.Update( 70 ); ::wxMilliSleep( 150 );

  // Transfer info. from one simulation object to the other
  if( ! m_oNetLst.bIsEmpty( ) )
  {
    if( GetMenuBar( )->IsChecked( ID_MNU_NGSPICE ) ) m_oSimnNgSpice = m_oSimnGnuCap;
    else                                             m_oSimnGnuCap  = m_oSimnNgSpice;
  }

  // Update the progress dialog
  oDlgProgress.Update( 80 ); ::wxMilliSleep( 150 );

  // Load the analysis notebook with the simulation values
  m_poNbkSimEng->bLoad( *m_poSimn );

  // Update the progress dialog
  oDlgProgress.Update( 90 ); ::wxMilliSleep( 150 );

  // Restore the frame's original size
  SetClientSize( iw, ih );

  // Update the progress dialog
  oDlgProgress.Update( 99 ); ::wxMilliSleep( 150 );
  oDlgProgress.Update( 100 );  // Kill the progress dialog

  // Show the GUI again and change the cursor to the default
  Show( true );
  ::wxEndBusyCursor( );
}

//**************************************************************************************************
// Select which EDA tool suite to use.
//
// Argument List :
//   roEvtCmd - The event to be processed

void  FrmMain::OnSelEDA_TS( wxCommandEvent & roEvtCmd )
{
  UNUSED( roEvtCmd );  // Suppresses a compiler warning about unused function parameters

  // Record the EDA tool suite change in the configuration file
  if( GetMenuBar( )->IsChecked( ID_MNU_LEPTON ) ) g_oConfig.bSetEdaToolSuite( eEDA_LEPTON  );
  else                                            g_oConfig.bSetEdaToolSuite( eEDA_GEDAGAF );
  g_oConfig.bFlush( );  // Write changes to the configuration file

  // Based on the EDA tool suite selected, set the schematic capture / editor program binary name
  m_oPrcSchem.bSetEdaToolSuite( g_oConfig.eGetEdaToolSuite( ) );

  // Based on the EDA tool suite selected, set the netlister utility binary name
  m_oFileTasks.m_oPrcNetLstr.bSetEdaToolSuite( g_oConfig.eGetEdaToolSuite( ) );

  // Set the EDA tool suite in the status bar
  SetStatusText( " " + g_oConfig.rosGetEdaToolSuite( ), ePANE_EDATLS );
}

//**************************************************************************************************
// Select which waveform (data) viewer application to use.
//
// Argument List :
//   roEvtCmd - The event to be processed

void  FrmMain::OnSelDataVwr( wxCommandEvent & roEvtCmd )
{
  UNUSED( roEvtCmd );  // Suppresses a compiler warning about unused function parameters

  // Record the EDA tool suite change in the configuration file
  if(      GetMenuBar( )->IsChecked( ID_MNU_GAW   ) ) g_oConfig.bSetDataViewer( eDVWR_GAW   );
  else if( GetMenuBar( )->IsChecked( ID_MNU_GWAVE ) ) g_oConfig.bSetDataViewer( eDVWR_GWAVE );
  else if( GetMenuBar( )->IsChecked( ID_MNU_KST   ) ) g_oConfig.bSetDataViewer( eDVWR_KST   );
  else return;
  g_oConfig.bFlush( );  // Write changes to the configuration file

  // Based on the data viewer selected, set the schematic capture / editor program binary name
  m_oPrcDataVwr.bSetDataViewer( g_oConfig.eGetDataViewer( ) );

  // Set the data (waveform) viewer in the status bar
  SetStatusText( " " + rosEnumVwrToStr( g_oConfig.eGetDataViewer( ) ), ePANE_DATVWR );
}

//**************************************************************************************************
// Show the application preferences dialog.
//
// Argument List :
//   roEvtCmd - The event to be processed

void  FrmMain::OnPrefs( wxCommandEvent & roEvtCmd )
{
  bool  bSimEngChanged=false;
  uint  ui1;

  UNUSED( roEvtCmd );  // Suppresses a compiler warning about unused function parameters

  DlgPrefs  oDlgPrefs( this );

  oDlgPrefs.CenterOnParent( );
  oDlgPrefs.ShowModal( );

  if( oDlgPrefs.GetReturnCode( ) == wxID_OK )
  {
    // Determine if the simulation engine has been changed
    if( (g_oConfig.eGetSimEngine()==eSIMR_NGSPICE && GetMenuBar()->IsChecked( ID_MNU_GNUCAP  )) ||
        (g_oConfig.eGetSimEngine()==eSIMR_GNUCAP  && GetMenuBar()->IsChecked( ID_MNU_NGSPICE )) )
      bSimEngChanged = true;

    // Check the appropriate simulation engine radio button in the Settings menu
    if( g_oConfig.eGetSimEngine()    == eSIMR_NGSPICE ) GetMenuBar()->Check( ID_MNU_NGSPICE, true );
    else                                                GetMenuBar()->Check( ID_MNU_GNUCAP , true );

    // Check the appropriate EDA tool suite radio button in the Settings menu
    if( g_oConfig.eGetEdaToolSuite() == eEDA_LEPTON   ) GetMenuBar()->Check( ID_MNU_LEPTON , true );
    else                                                GetMenuBar()->Check( ID_MNU_GEDAGAF, true );

    // Check the appropriate data viewer radio button in the Settings menu
    switch( g_oConfig.eGetDataViewer( ) )
    {
      case eDVWR_GAW   : GetMenuBar( )->Check( ID_MNU_GAW  , true ); break;
      case eDVWR_GWAVE : GetMenuBar( )->Check( ID_MNU_GWAVE, true ); break;
      case eDVWR_KST   : GetMenuBar( )->Check( ID_MNU_KST  , true ); break;
      default          :                                             break; // Do nothing
    }

    // Set the EDA tool suite in the status bar
    SetStatusText( ' ' + g_oConfig.rosGetEdaToolSuite( ), ePANE_EDATLS );

    // Set the waveform data viewer in the status bar
    SetStatusText( ' ' + rosEnumVwrToStr( g_oConfig.eGetDataViewer( ) ), ePANE_DATVWR );

    // Based on the EDA tool suite set the schematic capture / editor program binary name
    m_oPrcSchem.bSetEdaToolSuite( g_oConfig.eGetEdaToolSuite( ) );

    // Based on the EDA tool suite set the netlister utility binary name
    m_oFileTasks.m_oPrcNetLstr.bSetEdaToolSuite( g_oConfig.eGetEdaToolSuite( ) );

    // Set the waveform data viewer program binary name
    m_oPrcDataVwr.bSetDataViewer( g_oConfig.eGetDataViewer( ) );

    // Set the calculator application binary name
    m_oPrcCalculator.bSetBinFile( g_oConfig.rosGetCalculator( ) );

    // Set the PDF viewer application binary name
    m_oPrcManUser   .bSetBinFile( g_oConfig.rosGetPdfViewer( ) );
    m_oPrcManNgSpice.bSetBinFile( g_oConfig.rosGetPdfViewer( ) );
    m_oPrcManGnuCap .bSetBinFile( g_oConfig.rosGetPdfViewer( ) );

    // Set the precision of the results values
    CnvtType::bSetFltRes( g_oConfig.uiGetPrecision( ) );

    // Set the netlister utility Guile procedure to use for the backend processing
    m_oFileTasks.bSetGuileProc( g_oConfig.rosGetGuileProc( ) );

    // Set the text control maximum lines
    m_oNbkTxtCtls.bSetLinesMax( g_oConfig.uiGetNbkMaxLns( ) );

    // Set the spin control update rate
    PnlTxtSpn::bSetSpnPeriod( g_oConfig.uiGetSpnPeriod( ) );

    // Set the state of tooltips
    wxToolTip::Enable( g_oConfig.bGetToolTips( ) );
    wxToolTip::SetDelay( g_oConfig.uiGetToolTipDly( ) );

    // Set the main frame layout to be used
    ui1 = g_oConfig.uiGetFrmLayout( );
    if( m_uiFrmLayout != ui1 )
    {
      m_uiFrmLayout = ui1;
      DoLayout( );
      InitPosnSize( );
    }
  }

  // Now that everything else is done update the simulation engine if required
  if( bSimEngChanged )
  {
    wxCommandEvent  oEvtCmd;
    OnSelSimEng( oEvtCmd );
  }
}

//**************************************************************************************************
// Display the gSpiceUI user manual.
//
// Argument List :
//   roEvtCmd - The event to be processed

void  FrmMain::OnManUser( wxCommandEvent & roEvtCmd )
{
  wxString    osTitle = "PDF Viewer";
  SysScan     oSysScan;
  wxFileName  ofn1, ofn2;
  wxString    os1;

  UNUSED( roEvtCmd );  // Suppresses a compiler warning about unused function parameters

  // Check if the PDF viewer process for gSpiceUI documentation is already running
  if( m_oPrcManUser.bIsExec( ) )
  {
    DlgMsg( eDMSG_INFO, "The PDF viewer process is already running." );
    return;
  }

  // Check that a PDF viewer application is available
  if( ! m_oPrcManUser.bBinExists( ) )
  {
    os1 << "Can't find the binary file required to view PDF documentation :\n\n     "
        << m_oPrcManUser.rofnGetBinFile( ).GetFullPath( );
    DlgMsg( eDMSG_WARN, os1, osTitle );
    return;
  }

  // Check that the PDF documentation is available
  os1 = oSysScan.rosGetPathDoc( ) + "/gspiceui-man.pdf";
  if( ! m_oPrcManUser.bSetPdfFile( os1 ) )
  {
    // Display an error message
    os1.Empty( );
    os1 << "\nThe gSpiceUI PDF user manual \"" << ofn1.GetFullName( ) << '\"'
        << " couldn't be found in either of the following locations :\n\n"
        << "     " << ofn1.GetPath( ) << "/\n"
        << "     " << ofn2.GetPath( ) << "/\n";
    DlgMsg( eDMSG_WARN, os1, osTitle );
    return;
  }

  if( ! m_oPrcManUser.bExec( ) ) DlgMsg( eDMSG_ERROR, m_oPrcManUser.rosGetErrMsg( ), osTitle );
}

//**************************************************************************************************
// Display the NG-SPICE user manual.
//
// Argument List :
//   roEvtCmd - The event to be processed

void  FrmMain::OnManNgSpice( wxCommandEvent & roEvtCmd )
{
  wxString  osTitle = "PDF Viewer";
  wxString  osPdfFile = "/usr/share/doc/ngspice/manual.pdf";
  wxString  os1;

  UNUSED( roEvtCmd );  // Suppresses a compiler warning about unused function parameters

  // Check if the PDF viewer process for NG-SPICE documentation is already running
  if( m_oPrcManNgSpice.bIsExec( ) )
  {
    DlgMsg( eDMSG_INFO, "The PDF viewer process is already running." );
    return;
  }

  // Check that a PDF viewer application is available
  if( ! m_oPrcManNgSpice.bBinExists( ) )
  {
    os1 << "Can't find the binary file required to view PDF documentation :\n\n     "
        << m_oPrcManNgSpice.rofnGetBinFile( ).GetFullPath( );
    DlgMsg( eDMSG_WARN, os1, osTitle );
    return;
  }

  // Check that the PDF documentation is available
  if( ! m_oPrcManNgSpice.bSetPdfFile( osPdfFile ) )
  {
    os1 << "Can't find the PDF documentation for NG-SPICE :\n\n     " << osPdfFile;
    DlgMsg( eDMSG_WARN, os1, osTitle );
    return;
  }

  if( ! m_oPrcManNgSpice.bExec( ) ) DlgMsg( eDMSG_ERROR, m_oPrcManNgSpice.rosGetErrMsg(), osTitle );
}

//**************************************************************************************************
// Display the GNU-CAP user manual.
//
// Argument List :
//   roEvtCmd - The event to be processed

void  FrmMain::OnManGnuCap( wxCommandEvent & roEvtCmd )
{
  wxString  osTitle = "PDF Viewer";
  wxString  osPdfFile = "/usr/share/doc/gnucap/gnucap-man.pdf";
  wxString  os1;

  UNUSED( roEvtCmd );  // Suppresses a compiler warning about unused function parameters

  // Check if the waveform viewer process is already running
  if( m_oPrcManGnuCap.bIsExec( ) )
  {
    DlgMsg( eDMSG_INFO, "The PDF viewer process is already running." );
    return;
  }

  // Check that a PDF viewer application is available
  if( ! m_oPrcManGnuCap.bBinExists( ) )
  {
    os1 << "Can't find the binary file required to view PDF documentation :\n\n     "
        << m_oPrcManGnuCap.rofnGetBinFile( ).GetFullPath( );
    DlgMsg( eDMSG_WARN, os1, osTitle );
    return;
  }

  // Check that the PDF documentation is available
  if( ! m_oPrcManGnuCap.bSetPdfFile( osPdfFile ) )
  {
    os1 << "Can't find the PDF documentation for GNU-CAP :\n\n     " << osPdfFile;
    DlgMsg( eDMSG_WARN, os1, osTitle );
    return;
  }

  if( ! m_oPrcManGnuCap.bExec( ) ) DlgMsg( eDMSG_ERROR, m_oPrcManGnuCap.rosGetErrMsg( ), osTitle );
}

//**************************************************************************************************
// Display about message dialog.
//
// Argument List :
//   roEvtCmd - The event to be processed

void  FrmMain::OnAbout( wxCommandEvent & roEvtCmd )
{
  wxString  os1;
  long      lStyle;

  UNUSED( roEvtCmd );  // Suppresses a compiler warning about unused function parameters

  os1 << "\n                                     "  << APP_NAME
      << "\n                     Version "          << APP_VERSION
                                            << " (" << APP_DATE << ")"
      << "\n               "                        << APP_COPYRIGHT
      << "\n\n"

      << "\nThis application provides a GUI front-end to the following        "
      << "\nfreely available electronic circuit simulation engines :\n"
      << "\n"
      << "\n                 -  " << rosEnumEngToStr( eSIMR_NGSPICE )
      << "\n                 -  " << rosEnumEngToStr( eSIMR_GNUCAP  ) << '\n'

      << "\nThe EDA tool suites supported are :\n"
      << "\n                 -  " << rosEnumEdaToStr( eEDA_LEPTON  )
      << "\n                 -  " << rosEnumEdaToStr( eEDA_GEDAGAF ) << '\n'

      << "\nSimulation results can be viewed using the following "
      << "\nwaveform data viewer applications :\n"
      << "\n                 -  " << rosEnumVwrToStr( eDVWR_GAW   )
      << "\n                 -  " << rosEnumVwrToStr( eDVWR_GWAVE )
      << "\n                 -  " << rosEnumVwrToStr( eDVWR_KST   ) << '\n'

      << "\n" << APP_NAME << " is written in C++ and uses the wxWidgets"
      << "\nlibrary (version " << wxMAJOR_VERSION << '.' << wxMINOR_VERSION << '.'
      << wxRELEASE_NUMBER << '.' << wxSUBRELEASE_NUMBER << "), a free and open source widget"
      << "\ntoolkit and tools library. wxWidgets is itself built on the \n"
#ifdef __WXGTK__
      << "GTK"
#endif
#ifdef __WXQT__
      << "Qt"
#endif
      << " widget toolkit.\n"

      << "\n" << APP_NAME << " is free software; you can redistribute it and/or"
      << "\nmodify it under the terms of the GNU Library General"
      << "\nPublic Licence as published by the Free Software"
      << "\nFoundation; either version 3 of the Licence, or (at your"
      << "\noption) any later version.\n"

      << "\n" << APP_NAME << " is distributed in the hope that it will be useful,"
      << "\nbut WITHOUT ANY WARRANTY; without even the implied"
      << "\nwarranty of MERCHANTABILITY or FITNESS FOR A"
      << "\nPARTICULAR PURPOSE.";

  lStyle = wxOK | wxICON_INFORMATION;

  wxMessageBox( os1, "About gSpiceUI", lStyle, this );
}

//**************************************************************************************************
// Change the cursor when a simulation is running and the mouse is over the stop button.
//
// Argument List :
//   roEvtCmd - The event to be processed

void  FrmMain::OnToolEnter( wxCommandEvent & roEvtCmd )
{
  UNUSED( roEvtCmd );  // Suppresses a compiler warning about unused function parameters

  if( roEvtCmd.GetSelection( ) == ID_TBR_STOP )
  {
    if( m_poPrcSimEng->bIsExec( ) &&  ::wxIsBusy( ) )
    {
      ::wxEndBusyCursor( );     // Change the cursor to the default
      wxTheApp->Yield( true );  // Allow the display to update
    }
  }
  else
  {
    if( m_poPrcSimEng->bIsExec( ) && !::wxIsBusy( ) )
    {
      ::wxBeginBusyCursor( );   // Change the cursor to the hour glass
      wxTheApp->Yield( true );  // Allow the display to update
    }
  }
}

//**************************************************************************************************
// Event handler for system close.
//
// Argument List :
//   roEvtClose - The event to be processed

void  FrmMain::OnSysExit( wxCloseEvent & roEvtClose )
{
  wxMessageDialog * poDlgMsg;
  wxString          osTitle, osMsg;
  int               iPosnX, iPosnY, iSizeW, iSizeH;
  long              liStyle;
  int               iRtn;

  UNUSED( roEvtClose );  // Suppresses a compiler warning about unused function parameters

  // Display message to user if a schematic capture and/or a data viewer app. is running
  if( m_oPrcSchem     .bIsExec( ) || m_oPrcDataVwr  .bIsExec( ) || m_oPrcCalculator.bIsExec( ) ||
      m_oPrcManNgSpice.bIsExec( ) || m_oPrcManGnuCap.bIsExec( ) )
  {
    osTitle = "Running Processes Warning";
    osMsg  = "\n   The following processes are running and   ";
    osMsg += "\n   are about to be killed :\n";
    if( m_oPrcSchem     .bIsExec( ) )
      osMsg += "\n \t " + m_oPrcSchem     .rofnGetBinFile( ).GetFullName( );
    if( m_oPrcDataVwr   .bIsExec( ) )
      osMsg += "\n \t " + m_oPrcDataVwr   .rofnGetBinFile( ).GetFullName( );
    if( m_oPrcCalculator.bIsExec( ) )
      osMsg += "\n \t " + m_oPrcCalculator.rofnGetBinFile( ).GetFullName( );
    if( m_oPrcManNgSpice.bIsExec( ) )
      osMsg += "\n \t " + m_oPrcManNgSpice.rofnGetBinFile( ).GetFullName( );
    if( m_oPrcManGnuCap .bIsExec( ) )
      osMsg += "\n \t " + m_oPrcManGnuCap .rofnGetBinFile( ).GetFullName( );
    liStyle = wxOK | wxCANCEL | wxICON_WARNING | wxCENTRE;
    poDlgMsg = new wxMessageDialog( this, osMsg, osTitle, liStyle );
    iRtn = poDlgMsg->ShowModal( );
    delete poDlgMsg;
    if( iRtn == wxID_CANCEL ) return;
  }

  bClear( );              // Kill any processes currently running

  m_oFileTasks.bExit( );  // Delete temporary files

  Hide( );                // Hide the main frame while exiting

  // Save the frame size and position
  GetPosition( &iPosnX, &iPosnY );
  GetClientSize( &iSizeW, &iSizeH );
  g_oConfig.bSetMainPosnX( iPosnX );
  g_oConfig.bSetMainPosnY( iPosnY );
  g_oConfig.bSetMainSizeW( iSizeW );
  g_oConfig.bSetMainSizeH( iSizeH );

  // Record the simulation engine currently selected
  g_oConfig.bSetSimEngine( m_poPrcSimEng->eGetSimEngine( ) );

  // Record the analysis type last used
  g_oConfig.bSetAnalysis( m_poNbkSimEng->eGetPage( ) );

  g_oConfig.bFlush( );    // Write changes to the configuration file

  m_bIsOpen = false;      // Indicate that the main frame has been closed

  Destroy( );             // Destroys the window safely (used instead of the delete operator)
}

//**************************************************************************************************
