/* TODO: implement this within a single (or several) processes, polling for
 * each child socket */

#include "bool.h"
#include <fcntl.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <sys/socket.h>
#ifdef HAVE_SYS_SELECT_H
# include <sys/select.h>
#endif
#include <sys/stat.h>
#include <netinet/in.h>
#ifdef HAVE_ARPA_INET_H
#include <arpa/inet.h>
#endif
#include <signal.h>
#include <unistd.h>
#include <errno.h>
#ifdef HAVE_SYSLOG
# include <syslog.h>
#endif
#include <limits.h>
#include <stdio.h>
#include <time.h>
#include "util.h"
#include "auth.h"
#include "child.h"
#include "network.h"
#include "socket.h"
#include "buffer.h"
#include "parse.h"
#include "main.h"

/* Misc functions */
static bool sig_ign(int sig);

/* Main input handler functions */
static bool inputhandler(mmap_area authfile);
static int getcommand(char *arg, size_t maxargsize, int level);
static struct cmdlist **level2usergroup(int level);
static int parse(const char *command, int level, bool hasarg);
static void invalid_command(const char *reason);
static bool sendhelp(int level);

/* List/status functions */
static bool pid2user(char *user, pid_t pid, struct clientlist *list);
static bool get_victim(char *arg, char *victim_name, pid_t *victim_pid,
		struct clientlist **list);
static bool update_list(struct clientlist **list);
static void kill_user(char *user, struct clientlist *list);
static bool print_list(struct clientlist *list);
static bool print_uptime(void);
static void getprint(bool with_STATUS);
#define getprint_STATUS() getprint(TRUE)
#define getprint_CONNECT() getprint(FALSE)

/* Account functions */
static bool authentication(mmap_area authfile, char *user, char *pass);
static bool user_disabled(char *user, mmap_area authfile);
static bool lock(char *user, mmap_area authfile);
static bool unlock(char *user, mmap_area authfile);
static bool authwrite(bool writeable, mmap_area authfile);
static void print_locks(mmap_area curr);

/* Socket/Pipe functions */
static bool sendnline(const char *msg, size_t len);
static bool sendline(const char *msg);
static bool pipe_send(struct pipe_msg_buf msg);
static bool pipen_recv_wait(void *msg, size_t size);
static bool pipe_recv_wait(struct pipe_msg_buf *msg);
static bool sendcommand(int command, const char *user);

/* Socket globals. We make STDOUT_FILENO and STDIN_FILENO for use with
 * parent/child IPC. */
static const char *peername;
static SOK *stream;
static pid_t ourpid;

bool child(int sok, const int pwrite, const int pread, mmap_area authfile,
		const char *name)
{
	ourpid = getpid();
	if (-1==fcntl(sok, F_SETFL, O_NONBLOCK)) {
		notice_err("setting socket non-blocking failed");
		return TRUE;
	}
	if (!(stream = sdopen(sok)))
		return TRUE;
	stream->in.timeout = set.keep_alive_timeout;
	stream->out.timeout = 0; /* no timeout */

	if (sig_ign(SIGPIPE))
		return TRUE;
	if (-1==dup2(pwrite, STDOUT_FILENO) || -1==dup2(pread, STDIN_FILENO)) {
		notice_err("dup2 failed");
		return TRUE;
	}
	close(pwrite); close(pread);
	peername = name;
	notice("%s: connected\n", peername);
	sendline("#Hello DWUN-" VERSION "\n");
	if (user_disabled("%unknown", authfile)) {
		sendline("Admin has disabled new connections\n");
		notice_debug("%s: %%unknown disabled\n", peername);
	} else {
		(void)inputhandler(authfile);
	}
	notice("%s: closing\n", peername);
	close(STDOUT_FILENO); close(STDIN_FILENO);
#ifndef DEBUG_GENERAL
	shutdown(stream->sd, 2); /* Just to make sure :) */
#endif
	(void)sclose(stream);
	return FALSE;
}

bool sig_ign(int sig)
{
	int ret;
	struct sigaction cursig;
	memset(&cursig, 0, sizeof(struct sigaction));
	cursig.sa_handler = SIG_IGN;
	do {
		ret = sigaction(sig, &cursig, NULL);
	} while (ret==-1 && errno==EINTR);
	if (-1==ret) {
		notice_err("%s: ignoring signal %d failed", peername, sig);
		return TRUE;
	}
	return FALSE;
}


/* Main input handler functions
 *
 * inputhandler		- handles commands from user.
 * getcommand		- gets commands from user.
 * level2usergroup	- get the parse.h usergroup for the given level.
 * 			  (what groups of commands users at that level are
 * 			  allowed to use).
 * parse		- find the command for the given string.
 * invalid_command	- tells user their command was invalid.
 * sendhelp		- tells the user information about the commands they can
 * 			  use at their current level.
 */

/* The parent process handles us doing cDISCONNECT or cAUTH when unnecessary,
 * and returning without doing those. */
bool inputhandler(mmap_area authfile)
{
#define help_available() sendline("Enter HELP to see the available commands\n")
	/* Note: various functions assume that arg, user and victim_name are
	 * the same max length. (Search for "strcpy"). */
	char arg[MAXLINE];
	char user[MAXLINE]; /* undefined when level==AUTH_NONE */
	char victim_name[MAXLINE];
	pid_t victim_pid = -1;
	int level = AUTH_NONE;
	struct clientlist *list = NULL;

	while (1) {
		switch(getcommand(arg, MAXLINE, level)) {
	/* errors */
		case FATAL_TERM:
			free(list);
			return TRUE;
		case FATAL_TIMEOUT:
			sendline("Timeout waiting for message\n");
			free(list);
			return TRUE;
		case FATAL_SIGNAL:
			sendline("Terminated by signal\n");
			notice("%s: terminated by signal\n", peername);
			free(list);
			return TRUE;
		case COM_INVALID:
			invalid_command(NULL);
			help_available();
			break;
		case COM_DONT_WANT_ARG:
			invalid_command("no argument wanted");
			help_available();
			break;
		case COM_NEED_ARG:
			invalid_command("need an argument");
			help_available();
			break;
		case COM_TOOLONG:
			invalid_command("too long");
			help_available();
			break;
	/* misc, with arg */
		case COM_TIMEOUT:
			if (mystrtol(arg, &set.keep_alive_timeout)) {
				invalid_command("timeout out of range");
				break;
			}
			notice_debug("%s: changed timeout to %ld\n", peername,
					set.keep_alive_timeout);
			stream->in.timeout = set.keep_alive_timeout;
			sendline("OKAY\n");
			break;
	/* auth */
		case COM_USER:
			if (level!=AUTH_NONE && level!=AUTH_HALF) {
				sendcommand(cDISCONNECT, NULL);
				sendcommand(cDEAUTH, NULL);
			}
			level = AUTH_HALF;
			strcpy(user, arg);
			sendline("OKAY\n");
			break;
		case COM_PASS:
			if (level==AUTH_NONE) {
				memset(arg, 0, MAXLINE);
				invalid_command("need USER first");
				help_available();
				break;
			}
			if (authentication(authfile, user, arg)) {
				memset(arg, 0, MAXLINE);
				level = AUTH_NONE; /* start back at USER */
				break;
			}
			memset(arg, 0, MAXLINE);
			sendcommand(cAUTH, user);
			/* TODO: more AUTH_ stuff different depending on
			 * username. (Make configurable from rcfile.) */
			if (0==strcmp(user, "admin"))
				level = AUTH_ADMIN;
			else
				level = AUTH_NORMAL;
			sendline("OKAY\n");
			break;
	/* no argument */
		case COM_HELP:
			sendhelp(level);
			sendline("OKAY\n");
			break;
		case COM_QUIT:
			free(list);
			sendline("OKAY\n");
			return FALSE;
		case COM_ALIVE: /* Do-nothing: prevent keepalive timeout */
			sendline("OKAY\n");
			break;
		case COM_CONNECT:
			if (user_disabled("%con", authfile)) {
				invalid_command("admin has disabled CONNECT");
				notice_debug("%s: CONNECT disabled\n",peername);
				break;
			}
			getprint_CONNECT();
			notice_debug("%s: CONNECT\n", peername);
			sendline("OKAY\n");
			break;
		case COM_DISCONNECT:
			sendcommand(cDISCONNECT, NULL);
			notice_debug("%s: DISCONNECT\n", peername);
			sendline("OKAY\n");
			break;
		case COM_CLOSE:
			sendcommand(cDISCONNECT, NULL);
			sendcommand(cDEAUTH, NULL);
			level = AUTH_NONE;
			sendline("OKAY\n");
			break;
	/* admin */
		case COM_KILL:
			notice("%s: KILL '%s' attempt\n", peername, arg);
			/* Always update; if the pid has since exited we don't
			 * mind that we can't KILL them. */
			if (update_list(&list))
				break;
			if (get_victim(arg, victim_name, &victim_pid, &list))
				break;
			if (victim_pid > 0) {
				/* get_victim checks the pid is in the list */
				if (-1==kill(victim_pid, SIGTERM))
					sendline("KILL failed\n");
				else
					sendline("OKAY\n");
			} else {
				kill_user(victim_name, list);
				sendline("OKAY\n");
			}
			break;
		/* For LOCK and UNLOCK, don't update list if we already
		 * have it. This is so things like "LOCK 1234" works,
		 * even if 1234 has since left.
		 */
		case COM_LOCK: 
			notice("%s: LOCK '%s' attempt\n", peername, arg);
			if (get_victim(arg, victim_name, NULL, &list))
				break;
			authwrite(TRUE, authfile);
				lock(victim_name, authfile);
			authwrite(FALSE, authfile);
			sendline("OKAY\n");
			break;
		case COM_UNLOCK:
			notice("%s: UNLOCK '%s' attempt\n", peername, arg);
			if (get_victim(arg, victim_name, NULL, &list))
				break;
			authwrite(TRUE, authfile);
				unlock(victim_name, authfile);
			authwrite(FALSE, authfile);
			sendline("OKAY\n");
			break;
		case COM_ZAP:
			notice("%s: ZAP '%s' attempt\n", peername, arg);
			if (get_victim(arg, victim_name, NULL, &list))
				break;
			/* Unlike COM_KILL, we kill all clients belonging to
			 * a user even if a pid is given */
			if (update_list(&list))
				break;
			kill_user(victim_name, list);
			authwrite(TRUE, authfile);
				lock(victim_name, authfile);
			authwrite(FALSE, authfile);
			sendline("OKAY\n");
			break;
	/* admin, no arg */
		case COM_SHUTDOWN:
			notice("%s: SHUTDOWN attempt\n", peername);
			if (-1==kill(getppid(), SIGHUP)) {
				notice("%s: shutting down failed\n", peername);
				sendline("SHUTDOWN failed\n");
				break;
			}
			sendline("OKAY\n");
			break;
		case COM_RESTART:
			notice("%s: RESTART attempt\n", peername);
			sendcommand(cRESTART, NULL);
			sendline("OKAY\n");
			break;
		case COM_RECONNECT:
			notice("%s: RECONNECT attempt\n", peername);
			if (-1==kill(getppid(), SIGUSR2)) {
				notice("%s: RECONNECT failed\n", peername);
				sendline("RECONNECT failed\n");
				break;
			}
			sendline("OKAY\n");
			break;
		case COM_UPTIME:
			if (print_uptime())
				sendline("UPTIME failed\n");
			else
				sendline("OKAY\n");
			break;
		case COM_LIST:
			if (update_list(&list))
				break;
			if (print_list(list))
				sendline("LIST failed\n");
			else
				sendline("OKAY\n");
			break;
		case COM_STATUS:
			getprint_STATUS();
			sendline("OKAY\n");
			break;
		case COM_SHOWLOCKS:
			print_locks(authfile);
			sendline("OKAY\n");
			break;
		}
	} /* while (1) */
	free(list); return TRUE; /* Just to prevent compiler warning. */
}

int getcommand(char *arg, size_t maxargsize, int level)
{
	bool hasarg;
	char command[maxargsize];
	int com_len;
	int ret;
	char *sep;

	switch(srecv(stream, command, maxargsize)) {
	case SOK_FAILED: 
		notice_debug("%s: connection reset by peer\n", peername);
		return FATAL_TERM;
	case SOK_TIMEOUT: 
		notice("%s: timeout waiting for message\n", peername);
		return FATAL_TIMEOUT;
	case SOK_TOOLONG: 
		notice("%s: garbage received from peer\n", peername);
		if (discard(stream, arg, maxargsize)) {
			notice_debug("%s: partial message left in buffer\n",
					peername);
		}
		return COM_TOOLONG;
	}
	sep = strchr(command, ' ');
	if (sep) {
		hasarg = TRUE;
		com_len = sep - command;
		term_strncpy(arg, command+com_len+1, maxargsize);
		*(command+com_len) = '\0';
	} else {
		hasarg = FALSE;
		*arg = '\0';
	} 
	ret = parse(command, level, hasarg);
	if (ret==COM_PASS)
		memset(command, 0, MAXLINE); /* past the '\0' */
	return ret;
}

struct cmdlist **level2usergroup(int level)
{
	switch(level) {
	case AUTH_NORMAL: return normal_user;
	case AUTH_ADMIN : return admin_user;
	case AUTH_NONE  :
	case AUTH_HALF  : return unauthed_user;
	}
	return NULL;
}
	
int parse(const char *command, int level, bool hasarg)
{
	struct cmdlist **who = level2usergroup(level);
	int i, j;
	for (i=0; who[i]; ++i) {
		for (j=0; who[i][j].command; ++j) {
			if (0==strcasecmp(command, who[i][j].command)) {
				if (who[i][j].hasarg==hasarg)
					return who[i][j].tag;
				if (hasarg)
					return COM_DONT_WANT_ARG;
				else
					return COM_NEED_ARG;
			}
		}
	}
	return COM_INVALID;
}

void invalid_command(const char *reason)
{
	if (reason) {
		sendline("Invalid command: ");
		sendline(reason);
		sendline("\n");
	} else {
		sendline("Invalid command\n");
	}
	return;
}

/* XXX: should we include the trailing newline? */
bool sendhelp(int level)
{
	struct cmdlist **who = level2usergroup(level);
	int i, j;
	for (i=0; who[i]; ++i) {
		for (j=0; who[i][j].command; ++j) {
			if (who[i][j].help)
				sendline(who[i][j].help);
			else
				sendline(who[i][j].command);
			if (who[i][j+1].command)
				sendline(", ");
		}
		if (who[i])
			sendline("\n");
	}
#if 0
	sendline("\n");
#endif
	return FALSE;
}


/* Socket/Pipe functions
 *
 * sendnline		- send message of specified length to user.
 * sendline		- send message to user.
 * pipe_send		- send pipe_msg_buf to parent.
 * pipen_recv_wait	- receive data of the specified size from parent with
 * 			  no timeout.
 * pipe_recv_wait	- receive pipe_msg_buf from parent with no timeout.
 * sendcommand		- send command to parent.
 */

bool sendnline(const char *msg, size_t len)
{
	switch(ssendn(stream, msg, len)) {
	case SOK_SUCCESS: break;
	case SOK_FAILED:
		notice_debug("%s: sendline failed\n", peername);
      		return TRUE;
	case SOK_TIMEOUT: 
		notice_debug("%s: sendline timeout\n", peername);
		return TRUE;
	}
	return FALSE;
}

/* It's good that callers don't check the return status - that way if the
 * remote end sends us a load of commands and then tears down the connection,
 * we will still obey the commands. */
bool sendline(const char *msg)
{
	return sendnline(msg, strlen(msg));
}

bool pipe_send(struct pipe_msg_buf msg)
{
	struct timeval tv;

	tv.tv_sec = IPC_SEND_TIMEOUT_SECS;
	tv.tv_usec = 0;
	switch(sok_send(STDOUT_FILENO, &tv, &msg, sizeof(msg))) {
	case SOK_SUCCESS: break;
	case SOK_FAILED: 
		notice_debug("%s: pipe_send failed\n", peername);
		return TRUE;
	case SOK_TIMEOUT: 
		notice_debug("%s: pipe_send timeout\n", peername);
		return TRUE;
	}
	return FALSE;
}

bool pipe_recv_wait(struct pipe_msg_buf *msg)
{
	return pipen_recv_wait(msg, sizeof(struct pipe_msg_buf));
}

bool pipen_recv_wait(void *msg, size_t size)
{
	if (sok_recv(STDIN_FILENO, NULL, msg, size)!=SOK_SUCCESS) {
		notice_debug("%s: pipe_recv failed\n", peername);
		return TRUE;
	}
	return FALSE;
}

/* username only needed with cAUTH, otherwise just use NULL for user. */
bool sendcommand(int command, const char *user)
{
	struct pipe_msg_buf msg;
	msg.from = ourpid;
	msg.command = command;
	if (user)
		term_strncpy(msg.username, user, USER_LEN);
	else
		msg.username[0] = '\0';
	if (pipe_send(msg)) {
		/* Since we're failing to check the return status of
		 * sendcommand, we're probably stuffed now. */
		return TRUE;
	}
	return FALSE;
}


/* List/status functions
 * 
 * pid2user		- get the username for the client with the given pid.
 * get_victim		- get username and/or pid for the given argument.
 * string_isdigit	- checks that a string contains only digits.
 * update_list		- free's list and gets new one.
 * kill_user		- kill all pids belonging to the given user.
 * print_list		- output for the LIST command.
 * print_uptime		- output for the UPTIME command.
 * getprint		
 * 	getprint_CONNECT	- tell parent the user is CONNECTing and
 * 				  print status.
 * 	getprint_STATUS		- tell user connection status.
 */

bool pid2user(char *user, pid_t pid, struct clientlist *list)
{
	while (list->pid!=-1) {
		if (list->pid==pid) {
			term_strncpy(user, list->username, MAXLINE);
			return FALSE;
		}
		++list;
	}
	return TRUE;
}

/* victim_pid may be NULL, in which case only victim_name is set.
 *
 * If arg contains a username, *victim_pid==-1.
 * If it contains a pid, *victim_pid > 0. (If the pid doesn't exist in the list
 * we return TRUE).
 *
 * If *list is NULL and arg is a pid, we get the list. That is the only
 * reason we're using "**list" instead of "*list".
 */
bool get_victim(char *arg, char *victim_name, pid_t *victim_pid,
		struct clientlist **list)
{
	u_long pid;
#if 1
	if (was_a_username(arg)) {
		sendline("Usernames should no longer have % added to the "
				"start\n");
#if 0
		strcpy(victim_name, arg+1);
		if (victim_pid)
			*victim_pid = -1;
		return FALSE;
#endif
		return TRUE;
	}
#endif
	if (is_a_username(arg[0])) {
		strcpy(victim_name, arg);
		if (victim_pid)
			*victim_pid = -1;
		return FALSE;
	}

	if (string_isdigit((unsigned char *)arg)) {
		invalid_command("not a username or PID");
		return TRUE;
	}
	if (mystrtoul(arg, &pid)) {
		invalid_command("number out of range");
		return TRUE;
	}
	if (victim_pid)
		*victim_pid = (pid_t)pid;
	if (!*list) {
		if (update_list(list))
			return TRUE;
	}
	if (pid2user(victim_name, pid, *list)) {
		invalid_command("no client has that PID");
		return TRUE;
	}
	return FALSE;
}

bool string_isdigit(const unsigned char *s)
{
	while (*s) {
		if (!isdigit(*s))
			return TRUE;
		++s;
	}
	return FALSE;
}

/* Free's old list, gets new one */
bool update_list(struct clientlist **list)
{
	struct pipe_msg_buf msg;
	int numclients;
	size_t size = sizeof(struct clientlist);
	
	sendcommand(cLIST, NULL);
	if (pipe_recv_wait(&msg))
		goto bad;
	numclients = msg.command;
	
	free(*list);
	if (!(*list = malloc( (numclients+1) * size ))) {
		notice_err("%s: malloc failed", peername);
		goto bad;
	}
	if (pipen_recv_wait(*list, numclients * size)) {
		free(*list); *list = NULL;
		goto bad;
	}
	((*list) + numclients)->pid = -1;
	return FALSE;
bad:
	sendline("Internal server error\n"); /* XXX */
	return TRUE;
}

/* Kills all the pid's belonging to the given user. Update list before calling
 * this. */
void kill_user(char *user, struct clientlist *list)
{
	u_long succeeded = 0;
	u_long failed = 0;
	char buf[MAXLINE];

	if (!list) {
		notice_debug("kill_user: list==NULL\n");
		return;
	}
	/* :P */
#define KILL_THOSE(x)							\
	do {								\
		while (list->pid!=-1) {					\
			if ((x) && list->pid!=ourpid) {			\
				if (-1==kill(list->pid, SIGTERM))	\
					++failed;			\
				else					\
					++succeeded;			\
			}						\
			++list;						\
		}							\
	} while(0)

	if (0==strcmp(user, "%con")) {
		KILL_THOSE(list->connected==TRUE);
	} else if (0==strcmp(user, "%all")) {
		KILL_THOSE(TRUE);
	} else {
		KILL_THOSE(0==strcmp(user, list->username));
	}
	snprintf(buf, MAXLINE, "KILL %lu success: %lu failed\n",
			succeeded, failed);
	sendline(buf);
	return;
}

bool print_list(struct clientlist *list)
{
	char buf[MAXLINE];
	while (list->pid!=-1) {
		/* XXX: should we show ourselves? */
		if (list->pid!=ourpid) { 
			snprintf(buf, MAXLINE, "%lu %s %s %s\n",
					(u_long)list->pid,
					list->peername,
					list->username,
					list->connected?"CON":"!CON");
			sendline(buf);
		}
		++list;
	}
	return FALSE;
}

bool print_uptime(void)
{
	char buf[MAXLINE];
	struct tm *timep;
	time_t connect_time;
	time_t diff;

	sendcommand(cUPTIME, NULL);
	if (pipen_recv_wait(&connect_time, sizeof(connect_time)))
		return TRUE;
	sendline("Connected for: ");
	if (connect_time==-1) {
		sendline("not connected\n");
		return FALSE;
	}
	diff = time(NULL) - connect_time; 
	timep = gmtime(&diff);
	snprintf(buf, MAXLINE, "%u DAYS, %u HOURS, %u MINS, %u SECS\n",
			/* time_t starts at 1970, tm_year starts at 1900 */
			timep->tm_yday * (timep->tm_year - 70 + 1),
			timep->tm_hour,
			timep->tm_min,
			timep->tm_sec);
	sendline(buf);
	return FALSE;
}

/* This function is misnamed.

   We do above:
	#define getprint_STATUS() getprint(TRUE)
	#define getprint_CONNECT() getprint(FALSE)
*/
void getprint(bool with_STATUS)
{
	struct pipe_msg_buf msg;

	if (with_STATUS)
		sendcommand(cSTATUS, NULL);
	else
		sendcommand(cCONNECT, NULL);
	if (pipe_recv_wait(&msg))
		return;
	switch(msg.command) {
	case CONS_DISCONNECTED:
		if (with_STATUS)
			sendline("#Disconnected\n");
		/* else parent will send "#Connecting" shortly */
	        break;
	case CONS_DISCONNECTING:
		sendline("#Disconnecting\n");
		break;
#if 0 /* "fatal admin" is no longer implemented. If we do re-implement it, say
	 something like "#Connecting paused" instead so clients don't exit */
    	case CONS_FAILED:
		sendline("#Connect failed\n");
		break;
#endif
	case CONS_CONNECTING: 
		sendline("#Connecting\n");
		break;
	case CONS_CONNECTED: 
		sendline("#Connected\n");
		break;
	}
	return;
}


/* Account functions
 *
 * authentication	- check user/pass is valid.
 * user_disabled	- is the user account locked or non-existant?
 * setlock
 * 	lock		- lock an account.
 * 	unlock		- unlock an account.
 * authwrite		- set the authfile area writeable or read-only.
 * print_locks		- tells user the locked accounts.
 */

bool authentication(mmap_area authfile, char *user, char *pass)
{
	struct account ac;
	if (strlen(user) > USER_LEN) {
		/* We need to send the username via pipe_msg_buf to parent. */
		invalid_command("username is too long");
		notice("%s: username too long\n", peername);
		return TRUE;
	}
	/* Ideally, we'd do auth_checkpass() even if auth_getaccount() failed
	 * to prevent an attacker from detecting why authentication failed via
	 * timing. However, if auth_getaccount() failed, ac.crypass is
	 * undefined.
	 */
	if (user_disabled("%all", authfile)) {
		invalid_command("admin has disabled all accounts");
		notice_debug("%s: %%all disabled\n", peername);
		return TRUE;
	} else if (	user[0]=='%'				||
			auth_getaccount(user, authfile, &ac)	||
			auth_checkpass(pass, ac.crypass)) {
		sendline("Invalid username/password\n");
		notice("%s: invalid auth\n", peername);
		return TRUE;
	} else if (*ac.lock=='!') {
		invalid_command("admin has disabled your account");
		notice("%s: invalid auth: account locked\n", peername);
		return TRUE;
	}
	notice("%s: %s authenticated\n", peername, user);
	return FALSE;
}

bool user_disabled(char *user, mmap_area authfile)
{
	struct account ac;
	if (auth_getaccount(user, authfile, &ac))
		return TRUE; /* account does not exist */
	if (*ac.lock=='!')
		return TRUE; /* account is locked */
	return FALSE;
}

bool lock(char *user, mmap_area authfile)
{
	struct account ac;
	if (auth_getaccount(user, authfile, &ac)) {
		invalid_command("account not found");
		return TRUE;
	}
	if (*ac.lock=='!') {
		invalid_command("account already locked");
		return TRUE;
	}
	*ac.lock = '!';
	return FALSE;
}

bool unlock(char *user, mmap_area authfile)
{
	struct account ac;
	if (auth_getaccount(user, authfile, &ac)) {
		invalid_command("account not found");
		return TRUE;
	}
	if (*ac.lock==':') {
		invalid_command("account not locked");
		return TRUE;
	}
	*ac.lock = ':';
	return FALSE;
}

/* encapsulate calls that need to write to authfile with
 * authwrite(TRUE, authfile);
 * 	call_that_writes_to(authfile);
 * authwrite(FALSE, authfile);
 */
bool authwrite(bool writeable, mmap_area authfile)
{
	if (writeable_authfile(writeable, authfile)) {
		notice_err("%s: setting shared mem %s failed", peername,
				writeable?"writeable":"read only");
		return TRUE;
	}
	return FALSE;
}
	
void print_locks(mmap_area curr)
{
	struct account ac;
	curr = auth_getline(&ac, curr);
	while (curr.data) {
		if (*ac.lock=='!') {
			sendnline(ac.user.data, ac.user.len);
			sendline(" ");
		}
		curr = auth_getline(&ac, curr);
	}
	sendline("\n");
	return;
}
