/*
   Copyright (C) 2004 by James Gregory
   Part of the GalaxyHack project
 
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License.
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY.
 
   See the COPYING file for more details.
*/

#include "Menu_Base.h"
#include "Globals.h"
#include "Group.h"
#include "ForceSelect.h"

//static member
SideInfoType SidePU::infoType = SIT_Number;


PopupMenu::PopupMenu(int ix, int iy, int iParentID, int flags):
Menu_Base(ix, iy, iParentID, flags) {}

bool PopupMenu::MouseD(Uint8 button, Uint16 x, Uint16 y) {
	bool ret = Menu_Base::MouseD(button, x, y);

	if (ret != true && !cantClose)
		closed = true;

	return ret;
}

////

SidePU::SidePU(int iMySide, int iMyGroup, WindowChoice iChoiceType, int iWhichSaveGroup):
PopupMenu(0, 0, none_constant, 0), mySide(iMySide), myGroup(iMyGroup), choiceType(iChoiceType), whichSaveGroup(iWhichSaveGroup) {
	MessageWindows(WC_CloseSidePU, 0, 0, all_constant, all_constant);

	transparent = true;

	rect = screenRect;

	Update();
}

void SidePU::Update() {
	ClearItems();

	MenuItem tempItem;
	newItemx = colorRect.w + (bigBorderSize << 1);
	newItemy = bigBorderSize;
	tempItem.rect.h = lineGap * 3;

	tempItem.desc = sides[mySide].name;

	switch (choiceType) {
	case FS_EditGroup:
		tempItem.desc+= " - Edit group:";
		break;

	case FS_DeleteGroup:
		tempItem.desc += " - Delete group:";
		break;

	case FS_ChangeParentCap:
		tempItem.desc += " - Select parent capital ship:";
		break;

	case FS_ChooseGSaveGroup: {
		char descStr[60];
		sprintf(descStr, "%s - Select global saved group %d:", sides[mySide].name.c_str(), whichSaveGroup);
		tempItem.desc = descStr;
		break;
	}

	case FS_ChooseSaveGroup: {
		char descStr[80];
		sprintf(descStr, "%s - Select saved group %d for group %d:", sides[mySide].name.c_str(), whichSaveGroup, myGroup+1);
		tempItem.desc = descStr;
		break;
	}
	}

	tempItem.choice = WC_Nothing;
	AddItem(tempItem);

	tempItem.choice = choiceType;
	tempItem.rect.h = (lineGap << 1) + bigBorderSize;

	int i = 0;
	for (; i != sides[mySide].groups.size(); ++i) {
		tempItem.flags = 0;
		string info;
		
		int numberLeft = sides[mySide].groups[i].GetUnitsLeft();
		if (!numberLeft)
			tempItem.flags = MFLAG_FADED;
		
		switch (infoType)
		{
		case SIT_Number:			
			if (numberLeft)
				info.insert(info.end(), numberLeft, '*');
			else
				info = "Destroyed";
			break;
		
		case SIT_Type:
			info = uTypeToString[sides[mySide].groups[i].GetType()];
			break;
		
		case SIT_Script:
			info = sides[mySide].groups[i].GetAIFilename();
			break;
		
		case SIT_Points:{
			char points[20];
			sprintf(points, "Points: %d", sides[mySide].groups[i].GetPointsValue());
			info = points;
			}
			break;
		}

		//substring because may be too long
		string name = sides[mySide].groups[i].GetUnitName().substr(0, 14);

		char tempStr[7];
		//+1 because we don't say "Group 0"
		sprintf(tempStr, "%d-%d: ", mySide + 1, i + 1);

		tempItem.desc = tempStr + name + '\n' + info;
		tempItem.parem = i;
		AddItem(tempItem);

		if ((i + 1) % 12 == 0) {
			newItemy = bigBorderSize + (lineGap * 3);
			newItemx += menuItemWidth;
		}
	}

	//no border
}

void SidePU::DrawSelf() {
	PopupMenu::DrawSelf();

	colorRect.x = rect.x + bigBorderSize;
	colorRect.y = rect.y + bigBorderSize;
	JSDL.BltFill(colorRect, sides[mySide].color);
}

bool SidePU::MouseD(Uint8 button, Uint16 x, Uint16 y) {
	bool ret = PopupMenu::MouseD(button, x, y);
	
	//always close whether or not they made a selection, unless deleting groups
	if ((currentSelection.choiceType == MCT_None || currentSelection.choice != FS_DeleteGroup) && button == SDL_BUTTON_LEFT)
		closed = true;
	
	//if we've just deleted a group then we need to update what is highlighted
	if (currentSelection.choice == FS_DeleteGroup) {
		Update();
		MouseM(0, x, y);
	}
		
	return ret;
}

bool SidePU::SwitchOnChoice(Uint16 x, Uint16 y) {
	if (currentSelection.choiceType == MCT_LeftCursor) {
		switch (currentSelection.choice) {
		case RTS_GroupStats:
			SelectGroup(mySide, currentSelection.parem, RTS_GroupStats);
			break;

		case FS_EditGroup:
			myWindows.push_back(GenWindow(rect.x, rect.y, FS_EditGroup, currentSelection.parem, 0, WFLAG_TILED));
			break;

		case FS_DeleteGroup:
			ForceSelect::DeleteGroupCheck(currentSelection.parem);
			break;

		case FS_ChooseGSaveGroup:
			sides[0].saveGroups[whichSaveGroup].y = currentSelection.parem;
			break;

		case FS_ChooseSaveGroup: {
				CoordsInt tempCoords = {0, currentSelection.parem};
				sides[0].groups[myGroup].ChangeSaveGroup(whichSaveGroup, tempCoords);
			}
			break;
		}
	} else if (currentSelection.choiceType == MCT_RightCursor && gsCurrent != GST_ForceSelect) {
		myWindows.push_back(GenWindow(x, y, RTS_InfoChoicePU, mySide, currentSelection.parem, 1, myID, 0));
	}

	return false;
}

void SidePU::WinMessage(WindowChoice theMessage, int paremOne, int paremTwo, int targetID, int sourceID) {
GenWindow_Base::WinMessage(theMessage, paremOne, paremTwo, targetID, sourceID);

	if (targetID == all_constant || targetID == myID) {

		switch (theMessage) {
		case WC_CloseSidePU:
			closed = true;
			break;
		}
	}
}




