/*	$NetBSD: lpt_pccreg.h,v 1.4 2008/04/28 20:23:29 martin Exp $ */

/*-
 * Copyright (c) 1999 The NetBSD Foundation, Inc.
 * All rights reserved.
 *
 * This code is derived from software contributed to The NetBSD Foundation
 * by Steve C. Woodford.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE NETBSD FOUNDATION, INC. AND CONTRIBUTORS
 * ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED
 * TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE FOUNDATION OR CONTRIBUTORS
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

/*
 * MVME147 Parallel Port Register Definitions
 */
#ifndef _MVME68K_LPT_PCCREG_H
#define _MVME68K_LPT_PCCREG_H

/*
 * The mvme147's PCC chip has two status/control registers for the
 * printer port:
 *
 * PCCREG_PRNT_INTR_CTRL Printer interrupt control register
 *                0 - 2 Interrupt Level
 *                  3   Interrupt Enable
 *                  4   ACK Polarity. If set, falling edge of ACK generates
 *                      the interrupt. If clear, rising edge of ACK generates
 *                      the interrupt.
 *                  5   Indicates an ACK interrupt in progress. Cleared by
 *                      writing a one, or disabling lpt interrupts.
 *                  6   Indicates a FAULT interrupt. Set on falling edge
 *                      of printer's fault signal. Cleared by writing a one.
 *                  7   Printer Interrupt in progress. Basically just the
 *                      logical OR of bits 5 and 6.
 */
#define	LPI_ENABLE	(1 << 3)
#define	LPI_ACKPOL	(1 << 4)
#define	LPI_ACKINT	(1 << 5)
#define	LPI_FAULTINT	(1 << 6)
#define	LPI_INTERRUPT	(1 << 7)


/*
 * PCCREG_PRNT_CONTROL  Printer Control Register
 *                  0   Selects auto or manual strobe mode. When low, strobe
 *                      is automatically generated by a write to the printer
 *                      data register. When set, strobe must be generated
 *                      manually using bit 2 of this register.
 *                  1   Controls strobe timing in auto mode. When low, strobe
 *                      time is 6.4uS. When high, strobe time is 1.6uS.
 *                  2   Controls strobe in manual mode.
 *                  3   Control Input Prime signal. When set, Input Prime
 *                      is activated.
 *
 * Two other registers which are not addressed via the global PCC structure,
 * live at 0xfffe2800. This address is virtualised and passed to the driver
 * in the pcc_attach_args structure:
 */
#define	LPC_STROBE_MODE	(1 << 0)
#define	LPC_FAST_STROBE	(1 << 1)
#define	LPC_STROBE	(1 << 2)
#define	LPC_INPUT_PRIME	(1 << 3)

#define lpt_control_read()	pcc_reg_read(sys_pcc, PCCREG_PRNT_CONTROL)
#define lpt_control_write(v)	pcc_reg_write(sys_pcc, PCCREG_PRNT_CONTROL, v)

/*
 * Data and status registers appear at the same offset.
 * Write to access the data register. Read to access the status register.
 */
#define LPREG_DATA	0x00	/* Write only data register */
#define LPREG_STATUS	0x00	/* Read only status register */

#define LPREG_SIZE	0x1

/*
 * Access macros for the status register
 */
#define	LPS_BUSY	(1 << 3)
#define	LPS_PAPER_EMPTY	(1 << 4)
#define	LPS_SELECT	(1 << 5)
#define	LPS_FAULT	(1 << 6)
#define	LPS_ACK		(1 << 7)

#define lpt_data_write(sc,v)	\
	    bus_space_write_1((sc)->sc_bust, (sc)->sc_bush, LPREG_DATA, (v))
#define lpt_status_read(sc)	\
	    bus_space_read_1((sc)->sc_bust, (sc)->sc_bush, LPREG_STATUS)

#endif /* _MVME68K_LPT_PCCREG_H */
