# zwiki page hierarchy functionality
# original code by Ken Manheimer

from __future__ import nested_scopes
import string, re
from string import join
from types import *
from urllib import quote, unquote
import Acquisition
import Permissions
import Globals
from AccessControl import ClassSecurityInfo
from App.Common import absattr
from Utils import flatten, DLOG
from Defaults import PAGE_METATYPE
from Regexps import bracketedexpr

#XXX 20030917 time for the long-awaited, first-ever, Parents.py cleanup!!
#
# overview: we have
#
# ParentsSupport       mixin providing page hierarchy management for ZWikiPage
# ui related:
# contents/map         get html showing hierarchy in contentspage template
# context              get html showing current page + ancestors & maybe siblings
# offspring            get html showing current page + offspring
# children             get html listing current page's immediate children
# renderNesting        format a nest list as a html outline of links
# non-ui:
# reparent             change page parent(s)
# offspringAsList      get flat list of offspring names
# offspringIdsAsList   get flat list of offspring ids
# ancestorsAsList      get flat list of ancestor names
# wikiNesting          get a nested list including whole hierarchy
# offspringNesting     get a nested list including offspring (possibly of multiple pages)
# childrenNesting      get a nested list including immediate children
# ancestorsNesting     get a nested list including ancestors
# ancestorsAndSiblingsNesting get a nested list including ancestors & siblings
# ancestorsAndChildrenNesting get a nested list including ancestors, siblings & children
# getWikiParentInfo    get a WikiParentInfo object with all parentage in easy-to-process form
#
# descend_ancestors    some kind of helper for ancestors nestings
# deepappend           append a page to the bottom of a nested list
#
# (no more:)
# WikiNesting          generates hierarchy structures (nested lists) from a wiki folder
# __init__             set container, call save_nesting_info
# save_nesting_info    save nesting-related info for all pages in folder (or use catalog)
# get_map              get nested list for all pages
# get_ancestors        get nested list for ancestors of a page
# get_up_and_back      get nested list for ancestors and children of a page
# get_offspring        get nested list for offspring of one or more pages


class ParentsSupport:
    """
    This mix-in class encapsulates (most of) ZWikiPage's page hierarchy
    functionality.

    A page can name one or more others as it's parents; based on this we
    calculate the overall hierarchy in the wiki and can tell a page it's
    ancestors, offspring, siblings etc. 
    
    glossary --
    ancestors, offspring: all pages above and below in the hierarchy
    parents, children: the immediate ancestors and immediate offspring
    siblings: pages with the same parent
    nesting: a nested list representing a piece of the hierarchy
    """
    parents = []
    def addParent(self,parent):
        if parent:
            parent = string.strip(parent)
            if parent and not parent in self.parents:
                self.parents.append(parent)
    def removeParent(self,parent):
        try: self.parents.remove(parent)
        except ValueError:
            DLOG("failed to remove %s from %s's parents (%s)" \
                 % (parent,self.getId(),self.parents))

    _properties=(
        {'id':'parents', 'type': 'lines', 'mode': 'w'},
        )
    security = ClassSecurityInfo()

    # UI related methods, returning HTML:
    # XXX see also the methods in UI.py.. be more like them

    security.declareProtected(Permissions.View, 'contents')
    def contents(self, REQUEST=None, here=None):
        """
        Show the hierarchy of the entire wiki, using the contentspage template.

        Includes all the branches in the wiki - from the possibly multiple
        roots - and all singletons, ie those without parents or children.
        The page named by here will be highlighted with "you are here", 
        or the current page.

        """
        #nesting = WikiNesting(self.folder()).get_map()
        nesting = self.wikiNesting()
        singletons = []
        combos = []
        baseurl = self.wiki_url()
        for i in nesting:
            if type(i) == StringType:
                #try:
                #    # XXX poor caching ?
                #    linktitle = self.folder()[i].linkTitle()
                #except:
                linktitle = ''
                singletons.append(\
                    '<a href="%s/%s" name="%s" title="%s">%s</a>'\
                    % (baseurl, self.canonicalIdFrom(i), quote(i),
                       linktitle,
                       i))
            else:
                combos.append(i)
        here = (here and unquote(here)) or self.pageName()
        return self.contentspage(
            self.renderNesting(combos, here),
            #self.renderNestingX(combos, here),  # dan's skinnable version
            singletons,
            REQUEST=REQUEST)

    map = contents # backwards compatibility

    security.declareProtected(Permissions.View, 'context')
    def context(self, REQUEST=None, with_siblings=0, enlarge_current=0):
        """
        Return HTML showing this page's ancestors and siblings.

        XXX how can we use a page template for this ? macro ?
        If not should probably at least use a dtml method template
        
        """
        # get the nesting structure
        here = self.pageName()
        if with_siblings:
            #nesting = WikiNesting(self.folder()).get_up_and_back(here)
            nesting = self.ancestorsAndSiblingsNesting()
        else:
            # why does the above require a nesting and not this one ?
            # nesting = self.get_ancestors()
            #nesting = WikiNesting(self.folder()).get_ancestors(here,self)
            nesting = self.ancestorsNesting()
            # XXX looks like cruft
            if (len(nesting) == 0  or
                (len(nesting) == 1 and len(nesting[0]) == 1)) and not enlarge_current:
                return "&nbsp;"

        # format and link it
        # backwards compatibility: in case of an old editform template
        # which shows context, include the new page name at the bottom (unlinked)
        if REQUEST and REQUEST.has_key('page') and REQUEST['page'] is not here:
            here = REQUEST['page']
            nesting = deepappend(nesting, here)
            suppress_hyperlink=1
        else:
            suppress_hyperlink=0
        hierarchy = self.renderNesting(
            nesting, here,
            enlarge_current=enlarge_current,
            suppress_hyperlink=suppress_hyperlink)
        # special case: if parent seems to be missing, reset
        if hierarchy == '<ul>\n</ul>':
            self.parents = []
            self.index_object()
            hierarchy = self.renderNesting(
                nesting, here, 
                enlarge_current=enlarge_current,
                suppress_hyperlink=suppress_hyperlink)
        # if a SiteMap page exists, point the contents link there
        contentsurl = self.contentsUrl()
        contentslink = \
          '<a href="%s" title="show wiki contents" accesskey="c">%s contents</a>' \
          % (contentsurl, self.folder().title)
        #return '<small><ul>%s\n%s\n</ul></small>' % (contentslink,hierarchy)
        #XXX try no contents link in context
        return '<small>%s\n</small>' % (hierarchy)

    security.declareProtected(Permissions.View, 'offspring')
    def offspring(self, REQUEST=None, info=None, exclude_self=0):
        """
        Return HTML displaying all my offspring.
        """
        here = self.pageName()
        return self.renderNesting(
            self.offspringNesting([here],info=info),
            here,
            suppress_current=exclude_self)

    security.declareProtected(Permissions.View, 'children')
    def children(self):
        """
        Return HTML presenting a list of my immediate children, if any.
        """
        children = self.childrenAsList()
        if children:
            return self.renderNesting(children)
        else:
            return ''

    security.declareProtected(Permissions.View, 'subtopics')
    def subtopics(self,deep=0):
        """
        Render subtopics links for this page.
        """
        subtopics = ((deep and self.offspring(exclude_self=1)) or
                     self.children())
        if subtopics:
            return '\n\n<a name="subtopics"><br /></a>\n<p><table id="subtopicspanel"><tr><td>subtopics of this page:\n%s</td></tr></table>' \
                   % (subtopics)
        else: return ''

    security.declareProtected(Permissions.View, 'navlinks')
    def navlinks(self,contents=0):
        """
        Render next/prev/up/contents links for this page.
        """
        none = 'none'
        t = ''
        hierarchy = self.wikiNesting() 
        prev, next = self.previousPage(hierarchy), self.nextPage(hierarchy)
        if prev: prev = self.renderLink('['+prev+']',access_key='P')
        else: prev = none
        if next: next = self.renderLink('['+next+']',access_key='N')
        else: next = none
        t += '<span class="accesskey">n</span>ext:&nbsp;%s <span class="accesskey">p</span>revious:&nbsp;%s' \
             % (next,prev) # Info style!
        t += ' <span class="accesskey">u</span>p:&nbsp;%s' \
             % ((self.parents and self.renderLink('['+self.parents[0]+']',
                                                  access_key='U'))
                or none)
        if contents:
            contentsurl = self.contentsUrl()
            contentslink = \
              '<a href="%s" title="show wiki contents" accesskey="c"><span class="accesskey">c</span>ontents</a>'\
              % (contentsurl)
            t += ' contents:&nbsp;%s' % contentslink
        return t

    security.declareProtected(Permissions.View, 'renderNesting')
    def renderNesting(self, nesting, here=None, enlarge_current=0,
                      suppress_hyperlink=0, suppress_current=0,
                      did=None, got=None, indent=''):
        """
        Format a nesting structure as HTML unordered lists of wiki links.

        - nesting is the nesting to be formatted
        - here is the page name to highlight with "you are here", if any
        - if enlarge_current is true, here will be enlarged instead
        - if suppress_hyperlink is true, here will not be linked
          (backwards compatibility for old editforms)
        - if suppress_current is true, here will not be shown at all
        - did, got & indent are for recursion, callers should not use
        
        """
        if suppress_current and nesting[0] == here: # a single childless page
            return ''
        if did is None: did = []
        if got is None:
            got = ['<ul>']
            recursing = 0
        else:
            recursing = 1
        for n in nesting:
            if type(n) == ListType:
                if not (n[0]==here and suppress_current): #XXX temp
                    got.append('%s <li>[%s]' % (indent,n[0]))
                if len(n) > 1:
                    if not (n[0]==here and suppress_current): #XXX temp
                        got.append("<ul>")
                    for i in n[1:]:
                        if type(i) == ListType:
                            got = self.renderNesting(
                                [i],here,did=did,got=got,indent=indent+' ')
                        else:
                            got.append('%s <li>[%s]' % (indent,i))
                    if not (n[0]==here and suppress_current): #XXX temp
                        got.append("</ul>")
                else:
                    got[-1] += ' ...' # a parent whose children were omitted
            else:
                got.append('%s <li>[%s]' % (indent,n))
        if recursing: return got

        # finish up, do pretty printing options and wiki links
        got.append("</ul>")
        t = join(got, "\n")
        if here:
            if enlarge_current:
                t = re.sub(r'(\[%s\])' % re.escape(here),
                           r'<big><big><big><big><strong>\1</strong></big></big></big></big>',
                           t)
                # XXX temporary kludge.. assume we are in the page header here
                t = re.sub(r'(\[%s\])' % re.escape(self.pageName()),
                           '<a href="%s/backlinks" title="show backlinks for this page" accesskey="l">%s</a>' \
                           % (self.page_url(),self.pageName()),
                           t)
            else:
                t = re.sub(r'(\[%s\])' % re.escape(here),
                           r'\1 <b><-- You are here.</b>',t)
            if suppress_hyperlink:
                t = re.sub(r'(\[%s\])' % re.escape(here), r'!\1', t)
        #t = self.renderLinksIn(t) # too expensive for now.. do it on the cheap
        wikiurl = self.wiki_url()
        def quicklink(match):
            page = match.group(1)
            return '<a href="%s/%s" name="%s">%s</a>' \
                   % (wikiurl,self.canonicalIdFrom(page),quote(page),page)
        t = re.sub(bracketedexpr,quicklink,t)
        return t

    # dan mcmullen's skinnable versions of these, used by the wikipagex skin
    security.declareProtected(Permissions.View, 'contextX')
    def contextX(self, REQUEST=None, with_siblings=0):
        """
        Return HTML showing this page's ancestors and siblings.
        """
        # get the nesting structure
        here = self.pageName()
        if with_siblings:
            #nesting = WikiNesting(self.folder()).get_up_and_back(here)
            nesting = self.ancestorsAndChildrenNesting()
        else:
            # why does the above require a nesting and not this one ?
            # nesting = self.get_ancestors()
            #nesting = WikiNesting(self.folder()).get_ancestors(here,self)
            nesting = self.ancestorsNesting()
            # XXX looks like cruft
            #if (len(nesting) == 0  or
            #    (len(nesting) == 1 and len(nesting[0]) == 1)):
            #    return {'contentsUrl':self.contentsUrl(),
            #        'hierarchy':{}}

        # format and link it
        # backwards compatibility: in case of an old editform template
        # which shows context, include the new page name at the bottom (unlinked)
        if REQUEST.has_key('page') and REQUEST['page'] is not here:
            here = REQUEST['page']
            nesting = deepappend(nesting, here)
            suppress_hyperlink=1
        else:
            suppress_hyperlink=0

        hierarchy = self.renderNestingX( nesting, here )
            
        # special case: if parent seems to be missing, reset
        if len(hierarchy) == 2 :
            self.parents = []
            self.index_object()
            hierarchy = self.renderNestingX(
                nesting, here)
                
        # if a SiteMap page exists, point the contents link there
        return {'contentsUrl':self.contentsUrl(), 'hierarchy':hierarchy}

    security.declareProtected(Permissions.View, 'renderNestingX')
    def renderNestingX(self, nesting, here=None, suppress_current=0,
                      did=None, got=None, indent=''):
        """
        Unpack a nesting structure into a list.

        - nesting is the nesting to be formatted
        - here is the page name to highlight with "you are here", if any
        - if suppress_current is true, here will not be shown at all
        - did, got & indent are for recursion, callers should not use
        
        """
        if suppress_current and nesting[0] == here: # a single childless page
            return []
        if did is None: did = []
        if got is None:
            got = [ {'type':'+'} ]
            recursing = 0
        else:
            recursing = 1
        for n in nesting:
            if type(n) == ListType:
                if not (n[0]==here and suppress_current): #XXX temp
                    t = (n[0]==here and '=!' or '=')
                    got.append( {'type':t, 'page':str(n[0])} )
                if len(n) > 1:
                    if not (n[0]==here and suppress_current): #XXX temp
                        got.append( {'type':'+'} )
                    for i in n[1:]:
                        if type(i) == ListType:
                            got = self.renderNestingX(
                                [i],here,did=did,got=got,indent=indent+' ')
                        else:
                            t = (i==here and '=!' or '=')
                            got.append( {'type':t, 'page':str(i)} )
                    if not (n[0]==here and suppress_current): #XXX temp
                        got.append( {'type':'-'} )
                else:
                    
                    got[-1]['type'] += '.' # a parent whose children were omitted
            else:
                t = (n==here and '=!' or '=')
                got.append( {'type':t, 'page':str(n)} )
        if recursing: return got

        # finish up, do pretty printing options and wiki links
        got.append( {'type':'-'} )

        wikiurl = self.wiki_url()
        for g in got :
            if '=' in g['type'] :
                g['href'] = wikiurl + '/' + self.canonicalIdFrom(g['page'])
                g['name'] = quote(g['page'])
                
        return got

    # non-UI methods:

    def navlinksEnabled(self):
        """
        Should this page display extra navigation links ?

        True when there is a true show_navlinks page or folder property
        and user is in full UI mode.
        """
        return (getattr(self,'show_navlinks',0) and
                self.REQUEST.get('zwiki_displaymode',0) == 'full')

    def subtopicsEnabled(self):
        """
        Decide in a complicated way if this page should display it's subtopics.

        First, the folder must have a true show_subtopics property (can
        acquire). Then, we look for another show_subtopics property in:
        - REQUEST
        - the current page
        - our primary ancestor pages, all the way to the top
        returning the first one we find. If it was not found in any of
        those places (the default case), we return true unless the user is
        in minimal display mode.
        """
        prop = 'show_subtopics'
        if getattr(self.folder(),prop,0):
            if hasattr(self,'REQUEST') and hasattr(self.REQUEST,prop):
                return getattr(REQUEST,prop) and 1
            elif hasattr(self.aq_base,prop):
                return getattr(self,prop) and 1
            elif self.primaryParent():
                # poor caching
                try: return self.primaryParent().subtopicsEnabled() 
                except:
                    # experimental: support all-brains
                    try: return self.primaryParent().getObject().subtopicsEnabled() 
                    except: # XXX still run into errors here, investigate
                        DLOG('DEBUG: error in subtopicsEnabled for %s, primaryParent is: %s'\
                             % (self.id(),`self.primaryParent`))
                        return not (getattr(getattr(self,'REQUEST',None),
                                            'zwiki_displaymode',
                                            None) == 'minimal')
            else:
                return not (getattr(getattr(self,'REQUEST',None),
                                    'zwiki_displaymode',
                                    None) == 'minimal')
        else:
            return 0

    def subtopicsPropertyStatus(self):
        """
        Get the status of the show_subtopics property on this page.

        no property:    -1 ("default")
        true property:   1 ("always")
        false property:  0 ("never")
        """
        if not hasattr(self.aq_base,'show_subtopics'): return -1
        else: return self.show_subtopics and 1

    def setSubtopicsPropertyStatus(self,status,REQUEST=None):
        """
        Set, clear or remove this page's show_subtopics property.

        Same values as getSubtopicsStatus.
        """
        props = map(lambda x:x['id'], self._properties)
        if status == -1:
            if 'show_subtopics' in props:
                self.manage_delProperties(ids=['show_subtopics'],
                                          REQUEST=REQUEST)
        elif status:
            if not 'show_subtopics' in props:
                self.manage_addProperty('show_subtopics',1,'boolean',
                                        REQUEST=REQUEST)
            else:
                self.manage_changeProperties(show_subtopics=1,
                                             REQUEST=REQUEST)
        else:
            if not 'show_subtopics' in props:
                self.manage_addProperty('show_subtopics',0,'boolean',
                                        REQUEST=REQUEST)
            else:
                self.manage_changeProperties(show_subtopics=0,
                                             REQUEST=REQUEST)

    def primaryParent(self):
        for p in self.parents:
            if p: return self.pageWithName(p)
        return None
    
    def primaryParentName(self):
        p = self.primaryParent()
        if p: return p.pageName()
        else: return None
    
    def primaryParentUrl(self):
        p = self.primaryParent()
        if p: return p.pageUrl()
        else: return self.wikiUrl()
        
    security.declareProtected(Permissions.Reparent, 'reparent')
    def reparent(self, parents=None, REQUEST=None, pagename=None):
        """
        Make this page a child of the named parent pages.

        parent page names may be passed in several ways:
        - in the parents argument (a list or string of page names)
        - in a parents REQUEST attribute (ditto) #XXX needed ?
        - in the pagename argument (a single name)
        (the last is to support the page management form, whose field name
        is constrained).

        Page names may be fuzzy, or ids; any which do not resolve to an
        existing page or are duplicates will be removed. 
        
        """
        if pagename: parents = [pagename]
        if parents is None: parents = REQUEST.get('parents', [])
        if type(parents) != ListType: parents = [parents]
        parents = map(lambda x:absattr(x.Title),
                      filter(lambda x:x,
                             map(lambda x:self.pageWithFuzzyName(x),
                                 parents)))
        uniqueparents = []
        for p in parents:
            if not p in uniqueparents: uniqueparents.append(p)
        uniqueparents.sort()
        self.parents = uniqueparents
        self.index_object()
        if REQUEST is not None:
            REQUEST.RESPONSE.redirect(REQUEST['URL1'])

    security.declareProtected(Permissions.View, 'childrenAsList')
    def childrenAsList(self):
        """
        Return the list of names of my immediate children, if any (sorted).

        And do it fast.
        """
        #return self.childrenNesting()
        #return self.pages(parents=self.pageName()) # requires a catalog
        here = self.pageName()
        children = map(lambda x:x.Title,
                       filter(lambda x:here in x.parents, self.pages()))
        children.sort()
        return children

    security.declareProtected(Permissions.View, 'childrenIdsAsList')
    def childrenIdsAsList(self, REQUEST=None):
        """
        Return all my children's page ids as a flat list.
        """
        return map(lambda x:absattr(self.pageWithNameOrId(x).id),
                   self.childrenAsList())

    security.declareProtected(Permissions.View, 'siblingsAsList')
    def siblingsAsList(self):
        """
        Return the names of other pages sharing my first parent (sorted).

        Siblings by my other parents are ignored.
        """
        if self.parents:
            sibs = self.pageWithName(self.parents[0]).childrenAsList()
            sibs.remove(self.pageName())
            sibs.sort()
            return sibs
        else:
            return []

    # performance-sensitive
    # XXX too expensive right now
    security.declareProtected(Permissions.View, 'nextPage')
    def nextPage(self, nesting=None, depth_first=1):
        """
        Get the name of the next page in the hierarchy.

        If nesting is passed, use that instead of generating the wiki's.
        #When depth_first is true, step into children if any.
        """
        list = flatten(nesting or self.wikiNesting())
        # if we have bogus parents, we won't appear in the wiki nesting
        if self.pageName() in list:
            i = list.index(self.pageName())
            if i < len(list)-1: return list[i+1]
        return None

        ## faster ?
        #if depth_first:
        #    c = self.childrenAsList()
        #    if c: return c[0]
        #s = self.siblingsAsList()
        #if s:
        #    s.append(self.pageName())
        #    s.sort()
        #    i = s.index(self.pageName())
        #    if i < len(s)-1: return s[i+1]
        #p = self.parents
        #if p: return self.pageWithName(p[0]).nextPage(depth_first=0)
        ## XXX doesn't handle root pages
        #return None # last page!

    # performance-sensitive
    # XXX too expensive right now
    security.declareProtected(Permissions.View, 'previousPage')
    def previousPage(self, nesting=None):
        """
        Get the name of the previous page in the hierarchy.

        If nesting is passed, use that instead of generating the wiki's.
        """
        list = flatten(nesting or self.wikiNesting())
        # if we have bogus parents, we won't appear in the wiki nesting
        if self.pageName() in list:
            i = list.index(self.pageName())
            if i > 0: return list[i-1]
        return None

        #s = self.siblingsAsList()
        #if s:
        #    s.append(self.pageName())
        #    s.sort()
        #    i = s.index(self.pageName())
        #    if i > 0: prevsib = s[i-1]
        #c = self.childrenAsList()
        #if c: return c[0]
        #p = self.parents
        #if p: return p[0]
        #return None # first page!

    security.declareProtected(Permissions.View, 'offspringAsList')
    def offspringAsList(self, REQUEST=None):
        """
        Return all my offspring's page names as a flat list.
        """
        here = self.pageName()
        list = flatten(self.offspringNesting([here]))
        list.remove(here)
        return list

    security.declareProtected(Permissions.View, 'offspringIdsAsList')
    def offspringIdsAsList(self, REQUEST=None):
        """
        Return all my offspring's page ids as a flat list.
        """
        return map(lambda x:absattr(self.pageWithNameOrId(x).id),
                   self.offspringAsList())

    security.declareProtected(Permissions.View, 'ancestorsAsList')
    def ancestorsAsList(self, REQUEST=None):
        """
        Return the names of all my ancestor pages as a flat list, eldest first.

        If there are multiple lines of ancestry, return only the first.
        """
        try:
            return flatten(
                #self.get_ancestors(None,context)[0]
                #WikiNesting(self.folder()).get_ancestors(None,self)[0]
                self.ancestorsNesting()
                )[:-1]
        except: return [] # XXX temp, get rid of

    # performance-sensitive
    def wikiNesting(self,info=None):
        """
        Return a nesting representing the entire wiki.
        """
        info = info or self.getWikiParentInfo()
        roots = info.roots.keys()
        roots.sort()
        return self.offspringNesting(roots,info=info)

    def offspringNesting(self, pages, did=None, info=None):
        """
        Return a nesting including all offspring of a list of wiki page names.

        did is used for recursion, callers should not use.
        """
        if not info: info = self.getWikiParentInfo()
        if did is None: did = {}
        got = []
        for p in pages:
            been_there = did.has_key(p)
            did[p] = None
            if info.childmap.has_key(p):
                children = info.childmap[p]
                if children:
                    subgot = [p]
                    if not been_there:
                        subgot.extend(self.offspringNesting(children,
                                                            did=did,
                                                            info=info))
                    got.append(subgot)
                else:
                    got.append(p)
            else:
                got.append(p)
        return got

    def childrenNesting(self,info=None):
        """
        Return a nesting (a list, really) of all this page's immediate children.

        did is used for recursion, callers should not use.
        """
        if not info: info = self.getWikiParentInfo()
        here = self.pageName()
        if info.childmap.has_key(here): return info.childmap[here][:]
        else: return []

    def ancestorsNesting(self):
        """
        Return a nesting representing this page's ancestors.
        """
        info = self.getWikiParentInfo()
        container = self.folder() # XXX clean up
        ancestors = {}
        offspring = {}
        tops = {}                       # Ancestors that have no parents.
        todo = {self.pageName(): None}
        while todo:
            doing = todo
            todo = {}
            for i in doing.keys():
                if ancestors.has_key(i):
                    continue            # We've already collected this one.
                else:
                    pagewiththisname = None
                    if not hasattr(container, i):
                        pagewiththisname = self.pageWithName(i)
                        if not pagewiththisname:
                            continue        # Absent - ignore it.
                    ancestors[i] = None
                    #cf IssueNo0108
                    #obj = container[i]
                    obj = pagewiththisname or getattr(container,i)
                    if not hasattr(obj, 'parents'):
                        continue
                    parents = obj.parents
                    #if type(parents) != ListType: parents = [] # needed ?
                    if parents:
                        for p in parents:
                            if offspring.has_key(p):
                                offspring[p].append(i)
                            else:
                                offspring[p] = [i]
                            todo[p] = None
                    else: tops[i] = None
        # Ok, now go back down, unravelling each forebear only once:
        tops = tops.keys()
        tops.sort
        did = {}; got = []
        for t in tops:
            got.append(descend_ancestors(t, ancestors, did, offspring))
        return got

    def ancestorsAndSiblingsNesting(self):
        """
        Return a nesting representing this page's ancestors and siblings.
        """
        info = self.getWikiParentInfo()
        # Go up, identifying all and topmost forebears:
        ancestors = {}                  # Ancestors of page
        tops = {}                       # Ancestors that have no parents
        parents = {}
        todo = {self.pageName(): None}
        parentmap = info.parentmap
        while todo:
            doing = todo
            todo = {}
            for i in doing.keys():
                if ancestors.has_key(i):
                    continue            # We already took care of this one.
                else:
                    ancestors[i] = None
                    if parentmap.has_key(i):
                        parents = parentmap[i]
                    if parents:
                        for p in parents:
                            todo[p] = None
                    else: tops[i] = None
        #ancestors[self.pageName()] = None      # Finesse inclusion of page's offspring
        # Ok, now go back down, showing offspring of all intervening ancestors:
        tops = tops.keys()
        tops.sort
        did = {}; got = []
        childmap = info.childmap
        for t in tops:
            if t == self.pageName(): got.append([t])
            else: got.append(descend_ancestors(t, ancestors, did, childmap))
        return got

    def ancestorsAndChildrenNesting(self):
        """
        Return a nesting representing this page's ancestors and siblings.
        """
        info = self.getWikiParentInfo()
        # Go up, identifying all and topmost forebears:
        ancestors = {}                  # Ancestors of page
        tops = {}                       # Ancestors that have no parents
        parents = {}
        todo = {self.pageName(): None}
        parentmap = info.parentmap
        while todo:
            doing = todo
            todo = {}
            for i in doing.keys():
                if ancestors.has_key(i):
                    continue            # We already took care of this one.
                else:
                    ancestors[i] = None
                    if parentmap.has_key(i):
                        parents = parentmap[i]
                    if parents:
                        for p in parents:
                            todo[p] = None
                    else: tops[i] = None
        ancestors[self.pageName()] = None      # Finesse inclusion of page's offspring
        # Ok, now go back down, showing offspring of all intervening ancestors:
        tops = tops.keys()
        tops.sort
        did = {}; got = []
        childmap = info.childmap
        for t in tops:
            got.append(descend_ancestors(t, ancestors, did, childmap))
        return got

    # performance-sensitive ?
    def getWikiParentInfo(self):
        """
        Get current wiki parentage information in an easy-to-process form.

        Returns a WikiParentInfo object with:
          parentmap = {'node1': ['parent1', ...], ...}
          childmap = {'node1': ['child1', ...], ...}
          roots = {'root1': None, ...}

        Just regenerates this information from the wiki whenever needed;
        could trigger a whole-wiki walk if catalog optimization is not
        being used.
        """
        class WikiParentInfo:
            parentmap = {}
            childmap = {}
            roots = {}
        parentmap = {}
        childmap = {}
        roots = {}
        for p in self.pages():
            name, parents = p.Title, p.parents
            #if type(parents) != ListType: parents = [] # needed ?
            parentmap[name] = parents
            if not parents: roots[name] = None
            if not childmap.has_key(name): childmap[name] = []
            for parent in parents:
                # do an attr assignment in case we ever cache this as a
                # persistent object
                if childmap.has_key(parent): pchildren = childmap[parent]
                else: pchildren = []
                if name not in pchildren: pchildren.append(name)
                childmap[parent] = pchildren
        for k in parentmap.keys(): parentmap[k].sort()
        for k in childmap.keys(): childmap[k].sort()
        i = WikiParentInfo()
        i.parentmap = parentmap
        i.childmap = childmap
        i.roots = roots
        return i

Globals.InitializeClass(ParentsSupport) # install permissions
    
# helper functions

def descend_ancestors(page, ancestors, did, children):
    """
    Create nesting of ancestors leading to page.

    page is the name of the subject page.
    ancestors is a mapping whose keys are pages that are ancestors of page
    children is a mapping whose keys are pages children, and the values
       are the children's parents.

    Do not repeat ones we already did.
    """
    got = []
    for c in ((children.has_key(page) and children[page]) or []):
        if not ancestors.has_key(c):
            # We don't descend offspring that are not ancestors.
            got.append(c)
        elif ((children.has_key(c) and children[c]) or []):
            if did.has_key(c):
                # We only show offspring of ancestors once...
                got.append([c])
            else:
                # ... and this is the first time.
                did[c] = None
                got.append(descend_ancestors(c, ancestors, did, children))
        else:
            got.append(c)
    got.sort()                  # Terminals will come before composites.
    got.insert(0, page)
    return got

def deepappend(nesting, page):
    """
    Append a page to the very bottom of a nesting.
    """
    if type(nesting[-1]) is type([]):
        nesting[-1] = deepappend(nesting[-1], page)
    else:
        if len(nesting) is 1: nesting.append(page)
        else: nesting[-1] = [nesting[-1],page]
    return nesting



# helper class
# XXX We could make this a persistent object and minimize recomputes.
#     Put it in a standard place in the wiki's folder, or have the
#     pages in a folder share an instance, but use a single
#     persistent one which need not recompute all the relationship
#     maps every time - just needs to compare all pages parents
#     settings with the last noticed parents settings, and adjust
#     the children, roots, and parents maps just for those that
#     changed.  On this first cut we just recompute it all...
# XXX how slow is it ? not very - at least when using catalog
# XXX easier to include all this in ParentsSupport ?
# XXX why does this need Acquisition.Implicit ?
#class WikiNesting(Acquisition.Implicit):
#    """
#    Given a wiki folder, generate a nesting structure from parents info.
#
#    In a nesting, nodes are represented as:
#    - Leaves: the string name of the page
#    - Nodes with children: a list beginning with the parent node's name
#    - Nodes with omitted children (for brevity): list with one string.
#
#    """
#    def __init__(self, container):
#        self.container = container
#        self.save_nesting_info()
#
#    # XXX performance sensitive ?
#    def save_nesting_info(self):
#        """
#        Pre-process and save info for easy derivation of nesting structure.
#
#        We set:
#          - .parentmap: {'node1': ['parent1', ...], ...}
#          - .childmap:  {'node1': ['child1', ...], ...}
#          - .roots: {'root1': None, ...}
#        """
#        pagenames = []
#        parentmap = {}
#        childmap = {}
#        roots = {}
#
#        # we need the name and parents of all pages..
#        # optimization: try to use catalog for memory efficiency
#        # XXX some problems and duplication since this is not a ZWikiPage
#        # XXX catalog must be called "Catalog", look up SITE_CATALOG instead
#        # XXX does not filter out other wikis when using the catalog
#        try:
#            catalog = self.container.Catalog
#            indexes, metadata = catalog.indexes(), catalog.schema()
#            if not ('meta_type' in indexes and
#                    'path' in indexes and
#                    'Title' in metadata and
#                    'parents' in metadata):
#                raise AttributeError
#            pages = catalog(meta_type=PAGE_METATYPE)
#            #pages = self.pages()
#            #pages = filter(
#            #    lambda x:x.getPath()[:x.getPath().rfind('/')]==self.wikiPath(),
#            #    catalog(meta_type=PAGE_METATYPE,path=self.wikiPath()))
#        except AttributeError:
#            pages = self.container.objectValues(spec=PAGE_METATYPE)
#
#        for pg in pages:
#            parents = pg.parents
#            if type(parents) != ListType: parents = [] # SKWM XXX still needed ?
#            try: name = pg.Title() 
#            except TypeError: name = pg.Title
#            pagenames.append(name)
#
#            # Parents is direct:
#            parentmap[name] = parents[:]
#
#            # We have a root if node acknowledges no parents:
#            if not parents:
#                roots[name] = None
#
#            if not childmap.has_key(name):
#                childmap[name] = []
#
#            # Register page as child for all its parents:
#            for p in parents:
#                # We can't just append, in case we're a persistent object
#                # (which recognizes the need to update by an attr assignment).
#                if childmap.has_key(p): pchildren = childmap[p]
#                else: pchildren = []
#                if name not in pchildren: pchildren.append(name)
#                childmap[p] = pchildren
#
#        for k in parentmap.keys():
#            parentmap[k].sort()
#        for k in childmap.keys():
#            childmap[k].sort()
#        self.parentmap = parentmap
#        self.childmap = childmap
#        self.roots = roots
#
#    def get_map(self):
#        """
#        Return nesting of entire wiki.
#        """
#        roots = self.roots.keys()
#        roots.sort()
#        return self.get_offspring(roots)
#
#    def get_up_and_back(self, page):
#        """
#        Return nesting showing page containment and immediate offspring.
#        """
#        # Go up, identifying all and topmost forebears:
#        ancestors = {}                  # Ancestors of page
#        tops = {}                       # Ancestors that have no parents
#        todo = {page: None}
#        parentmap = self.parentmap
#        while todo:
#            doing = todo
#            todo = {}
#            for i in doing.keys():
#                if ancestors.has_key(i):
#                    continue            # We already took care of this one.
#                else:
#                    ancestors[i] = None
#                    if parentmap.has_key(i):
#                        parents = parentmap[i]
#                    if parents:
#                        for p in parents:
#                            todo[p] = None
#                    else: tops[i] = None
#        ancestors[page] = None      # Finesse inclusion of page's offspring
#        # Ok, now go back down, showing offspring of all intervening ancestors:
#        tops = tops.keys()
#        tops.sort
#        did = {}; got = []
#        childmap = self.childmap
#        for t in tops:
#            got.append(descend_ancestors(t, ancestors, did, childmap))
#        return got
#
#    def get_ancestors(self, page, context):
#        """
#        Return a trimmed nesting structure representing page's ancestors.
#
#        Callers pass in a wiki page object as context to allow us to call
#        various page lookup methods.
#        """
#        container = context.folder()
#        ancestors = {}
#        offspring = {}
#        tops = {}                       # Ancestors that have no parents.
#        todo = {context.pageName(): None}
#        while todo:
#            doing = todo
#            todo = {}
#            for i in doing.keys():
#                if ancestors.has_key(i):
#                    continue            # We've already collected this one.
#                else:
#                    pagewiththisname = None
#                    if not hasattr(container, i):
#                        pagewiththisname = context.pageWithName(i)
#                        if not pagewiththisname:
#                            continue        # Absent - ignore it.
#                    ancestors[i] = None
#                    #cf IssueNo0108
#                    #obj = container[i]
#                    obj = pagewiththisname or getattr(container,i)
#                    if not hasattr(obj, 'parents'):
#                        continue
#                    parents = obj.parents
#                    if type(parents) != ListType: # SKWM XXX still needed ?
#                        parents = []
#                    if parents:
#                        for p in parents:
#                            if offspring.has_key(p):
#                                offspring[p].append(i)
#                            else:
#                                offspring[p] = [i]
#                            todo[p] = None
#                    else: tops[i] = None
#        # Ok, now go back down, unravelling each forebear only once:
#        tops = tops.keys()
#        tops.sort
#        did = {}; got = []
#        for t in tops:
#            got.append(descend_ancestors(t, ancestors, did, offspring))
#        return got
#
#    def get_offspring(self, pages, did=None):
#        """
#        Return a nesting including all offspring of a list of wiki page names.
#
#        did is used for recursion, to prune already elaborated pages.
#        """
#        if did is None: did = {}
#        got = []
#        for p in pages:
#            been_there = did.has_key(p)
#            did[p] = None
#            if self.childmap.has_key(p):
#                children = self.childmap[p]
#                if children:
#                    subgot = [p]
#                    if not been_there:
#                        subgot.extend(self.get_offspring(children, did=did))
#                    got.append(subgot)
#                else:
#                    got.append(p)
#            else:
#                got.append(p)
#        return got
