/*
 * Top-level interpreter API
 * (C) 2006, Pascal Schmidt <arena-language@ewetel.net>
 * see file doc/LICENSE for license
 */

#include "libmisc/misc.h"
#include "libruntime/runtime.h"
#include "libparser/parser.h"
#include "libeval/eval.h"
#include "libstdlib/stdlib.h"

#include "api.h"

/*
 * Create new interpreter context
 */
arena_ctx arena_new_ctx(int argc, char **argv)
{
  arena_state *state = state_alloc();
  
  symtab_stack_init(state);
  stdlib_register(state, argc, argv);
  
  return state;
}

/* 
 * Free interpreter context
 */
void arena_free_ctx(arena_ctx ctx)
{
  arena_state *state = ctx;
  
  symtab_stack_teardown(state);
  eval_free_floats(state);
  state_free(state);
}

/*
 * Create new script by parsing source file
 */
arena_script arena_new_script(char *name)
{
  arena_state *state = state_alloc();
  stmt_list *list;
  
  state->source_line = 1;
  state->source_col  = 0;
  
  list = parse_file(state, name);
  state_free(state);

  return list;
}

/*
 * Free script data
 */
void arena_free_script(arena_script script)
{
  stmt_list *list = script;
  
  stmt_list_free(list);
}

/*
 * Execute a script inside a given interpreter context
 */
void arena_execute_script(arena_ctx ctx, arena_script script)
{
  arena_state *state = ctx;
  stmt_list *list = script;

  state->source_line = 0;
  state->source_col  = 0;
  
  eval_stmt_list(state, list, 0);
}

/*
 * Dump a script to standard output
 */
void arena_dump_script(arena_script script)
{
  stmt_list *list = script;
  
  stmt_list_dump(list, 0);
}
