/*
 * Foreign function interface for x86 with GCC
 * (C) 2006, Pascal Schmidt <arena-language@ewetel.net>
 * see file ../../doc/LICENSE for license
 */

#include <dlfcn.h>
#include <stdio.h>
#include <stdlib.h>

#include "../stdlib.h"

/*
 * Compute memory size requirement of a value
 */
static int value_size(value *val)
{
  switch (val->type) {
    case VALUE_TYPE_VOID:
    case VALUE_TYPE_BOOL:
    case VALUE_TYPE_INT:
    case VALUE_TYPE_STRING:
    case VALUE_TYPE_RES:
      return 4;
    case VALUE_TYPE_FLOAT:
      return 8;
    default:
      return 0;
  }
}

/*
 * Create stack image from argument values
 */
static char *mkstack(unsigned int argc, value **argv, int *sizeout)
{
  char *stack;
  int dbuf[2];
  double *dbufptr = (double *) dbuf;
  int *inext;
  int size = 0;
  int i;
  
  for (i = 0; i < argc; i++) {
    if (argv[i]->type == VALUE_TYPE_ARRAY  ||
        argv[i]->type == VALUE_TYPE_STRUCT ||
        argv[i]->type == VALUE_TYPE_FN) {
      return NULL;
    }
    size += value_size(argv[i]);
  }
  
  stack = malloc(size);
  if (!stack) {
    return NULL;
  }
  
  inext = (int *) (stack + size - 4);
  
  for (i = 0; i < argc; i++) {
    switch (argv[i]->type) {
      case VALUE_TYPE_VOID:      
        *inext-- = 0;
        break;
      case VALUE_TYPE_BOOL:
        *inext-- = argv[i]->value_u.bool_val;
        break;
      case VALUE_TYPE_INT:
        *inext-- = argv[i]->value_u.int_val;
        break;
      case VALUE_TYPE_FLOAT:
        *dbufptr = argv[i]->value_u.float_val;
        *inext-- = dbuf[0];
        *inext-- = dbuf[1];
        break;
      case VALUE_TYPE_STRING:
        if (argv[i]->value_u.string_val.value) {
          *inext-- = (int) argv[i]->value_u.string_val.value;
        } else {
          *inext-- = (int) "";
        }
        break;
      case VALUE_TYPE_RES:
        *inext-- = (int) argv[i]->value_u.res_val->data;
        break;
      default:
        sanity(0);
    }
  }
  *sizeout = size;
  return stack;
}

/*
 * Call C function via inline assembly magic
 */
static void call(char *stack, int size, void *f, int *ires, double *fres)
{
  asm volatile (
    "movl %2, %%ecx \n\t"
    "movl %4, %%esi \n\t"
    "copy: lodsl \n\t"
    "pushl %%eax \n\t"
    "loop copy \n\t"
    "call *%5 \n\t"
    "addl %3, %%esp \n\t"
    "movl %%eax, %0 \n\t"
    "fstl %1 \n\t"
    : "=m" (*ires), "=m" (*fres)
    : "g" (size >> 2), "g" (size), "g" (stack), "g" (f)
    : "cc", "memory", "%eax", "%ecx", "%esi", "%st" );
}

/*
 * Call C function by name and argument list
 */
static value *ccall(unsigned int argc, value **argv, char type, int mfree)
{
  void *handle = RESDATA_OF(argv[0]);
  char *name = STR_OF(argv[1]);
  char *stack;
  int stacksize;
  void *func;
  int ires = 0;
  double fres = 0.0;
  value *res;

  if (RESTYPE_OF(argv[0]) != RESOURCE_LIB || !name || !handle) {
    return value_make_void();
  }

  func = dlsym(handle, name);
  if (!func) {
    return value_make_void();
  }

  stack = mkstack(argc - 2, argv + 2, &stacksize);
  if (!stack) {
    return value_make_void();
  }

  call(stack, stacksize, func, &ires, &fres);
  free(stack);

  switch (type) {
    case 'i':
      res = value_make_int(ires);
      break;
    case 'f':
      res = value_make_float(fres);
      break;
    case 'p':
      res = mem_make_pointer((void *) ires, mfree);
      break;
    default:
      res = value_make_void();
      break;
  }
  
  return res;
}

/*
 * Call C function that returns nothing
 */
value *dyn_call_void(arena_state *s, unsigned int argc, value **argv)
{
  return ccall(argc, argv, 'v', 0);
}

/*
 * Call C function that returns an int (or compatible value)
 */
value *dyn_call_int(arena_state *s, unsigned int argc, value **argv)
{
  return ccall(argc, argv, 'i', 0);
}

/*
 * Call C function that returns a float
 */
value *dyn_call_float(arena_state *s, unsigned int argc, value **argv)
{
  return ccall(argc, argv, 'f', 0);
}

/*
 * Call C function that returns a pointer
 */
value *dyn_call_ptr(arena_state *s, unsigned int argc, value **argv)
{
  value *freeval;
  int free = 0, nargc = argc;
  
  freeval = argv[argc - 1];
  if (freeval->type == VALUE_TYPE_BOOL) {
    free = BOOL_OF(freeval);
    --nargc;
  }

  return ccall(nargc, argv, 'p', free);
}

/*
 * Free C library resource
 */
static void dyn_free(void *data)
{
  if (data) dlclose(data);
}

/*
 * Dynamically load a C library
 */
value *dyn_open(arena_state *s, unsigned int argc, value **argv)
{
  char *name = STR_OF(argv[0]);
  void *handle;
  
  if (!value_str_compat(argv[0])) {
    return value_make_void();
  }

  handle = dlopen(name, RTLD_NOW | RTLD_GLOBAL);
  if (!handle) {
    return value_make_void();
  }

  return value_make_resource(RESOURCE_LIB, handle, dyn_free);
}

/*
 * Unload a previously loaded C library
 */
value *dyn_close(arena_state *s, unsigned int argc, value **argv)
{
  void *handle = RESDATA_OF(argv[0]);
  
  if (RESTYPE_OF(argv[0]) == RESOURCE_LIB && handle) {
    dlclose(handle);
    RESDATA_OF(argv[0]) = NULL;
  }
  return value_make_void();
}

/*
 * Indicate that foreign function calls are supported
 */
value *dyn_supported(arena_state *s, unsigned int argc, value **argv)
{
  return value_make_bool(1);
}

/*
 * Get pointer to C function
 */
value *dyn_fn_pointer(arena_state *s, unsigned int argc, value **argv)
{
  void *handle = RESDATA_OF(argv[0]);
  char *name = STR_OF(argv[1]);
  void *func;
  
  if (RESTYPE_OF(argv[0]) != RESOURCE_LIB || !value_str_compat(argv[1])) {
    return value_make_void();
  }
  
  func = dlsym(handle, name);
  if (!func) {
    return value_make_void();
  }
  return mem_make_pointer(func, 0);
}
