%  Copyright (C) 2002-2003 David Roundy
%
%  This program is free software; you can redistribute it and/or modify
%  it under the terms of the GNU General Public License as published by
%  the Free Software Foundation; either version 2, or (at your option)
%  any later version.
%
%  This program is distributed in the hope that it will be useful,
%  but WITHOUT ANY WARRANTY; without even the implied warranty of
%  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%  GNU General Public License for more details.
%
%  You should have received a copy of the GNU General Public License
%  along with this program; if not, write to the Free Software Foundation,
%  Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
\section{darcs add}
\begin{code}
module Add ( add ) where

import List ( (\\) )

import DarcsCommands
import DarcsArguments
import Directory
import IO
import System
import Repository
import Patch
import SlurpDirectory
import RegexString
import Monad ( liftM, unless )
import RepoPrefs ( darcsdir_filter, boring_file_filter )
\end{code}

\begin{code}
add_description =
 "Add a one or more new files or directories."
\end{code}

\options{add}

\haskell{add_help}

\begin{code}
add_help =
 "Add needs to be called whenever you add a new file or directory to
your project.  Of course, it also needs to be called when you first
create the project, to let darcs know which files should be kept
track of.
"
\end{code}

\begin{code}
add = DarcsCommand {command_name = "add",
                    command_help = add_help,
                    command_description = add_description,
                    command_extra_args = -1,
                    command_command = add_cmd,
                    command_prereq = am_in_repo,
                    command_get_arg_possibilities = list_unregistered_files,
                    command_argdefaults = nodefaults,
                    command_darcsoptions =
                    [verbose, noskip_boring, allow_caseonly,
                     recursive "add contents of subdirectories"]}
\end{code}

Darcs will refuse to add a file or directory that differs from an existing
one only in case.  This is because the HFS+ file system used on under MacOS
treats such files as being one and the same.

\begin{code}
add_cmd opts args = do
    cur <- slurp_current "."
    flist <- if Recursive `elem` opts
             then expand_dirs args
             else return args
    -- refuse to add boring files recursively:
    nboring <- if Boring `elem` opts
               then return $ darcsdir_filter
               else boring_file_filter
    sequence_ $ map (putStrLn . ("Skipping boring file "++)) $ flist \\ nboring flist
    ps <- addp (AllowCaseOnly `elem` opts) cur $ nboring flist
    pend <- read_pending
    case pend of
        Nothing -> write_pending $ join_patches $ filter (/= join_patches []) ps
        Just op -> write_pending $ join_patches $
                   flatten $ join_patches [op,join_patches ps]

addp :: Bool -> Slurpy -> [FilePath] -> IO [Patch]
addp _ _ [] = return []
addp allowcaseonly cur (f:fs) =
  if (if allowcaseonly
      then slurp_has f cur
      else slurp_has_anycase f cur)
  then do putStr $ "A file named "++f++" is already in the repository!\n"
          unless allowcaseonly $ do
            putStr $ "Note that to ensure portability we don't allow files"
            putStr $ " that differ\nonly in case.\n"
          addp allowcaseonly cur fs
  else do
    isdir <- doesDirectoryExist f
    if isdir
       then trypatch cur $ adddir f
       else do isfile <- doesFileExist f
               if isfile
                  then trypatch cur $ addfile f
                  else do putStr $ "File "++ f ++" does not exist!\n"
                          addp allowcaseonly cur fs
    where trypatch cur p = case apply_to_slurpy p cur of
                           Nothing -> do putStr $ "Skipping:\n"++show p
                                         addp allowcaseonly cur fs
                           Just s' -> do (p:) `liftM` addp allowcaseonly s' fs
\end{code}

\begin{code}
expand_dirs :: [FilePath] -> IO [FilePath]
expand_dirs fs = liftM (foldl (++) []) $ sequence $ map expand_one fs
expand_one f = do
  isdir <- doesDirectoryExist f
  if not isdir then return [f]
     else do formerdir <- getCurrentDirectory
             setCurrentDirectory f
             fs <- list_files
             setCurrentDirectory formerdir
             return $ f: map (\f'->f++"/"++f') fs
\end{code}
