%  Copyright (C) 2002-2003 David Roundy
%
%  This program is free software; you can redistribute it and/or modify
%  it under the terms of the GNU General Public License as published by
%  the Free Software Foundation; either version 2, or (at your option)
%  any later version.
%
%  This program is distributed in the hope that it will be useful,
%  but WITHOUT ANY WARRANTY; without even the implied warranty of
%  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%  GNU General Public License for more details.
%
%  You should have received a copy of the GNU General Public License
%  along with this program; if not, write to the Free Software Foundation,
%  Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
\section{darcs check}
\begin{code}
module Check ( check ) where
import DarcsCommands
import DarcsArguments
import Directory
import IO
import System
import Repository
import Patch
import PatchInfo
import SlurpDirectory
import Diff
import Monad
import Test
import Lock ( withTempDir )
\end{code}

\options{check}

\haskell{check_description}
\begin{code}
check_description = "Check the repository for consistency.
"
\end{code}
Check verifies that the patches stored in the repository, when successively
applied to an empty tree, properly recreate the stored current tree.

\begin{code}
check_help =
 "Check verifies that the patches stored in the repository, when successively\n"++
 "applied to an empty tree, properly recreate the stored current tree.\n"
\end{code}

\begin{code}
check = DarcsCommand {command_name = "check",
                      command_help = check_help,
                      command_description = check_description,
                      command_extra_args = 0,
                      command_command = check_cmd,
                      command_prereq = am_in_repo,
                      command_get_arg_possibilities = return [],
                      command_argdefaults = nodefaults,
                      command_darcsoptions = [verbose,notest]}
\end{code}

\begin{code}
check_cmd opts args = do
  am_verbose <- return $ Verbose `elem` opts
  patches <- read_repo "."
  formerdir <- getCurrentDirectory
  withTempDir "_darcs/checking" $ \chd -> do
    when am_verbose $ putStr "Applying patches...\n"
    apply_patches am_verbose $ reverse $ concat patches
    is_same <- cmp (formerdir++"/_darcs/current") chd
    if is_same
      then do
        putStr "The repo is consistent!\n"
        if NoTest `elem` opts
           then exitWith ExitSuccess
           else do setCurrentDirectory formerdir
                   ec <- run_test chd
                   exitWith ec
      else do putStr $ "Inconsistent repo!\n"
              exitWith $ ExitFailure 1
\end{code}

\begin{code}
apply_patches :: Bool -> [(PatchInfo, Maybe Patch)] -> IO ()
apply_patches _ [] = return ()
apply_patches v ((pi,mp):ps) = do
  s <- slurp "."
  when v $ putStr $ "Applying patch "++show pi++"\n"
  case mp of
    Nothing -> do putStr $ "Unreadable patch: "++show pi++"\nExiting...\n"
                  exitWith $ ExitFailure 1
    Just p -> do
        when v $ putStr $ "Patch read...\n"
        case apply_to_slurpy p s of
          Nothing -> do putStr "Unapplicable patch!\n"
                        putStr $ show p
                        exitWith $ ExitFailure 1
          Just s' -> do when v $ putStr $ "Patch applies cleanly...\n"
                        slurp_write_dirty s'
                        apply_patches v ps
\end{code}

\input{Test.lhs}

If you just want to check the consistency of your repository without
running the test, you can call darcs check with the \verb!--no-test!
option.
