%  Copyright (C) 2002-2003 David Roundy
%
%  This program is free software; you can redistribute it and/or modify
%  it under the terms of the GNU General Public License as published by
%  the Free Software Foundation; either version 2, or (at your option)
%  any later version.
%
%  This program is distributed in the hope that it will be useful,
%  but WITHOUT ANY WARRANTY; without even the implied warranty of
%  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%  GNU General Public License for more details.
%
%  You should have received a copy of the GNU General Public License
%  along with this program; if not, write to the Free Software Foundation,
%  Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

\begin{code}
module PatchCheck ( PatchCheck(), do_check, file_exists, dir_exists,
                    remove_file, remove_dir, create_file, create_dir,
                    insert_line, delete_line, is_valid, do_verbose_check,
                    file_empty,
                    check_move, modify_file, Possibly(..)
                  ) where

import RegexString
import IO
import IOExts
import FastPackedString

newtype PatchCheck a = PC( KnownState -> (KnownState, a) )

data Possibly a = PJust a | PNothing | PSomething
                  deriving (Eq, Show)
data Prop = FileEx String | DirEx String
          | FileLines String [Possibly PackedString]
            deriving (Eq)
data KnownState = P [Prop] [Prop]
                | Inconsistent
                  deriving (Show)
instance  Show Prop  where
    show (FileEx f) = "FileEx "++f
    show (DirEx d)  = "DirEx "++d
    show (FileLines f l)  = "FileLines "++f++" "++show (take 10 l)
\end{code}

\section{Patch Consistency Checking}

\begin{code}
instance Monad PatchCheck where
    (PC p) >>= k  =  PC( \s0 -> let (s1, a) = p s0
                                    (PC q) = k a
                                in q s1 )
    return a = PC( \s -> (s, a) )
\end{code}

\begin{code}
do_check :: PatchCheck a -> a
do_check (PC p) = snd $ p (P [] [])

do_verbose_check :: PatchCheck a -> a
do_verbose_check (PC p) =
    case p (P [] []) of
    (pc, b) -> unsafePerformIO $ do putStr $ show pc++"\n"
                                    return b

is_valid :: PatchCheck Bool
is_valid = PC iv
iv Inconsistent = (Inconsistent, False)
iv m = (m, True)
\end{code}

\begin{code}
has :: Prop -> [Prop] -> Bool
has k [] = False
has k (k':ks) = if k == k' then True else has k ks

modify_file :: String
            -> ([Possibly PackedString]-> Maybe [Possibly PackedString])
            -> PatchCheck Bool
modify_file f change = do
    file_exists f
    c <- file_contents f
    case change c of
      Nothing -> assert_not $ FileEx f -- shorthand for "FAIL"
      Just c' -> do set_contents f c'
                    is_valid

insert_line :: String -> Int -> PackedString -> PatchCheck Bool
insert_line f n l = do
    c <- file_contents f
    case il n l c of
       [] -> assert_not $ FileEx f
       c' -> do
             set_contents f c'
             return True
il 1 l mls = (PJust l:mls)
il n l (ml:mls) = ml : il (n-1) l mls
il _ _ [] = []

delete_line :: String -> Int -> PackedString -> PatchCheck Bool
delete_line f n l = do
    c <- file_contents f
    case dl [] n l c of
        Nothing -> assert_not $ FileEx f
        Just c' -> do
            set_contents f c'
            is_valid
dl o n l [] = Nothing
dl o 1 l (ml':ls) =
    case ml' of
    PSomething -> Just $ o++ls
    PNothing -> Just $ o++ls
    PJust l' -> if l' == l then Just $ o++ls
                          else Nothing
dl o n l (ml:mls) =
    case ml of
    PNothing -> dl (o++[PSomething]) (n-1) l mls
    _ -> dl (o++[ml]) (n-1) l mls

set_contents :: String -> [Possibly PackedString] -> PatchCheck ()
set_contents f mss = PC (sc f mss)
sc :: String -> [Possibly PackedString] -> KnownState -> (KnownState,())
sc f mss (P ks nots) = (P (scl [] f mss ks) nots, ())
sc _ _ Inconsistent = (Inconsistent, ())
scl :: [Prop] -> String -> [Possibly PackedString] -> [Prop] -> [Prop]
scl olds f mss [] = FileLines f mss : olds
scl olds f mss (FileLines f' mss':ks)
    | f == f' = FileLines f mss : (olds++ks)
    | f /= f' = scl (FileLines f' mss':olds) f mss ks
scl olds f mss (k:ks) = scl (k:olds) f mss ks

file_contents :: String -> PatchCheck [Possibly PackedString]
file_contents f = PC (fc f)
unknown_contents :: [Possibly PackedString]
unknown_contents = (PNothing: unknown_contents)
fc f Inconsistent = (Inconsistent, [])
fc f (P ks nots) = (P ks nots, fic f ks)
fic f (FileLines f' mss:ks) | f == f' = mss
fic f (_:ks) = fic f ks
fic f [] = unknown_contents

file_empty :: String -> PatchCheck Bool
file_empty f = do
  c <- file_contents f
  if (cempty 100 c)
     then do set_contents f []
             is_valid
     -- Crude way to make it inconsistent and return false:
     else assert_not $ FileEx f

  return $ cempty 100 c
cempty _ [] = True
cempty _ (PJust _:_) = False
cempty _ (PSomething:_) = False
cempty 0 (PNothing:_) = True
cempty n (PNothing:ls) = cempty (n-1) ls

movedirfilename :: String -> String -> String -> String
movedirfilename d d' f =
    if length f > length d && take (length d+1) f == d ++ "/"
    then d'++drop (length d) f
    else if f == d
         then d'
         else f

is_soe d1 d2 = -- is_superdir_or_equal
    d1 == d2 ||
           (length d2 >= length d1 + 1 && take (length d1 + 1) d2 == d1 ++ "/")

do_swap :: String -> String -> PatchCheck Bool
do_swap f f' = PC swfn
  where swfn Inconsistent = (Inconsistent, False)
        swfn (P ks nots) = (P (map sw ks) (map sw nots), True)
        sw (FileEx a) | f  `is_soe` a = FileEx $ movedirfilename f f' a
                      | f' `is_soe` a = FileEx $ movedirfilename f' f a
        sw (DirEx a) | f  `is_soe` a = DirEx $ movedirfilename f f' a
                     | f' `is_soe` a = DirEx $ movedirfilename f' f a
        sw (FileLines a ls) | f  `is_soe` a = FileLines (movedirfilename f f' a) ls
                            | f' `is_soe` a = FileLines (movedirfilename f' f a) ls
        sw p = p

assert :: Prop -> PatchCheck Bool
assert p = PC (assertfn p)
assertfn k Inconsistent = (Inconsistent, False)
assertfn k (P ks nots) =
    if has k nots then (Inconsistent, False)
    else if has k ks then (P ks nots, True)
    else (P (k:ks) nots, True)

assert_not :: Prop -> PatchCheck Bool
assert_not p = PC (assertnfn p)
assertnfn k Inconsistent = (Inconsistent, False)
assertnfn k (P ks nots) =
    if has k ks then (Inconsistent, False)
    else if has k nots then (P ks nots, True)
    else (P ks (k:nots), True)

change_to_true :: Prop -> PatchCheck Bool
change_to_true p = PC (chtfn p)
chtfn k Inconsistent = (Inconsistent, False)
chtfn k (P ks nots) = (P (k:ks) (filter (\k'->k /= k') nots), True)

change_to_false :: Prop -> PatchCheck Bool
change_to_false p = PC (chffn p)
chffn k Inconsistent = (Inconsistent, False)
chffn k (P ks nots) = (P (filter (\k'->k /= k') ks) (k:nots), True)
\end{code}

\begin{code}
create_file :: String -> PatchCheck Bool
create_file fn = do
  superdirs_exist fn
  assert_not (FileEx fn)
  assert_not (DirEx fn)
  change_to_true (FileEx fn)

create_dir :: String -> PatchCheck Bool
create_dir fn = do
  substuff_dont_exist fn
  superdirs_exist fn
  assert_not (FileEx fn)
  assert_not (DirEx fn)
  change_to_true (DirEx fn)

remove_file :: String -> PatchCheck Bool
remove_file fn = do
  superdirs_exist fn
  assert (FileEx fn)
  file_empty fn
  assert_not (DirEx fn)
  change_to_false (FileEx fn)

remove_dir :: String -> PatchCheck Bool
remove_dir fn = do
  substuff_dont_exist fn
  superdirs_exist fn
  assert_not (FileEx fn)
  assert (DirEx fn)
  change_to_false (DirEx fn)

check_move :: String -> String -> PatchCheck Bool
check_move f f' = do
  superdirs_exist f
  superdirs_exist f'
  assert_not (FileEx f')
  assert_not (DirEx f')
  do_swap f f'
\end{code}

\begin{code}
substuff_dont_exist :: String -> PatchCheck Bool
substuff_dont_exist d = PC (ssde d)
ssde d Inconsistent = (Inconsistent, False)
ssde d (P ks nots) = if noss d ks
                     then (P ks nots, True)
                     else (Inconsistent, False)
is_within_dir d f = d == take l f && l < length f && '/' == (head $ drop l f)
    where l = length d
noss d (FileEx f:ks) = if is_within_dir d f
                       then False else noss d ks
noss d [] = True
noss d (DirEx f:ks) = if is_within_dir d f
                      then False else noss d ks
noss d (k:ks) = noss d ks

superdirs_exist :: String -> PatchCheck Bool
superdirs_exist fn =
  case matchRegex (mkRegex "\\./(.+)/[^/]+") fn of
  Just ["."] -> return True
  Just [d] -> do
                a <- assert (DirEx ("./"++d))
                c <- assert_not (FileEx ("./"++d))
                b <- superdirs_exist ("./"++d)
                return $! a && b && c
  _ -> is_valid

file_exists :: String -> PatchCheck Bool
file_exists fn = do
  superdirs_exist fn
  assert (FileEx fn)
  assert_not (DirEx fn)

dir_exists :: String -> PatchCheck Bool
dir_exists fn = do
  superdirs_exist fn
  assert (DirEx fn)
  assert_not (FileEx fn)
\end{code}
