/*
 * categories.c
 * Thomas Nemeth, le 22.02.2000 (from acclists.c 18.09.1999)
 *
 * Gestion de la liste des donnes de gAcc, gestionnaire de comptes
 * banquaires personnels avec GTK+.
 *
 *   Copyright (C) 1999  Thomas Nemeth
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

#include <config.h>
#include <string.h>
#include <strings.h>
#include "structs.h"
#include "categories.h"
#include "fileaccess.h"
#include "usefull.h"
#include "oldversions.h"


/* ------++++++======*** CATGORIES * CATEGORIES ***======++++++----- */

void read_cat_from_last_version (FILE *file) {
    CAT_ELT *lst, *prev = NULL;
    char     tmpstr[MAXSTRLEN+1];

	fgets (tmpstr, (int) MAXSTRLEN, file); // read separator
	while ( ! feof (file) ) {
		if (tmpstr[0] != 0 && strlen(tmpstr)>5) {
#if DEBUG_LEVEL>4
			printf ("read_cat_from_last_version %s",tmpstr);
#endif
			MY_ALLOC (lst, 1, CAT_ELT);
			lst->next = NULL;
			MY_ALLOC (lst->category, 1, CATEGORY);
			lst->category = get_category_from_line (tmpstr);
			if(prev != NULL) {
				prev->next = lst;
			}
			prev = lst;
			if(config.cat_list_head==NULL) {
				config.cat_list_head = lst;
			}
		}
		fgets (tmpstr, (int) MAXSTRLEN, file);
	}
}

void read_categories_from_file () {
    FILE    *file;
    char    *FileVersion;

    FileVersion = open_file (config.catfilename, &file);
	if(FileVersion != NULL) {
#if DEBUG_LEVEL>2
		printf ("Version du fichier   : %s\n", FileVersion);
		printf ("Version du programme : %s\n", VERSION);
#endif
		if (in_versions ("0.6.5", VERSION, FileVersion) == VRAI) {
			read_cat_from_last_version (file);
		} else if (in_versions ("0.0.0", "0.6.4", FileVersion) == VRAI) {
			read_cat_from_000_to_064 (file);
		} else if (strcmp (FileVersion, VERSION) > 0) {
			printf ("File version : %s -- Program version : %s\n",
					FileVersion, VERSION);
			printf (_("Category file version not supported by this one. Sorry !\n") );
		} else {
			printf ("File version : %s -- Program version : %s\n",
					FileVersion, VERSION);
			printf ("Cannot open %s\n",config.catfilename);
		}
		// do not change CatDataModified status
		//    config.CatDataModified = FAUX;
		if(strlen(FileVersion)>3)
			fclose (file);
		free (FileVersion);
	}
}

void write_categories_to_file () {
    CAT_ELT *lst;
    FILE    *file;
    char    *line;

    lst = config.cat_list_head;
    if ( (file = fopen(config.catfilename, "w") ) == NULL) {
        fatal_error (config.catfilename);
    } else {
        create_file_header (file, "Categories description file");
        while (lst != NULL) {
            line = create_line_from_category (lst->category);
            /*
            fprintf (file, "CATNAM: %s\n",   lst->category->name);
            fprintf (file, "CATNUM: %d\n",   lst->category->num);
            fprintf (file, "CATTYP: %d\n",   lst->category->type);
            */
            fprintf (file, "%s", line);
            free (line);
            lst = lst->next;
        }
        fclose (file);
        config.CatDataModified = FAUX;
    }
}

void add_category (CATEGORY *category) {
    CAT_ELT *cat = config.cat_list_head;
    CAT_ELT *prev_cat = NULL;
    CAT_ELT *support = NULL;
    int cmp = 0;

	if(cat!=NULL) {
		cmp = strcasecmp (category->name, cat->category->name);
	}

#if DEBUG_LEVEL>4
	printf ("add_category %s\n",category->name);
#endif
    if (category->type == CREDIT) {
#if DEBUG_LEVEL>4
		printf ("STORING a CREDIT category :\n");
		if(cat!=NULL && cat->category!=NULL) {
			printf ("Category \"%s\" is a %s category\n",
					cat->category->name,
					cat->category->type == CREDIT ? "CREDIT" : "DEBIT");
		}
#endif
        while ( (cat != NULL) && (cat->category->type != CREDIT) ) {
            prev_cat = cat;
            cat      = cat->next;
#if DEBUG_LEVEL>4
			if (cat != NULL && cat->category!=NULL) {
				printf ("Category \"%s\" is a %s category\n",
						cat->category->name,
						cat->category->type == CREDIT ? "CREDIT" : "DEBIT");
			} else {
				printf ("There is no other category...\n");
			}
#endif
        }
    }
	while ( (cat != NULL) && (cmp > 0) ) {
        prev_cat = cat;
        cat      = cat->next;
        if (cat != NULL) cmp = strcasecmp (category->name, cat->category->name);
    }
    MY_ALLOC (support, 1, CAT_ELT);
    support->category      = category;
    support->next          = cat;
    if (prev_cat != NULL)
        prev_cat->next     = support;
    else
        config.cat_list_head = support;
    config.CatDataModified   = VRAI;
#if DEBUG_LEVEL>4
	printf ("add_category ok\n");
#endif
}

void remove_category (CATEGORY *category) {
    CAT_ELT *cat, *prev;

    cat  = config.cat_list_head;
    prev = NULL;
    while ( (cat != NULL) && (cat->category != category) ) {
        prev = cat;
        cat  = cat->next;
    }
    if (cat->category == category) {
        #ifdef DEBUG_GACC
            printf ("REMOVING \"%s\" before \"%s\"...\n",
                    cat->category->name,
                    cat->next != NULL ? cat->next->category->name : "END");
        #endif
        if (prev != NULL)
            prev->next = cat->next;
        else
            config.cat_list_head = cat->next;

        free (cat);
        config.CatDataModified = VRAI;
    } else
        printf ("remove_category : category not found !\n");
}

void del_category (CAT_ELT *element) {
    CAT_ELT *cat, *prev;

    prev = NULL;
    cat  = config.cat_list_head;
    while ( (cat->next != NULL) && (cat != element) ) {
        prev = cat;
        cat  = cat->next;
    }

    if (prev != NULL) prev->next = element->next;

    free (element->category->name);
    free (element->category);
    free (element);
    config.CatDataModified = VRAI;
}

CAT_ELT *get_category (int cat_num) {
    CAT_ELT *cat = config.cat_list_head;
    int      i = 0;

    while ( (i != cat_num) && (cat != NULL) ){
        cat = cat->next;
        i++;
    }

    return cat;
}

int get_category_next_num () {
    CAT_ELT *cat     = config.cat_list_head;
    int      max_num = UNKNOWN_CATEGORY;

    while (cat != NULL) {
        if (cat->category->num > max_num) max_num = cat->category->num;
        cat = cat->next;
    }

    return max_num + 1;
}

char *category_name (int cat_num) {
    CAT_ELT *cat = config.cat_list_head;
    char    *name = NULL;

	while ( (cat != NULL) && (cat->category->num != cat_num) )
		cat = cat->next;

    if (cat != NULL)
        name = cat->category->name;
    else if (cat_num != UNKNOWN_CATEGORY) {
        MY_ALLOC (name, 6, char);
        sprintf (name , "%d", cat_num);
    }

    return name;
}

int category_number (char *cat_name) {
    CAT_ELT *cat = config.cat_list_head;
	int num = UNKNOWN_CATEGORY;
	
	if(cat_name == NULL || strlen(cat_name)==0) {
		return category_number(_("Unknown"));
	} else {
#if DEBUG_LEVEL>4
		printf ("Recherche du numro de la catgorie : %s\n", cat_name);
#endif
		while(cat != NULL
			  && (cat->category == NULL
				  || cat->category->name == NULL
				  || strcmp (cat->category->name, cat_name) != 0 )) {
#if DEBUG_LEVEL>4
			//			printf ("La catgorie %s n'est pas %s\n",
			//					cat->category->name, cat_name);
#endif
			cat = cat->next;
		}
		if (cat != NULL) {
			num = cat->category->num;
		}
#if DEBUG_LEVEL>4
		printf ("numro de la catgorie=%d\n", num);
#endif
		return num;
	}
}

void free_categories () {
    CAT_ELT *p = config.cat_list_head, *q;
	
    while (p != NULL) {
        q = p->next;
        free (p->category->name);
        free (p->category);
        free (p);
        p = q;
    }
    config.cat_list_head = NULL;
}

CATEGORY *category_new () {
    CATEGORY *cat;

    MY_ALLOC (cat, 1, CATEGORY);

    cat->name = NULL;
    cat->num  = UNKNOWN_CATEGORY;
    cat->type = DEBIT;

    return cat;
}

CATEGORY *get_category_from_line (const char *line) {
    CATEGORY *cat = category_new ();
    char     *elt = NULL;
    int       i = 0, j = 0, k = 0;

    MY_ALLOC (elt, MAXSTRLEN + 1, char);
    for (i = 0 ; i < strlen (line) ; i++) {
        if ( (line[i] != '\t') && (line[i] != '\n') ){
            elt[j] = line[i];
            j++;
        } else {
            elt[j] = 0;
            j      = 0;
            #ifdef DEBUG_GACC
                printf ("Niveau %s -- ",
                        k ==  0 ? "NAME" :
                        k ==  1 ? "NUM " : "TYPE");
                printf ("lement reconnu : %s\n", elt);
            #endif
            switch (k) {
                case  0 : if (elt[0] != 0) {
                              MY_ALLOC (cat->name, strlen (elt) + 1, char);
                              strcpy (cat->name, elt);
                          } else cat->name = NULL;
                          break;
                case  1 : cat->num  = atoi (elt);
                          break;
                case  2 : cat->type = atoi (elt);
                          break;
                default : MY_STOP;
            }
            k++;
        }
    }
    free (elt);

    return cat;
}

char *create_line_from_category (CATEGORY *cat) {
    char *line;

    MY_ALLOC (line, MAXSTRLEN + 1, char);
    sprintf (line, "%s\t%d\t%d\n",
             cat->name != NULL ? cat->name : "",
             cat->num,
             cat->type);
    return line;
}

CAT_ELT *cat_list_elt_new () {
    CAT_ELT *elt;

    MY_ALLOC (elt, 1, CAT_ELT);

    elt->category = NULL;
    elt->next     = NULL;

    return elt;
}

void cat_set_name (CATEGORY *cat, const char *name) {
	if(name == NULL || strlen(name)==0) {
		cat_set_name(cat,_("Unknown"));
	} else {
        if (cat->name != NULL) free (cat->name);
        MY_ALLOC (cat->name, (strlen (name) + 1), char);
        strcpy (cat->name, name);
        config.CatDataModified = VRAI;
    }
}

void cat_set_num (CATEGORY *cat, int num) {
    if (num > UNKNOWN_CATEGORY) {
        cat->num = num;
        config.CatDataModified = VRAI;
    }
}

void cat_set_type (CATEGORY *cat, int type) {
    if ( (type == DEBIT) || (type == CREDIT) ) {
        cat->type = type;
        config.CatDataModified = VRAI;
    }
}

