/* GKrellM
|  Copyright (C) 1999-2002 Bill Wilson
|
|  Author:  Bill Wilson    bill@gkrellm.net
|  Latest versions might be found at:  http://gkrellm.net
|
|  This program is free software which I release under the GNU General Public
|  License. You may redistribute and/or modify this program under the terms
|  of that license as published by the Free Software Foundation; either
|  version 2 of the License, or (at your option) any later version.
|
|  This program is distributed in the hope that it will be useful,
|  but WITHOUT ANY WARRANTY; without even the implied warranty of
|  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
|  GNU General Public License for more details.  Version 2 is in the
|  COPYRIGHT file in the top level directory of this distribution.
| 
|  To get a copy of the GNU General Puplic License, write to the Free Software
|  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/


#include "gkrellm.h"
#include "gkrellm_private_proto.h"


static GList	*gkrellm_alert_list;
static gint		voice_fifo = -1;

static gboolean
gvoice_fifo_open(void)
	{
	gchar	buf[32];

	if (voice_fifo < 0)
		{
		sprintf(buf, "/tmp/gvoice_%08x", getpid());
		if (!access(buf, W_OK))
			voice_fifo = open(buf, O_NONBLOCK|O_WRONLY);
		}
	if (voice_fifo >= 0)
		return TRUE;
	return FALSE;
	}
	
static void
send_gvoice(gchar *str)
	{
	if (!str || !*str || !gvoice_fifo_open())
		return;
	write(voice_fifo, str, strlen(str));
	}

static void
run_command(gchar *command)
	{
	gchar *cmd;

	if (!command || !*command)
		return;
	cmd = g_strdup_printf("%s &", command);
	gkrellm_system(cmd);
	g_free(cmd);
	}

void
gkrellm_render_default_alert_decal(Alert *alert)
	{
	GdkImlibImage	*im;
	AlertDecal		*ad;

	if (!alert)
		return;
	ad = &alert->ad;
	if (ad->w <= 0 || ad->h <= 0)
		return;
	if (alert->high.alarm_on || alert->low.alarm_on)
		gkrellm_get_decal_alarm_image(&im, &ad->nframes);
	else if (alert->high.warn_on || alert->low.warn_on)
		gkrellm_get_decal_warn_image(&im, &ad->nframes);
	else
		return;
	gkrellm_render_to_pixmap(im, &ad->pixmap, &ad->mask, ad->w,
				ad->h * ad->nframes);
	}

static gboolean
create_alert_objects(Alert *alert)
	{
	AlertDecal		*ad;
//	Krell			*k;

	/* Whichever monitor created the alert that is being triggered:
	|  1) has set a panel pointer in the create and is done, so I will make
	|     here a default panel sized alert decal for him, or
	|  2) has not set a panel ptr, but will do so in the callback and then
	|     let me make the default alert, or
	|  3) has done one of the 2 above panel possibilities, and will also
	|     render a custom size/position for the alert decal and set the pixmap.
	|     For this case, the alert decal may be a render of the default decal
	|     image or may be a custom image from the monitor.  The work needed is
	|     setting ad x,y,w,h and nframes and rendering to ad->pixmap.
	*/
	if (alert->cb_trigger)
		(*alert->cb_trigger)(alert, alert->cb_trigger_data);
	if (!alert->panel)
		return FALSE;		/* Monitor may need trigger deferred */
	ad = &alert->ad;
	if (!alert->ad.pixmap /* && style is decal alert */)
		{
		ad->x = 0;
		ad->y = 0;
		ad->w = alert->panel->w;
		ad->h = alert->panel->h;
		gkrellm_render_default_alert_decal(alert);
		}
	/* Don't let create_decal append the decal, I want to insert it first so
	|  it will appear under all other panel decals.
	*/
	ad->decal = gkrellm_create_decal_pixmap(NULL, ad->pixmap, ad->mask,
				ad->nframes, NULL, ad->x, ad->y);
	gkrellm_insert_decal(alert->panel, ad->decal, FALSE /* prepend */);

#if 0
	k = gkrellm_create_krell(NULL,
			gkrellm_krell_mini_image(), gkrellm_krell_mini_style());
	k->y0 = 5;
	k->full_scale = 100;

	alert->ak.krell = k;
	gkrellm_insert_krell(alert->panel, k, TRUE);
#endif
	return TRUE;
	}

static void
destroy_alert_objects(Alert *alert)
	{
//	if (alert->cb_stop)
//		(*alert->cb_stop)(alert, alert->cb_stop_data);
	gkrellm_remove_and_destroy_decal(alert->panel, alert->ad.decal);
	if (alert->ad.pixmap)
		gdk_pixmap_unref(alert->ad.pixmap);
	alert->ad.pixmap = NULL;
	alert->ad.mask = NULL;
	alert->ad.decal = NULL;
	gkrellm_remove_and_destroy_krell(alert->panel, alert->ak.krell);
	alert->ak.krell = NULL;
	}

static void
trigger_warn(Alert *alert, Trigger *trigger)
	{
	if (!trigger->warn_on)
		{
		trigger->warn_on = TRUE;
		create_alert_objects(alert);
		gkrellm_alert_list = g_list_append(gkrellm_alert_list, alert);
		if (!alert->suppress_command)
			run_command(alert->warn_command);
		alert->suppress_command = FALSE;
		alert->warn_repeat = alert->warn_repeat_set;
		send_gvoice(alert->voice_string);
		alert->voice_repeat = alert->voice_repeat_set;
		}
	}

static void
stop_warn(Alert *alert, Trigger *trigger)
	{
	if (trigger->warn_on)
		{
		trigger->warn_on = FALSE;
		destroy_alert_objects(alert);
		gkrellm_alert_list = g_list_remove(gkrellm_alert_list, alert);
		alert->warn_repeat = 0;
		alert->suppress_command = FALSE;
		}
	}

static void
trigger_alarm(Alert *alert, Trigger *trigger)
	{
	if (!trigger->alarm_on)
		{
		trigger->alarm_on = TRUE;
		create_alert_objects(alert);
		gkrellm_alert_list = g_list_append(gkrellm_alert_list, alert);
		if (!alert->suppress_command)
			run_command(alert->alarm_command);
		alert->suppress_command = FALSE;
		alert->alarm_repeat = alert->alarm_repeat_set;
		send_gvoice(alert->voice_string);
		alert->voice_repeat = alert->voice_repeat_set;
		}
	}

static void
stop_alarm(Alert *alert, Trigger *trigger)
	{
	if (trigger->alarm_on)
		{
		trigger->alarm_on = FALSE;
		destroy_alert_objects(alert);
		gkrellm_alert_list = g_list_remove(gkrellm_alert_list, alert);
		alert->alarm_repeat = 0;;
		alert->suppress_command = FALSE;
		}
	}

void
gkrellm_freeze_alert(Alert *alert)
	{
	if (alert)
		alert->freeze = TRUE;
	}

void
gkrellm_thaw_alert(Alert *alert)
	{
	if (alert)
		alert->freeze = FALSE;
	}

void
gkrellm_check_alert(Alert *alert, gfloat value)
	{
	if (!alert || !alert->activated || alert->freeze || !GK.initialized)
		return;
	if (alert->check_low)
		{
		if (value <= alert->low.alarm_limit)
			{				
			stop_alarm(alert, &alert->high);
			stop_warn(alert, &alert->high);
			stop_warn(alert, &alert->low);
			trigger_alarm(alert, &alert->low);
			}
		else
			{
			stop_alarm(alert, &alert->low);
			if (value <= alert->low.warn_limit)
				{
				stop_alarm(alert, &alert->high);
				stop_warn(alert, &alert->high);
				trigger_warn(alert, &alert->low);
				}
			else
				stop_warn(alert, &alert->low);
			}
		}
	if (alert->check_high)
		{
		if (value >= alert->high.alarm_limit)
			{
			stop_warn(alert, &alert->high);
			stop_warn(alert, &alert->low);
			stop_alarm(alert, &alert->low);
			trigger_alarm(alert, &alert->high);
			}
		else
			{
			stop_alarm(alert, &alert->high);
			if (value >= alert->high.warn_limit)
				{
				stop_warn(alert, &alert->low);
				stop_alarm(alert, &alert->low);
				trigger_warn(alert, &alert->high);
				}
			else
				stop_warn(alert, &alert->high);
			}
		}
	}

void
gkrellm_alert_trigger_connect(Alert *alert, void (*func)(), gpointer data)
	{
	if (!alert)
		return;
	alert->cb_trigger = func;
	alert->cb_trigger_data = data;
	}

void
gkrellm_alert_stop_connect(Alert *alert, void (*func)(), gpointer data)
	{
	if (!alert)
		return;
	alert->cb_stop = func;
	alert->cb_stop_data = data;
	}

static void
destroy_alert(Alert *alert)
	{
	if (g_list_find(gkrellm_alert_list, alert))
		{
		gkrellm_alert_list = g_list_remove(gkrellm_alert_list, alert);
		destroy_alert_objects(alert);
		}
	g_free(alert->name);
	g_free(alert->unit_string);
	g_free(alert->alarm_command);
	g_free(alert->warn_command);
	g_free(alert);
	}

void
gkrellm_alert_destroy(Alert **alert)
	{
	if (!alert || !*alert)
		return;
	gkrellm_alert_window_destroy(alert);
	destroy_alert(*alert);
	*alert = NULL;
	}

Alert *
gkrellm_alert_create(Panel *p, gchar *name, gchar *unit_string,
		gboolean check_high, gboolean check_low, gboolean do_updates,
		gfloat max_high, gfloat min_low,
		gfloat step0, gfloat step1, gint digits)
	{
	Alert	*alert;

	alert = g_new0(Alert, 1);
	alert->panel = p;
	alert->name = g_strdup(name);
	alert->unit_string = g_strdup(unit_string ? unit_string : "");
	alert->check_high = check_high;
	alert->check_low = check_low;
	alert->do_panel_updates = do_updates;
	alert->max_high = max_high;
	alert->min_low = min_low;
	alert->step0 = step0;
	alert->step1 = step1;
	alert->digits = digits;
	return alert;
	}

static void
reset_alert(Alert *alert)
	{
	Trigger	*th = &alert->high,
			*tl = &alert->low;

	if (   ((th->alarm_on || tl->alarm_on) && alert->alarm_repeat_set == 0)
		|| ((th->warn_on || tl->warn_on) && alert->warn_repeat_set == 0)
	   )
		alert->suppress_command = TRUE;
	th->alarm_on = FALSE;
	th->warn_on = FALSE;
	tl->alarm_on = FALSE;
	tl->warn_on = FALSE;
	alert->alarm_repeat = 0;
	alert->warn_repeat = 0;
	destroy_alert_objects(alert);
	}

void
gkrellm_reset_alert(Alert *alert)
	{
	GList	*list;

	if (!alert)
		return;
	list = g_list_find(gkrellm_alert_list, alert);
	if (list)
		{
		reset_alert(alert);
		gkrellm_alert_list = g_list_remove(gkrellm_alert_list, alert);
		}
	}

void
gkrellm_reset_panel_alerts(Panel *p)
	{
	GList		*list;
	Alert		*alert;
	gboolean	done = FALSE;

	if (!p)
		return;
	while (!done)
		{
		done = TRUE;	/* Assume won't find any */
		for (list = gkrellm_alert_list; list; list = list->next)
			{
			alert = (Alert *) list->data;
			if (alert->panel != p)
				continue;
			done = FALSE;
			reset_alert(alert);
			gkrellm_alert_list = g_list_remove(gkrellm_alert_list, alert);
			break;
			}
		}
	}

  /* At theme changes, turn all alerts off so there won't be any alert decals
  |  in monitor lists when they are destroyed.  The alerts should just get
  |  retriggered.
  */
void
reset_all_alerts(void)
	{
	GList		*list;

	for (list = gkrellm_alert_list; list; list = list->next)
		reset_alert((Alert *) list->data);
	g_list_free(gkrellm_alert_list);
	gkrellm_alert_list = NULL;
	}

void
update_alerts(void)
	{
	GList		*list;
	Alert		*alert;
	AlertDecal	*ad;
	AlertKrell	*ak;

	for (list = gkrellm_alert_list; list; list = list->next)
		{
		alert = (Alert *) list->data;
		ad = &alert->ad;
		ak = &alert->ak;
		if (ak->krell)
			{
			ak->krell_position = (ak->krell_position + 2) % 100;
			gkrellm_update_krell(alert->panel, ak->krell, ak->krell_position);
			}
		if (ad->decal)
			{
			if (ad->frame <= 0)
				ad->dir = 1;
			else if (ad->frame >= ad->nframes - 1)
				ad->dir = 0;
			ad->frame += (ad->dir) ? 1 : -1;
			gkrellm_draw_decal_pixmap(alert->panel, ad->decal, ad->frame);
			}
		if (alert->do_panel_updates)
			gkrellm_draw_panel_layers(alert->panel);
		if (GK.second_tick)
			{
			if (alert->alarm_repeat > 0 && --alert->alarm_repeat == 0)
				{
				run_command(alert->alarm_command);
				alert->alarm_repeat = alert->alarm_repeat_set;
				}
			if (alert->warn_repeat > 0 && --alert->warn_repeat == 0)
				{
				run_command(alert->warn_command);
				alert->warn_repeat = alert->warn_repeat_set;
				}
			if (alert->voice_repeat > 0 && --alert->voice_repeat == 0)
				{
				send_gvoice(alert->voice_string);
				alert->voice_repeat = alert->voice_repeat_set;
				}
			}
		}
	}

/* ------------------------------------------------------------ */
void
gkrellm_alert_config_connect(Alert *alert, void (*func)(), gpointer data)
	{
	if (!alert)
		return;
	alert->cb_config = func;
	alert->cb_config_data = data;
	}

static void
alert_delete(GtkWidget *widget, Alert **ap)
	{
	Alert	*alert;

	if (!ap)
		return;
	alert = *ap;
	alert->activated = FALSE;
	if (alert->cb_config)
		(*alert->cb_config)(alert, alert->cb_config_data);
	if (alert->config_window)
		gtk_widget_destroy(alert->config_window);
	destroy_alert(alert);
	*ap = NULL;
	}

static void
alert_close(GtkWidget *widget, Alert **alert)
	{
	Alert	*ap;

	if (!alert)
		return;
	ap = *alert;
	if (!ap->activated)
		alert_delete(NULL, alert);
	else if (ap->config_window)
		{
		gtk_widget_destroy(ap->config_window);
		ap->config_window = NULL;
		}
	}

void
gkrellm_alert_window_destroy(Alert **alert)
	{
	alert_close(NULL, alert);
	}

static gint
alert_config_window_delete_event(GtkWidget *widget, GdkEvent *ev,
		Alert **alert)
	{
	alert_close(widget, alert);
	return FALSE;
	}

static void
alert_apply(GtkWidget *widget, Alert **ap)
	{
	Alert	*alert;
	gchar	*s;

	GtkSpinButton	*spin;

	alert = *ap;
	if (alert->check_high)
		{
		spin = GTK_SPIN_BUTTON(alert->high.alarm_limit_spin_button);
		alert->high.alarm_limit = gtk_spin_button_get_value_as_float(spin);
		spin = GTK_SPIN_BUTTON(alert->high.warn_limit_spin_button);
		alert->high.warn_limit = gtk_spin_button_get_value_as_float(spin);
		}
	if (alert->check_low)
		{
		spin = GTK_SPIN_BUTTON(alert->low.alarm_limit_spin_button);
		alert->low.alarm_limit = gtk_spin_button_get_value_as_float(spin);
		spin = GTK_SPIN_BUTTON(alert->low.warn_limit_spin_button);
		alert->low.warn_limit = gtk_spin_button_get_value_as_float(spin);
		}
	spin = GTK_SPIN_BUTTON(alert->alarm_repeat_spin_button);
	alert->alarm_repeat_set = gtk_spin_button_get_value_as_int(spin);
	spin = GTK_SPIN_BUTTON(alert->warn_repeat_spin_button);
	alert->warn_repeat_set = gtk_spin_button_get_value_as_int(spin);
	if (alert->voice_repeat_spin_button)
		{
		spin = GTK_SPIN_BUTTON(alert->voice_repeat_spin_button);
		alert->voice_repeat_set = gtk_spin_button_get_value_as_int(spin);
		}
	s = gkrellm_entry_get_text(&alert->alarm_command_entry);
	gkrellm_dup_string(&alert->alarm_command, s);
	if (!*s)
		alert->alarm_repeat_set = 0;
	if (alert->high.alarm_on || alert->low.alarm_on)
		alert->alarm_repeat = alert->alarm_repeat_set;

	s = gkrellm_entry_get_text(&alert->warn_command_entry);
	gkrellm_dup_string(&alert->warn_command, s);
	if (!*s)
		alert->warn_repeat_set = 0;
	if (alert->high.warn_on || alert->low.warn_on)
		alert->warn_repeat = alert->warn_repeat_set;

	if (alert->voice_string_entry)
		{
		s = gkrellm_entry_get_text(&alert->voice_string_entry);
		gkrellm_dup_string(&alert->voice_string, s);
		if (!*s)
			alert->voice_repeat_set = 0;
		alert->voice_repeat = alert->voice_repeat_set;
		}
	alert->activated = TRUE;
	if (alert->cb_config)
		(*alert->cb_config)(alert, alert->cb_config_data);
	gkrellm_config_modified();
	}

static void
alert_ok(GtkWidget *widget, Alert **alert)
	{
	alert_apply(NULL, alert);
	alert_close(NULL, alert);
	}

static void
cb_high_alarm_spin_changed(GtkAdjustment *adjustment, Alert *alert)
	{
	GtkSpinButton	*spin;
	gfloat			alarm, warn;

	spin = GTK_SPIN_BUTTON(alert->high.alarm_limit_spin_button);
	alarm = gtk_spin_button_get_value_as_float(spin);
	spin = GTK_SPIN_BUTTON(alert->high.warn_limit_spin_button);
	warn = gtk_spin_button_get_value_as_float(spin);
	if (alarm < warn)
		gtk_spin_button_set_value(spin, alarm);
	}

static void
cb_high_warn_spin_changed(GtkWidget *adjustment, Alert *alert)
	{
	GtkSpinButton	*spin;
	gfloat			alarm, warn, low_warn;

	spin = GTK_SPIN_BUTTON(alert->high.warn_limit_spin_button);
	warn = gtk_spin_button_get_value_as_float(spin);
	spin = GTK_SPIN_BUTTON(alert->high.alarm_limit_spin_button);
	alarm = gtk_spin_button_get_value_as_float(spin);
	if (alarm < warn)
		gtk_spin_button_set_value(spin, warn);
	if (alert->check_low)
		{
		spin = GTK_SPIN_BUTTON(alert->low.warn_limit_spin_button);
		low_warn = gtk_spin_button_get_value_as_float(spin);
		if (low_warn > warn)
			gtk_spin_button_set_value(spin, warn);
		}
	}

static void
cb_low_warn_spin_changed(GtkWidget *adjustment, Alert *alert)
	{
	GtkSpinButton	*spin;
	gfloat			alarm, warn, high_warn;

	spin = GTK_SPIN_BUTTON(alert->low.warn_limit_spin_button);
	warn = gtk_spin_button_get_value_as_float(spin);
	spin = GTK_SPIN_BUTTON(alert->low.alarm_limit_spin_button);
	alarm = gtk_spin_button_get_value_as_float(spin);
	if (alarm > warn)
		gtk_spin_button_set_value(spin, warn);
	if (alert->check_high)
		{
		spin = GTK_SPIN_BUTTON(alert->high.warn_limit_spin_button);
		high_warn = gtk_spin_button_get_value_as_float(spin);
		if (high_warn < warn)
			gtk_spin_button_set_value(spin, warn);
		}
	}

static void
cb_low_alarm_spin_changed(GtkWidget *adjustment, Alert *alert)
	{
	GtkSpinButton	*spin;
	gfloat			alarm, warn;

	spin = GTK_SPIN_BUTTON(alert->low.alarm_limit_spin_button);
	alarm = gtk_spin_button_get_value_as_float(spin);
	spin = GTK_SPIN_BUTTON(alert->low.warn_limit_spin_button);
	warn = gtk_spin_button_get_value_as_float(spin);
	if (alarm > warn)
		gtk_spin_button_set_value(spin, alarm);
	}

void
gkrellm_alert_config_window(Alert **alert)
	{
	GtkWidget	*main_vbox, *vbox, *vbox1, *hbox;
	GtkWidget	*table;
	GtkWidget	*button;
	GtkWidget	*label;
	GtkWidget	*separator;
	Alert		*ap;
	gchar		*title;

	if (!alert || !*alert)
		return;
	ap = *alert;
	if (!ap->config_window)
		{
		ap->config_window = gtk_window_new(GTK_WINDOW_TOPLEVEL);
		gtk_signal_connect(GTK_OBJECT(ap->config_window), "delete_event",
				(GtkSignalFunc) alert_config_window_delete_event, alert);
		gtk_window_set_policy(GTK_WINDOW(ap->config_window),
				FALSE, FALSE, TRUE);
		gtk_window_set_title(GTK_WINDOW(ap->config_window),
				_("GKrellM Set Alerts"));
	
		main_vbox = gtk_vbox_new(FALSE, 0);
		gtk_container_add(GTK_CONTAINER(ap->config_window), main_vbox);
		
		title = g_strdup_printf("%s - %s", ap->name, ap->unit_string);
		vbox = gkrellm_framed_vbox(main_vbox, title, 4, FALSE, 4, 3);
		g_free(title);

		hbox = gtk_hbox_new(FALSE, 0);
		gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 0);
		if (ap->check_high)
			{
			vbox1 = gkrellm_framed_vbox(hbox, _("High Limits"), 2, FALSE, 2,2);
			gkrellm_spin_button(vbox1, &ap->high.alarm_limit_spin_button,
					ap->high.alarm_limit, ap->min_low, ap->max_high,
					ap->step0, ap->step1,
					ap->digits, 70, cb_high_alarm_spin_changed, ap, FALSE,
					_("High alarm limit"));
		
			gkrellm_spin_button(vbox1, &ap->high.warn_limit_spin_button,
					ap->high.warn_limit, ap->min_low, ap->max_high,
					ap->step0, ap->step1,
					ap->digits, 70, cb_high_warn_spin_changed, ap, FALSE,
					_("High warn limit"));
			}

		if (ap->check_low)
			{
			vbox1 = gkrellm_framed_vbox_end(hbox, _("Low Limits"),
					2, FALSE, 2, 2);
			gkrellm_spin_button(vbox1, &ap->low.warn_limit_spin_button,
					ap->low.warn_limit, ap->min_low, ap->max_high,
					ap->step0, ap->step1,
					ap->digits, 70, cb_low_warn_spin_changed, ap, FALSE,
					_("Low warn limit"));
			gkrellm_spin_button(vbox1, &ap->low.alarm_limit_spin_button,
					ap->low.alarm_limit, ap->min_low, ap->max_high,
					ap->step0, ap->step1,
					ap->digits, 70, cb_low_alarm_spin_changed, ap, FALSE,
					_("Low alarm limit"));
			}
		vbox1 = gkrellm_framed_vbox(vbox,
				_("Commands - with repeat intervals in seconds"),
				2, FALSE, 2, 2);

		table = gtk_table_new(3 /* across */, 3 /* down */, FALSE);
		gtk_table_set_col_spacings(GTK_TABLE(table), 4);
		gtk_box_pack_start(GTK_BOX(vbox1), table, FALSE, FALSE, 0);

		if (!ap->alarm_command)
			ap->alarm_command = g_strdup("");
		label = gtk_label_new(_("Alarm command:"));
		gtk_misc_set_alignment(GTK_MISC(label), 1.0, 0.5);
		gtk_table_attach_defaults(GTK_TABLE(table), label, 0, 1, 0, 1);
	    ap->alarm_command_entry = gtk_entry_new_with_max_length(255);
		gtk_table_attach_defaults(GTK_TABLE(table), ap->alarm_command_entry,
				1, 2, 0, 1);
		gtk_entry_set_text(GTK_ENTRY(ap->alarm_command_entry),
				ap->alarm_command);
		gkrellm_spin_button(NULL, &ap->alarm_repeat_spin_button,
				ap->alarm_repeat_set, 0, 1000,
				1, 10, 0, 60, NULL, NULL, FALSE, NULL);
		gtk_table_attach_defaults(GTK_TABLE(table),
				ap->alarm_repeat_spin_button, 2, 3, 0, 1);

		if (!ap->warn_command)
			ap->warn_command = g_strdup("");
		label = gtk_label_new(_("Warn command:"));
		gtk_misc_set_alignment(GTK_MISC(label), 1.0, 0.5);
		gtk_table_attach_defaults(GTK_TABLE(table), label, 0, 1, 1, 2);
	    ap->warn_command_entry = gtk_entry_new_with_max_length(255);
		gtk_table_attach_defaults(GTK_TABLE(table), ap->warn_command_entry,
				1, 2, 1, 2);
		gtk_entry_set_text(GTK_ENTRY(ap->warn_command_entry),
				ap->warn_command);
		gtk_widget_set_usize(ap->warn_command_entry, 300, 0);
		gkrellm_spin_button(NULL, &ap->warn_repeat_spin_button,
				ap->warn_repeat_set, 0, 1000,
				1, 10, 0, 60, NULL, NULL, FALSE, NULL);
		gtk_table_attach_defaults(GTK_TABLE(table),
				ap->warn_repeat_spin_button, 2, 3, 1, 2);

		if (!ap->voice_string)
			ap->voice_string = g_strdup("");
		if (gvoice_fifo_open())
			{
			label = gtk_label_new(_("Voice string:"));
			gtk_misc_set_alignment(GTK_MISC(label), 1.0, 0.5);
			gtk_table_attach_defaults(GTK_TABLE(table), label, 0, 1, 2, 3);
		    ap->voice_string_entry = gtk_entry_new_with_max_length(255);
			gtk_table_attach_defaults(GTK_TABLE(table), ap->voice_string_entry,
					1, 2, 2, 3);
			gtk_entry_set_text(GTK_ENTRY(ap->voice_string_entry),
					ap->voice_string);
			gtk_widget_set_usize(ap->voice_string_entry, 300, 0);
			gkrellm_spin_button(NULL, &ap->voice_repeat_spin_button,
					ap->voice_repeat_set, 0, 1000,
					1, 10, 0, 60, NULL, NULL, FALSE, NULL);
			gtk_table_attach_defaults(GTK_TABLE(table),
					ap->voice_repeat_spin_button, 2, 3, 2, 3);
			}
		else
			{
			ap->voice_string_entry = NULL;
			ap->voice_repeat_spin_button = NULL;
			}

		separator = gtk_hseparator_new();
		gtk_box_pack_start(GTK_BOX(vbox1), separator, FALSE, FALSE, 4);

		label = gtk_label_new(_(
			"A repeat of zero seconds executes the command once per alert."));
		gtk_misc_set_alignment(GTK_MISC(label), 0.0, 0.5);
		gtk_label_set_justify(GTK_LABEL(label), GTK_JUSTIFY_LEFT);
		gtk_box_pack_start(GTK_BOX(vbox1), label, FALSE, FALSE, 0);

		hbox = gtk_hbutton_box_new();
		gtk_button_box_set_layout(GTK_BUTTON_BOX(hbox), GTK_BUTTONBOX_END);
		gtk_button_box_set_spacing(GTK_BUTTON_BOX(hbox), 5);
		gtk_box_pack_start(GTK_BOX(main_vbox), hbox, FALSE, FALSE, 0);

		button = gtk_button_new_with_label(_("OK"));
		GTK_WIDGET_SET_FLAGS(button, GTK_CAN_DEFAULT);
		gtk_signal_connect(GTK_OBJECT(button), "clicked",
				(GtkSignalFunc) alert_ok, alert);
	    gtk_box_pack_start(GTK_BOX(hbox), button, TRUE, TRUE, 0);
		gtk_widget_grab_default(button);

		button = gtk_button_new_with_label(_("Apply"));
		GTK_WIDGET_SET_FLAGS(button, GTK_CAN_DEFAULT);
		gtk_signal_connect(GTK_OBJECT(button), "clicked",
				(GtkSignalFunc) alert_apply, alert);
	    gtk_box_pack_start(GTK_BOX(hbox), button, TRUE, TRUE, 0);

		button = gtk_button_new_with_label(_("Delete"));
		GTK_WIDGET_SET_FLAGS(button, GTK_CAN_DEFAULT);
		gtk_signal_connect(GTK_OBJECT(button), "clicked",
				(GtkSignalFunc) alert_delete, alert);
	    gtk_box_pack_start(GTK_BOX(hbox), button, TRUE, TRUE, 0);

		button = gtk_button_new_with_label(_("Close"));
		GTK_WIDGET_SET_FLAGS(button, GTK_CAN_DEFAULT);
		gtk_signal_connect(GTK_OBJECT(button), "clicked",
				(GtkSignalFunc) alert_close, alert);
	    gtk_box_pack_start(GTK_BOX(hbox), button, TRUE, TRUE, 0);

		gtk_widget_show_all(ap->config_window);
		}
	else
		gdk_window_raise(ap->config_window->window);
	}


/* Example:
sensor alert_config +5v alarm_command blah blah
sensor alert_config +5v warn_command blah blah
sensor alert_config +5v voice_string blah blah
sensor alert_config +5v values do_panel_update check_high check_low check_bool
sensor alert_config +5v repeat alarm_repeat warn_repeat voice_repeat
sensor alert_config +5v limits high.alarm_limit high.warm_limit low.warn_limit low.alarm_limit
*/

void
gkrellm_save_alertconfig(FILE *f, Alert *alert, gchar *mon_keyword,gchar *name)
	{
	gchar		*s;

	if (!f || !alert || !mon_keyword)
		return;
	if (name)
		s = g_strdup_printf("%s %s %s ", mon_keyword,
				GKRELLM_ALERTCONFIG_KEYWORD, name);
	else
		s = g_strdup_printf("%s %s ", mon_keyword,GKRELLM_ALERTCONFIG_KEYWORD);

	if (alert->alarm_command && *alert->alarm_command)
		fprintf(f, "%s alarm_command %s\n", s, alert->alarm_command);
	if (alert->warn_command && *alert->warn_command)
		fprintf(f, "%s warn_command %s\n", s, alert->warn_command);
	if (alert->voice_string && *alert->voice_string)
		fprintf(f, "%s voice_string %s\n", s, alert->voice_string);
	fprintf(f, "%s values %d %d %d %d\n", s, alert->do_panel_updates,
			alert->check_high, alert->check_low, alert->check_boolean);
	fprintf(f, "%s repeat %d %d %d\n", s, alert->alarm_repeat_set,
			alert->warn_repeat_set, alert->voice_repeat_set);
	fprintf(f, "%s limits %.2f %.2f %.2f %.2f\n", s,
			alert->high.alarm_limit, alert->high.warn_limit,
			alert->low.warn_limit, alert->low.alarm_limit);
	g_free(s);
	}

void
gkrellm_load_alertconfig(Alert **ap, gchar *config_line)
	{
	Alert		*alert;
	gchar		config[32], item[CFG_BUFSIZE];
	gint		n;

	if (!ap || !config_line)
		return;
	if (!*ap)
		*ap = g_new0(Alert, 1);
	alert = *ap;

	n = sscanf(config_line, "%31s %[^\n]", config, item);
	if (n != 2)
		return;

	if (!strcmp(config, "alarm_command"))
		gkrellm_dup_string(&alert->alarm_command, item);
	else if (!strcmp(config, "warn_command"))
		gkrellm_dup_string(&alert->warn_command, item);
	else if (!strcmp(config, "voice_string"))
		gkrellm_dup_string(&alert->voice_string, item);
	else if (!strcmp(config, "values"))
		sscanf(item, "%d %d %d %d", &alert->do_panel_updates,
				&alert->check_high, &alert->check_low, &alert->check_boolean);
	else if (!strcmp(config, "repeat"))
		sscanf(item, "%d %d %d", &alert->alarm_repeat_set,
				&alert->warn_repeat_set, &alert->voice_repeat_set);
	else if (!strcmp(config, "limits"))
		sscanf(item, "%f %f %f %f",
				&alert->high.alarm_limit, &alert->high.warn_limit,
				&alert->low.warn_limit, &alert->low.alarm_limit);

	alert->activated = TRUE;
	}
