/* GKrellM
|  Copyright (C) 1999-2002 Bill Wilson
|
|  Author:  Bill Wilson    bill@gkrellm.net
|  Latest versions might be found at:  http://gkrellm.net
|
|  This program is free software which I release under the GNU General Public
|  License. You may redistribute and/or modify this program under the terms
|  of that license as published by the Free Software Foundation; either
|  version 2 of the License, or (at your option) any later version.
|
|  This program is distributed in the hope that it will be useful,
|  but WITHOUT ANY WARRANTY; without even the implied warranty of
|  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
|  GNU General Public License for more details.  Version 2 is in the
|  COPYRIGHT file in the top level directory of this distribution.
| 
|  To get a copy of the GNU General Puplic License, write to the Free Software
|  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

/*
| 5/13/2002 Nick Mitchell <firepile@speakeasy.net> patch:
|   Fix IMAP login fail if password had certain chars.
| 12/28/2001 Patch from Hajimu UMEMOTO ume@mahoroba.org
|   CRAM-MD5 password authentication for POP3 and IMAP mail checking.
| 12/4/2001 Patch from Hajimu UMEMOTO ume@mahoroba.org
|   APOP password authentication for POP3 checking.
| 10/7/2001 Patch from Andrew Lombardi <andrew@mysticcoders.com>
|	Added Evolution mail status check to status_is_old().
| 6/7/2001 Patch from Jason Lunz <j@falooley.org>
|	Added multimua option.
| 2/6/2001 Patch from Laurent Caillat-Vallet <caillat@in2p3.fr>
|	POP3/IMAP connections can be on non-standard port numbers.
| 1/14/2001 Patch from Masaharu FUJITA <fujita@hpc.cs.ehime-u.ac.jp>
|	The new mail krell animation for BSD is now a Daemon.
| 11/30/2000 Patch from Adam Sampson <ats1@ukc.ac.uk>
|	Added MH mail dir checking, check_mh_dir() 
| 10/19/2000 Patch from Matt McClure <mlm@aya.yale.edu>
|	Added tooltip reporting of new mail counts for mailboxes checked via the
|	Mail fetch/check program.
|  7/4/2000 Patch from Michael Bell <mikeb@map.com>
|	Added check_mail_pop3() and check_mail_imap() mail box checking routines.
|	Also added reporting the new mail counts for these to the tooltip popup.
|	Hajimu UMEMOTO added IPv6 awareness to these routines.
|  4/25/2000 Patch from Stefan Schlott <stefan.schlott@student.uni-ulm.de>
|	Added option: If there is new mail, the names of the mailboxes
|	containing new mail can be displayed in a tooltip.
|  3/16/2000 Patch from Stefan Schlott <stefan.schlott@student.uni-ulm.de>
|	Added a check to avoid double counting old mail with status lines
|	from Netscape and standard "Status" mail readers.  Also added
|	is_multipart_mail() to avoid counting mime attachments as messages.
|  2/24/2000 Patch from Darko Koruga <darko.koruga@hermes.si>
|	Continuous mail animation option as long as there is new mail.
| 12/16/1999 Patch from Patrick Morris <pmorris@starnixsolutions.com>
|	check_maildir() accepts procmail '_' messages.
| 10/30/1999 Patch from Patrick Crosby <xb@dotfiles.com>
|	check_maildir() function.
*/

#include "gkrellm.h"
#include "gkrellm_private_proto.h"

#include <utime.h>
#include <dirent.h>
#include <sys/time.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netdb.h>
#include <errno.h>

#include "gkrellm_threads.h"

#include	"pixmaps/mail/decal_mail.xpm"

#if defined(__linux__)
#if defined(__GLIBC__) && ((__GLIBC__>2)||(__GLIBC__==2 && __GLIBC_MINOR__>=1))
#define HAVE_GETADDRINFO	1
#endif
#endif

#if defined(__FreeBSD__)
#if __FreeBSD_version >= 400000
#define HAVE_GETADDRINFO	1
#endif
#define HAVE_MD5_H
#endif

#if defined(__NetBSD__) || defined(__OpenBSD__)
#define HAVE_GETADDRINFO	1
#define HAVE_MD5_H
#endif

#if defined(__solaris__)
/* getaddrinfo is related to IPv6 stuff */
# include <netconfig.h>
# if defined(NC_INET6)
#  define HAVE_GETADDRINFO	1
# endif
#endif

#if defined(HAVE_MD5_H)
#include <md5.h>
#else
#include "md5.h"
#endif

#define MUTE_FLAG	-1

/* msg_count_mode has 3 states
*/
#define	MSG_NEW_TOTAL_COUNT	0
#define	MSG_NEW_COUNT		1
#define	MSG_NO_COUNT		2

/* animation_select_mode states
*/
#define	ANIMATION_NONE			0
#define	ANIMATION_ENVELOPE		1
#define	ANIMATION_PENGUIN		2
#define	ANIMATION_BOTH			3

/* # of seconds to wait for a response from a POP3 or IMAP server
*/
#define TCP_TIMEOUT			30
#define	DEFAULT_POP3_PORT	"110"
#define	DEFAULT_IMAP_PORT	"143"


  /* A mailbox type has bits encoding how to check the mailbox (inline code
  |  check or threaded check).
  |  Threaded checks and the fetch program are remote checks
  */
#define	MBOX_CHECK_FETCH		0x1000
#define	MBOX_CHECK_INLINE		0x2000
#define	MBOX_CHECK_THREADED		0x4000
#define MBOX_CHECK_TYPE_MASK	0xf000

  /* Counts for mailboxes created and checked in other plugins can be shown */
#define	MBOX_EXTERNAL		0x10

  /* Mailboxes internally checked and created via the Mail->Mailboxes config */
#define MBOX_INTERNAL		0x20


  /* Here's the list of all the mailbox types the Mail monitor knows about.
  |  The MBOX_FETCH is a pseudo internal mailbox where the counts read from
  |  the fetch program are kept.  Additionally MBOX_FETCH_TOOLTIP types
  |  are constructed just so the fetch programs output lines can be 
  |  reported in a tooltip.  Real mailboxes that GKrellM creates in its
  |  config and knows how to check have MBOX_INTERNAL set.  And
  |  finally there can be external (plugin) mailboxes created which
  |  can have their check function called at the update intervals.  If the
  |  plugin reports back the count results, the animation/sound can be
  |  triggered for the plugin.  (Don't know if EXTERNAL guys will ever be used)
  |  Internal mailboxes can be remote or local.  Remote mailboxes have an
  |  authorization protocol that subdivides them into types.  Local mailboxes
  |  currently have separate mboxtype values but I may later group them
  |  into a MBOX_LOCAL type with a subdivision protocol like is currently
  |  done for remote mailboxes.
  */
#define	MBOX_FETCH		(MBOX_CHECK_FETCH)
#define	MBOX_MBOX		(MBOX_CHECK_INLINE   | MBOX_INTERNAL | 0)
#define	MBOX_MAILDIR	(MBOX_CHECK_INLINE   | MBOX_INTERNAL | 1)
#define	MBOX_MH_DIR		(MBOX_CHECK_INLINE   | MBOX_INTERNAL | 2)
#define	MBOX_REMOTE		(MBOX_CHECK_THREADED | MBOX_INTERNAL | 3)
#define	MBOX_FETCH_TOOLTIP	(6)

#define	MBOX_LOCAL_PLUGIN	(MBOX_CHECK_INLINE   | MBOX_EXTERNAL)
#define	MBOX_REMOTE_PLUGIN	(MBOX_CHECK_THREADED | MBOX_EXTERNAL)

#define	AUTH_POP3_PLAIN		0
#define	AUTH_POP3_APOP		1
#define	AUTH_POP3_CRAM		2
#define	AUTH_IMAP_PLAIN		3
#define	AUTH_IMAP_CRAM		4
#define	N_AUTH_PROTOCOLS	5

  /* Authorization protocol strings to write into the config for remote
  |  mailboxes.  The order must match the above AUTH_ defines.
  */
static gchar	*auth_strings[N_AUTH_PROTOCOLS] =
	{
	"POP3",
	"POP3_(APOP)",
	"POP3_(CRAM-MD5)",
	"IMAP",
	"IMAP_(CRAM-MD5)"
	};

  /* Save local mailbox type strings in the config in case I later change
  |  to an option_menu selection for subdividing a MBOX_LOCAL type.
  |  Currently local mailbox types are determined in get_local_mboxtype().
  */
static gchar	*mbox_strings[3] =
	{
	"mbox",
	"Maildir",
	"MH_mail"
	};

static Monitor	*mon_mail;

typedef struct
	{
	gchar	*path;
	gchar	*server;
	gchar	*username;
	gchar	*password;
	gchar	*imapfolder;
	gint	mboxtype;
	gint	authtype;
	gint	port;
	}
	MailAccount;

typedef struct
	{
	MailAccount	*account;
	gboolean	busy;
	GString		*tcp_in;
	gboolean	(*check_func)();
	gpointer	data;			/* For external mailboxes (in plugins) */

	gint		mail_count;
	gint		new_mail_count;
	gint		old_mail_count;
	gint		prev_mail_count,
				prev_new_mail_count;
	time_t		last_mtime;
	off_t		last_size;
	gboolean	is_internal;	/* Internal mail message (ie: localmachine) */
	gboolean	need_animation,
				prev_need_animation;
	gchar		*warn_msg;
	gboolean	warned;
	void		*private;
	}
	Mailbox;

static GList	*mailbox_list;

typedef struct
	{
	gchar	*command;
	GString	*read_gstring;			/* Bytes read from pipe stored here */
	FILE	*pipe;
	}
	Mailproc;

Mailbox			*mail_fetch;		/* Internal mailbox: fetch command */

static Mailproc	mail_user_agent;
static gchar	*mail_notify;		/* Sound		*/

static GdkImlibImage *decal_mail_image;
static GdkPixmap	*decal_mail_pixmap;
static GdkBitmap	*decal_mail_mask;

static gint		run_animation,
				decal_frame;

static gint		mail_fetch_timeout = 5;			/* Minutes */
static gint		mail_check_timeout = 4;			/* Seconds */

static Panel	*mail;

static GtkTooltips	*tooltip;

static DecalButton	*mua_button;

static gboolean	enable_mail,
				mute_mode,
				super_mute_mode,
				cont_animation_mode,
				mua_inhibit_mode,		/* Inhibit checking if MUA launched */
				enable_multimua,		/* allow multiple MUA instances */
				msg_count_mode,
				fetch_check_only_mode,
				reset_remote_mode,
				unseen_is_new;			/* Accessed but unread */

static gboolean	mh_seq_ignore,
				checking_mh_mail;

static gint		animation_select_mode	= ANIMATION_BOTH;

static gboolean	force_mail_check;
static gint		new_mail_count, total_mail_count;
static gint		check_timeout;
static gint		show_tooltip = FALSE;

static gint		anim_frame,
				anim_dir,
				anim_pause;

static gint		style_id;

static void
free_account(MailAccount *account)
	{
	if (!account)
		return;
	g_free(account->path);
	g_free(account->server);
	g_free(account->username);
	g_free(account->password);
	g_free(account->imapfolder);
	g_free(account);
	}

static void
free_mailbox(Mailbox *mbox)
	{
	/* If user changes mailbox config list while a mailbox thread is busy,
	|  freeing the mbox can cause a segfault.   Probably a rare enough
	|  event that I can just suffer the small memory leak.
	*/
	if (mbox->busy)
		{
		printf("freeing busy mbox\n");
		return;
		}
	free_account(mbox->account);
	g_free(mbox->warn_msg);
	g_free(mbox);
	}


static gboolean
format_remote_mbox_name(Mailbox *mbox, gchar *buf, size_t len)
	{
	MailAccount	*account = mbox->account;

	if (account->imapfolder)
		snprintf(buf, len, "%s-%s@%s", account->username,
			account->imapfolder, account->server);
	else if (account->server)
		snprintf(buf, len, "%s@%s", account->username, account->server);
	else if (account->username)
		snprintf(buf, len, "%s", account->username);
	else
		{
		snprintf(buf, len, "??");
		return FALSE;
		}
	return TRUE;
	}

  /* Make tooltip visible/invisible and fill it with mailbox names
  |  containing new mail.
  */
static void
update_tooltip(void)
	{
	GList		*list;
	Mailbox		*mbox;
	MailAccount	*account;
	GString		*mboxes = NULL;
	gchar		buf[128];
   
	if (show_tooltip)
		{
		mboxes = g_string_sized_new(512);
		for (list = mailbox_list; list; list = list->next)
			{
			mbox = (Mailbox *) list->data;
			account = mbox->account;
			if (mbox->new_mail_count > 0)
				{
				if ((    account->mboxtype == MBOX_MBOX
					  || account->mboxtype == MBOX_MAILDIR
					  || account->mboxtype == MBOX_MH_DIR
					  || account->mboxtype == MBOX_LOCAL_PLUGIN
					  || account->mboxtype == MBOX_REMOTE_PLUGIN
					) && account->path
				   )
					snprintf(buf, sizeof(buf), "%s", account->path);
				else if (! format_remote_mbox_name(mbox, buf, sizeof(buf)))
					continue;	/* Can't get a name, so no tooltip for you! */

				if (mboxes->len > 0) 
					g_string_append_c(mboxes, '\n');
				g_string_append(mboxes, buf);
							
				if (msg_count_mode == MSG_NEW_TOTAL_COUNT)
					snprintf(buf, sizeof(buf), "(%d/%d)",
						mbox->new_mail_count, mbox->mail_count);
				else
					snprintf(buf, sizeof(buf), "(%d)", mbox->new_mail_count);
				g_string_append(mboxes, buf);
 				}
			}
		}
	if (show_tooltip && mboxes && mboxes->len > 0)
		{
		gtk_tooltips_set_tip(tooltip, mail->drawing_area, mboxes->str, "");
		gtk_tooltips_enable(tooltip);
		}
	else
		gtk_tooltips_disable(tooltip);
	if (mboxes)
		g_string_free(mboxes, TRUE);
	}
	
	
  /* Look at a From line to see if it is valid, lines look like:
  |  From sending_address dayofweek month dayofmonth timeofday year
  |  eg: From bill@gkrellm.net Fri Oct 22 13:52:49 2010
  */
static gint
is_From_line(Mailbox *mbox, gchar *buf)
	{
	gchar	sender[512];
	gint	dayofmonth = 0;

	if (strncmp(buf, "From ", 5))
		return FALSE;

	/* In case sending address missing, look for a day of month
	|  number in field 3 or 4 (0 based).
	*/
	sender[0] = '\0';
	if (sscanf(buf, "%*s %*s %*s %d", &dayofmonth) != 1)
		{
		if (sscanf(buf, "%*s %511s %*s %*s %d", sender, &dayofmonth) != 2)
			return FALSE;
		}
	if (dayofmonth < 1 || dayofmonth > 31)
		return FALSE;
	if (strcmp(sender, "MAILER-DAEMON") == 0)
		mbox->is_internal = TRUE;
	return TRUE;
	}


  /* Check if this is a Content-Type-line. If it contains a boundary
  |  field, copy boundary string to buffer (including two leading and
  |  trailing dashes marking the end of a multipart mail) and return
  |  true. Otherwise, return false.
  */
static gint
is_multipart_mail(gchar *buf, gchar *separator)
	{
	gchar *fieldstart;
	gchar *sepstart;
	gint  seplen;
	
	if (strncmp(buf, "Content-Type: ", 14) != 0)
		return FALSE;
	if (strncmp(&buf[14], "multipart/", 10) != 0)
		return FALSE;
	fieldstart = &buf[14];
	while (*fieldstart!=0)
		{
		while (*fieldstart!=0 && *fieldstart!=';')
			fieldstart++;
		if (*fieldstart==';') fieldstart++;
		while (*fieldstart!=0 && *fieldstart==' ')
			fieldstart++;
		if (strncmp(fieldstart, "boundary=", 9) == 0)
			{
			sepstart = fieldstart + 9;
			if (sepstart[0]=='"')
				{
				sepstart++;
				seplen = 0;
				while (sepstart[seplen]!='"' && sepstart[seplen]>=32)
					seplen++;
				}
			else
				{
				seplen = 0;
				while (sepstart[seplen]!=';' && sepstart[seplen]>32)
					seplen++;
				}
			strcpy(separator,"--");
			strncpy(&separator[2],sepstart,seplen);
			strcpy(&separator[seplen+2],"--");
			return TRUE;
			}
		}
	return FALSE;
	}

  /* Hide a password that is embedded in a string.
  */
static void
hide_password(Mailbox *mbox, gchar *line, gint offset)
	{
	gint    n;

	n = strlen(mbox->account->password);
	while (n--)
		line[offset + n] = '*';
	}

#if defined (GKRELLM_THREADS)
static gint
read_select(gint fd, gchar *buf, size_t size, time_t timeout)
	{ 
	fd_set			readfds;
	struct timeval	tv;
	gint			n	= 0;
         
	FD_ZERO(&readfds); 
	FD_SET(fd, &readfds); 
	tv.tv_sec = timeout; 
	tv.tv_usec = 0; 

	if (select(fd+1, &readfds, NULL, NULL, &tv) > 0) 
		n = read(fd, buf, size);
	return n; 
	}

  /* Read \r\n terminated lines from a remote IMAP or POP3 mail server,
  */
static void
tcp_getline(Mailbox *mbox,  gint fd)
	{
	gchar	buf[256];
	gint	n;
	gchar	*s;

	if (mbox->tcp_in)
		mbox->tcp_in = g_string_truncate(mbox->tcp_in, 0);
	else
		mbox->tcp_in = g_string_new("");
	s = buf;;
	while ((n = read_select(fd, s, 1, TCP_TIMEOUT)) > 0)
		{
		*(s+1) = '\0';
		if (*s++ == '\n')
			break;
		if (s >= buf + sizeof(buf) - 2)
			{
			g_string_append(mbox->tcp_in, buf);
			s = buf;
			}
		}
	if (s > buf)
		g_string_append(mbox->tcp_in, buf);

	if (GK.debug_level & DEBUG_MAIL)
		{
		if (n < 0)
			perror("tcp_getline: ");
		format_remote_mbox_name(mbox, buf, sizeof(buf));
		printf("server_response( %s )<%d>:%s\n", buf,
			mbox->tcp_in->len, mbox->tcp_in->str);
		}
	}

  /* Get a server response line and verify the beginning of the line
  |  matches a string.
  */
static gboolean
server_response(Mailbox *mbox, gint fd, gchar *match)
	{
	tcp_getline(mbox, fd);
	return (!strncmp(match, mbox->tcp_in->str, strlen(match)) ? TRUE : FALSE);
	}

  /* Get a imap server completion result response for a tagged command.
  |  Skip over any untagged responses the server may send.
  */
static gboolean
imap_completion_result(Mailbox *mbox, gint fd, gchar *tag)
	{
	while (1)
		{
		tcp_getline(mbox, fd);
		if (*(mbox->tcp_in->str) == '*')	/* untagged response */
			continue;
		return (!strncmp(tag, mbox->tcp_in->str, strlen(tag)) ? TRUE : FALSE);
		}
	}

static void
server_command(Mailbox *mbox, gint fd, gchar *line)
	{
	gchar	buf[128];

	write(fd, line, strlen (line));
	if (GK.debug_level & DEBUG_MAIL)
		{
		format_remote_mbox_name(mbox, buf, sizeof(buf));
		printf("server_command( %s ):%s", buf, line);
		}
	}

static gchar	*tcp_error_message[]	=
	{
	N_("Unable to connect."),
	N_("Bad response after connect."),
	N_("Bad response after username."),
	N_("Bad response after password."),
	N_("Bad response after STAT or STATUS."),
	N_("Bad response after UIDL."),
	N_("Bad APOP response after connect."),
	N_("Bad CRAM_MD5 response after connect."),
	};

static gboolean
tcp_shutdown(gint fd, Mailbox *mbox, gchar *message, gboolean warn)
	{
	gchar	buf[128];

  	if (GK.debug_level & DEBUG_MAIL)
		{
		format_remote_mbox_name(mbox, buf, sizeof(buf));
		printf(_("Mail TCP Error: %s - %s\n"), buf, _(message));
		}
	if (warn && !mbox->warned)
		{
		g_free(mbox->warn_msg);
		format_remote_mbox_name(mbox, buf, sizeof(buf));
		mbox->warn_msg = g_strdup_printf("%s\n%s\n%s\n", buf,
				_(message), mbox->tcp_in->str);
		}
	if (fd != -1)
		close (fd);
	return FALSE;
	}

extern void to64frombits(unsigned char *, const unsigned char *, int);
extern int from64tobits(char *, const char *);

static void
hmac_md5(unsigned char *password,  size_t pass_len,
	 unsigned char *challenge, size_t chal_len,
	 unsigned char *response,  size_t resp_len)
	{
	int i;
	unsigned char ipad[64];
	unsigned char opad[64];
	unsigned char hash_passwd[16];

	MD5_CTX ctx;
    
	if (resp_len != 16)
		return;

	if (pass_len > sizeof(ipad))
		{
		MD5Init(&ctx);
		MD5Update(&ctx, password, pass_len);
		MD5Final(hash_passwd, &ctx);
		password = hash_passwd;
		pass_len = sizeof(hash_passwd);
		}

	memset(ipad, 0, sizeof(ipad));
	memset(opad, 0, sizeof(opad));
	memcpy(ipad, password, pass_len);
	memcpy(opad, password, pass_len);

	for (i = 0; i < 64; i++) {
		ipad[i] ^= 0x36;
		opad[i] ^= 0x5c;
	}

	MD5Init(&ctx);
	MD5Update(&ctx, ipad, sizeof(ipad));
	MD5Update(&ctx, challenge, chal_len);
	MD5Final(response, &ctx);

	MD5Init(&ctx);
	MD5Update(&ctx, opad, sizeof(opad));
	MD5Update(&ctx, response, resp_len);
	MD5Final(response, &ctx);
	}

/* authenticate as per RFC2195 */
static int
do_cram_md5(int sock, char *command, Mailbox  *mbox, char *strip)
	{
	int len;
	unsigned char buf1[1024];
	unsigned char msg_id[768];
	unsigned char response[16];
	unsigned char reply[1024];
	unsigned char *respdata;

	snprintf(buf1, sizeof(buf1), "%s CRAM-MD5\r\n", command);
	server_command(mbox, sock, buf1);

	/* From RFC2195:
	 * The data encoded in the first ready response contains an
	 * presumptively arbitrary string of random digits, a
	 * timestamp, and the * fully-qualified primary host name of
	 * the server.  The syntax of the * unencoded form must
	 * correspond to that of an RFC 822 'msg-id' * [RFC822] as
	 * described in [POP3].
	 */

	if (!server_response(mbox, sock, "+ "))
		return FALSE;

	/* caller may specify a response prefix we should strip if present */
	respdata = mbox->tcp_in->str;
	if (respdata[strlen(respdata) - 1] == '\n')
		respdata[strlen(respdata) - 1] = '\0';
	if (respdata[strlen(respdata) - 1] == '\r')
		respdata[strlen(respdata) - 1] = '\0';
	if (strip && strncmp(respdata, strip, strlen(strip)) == 0)
		respdata += strlen(strip);
	len = from64tobits(msg_id, respdata);

	if (len < 0)
		{
		if (GK.debug_level & DEBUG_MAIL)
			printf(_("could not decode BASE64 challenge\n"));
		return FALSE;
		}
	else if (len < sizeof(msg_id))
		{
		msg_id[len] = 0;
		}
	else
		{
		msg_id[sizeof(msg_id) - 1] = 0;
		}
	if (GK.debug_level & DEBUG_MAIL)
		{
		printf(_("decoded as %s\n"), msg_id);
		}

	/* The client makes note of the data and then responds with a string
	 * consisting of the user name, a space, and a 'digest'.  The latter is
	 * computed by applying the keyed MD5 algorithm from [KEYED-MD5] where
	 * the key is a shared secret and the digested text is the timestamp
	 * (including angle-brackets).
	 */

	hmac_md5(mbox->account->password, strlen(mbox->account->password),
		 msg_id, strlen(msg_id), response, sizeof(response));

	snprintf(reply, sizeof(reply),
		 "%s %02x%02x%02x%02x%02x%02x%02x%02x%02x%02x%02x%02x%02x%02x%02x%02x", 
		  mbox->account->username,
		  response[0], response[1], response[2], response[3],
		  response[4], response[5], response[6], response[7],
		  response[8], response[9], response[10], response[11],
		  response[12], response[13], response[14], response[15]);

	to64frombits(buf1, reply, strlen(reply));

	len = strlen(buf1);
	if (len + 3 > sizeof(buf1))
		return FALSE;
	strcpy(buf1 + len, "\r\n");
	server_command(mbox, sock, buf1);
	return TRUE;
	}

static gboolean
check_pop3(Mailbox *mbox)
	{
	MailAccount	*account = mbox->account;
	gint		fd	= -1;
	gchar		line[128], buf[128];
#ifdef HAVE_GETADDRINFO
	gint 		rv	= 0;
	struct addrinfo	hints, *res, *res0;
	gchar		portnumber[6];
#else
	struct hostent	*addr;
	struct sockaddr_in s;
#endif

#ifdef HAVE_GETADDRINFO
	snprintf (portnumber, sizeof(portnumber), "%d", account->port);
	memset(&hints, 0, sizeof(hints));
	hints.ai_family = PF_UNSPEC;
	hints.ai_socktype = SOCK_STREAM;
	hints.ai_protocol = 0;
	if ((rv = getaddrinfo(account->server, portnumber, &hints, &res0)) != 0)
		return tcp_shutdown(fd, mbox, tcp_error_message[0], FALSE);

	for (res = res0; res; res = res->ai_next)
		{
		if ((fd = socket(res->ai_family, res->ai_socktype,
				res->ai_protocol)) < 0)
			continue;
		if (GK.debug_level & DEBUG_MAIL)
			{
			format_remote_mbox_name(mbox, buf, sizeof(buf));
			printf("check_pop3 (%d,%d,%d): connecting to %s\n",
				res->ai_family, res->ai_socktype, res->ai_protocol, buf);
			}
		if (connect(fd, res->ai_addr, res->ai_addrlen) >= 0)
			break;
		close(fd);
		fd = -1;
		}
	freeaddrinfo(res0);
	if (fd == -1)
		return tcp_shutdown(fd, mbox, tcp_error_message[0], FALSE);
#else
	addr = gethostbyname (account->server);
	if (addr)
		memcpy (&s.sin_addr, addr->h_addr, sizeof (struct in_addr));  
	else 
		return tcp_shutdown(fd, mbox, tcp_error_message[0], FALSE);

	fd = socket (AF_INET, SOCK_STREAM, 0);
	if (fd < 0) 
		return FALSE;

	s.sin_family = AF_INET;
	s.sin_port = htons (account->port);

	if (GK.debug_level & DEBUG_MAIL)
		{
		format_remote_mbox_name(mbox, buf, sizeof(buf));
		printf("check_pop3: connecting to %s\n", buf);
		}
	if (connect (fd, (struct sockaddr *)&s, sizeof (s)) < 0)
		return tcp_shutdown(fd, mbox, tcp_error_message[0], FALSE);
#endif

	/* Is the machine we are connected to really a POP3 server?
	*/
	if (! server_response(mbox, fd, "+OK"))
		return tcp_shutdown(fd, mbox, tcp_error_message[1], FALSE);

	if (account->authtype == AUTH_POP3_APOP)
		{
		static const gchar hex[] = "0123456789abcdef";
		MD5_CTX	ctx;
		gint	i;
		gchar	*key, *p;
		guchar	digest[16];
		gchar	ascii_digest[33];

		if ((key = strchr(mbox->tcp_in->str + 3, '<')) == NULL)
			return tcp_shutdown(fd, mbox, tcp_error_message[6], TRUE);
		if ((p = strchr(key, '>')) == NULL)
			return tcp_shutdown(fd, mbox, tcp_error_message[6], TRUE);
		*(p + 1) = '\0';
		snprintf(line, sizeof(line), "%s%s", key, account->password);
		MD5Init(&ctx);
		MD5Update(&ctx, line, strlen(line));
		MD5Final(digest, &ctx);
		for (i = 0;  i < 16;  i++)
			{
			ascii_digest[i + i] = hex[digest[i] >> 4];
			ascii_digest[i + i + 1] = hex[digest[i] & 0x0f];
			}
		ascii_digest[i + i] = '\0';
		snprintf(line, sizeof(line),
			 "APOP %s %s\r\n", account->username, ascii_digest);
		server_command(mbox, fd, line);
		}
	else if (account->authtype == AUTH_POP3_CRAM)
		{
		if (!do_cram_md5(fd, "AUTH", mbox, NULL))
			{
			/* SASL cancellation of authentication */
			server_command(mbox, fd, "*\r\n");
			return tcp_shutdown(fd, mbox, tcp_error_message[7], TRUE);
			}
		}
	else	/* AUTH_POP3_PLAIN */
		{
		snprintf (line, sizeof (line), "USER %s\r\n", account->username);
		server_command(mbox, fd, line);
		if (! server_response(mbox, fd, "+OK"))
			return tcp_shutdown(fd, mbox, tcp_error_message[2], TRUE);

		snprintf (line, sizeof (line), "PASS %s\r\n", account->password);
		write(fd, line, strlen (line));
		if (GK.debug_level & DEBUG_MAIL)
			{
			hide_password(mbox, line, 5);
			format_remote_mbox_name(mbox, buf, sizeof(buf));
			printf("server_command( %s ):%s", buf, line);
			}
		}
	if (! server_response(mbox, fd, "+OK"))
		return tcp_shutdown(fd, mbox, tcp_error_message[3], TRUE);

	server_command(mbox, fd, "STAT\r\n");
	if (! server_response(mbox, fd, "+OK"))
		return tcp_shutdown(fd, mbox, tcp_error_message[4], FALSE);

	sscanf(mbox->tcp_in->str, "+OK %d", &mbox->mail_count);
   	snprintf (line, sizeof (line), "UIDL %d\r\n", mbox->mail_count);
	server_command(mbox, fd, line);

	if (! server_response(mbox, fd, "+OK"))
		mbox->new_mail_count = mbox->mail_count;
	else
		/* Set the new_mail_count only if the UIDL is changed to avoid
		|  re-reporting mail is new after MUA button has been clicked.
		*/
		if (   sscanf(mbox->tcp_in->str, "+OK %*d %128s", line) == 1
			&& (   gkrellm_dup_string((gchar **) (&mbox->private), line)
				|| unseen_is_new
			   )
		   )
			mbox->new_mail_count = mbox->mail_count;

	server_command(mbox, fd, "QUIT\r\n");
	close (fd);

	return TRUE;
	}

static gboolean
check_imap(Mailbox *mbox)
	{
	MailAccount	*account = mbox->account;
	gint		fd	= -1;
	gint		messages = 0;
	gint		unseen = 0;
	gchar		line[128], *ss;
	gchar		buf[128];
#ifdef HAVE_GETADDRINFO
	gchar	portnumber[6];
	gint	rv	= 0;
	struct addrinfo hints, *res, *res0;
#else
	struct hostent		*addr;
	struct sockaddr_in	s;
#endif

#ifdef HAVE_GETADDRINFO
	snprintf (portnumber, sizeof(portnumber), "%d", account->port);
	memset(&hints, 0, sizeof(hints));
	hints.ai_family = PF_UNSPEC;
	hints.ai_socktype = SOCK_STREAM;
	hints.ai_protocol = 0;
	if ((rv = getaddrinfo(account->server, portnumber, &hints, &res0)) != 0)
		return tcp_shutdown(fd, mbox, tcp_error_message[0], FALSE);

	for (res = res0; res; res = res->ai_next)
		{
		if ((fd = socket(res->ai_family, res->ai_socktype,
				res->ai_protocol)) < 0)
			continue;
		if (GK.debug_level & DEBUG_MAIL)
			{
			format_remote_mbox_name(mbox, buf, sizeof(buf));
			printf("check_imap (%d,%d,%d): connecting to %s\n",
				res->ai_family, res->ai_socktype, res->ai_protocol, buf);
			}
		if (connect(fd, res->ai_addr, res->ai_addrlen) >= 0)
			break;
		close(fd);
		fd = -1;
		}
	freeaddrinfo(res0);
	if (fd == -1)
		return tcp_shutdown(fd, mbox, tcp_error_message[0], FALSE);
#else
	addr = gethostbyname (account->server);
	if (addr)
		memcpy (&s.sin_addr, addr->h_addr, sizeof (struct in_addr));  
	else 
		return tcp_shutdown(fd, mbox, tcp_error_message[0], FALSE);

	fd = socket (AF_INET, SOCK_STREAM, 0);
	if (fd < 0) 
		return FALSE;

	s.sin_family = AF_INET;
	s.sin_port = htons (account->port);

	if (GK.debug_level & DEBUG_MAIL)
		{
		format_remote_mbox_name(mbox, buf, sizeof(buf));
		printf("check_imap: connecting to %s\n", buf);
		}
	if (connect (fd, (struct sockaddr *)&s, sizeof (s)) < 0)
		return tcp_shutdown(fd, mbox, tcp_error_message[0], FALSE);
#endif

	/* Is the machine we are connected to really a IMAP server?
	*/
	if (! server_response(mbox, fd, "* OK"))
		return tcp_shutdown(fd, mbox, tcp_error_message[1], FALSE);

	if (account->authtype == AUTH_IMAP_CRAM)
		{
		if (!do_cram_md5(fd, "a001 AUTHENTICATE", mbox, NULL))
			{
			/* SASL cancellation of authentication */
			server_command(mbox, fd, "*\r\n");
			return tcp_shutdown(fd, mbox, tcp_error_message[7], TRUE);
			}
		}
	else	/* AUTH_IMAP_PLAIN */
		{
		snprintf (line, sizeof (line), "a001 LOGIN \"%s\" \"%s\"\r\n",
			  account->username, account->password);
		write(fd, line, strlen (line));
		if (GK.debug_level & DEBUG_MAIL)
			{
			line[10 + 2 + strlen(account->username)] = '\0';
			hide_password(mbox, line, 11 + 2 + strlen(account->username));
			format_remote_mbox_name(mbox, buf, sizeof(buf));
			printf("server_command( %s ):%s", buf, line);
			}
		}
	if (! imap_completion_result(mbox, fd, "a001 OK"))
		return tcp_shutdown(fd, mbox, tcp_error_message[2], TRUE);

	/* I expect the only untagged response to STATUS will be "* STATUS ..."
	*/
	snprintf (line, sizeof (line), "a002 STATUS %s (MESSAGES UNSEEN)\r\n",
			account->imapfolder);
	server_command(mbox, fd, line);
	if (! server_response(mbox, fd, "*"))
		return tcp_shutdown(fd, mbox, tcp_error_message[4], FALSE);

	if ((ss = strstr(mbox->tcp_in->str, "MESSAGES")) == NULL)
		return tcp_shutdown(fd, mbox, tcp_error_message[4], FALSE);

	if (sscanf(ss, "MESSAGES %d", &messages) == 1)
		{
		if ((ss = strstr(mbox->tcp_in->str, "UNSEEN")) != NULL)
			sscanf(ss, "UNSEEN %d", &unseen);
		if (GK.debug_level & DEBUG_MAIL)
			printf(_("Messages: %d\nUnseen: %d\n"), messages, unseen);
		}
	mbox->mail_count = messages;
	mbox->new_mail_count = unseen;
	imap_completion_result(mbox, fd, "a002 OK");

	server_command(mbox, fd, "a003 LOGOUT\r\n");
	imap_completion_result(mbox, fd, "a003 OK");

	close (fd);
	return TRUE;
	}
#endif	/* GKRELLM_THREADS */


  /* Check a mh directory for mail. The way that messages are marked as new
  |  depends on the MUA being using.  Only .mh_sequences is checked for
  |  now, otherwise all mail found is considered new mail.
  */
static gboolean
check_mh_dir(Mailbox *mbox)
	{
	FILE			*f;
	struct dirent	*entry;
	DIR				*folder;
	gchar			*path, *tok;
	gchar			buf[1024];
	gint			n0, n1;

	mbox->mail_count = mbox->new_mail_count = 0;

	if (!(folder = opendir(mbox->account->path)))
		return FALSE;
	while ((entry = readdir(folder)) != NULL)
		{
		/* Files starting with a digit are messages. */
		if (isdigit(entry->d_name[0]))
			mbox->mail_count++;
		}
	closedir(folder);
	path = g_strdup_printf("%s/%s", mbox->account->path, ".mh_sequences");

	/* Some MH dir clients use .mh_sequences, others such as mutt or gnus
	|  do not.  For mixed cases, it's a user option to ignore .mh_sequences.
	*/
	if (!mh_seq_ignore && (f = fopen(path, "r")) != NULL)
		{
		while (fgets(buf, sizeof(buf), f))
			{
			/* Look for unseen sequence like "unseen: 4 7-9 23"
			*/
			if (strncmp(buf, "unseen:", 7))
				continue;
			tok = strtok(buf, " \t\n");
			while ((tok = strtok(NULL, " \t\n")) != NULL)
				{
				if (sscanf(tok, "%d-%d", &n0, &n1) == 2)
					mbox->new_mail_count += n1 - n0 + 1;
				else
					mbox->new_mail_count++;
				}
			break;
			}
		fclose(f);
		}
	else
		mbox->new_mail_count = mbox->mail_count;

	g_free(path);
	return TRUE;
	}


  /* A maildir has new, cur, and tmp subdirectories.  Any file in new
  |  or cur that does not begin with a '.' is a mail message.  It is
  |  suggested that messages begin with the output of time() (9 digits)
  |  but while mutt and qmail use this standard, procmail does not.
  |  maildir(5) says:
  |      It is a good idea for readers to skip all filenames in
  |      new and cur starting with a dot.  Other than this,
  |      readers should not attempt to parse filenames.
  |  So check_maildir() simply looks for files in new and cur.
  |  But if unseen_is_new flag is set, look for ":2,*S" file suffix where
  |  the 'S' indicates the mail is seen.
  |  See http://cr.yp.to/proto/maildir.html
  */
static gboolean
check_maildir(Mailbox *mbox)
	{
	gchar			path[256], *s;
	struct dirent	*dentry;
	DIR 			*dir;

	mbox->new_mail_count = 0;
	snprintf(path, sizeof(path), "%s/new", mbox->account->path);
	if ((dir = opendir(path)) != NULL)
		{
		while ((dentry = readdir(dir)) != NULL)
			{
			if (*dentry->d_name != '.' && dentry->d_ino > 0)
				mbox->new_mail_count++;
			}
		closedir(dir);
		}
	mbox->mail_count = mbox->new_mail_count;
	snprintf(path, sizeof(path), "%s/cur", mbox->account->path);
	if ((dir = opendir(path)) != NULL)
		{
		while ((dentry = readdir(dir)) != NULL)
			{
			if (*dentry->d_name == '.' || dentry->d_ino == 0)
				continue;
			mbox->mail_count++;
			if (   unseen_is_new
				&& (s = strchr(dentry->d_name, ':')) != NULL
				&& !strchr(s, 'S')
			   )
				mbox->new_mail_count++;
			}
		closedir(dir);
		}

	if (GK.debug_level & DEBUG_MAIL)
		printf(_("mdir %s total=%d old=%d new=%d\n"), mbox->account->path,
			mbox->mail_count, mbox->old_mail_count, mbox->new_mail_count);
    return TRUE;
	}


  /* Count total mail and old mail in a mailbox.  Old mail can be read
  |  with a Status: R0, or can be accessed and not read with Status: O
  |  So, new mail will be the diff - note that unread mail is not
  |  necessarily new mail.  According to stat() man page:
  |  st_atime is changed by mknod(), utime(), read(), write(), truncate()
  |  st_mtime is changed by mknod(), utime(), write()
  |  But, new mail arriving (writing mailbox) sets st_mtime while reading
  |  the mailbox (mail program reading) sets st_atime.  So the test
  |  st_atime > st_mtime is testing if mbox has been read since last new mail.
  |  Mail readers may restore st_mtime after writting status.
  |  And Netscape mail does status with X-Mozilla-Status: xxxS
  |    where S is bitwise or of status flags:
  |    1: read  2: replied  4: marked  8: deleted
  |  
  |  Evolution uses status with X-Evolution: 00000000-00x0 where x status is
  |    0: unread   1: read   2: unread + attachments  3: read + attachments
  */
static gboolean
status_is_old(gchar *buf)
	{
	gchar	c;

	if (buf[0] != 'S' && buf[0] != 'X')
		return FALSE;

	/* Standard mail clients
	*/
	if (   !strncmp(buf, "Status:", 7)
		&& (strchr(buf, 'R') || (!unseen_is_new && strchr(buf, 'O')))
	   )
		return TRUE;

	/* Netscape
	*/
	if (!strncmp(buf, "X-Mozilla-Status:", 17))
		{
		c = buf[21];
		if (c < '8')	/* Not deleted */
			c -= '0';
		if (c >= '8' || (c & 0x1))
			return TRUE;
		}

	/* Evolution
	*/
	if (   !strncmp(buf, "X-Evolution:", 12)
		&& (!strncmp(buf+24, "1", 1) || !strncmp(buf+24, "3", 1))
	   )
		return TRUE;

	return FALSE;
	}

static gboolean
check_mbox(Mailbox *mbox)
	{
	FILE			*f;
	struct utimbuf	ut;
	struct stat		s;
	gchar			buf[1024];
	gchar			mpart_sep[1024];
	gint			in_header	= FALSE;
	gint			marked_read = FALSE;
	gint			is_multipart = FALSE;

	if (stat(mbox->account->path, &s) != 0)
		{
		mbox->mail_count = mbox->old_mail_count = mbox->new_mail_count = 0;
		mbox->last_mtime = 0;
		mbox->last_size = 0;
		return FALSE;
		}

	/* Message no-counting mode reports new mail based on mailbox
	|  modified time and size.
	*/
	if (msg_count_mode == MSG_NO_COUNT)
		{
		if (   s.st_size > 0
			&& s.st_size >= mbox->last_size
			&& s.st_mtime >= s.st_atime
		   )
			mbox->new_mail_count = TRUE;
		else
			mbox->new_mail_count = FALSE;
		mbox->mail_count = (s.st_size > 0) ? 1 : 0;	/* boolean, not used */
		mbox->old_mail_count = 0;			/* not used */
		mbox->last_size = s.st_size;
		mbox->last_mtime = s.st_mtime;
		return TRUE;
		}

	/* If the mailboxes have been modified since last check, count
	|  the new/total messages.
	*/
	if (   s.st_mtime != mbox->last_mtime
		|| s.st_size  != mbox->last_size
		|| force_mail_check
	   )
		{
		if ((f = fopen(mbox->account->path, "r")) == NULL)
			return FALSE;
		mbox->mail_count = 0;
		mbox->old_mail_count = 0;
		while(fgets(buf, sizeof(buf), f))
			{
			if (is_multipart && !in_header)
				{
				/* Skip to last line of multipart mail */
				if (strncmp(buf,mpart_sep,strlen(mpart_sep))==0)
					is_multipart = FALSE;
				}
			else if (buf[0] == '\n')
				{
				in_header = FALSE;
				mbox->is_internal = FALSE;
				}
			else if (is_From_line(mbox, buf))
				{
				mbox->mail_count += 1;
				in_header = TRUE;
				marked_read = FALSE;
				}
			else if (in_header && status_is_old(buf) && !marked_read)
				{
				mbox->old_mail_count += 1;
				marked_read = TRUE;
				}
			else if (in_header && mbox->is_internal)
				{
				if (strncmp(buf, "From: Mail System Internal Data", 31) == 0)
					{
					in_header = FALSE;
					mbox->mail_count -= 1;
					mbox->is_internal = FALSE;
					}
				}
			else if (in_header && is_multipart_mail(buf,mpart_sep))
				{
				is_multipart = TRUE;
				}
			}
		fclose(f);

		/* Restore the mbox stat times for other mail checking programs and
		|  so the (st_atime > st_mtime) animation override below will work.
		*/
		ut.actime = s.st_atime;
		ut.modtime = s.st_mtime;
		utime(mbox->account->path, &ut);

		mbox->last_mtime = s.st_mtime;
		mbox->last_size = s.st_size;
		if (GK.debug_level & DEBUG_MAIL)
			printf(_("mbox read <%s> total=%d old=%d ==> "),
					mbox->account->path,
					mbox->mail_count, mbox->old_mail_count);
		}
	/* Set the animation state when new mail count changes, and override
	|  the animation to false if mbox has been accessed since last modify
	|  (A MUA has probably read the mbox).
	*/
	mbox->new_mail_count = mbox->mail_count - mbox->old_mail_count;
	if (s.st_atime > s.st_mtime)
		{
		mbox->need_animation = FALSE;
		mbox->prev_new_mail_count = mbox->new_mail_count;
		}
	if (GK.debug_level & DEBUG_MAIL)
		printf(_("mbox %s total=%d old=%d new=%d\n"), mbox->account->path,
			mbox->mail_count, mbox->old_mail_count, mbox->new_mail_count);

    return TRUE;
	}

static Decal	*mail_text_decal;
static Decal	*mail_icon_decal;

static void
draw_mail_text_decal(gint new_mail_count, gint mail_count)
	{
	TextStyle	ts, ts_save;
	gint		x, w;
	Panel		*p;
	Decal		*d;
	Style		*style;
	Margin		*m;
	glong		new_value;
	gchar		buf[32], nbuf[16], tbuf[16];

	p = mail;
	d = mail_text_decal;

	/* Decals have only a single value, but I need a two valued decal here.
	|  So, just pack the two numbers and use that as a test for value.
	*/
	new_value = (mail_count << 16) | new_mail_count;
	if (new_value == d->value)
		return;

	if (GK.debug_level & DEBUG_MAIL)
		printf(_("Draw mail: new=%d count=%d "), new_mail_count, mail_count);

	ts_save = d->text_style;
	if (new_mail_count == MUTE_FLAG)
		{
		sprintf(buf, _("mute"));
		ts = *gkrellm_meter_alt_textstyle(style_id);	/* Use the alt color */
		ts.font = d->text_style.font;
		}
	else
		{
		ts = d->text_style;
		if (msg_count_mode == MSG_NO_COUNT)
			buf[0] = '\0';
		else if (msg_count_mode == MSG_NEW_COUNT)
			{
			if (new_mail_count == 0)
				strcpy(buf, "-");
			else
				snprintf(buf, sizeof(buf), "%d", new_mail_count);
			}
		else /* MSG_NEW_TOTAL_COUNT */
			{
			if (new_mail_count == 0)
				strcpy(nbuf, "-");
			else
				snprintf(nbuf, sizeof(nbuf), "%d", new_mail_count);
			if (mail_count == 0)
				strcpy(tbuf, "-");
			else
				snprintf(tbuf, sizeof(tbuf), "%d", mail_count);
			snprintf(buf, sizeof(buf), "%s/%s", nbuf, tbuf); 
			}
		}
	if (GK.debug_level & DEBUG_MAIL)
		printf(_("string=<%s>\n"), buf);
	w = gdk_string_width(ts.font, buf);
	if (w > d->w)
		{
		ts.font = gkrellm_meter_alt_textstyle(style_id)->font;
		w = gdk_string_width(ts.font, buf);
		}
	style = gkrellm_meter_style(style_id);
	m = gkrellm_get_style_margins(style);
	x = gkrellm_chart_width() * p->label->position / GKRELLM_LABEL_MAX;
	x -= m->right + w / 2;
	if (p->label->position >= 50)
		x -= mail_icon_decal->w;
	if (x > d->w - w)
		x = d->w - w;
	if (x < 0)
		x = 0;
	d->text_style = ts;
	d->x_off = x;
	gkrellm_draw_decal_text(p, d, buf, new_value);
	d->text_style = ts_save;
	}

static void
mbox_set_animation_state(Mailbox *mbox)
	{
	if (mbox->new_mail_count != mbox->prev_new_mail_count)
		mbox->need_animation =
				(mbox->new_mail_count > mbox->prev_new_mail_count);
	}

static void
update_krell_animation_frame(void)
	{
	/* Run the animation.  Just go back and forth with a pause on
	|  frames 1 and full_scale - 1 (frames 0 and full_scale are cut off).
	|  Frame 0 is blank anyway, and frame full_scale is just not used.
	*/
	if (anim_pause-- <= 0)
		{
		if (anim_frame <= 1)
			anim_dir = 1;
		if (anim_frame >= KRELL(mail)->full_scale - 1)
			anim_dir = -1;
		anim_frame += anim_dir;
		anim_frame %= KRELL(mail)->full_scale;
		if (anim_frame == 1 || anim_frame == KRELL(mail)->full_scale - 1)
			anim_pause = 4;
		}
	}

  /* I popen the mail_user_agent so I can know if it is running and
  |  the mail_fetch so I can process fetchmail/flist/fetcho/...  output.
  |  Reading the pipes need to not block so GKrellM won't freeze.
  |  So, I need a special non-blocking pipe line reading routine.
  */
static void
pipe_command(Mailproc *mp)
	{
	gchar	buf[128];

	if (mp->pipe)	/* Still running?  */
		{
		if (GK.debug_level & DEBUG_MAIL)
			printf(_("mail pipe_command: old command <%s> still running.\n"),
					mp->command);
		return;
		}
	if (!mp->command || *mp->command == '\0')
		return;
	snprintf(buf, sizeof(buf), "%s 2>&1", mp->command);
	if (GK.debug_level & DEBUG_MAIL)
		printf(_("mail launching pipe_command <%s>\n"), buf);
	if ((mp->pipe = popen(buf, "r")) == NULL)
		{
		if (GK.debug_level & DEBUG_MAIL)
			printf(_("mail could not run pipe_command <%s>\n"), buf);
		return;
		}
	if (mp->read_gstring)
		mp->read_gstring = g_string_truncate(mp->read_gstring, 0);
	else
		mp->read_gstring = g_string_new("");
	fcntl(fileno(mp->pipe), F_SETFL, O_NONBLOCK);
	}


  /* Accumulate non-blocking reads from a pipe into a gstring.  Then
  |  try to read a single line from the gstring.  If a line is read,
  |  return the count of chars in the line.  But if no line can be read
  |  and the other end of the pipe has exited (pipe eof), return -1.
  |  Otherwise return 0 to indicate no line is available but the pipe
  |  producer is still alive.
  |  Why this trouble?: the stdio fgets() reading from a non-blocking
  |  stream is not guaranteed to return complete '\n' delimited lines.
  */
static gint
fgets_pipe(gchar *line, gint len, Mailproc *mp)
	{
	gchar	buf[512];
	gint	n;

	if (mp->pipe)
		{
		if (feof(mp->pipe))
			{
			n = pclose(mp->pipe);
			if (n == -1 && errno == EINTR)
				pclose(mp->pipe);
			mp->pipe = NULL;
			}
		else
			{
			n = fread(buf, 1, sizeof(buf) - 1, mp->pipe);
			if (n > 0)
				{
				buf[n] = '\0';
				mp->read_gstring = g_string_append(mp->read_gstring, buf);
				}
			}
		}
	if (getline_from_gstring(&mp->read_gstring, line, len))
		return strlen(line);
	if (!mp->pipe)
		{
		if (mp->read_gstring)
			g_string_free(mp->read_gstring, TRUE);
		mp->read_gstring = NULL;
		return -1;
		}
	return 0;
	}

static gboolean
mua_is_launched(void)
	{
	gchar	buf[128];
	gint	n;

	if (mail_user_agent.pipe == NULL)
		return FALSE;
	while ((n = fgets_pipe(buf, sizeof(buf), &mail_user_agent)) > 0)
		;
	if (n == 0 && mail_user_agent.pipe && feof(mail_user_agent.pipe))
		{
		n = pclose(mail_user_agent.pipe);
		if (n == -1 && errno == EINTR)
			pclose(mail_user_agent.pipe);
		mail_user_agent.pipe = NULL;
		n = -1;
		}
	return (n < 0) ? FALSE : TRUE;
	}

  /* Read mail_fetch pipe and if fetch_check_only_mode, look for output
  |  of fetchmail -c or flist so I can report mail from these programs.
  |
  | eg. for MH mail, the nmh program flist -all (from the man page):
  |   /work/Mail  has  5 in sequence unseen (private); out of  46
  |   inbox+      has 10 in sequence unseen          ; out of 153
  |   junklist    has  0 in sequence unseen          ; out of  63
  |
  |  For fetchmail, if no remote mail, I get:
  |		fetchmail: No mail for billw at mail.wt.net
  |  If remote mail i will get lines like:
  |		1 message for billw at mail.wt.net (32743 octets).
  |	  or, as reported by a user, there could be:
  |		26 messages (25 seen) for billw at mail.wt.net
  |  If the remote mail is fetched, I get additional lines like:
  |		reading message 1 of 1 (32743 octets) .................... flushed
  |  Note: above 26 messages (25 seen) should show as 1/26 on panel.
  |
  |  And fetchmail can't make up its mind.  A user gets this with 5.2.0:
  |  fetchmail: 5.2.0 querying xx.yy.net (protocol POP3) at Thu, 20 Jan 2000...
  |  fetchmail: 2 messages for uuu at xx.yy.net (20509 octets).
  */

  /* Since there is now internal POP3 and IMAP checking, this will not be
  |  used as much, but there is still flist, using fetchmail for ssl, etc.
  |  Anyway, here's the code to grok the above mess!
  */
static gboolean
parse_fetch_line(gchar *line, gint *msg, gint *seen)
	{
	gchar	*s, *p, *buf;
	gint	n, n1, n2;
	gint	tok_count	= 0;
	gint	state		= 0,
			seen_flag	= FALSE,
			unseen_flag	= FALSE;

	*msg = 0;
	*seen = 0;
	n1 = n2 = 0;
	buf = g_strdup(line);
	s = strtok(buf, " \t()\n");

	/* Trap out badly set Fetch/check option
	*/
	if (!s || !strcmp(s, "reading") || !strcmp(s, "skipping"))
		{
		g_free(buf);
		return FALSE;
		}

	if (GK.debug_level & DEBUG_MAIL)
		printf(_("  parse["));
	while (s)
		{
		if (++tok_count > 3 && state == 0)	/* need a int within 3 tokens */
			break;
		n = strtol(s, &p, 0);
		if (*p == ' ' || *p == '\t' || *p == '\0' || *p == '\n')
			{		/* Have an integer, and not a x.y version number */
			if (GK.debug_level & DEBUG_MAIL)
				printf(_("*<%s>,st=%d,"), s, state);
			if (state == 0)
				n1 = n;
			else if (state == 1)
				n2 = n;
			if (GK.debug_level & DEBUG_MAIL)
				printf(_("n1=%d,n2=%d state=%d*"), n1, n2, state);
			++state;
			}
		else if (strcmp(s, "seen") == 0)
			seen_flag = TRUE;
		else if (strcmp(s, "unseen") == 0)
			unseen_flag = TRUE;
		s = strtok(NULL, " \t()\n");
		}
	if (state > 1 && seen_flag)	/* 26 messages (25 seen) ... */
		{
		*msg = n1;
		*seen = n2;
		}
	else if (state > 1 && unseen_flag)	/* /xxx has 5 in sequence unseen ... */
		{
		*msg = n2;
		*seen = n2 - n1;
		}
	else if (state > 0)			/*     1 message for billw at ... */
		*msg = n1;				/* or  Fetchmail: 1 message for .... */
	if (GK.debug_level & DEBUG_MAIL)
		printf(_("]snf=%d sunf=%d msg=%d seen=%d STATE=%d\n"),
				seen_flag, unseen_flag, *msg, *seen, state);
	g_free(buf);
	return TRUE;
	}


static gint
compare_mailboxes(gconstpointer a, gconstpointer b)
	{
	gchar a_name[128];
	gchar b_name[128];

	format_remote_mbox_name((Mailbox *)a, a_name, sizeof(a_name));
	format_remote_mbox_name((Mailbox *)b, b_name, sizeof(b_name));

	return strcmp(a_name, b_name);
	}

static void
make_fetch_tooltip(gchar *line, gint msg, gint seen)
	{
	Mailbox		*mbox;
	MailAccount	*account;
	GList		*old_mbox_pointer;
	gchar		buf[64], *s;

	mbox = g_new0(Mailbox, 1);
	account = g_new0(MailAccount, 1);
	mbox->account = account;
	account->mboxtype = MBOX_FETCH_TOOLTIP;

	if ((s = strstr(line, "sequence unseen")) != NULL)	/* flist */
		{
		sscanf(line, "%64s", buf);
		account->username = g_strdup(buf);
		}
	else if ((s = strstr(line, " for ")) != NULL)		/* fetchmail */
		{
		sscanf(s + 5, "%64s", buf);
		account->username = g_strdup(buf);
		if ((s = strstr(line, " at ")) != NULL)
			{
			sscanf(s + 4, "%64s", buf);
			account->server = g_strdup(buf);
			}
		if ((s = strstr(line, "(folder ")) != NULL)
			{
			sscanf(s + 8, "%64[^)]", buf);
			account->imapfolder = g_strdup(buf);
			}
		}
	else
		{
		free_mailbox(mbox);
		return;
		}
	old_mbox_pointer = g_list_find_custom(mailbox_list, mbox,
				(GCompareFunc) compare_mailboxes);
	if (old_mbox_pointer)
		{
		free_mailbox(mbox);
		mbox = (Mailbox *) old_mbox_pointer->data;
		if (mbox->account->mboxtype == MBOX_FETCH_TOOLTIP)
			{
			if (GK.debug_level & DEBUG_MAIL)
				printf(_("fetch tooltip - found old mailbox: %s, %s, %s\n"),
						mbox->account->imapfolder, mbox->account->username,
						mbox->account->server);
			mbox->mail_count = msg;
			mbox->new_mail_count = msg - seen;
			}
		}
	else
		{
		if (GK.debug_level & DEBUG_MAIL)
			printf(_("fetch tooltip - inserting new mailbox: %s, %s, %s\n"),
					account->imapfolder, account->username, account->server);
		mbox->mail_count = msg;
		mbox->new_mail_count = msg - seen;
		mailbox_list = g_list_insert_sorted(mailbox_list, mbox,
								(GCompareFunc)(compare_mailboxes));
		}
	}

  /* Read output lines from the fetch/check program.  If fetch_check_only_mode
  |  is set, parse the lines to try to read message counts so they can be
  |  reported ("fetchmail -c" or equiv should be configured).  However, if
  |  the mode is not set, then just read lines and waste them because mail
  |  is presumably being downloaded into local mailboxes.
  */
static void
read_mail_fetch(void)
	{
	Mailproc	*mp		= (Mailproc *) mail_fetch->private;
	gchar		buf[128];
	gint		n, msg, seen;

	while ((n = fgets_pipe(buf, sizeof(buf), mp)) > 0)	/* non-blocking */
		{
		if (GK.debug_level & DEBUG_MAIL)
			printf(_("read_mail_fetch(%d): %s\n"), fetch_check_only_mode, buf);
		if (fetch_check_only_mode)
			{
			if (parse_fetch_line(buf, &msg, &seen))
				make_fetch_tooltip(buf, msg, seen);
			if (msg > 0)
				{
				mail_fetch->mail_count += msg;
				mail_fetch->old_mail_count += seen;
				if (GK.debug_level & DEBUG_MAIL)
					printf(_("read_mail_fetch: fetch_messages=%d fetch_seen=%d\n"),
						mail_fetch->mail_count, mail_fetch->old_mail_count);
				}
			}
		}
	if (n == 0 && (GK.debug_level & DEBUG_MAIL))
		printf(_("read_mail_fetch: no read data.\n"));

	/* When fetch program is done, flag not busy so new counts will be used.
	*/
	if (n < 0)
		{
		if (fetch_check_only_mode)
			{
			mail_fetch->new_mail_count =
					mail_fetch->mail_count - mail_fetch->old_mail_count;
			mbox_set_animation_state(mail_fetch);
			if (GK.debug_level & DEBUG_MAIL)
				printf(_("read_mail_fetch: total=%d old=%d new=%d\n"),
					mail_fetch->mail_count,
					mail_fetch->old_mail_count, mail_fetch->new_mail_count);
			}
		mail_fetch->busy = FALSE;
		}
	}

static void
reset_mail_fetch(void)
	{
	Mailproc	*mp	= (Mailproc *) mail_fetch->private;
	gint		n;

	if (mp->pipe)
		{
		n = pclose(mp->pipe);
		if (n == -1 && errno == EINTR)
			pclose(mp->pipe);
		}
	mp->pipe = NULL;
	if (mp->read_gstring)
		g_string_free(mp->read_gstring, TRUE);
	mp->read_gstring = NULL;
	mail_fetch->mail_count = 0;
	mail_fetch->new_mail_count = 0;
	mail_fetch->old_mail_count = 0;
	mail_fetch->need_animation = FALSE;
	mail_fetch->busy = FALSE;
	force_mail_check = TRUE;
	}

static gboolean
run_fetch_program(void)
	{
	Mailbox		*mbox;
	Mailproc	*mp	= (Mailproc *) mail_fetch->private;
	GList		*list;

	if (   !mail_fetch->busy && !mp->read_gstring
		&& mp->command && *(mp->command) != '\0'
	   )
		{
		mail_fetch->busy = TRUE;
		for (list = mailbox_list; list; list = list->next)
			{
			mbox = (Mailbox *) list->data;
			if (   mbox->account->mboxtype == MBOX_FETCH
				|| mbox->account->mboxtype == MBOX_FETCH_TOOLTIP
			   )
				{
				mbox->mail_count = 0;
				mbox->new_mail_count = 0;
				mbox->old_mail_count = 0;
				}
			}
		pipe_command(mp);
		if (mp->pipe)
			return TRUE;
		}
	return FALSE;
	}

#if defined (GKRELLM_THREADS)
  /* This routine is run as a thread so GKrellM does not have to wait
  |  for the results of a mail checking function.
  */
static void
remote_mail_check_func(void *data)
	{
	Mailbox	*mbox	= (Mailbox *) data;
	gint	external;

	external = mbox->account->mboxtype & MBOX_EXTERNAL;
	if ( (*(mbox->check_func))(external ? mbox->data : mbox) == FALSE)
		{
		mbox->mail_count = 0;
		mbox->new_mail_count = 0;
		mbox->need_animation = FALSE;
		}
	else
		mbox_set_animation_state(mbox);
	mbox->busy = FALSE;
	}

static void
run_mail_check_thread(Mailbox *mbox)
	{
	gpointer	thread_id;

	if (!mbox->busy && mbox->check_func)
		{
		mbox->busy = TRUE;
		thread_id = new_thread(remote_mail_check_func, mbox);
		}
	}
#endif

static void
update_mail(void)
	{
	Mailbox		*mbox;
	GList		*list;
	gchar		cmd[256];
	gint		external, prev_new_mail_count, any_busy;
	gint		local_check		= FALSE,
				remote_check	= FALSE;
	static gint	second_count,
				minute_count,
				sound_inhibit;

	if (enable_mail == FALSE || mailbox_list == NULL)
		return;
	if (GK.minute_tick && (++minute_count % mail_fetch_timeout) == 0)
		remote_check = TRUE;
	if (GK.second_tick && (++second_count % mail_check_timeout) == 0)
		local_check = TRUE;
	if (remote_check || local_check)
		mua_is_launched();		/* update pipe, avoid lingering zombie */

	if (   force_mail_check
		|| (   GK.second_tick
			&& !(mute_mode && super_mute_mode)
			&& !(mua_inhibit_mode && mail_user_agent.pipe)
		   )
	   )
		{
		prev_new_mail_count = new_mail_count;
		total_mail_count = 0;
		new_mail_count = 0;
		run_animation = FALSE;
		remote_check |= force_mail_check;
		local_check |= force_mail_check;
		if (force_mail_check)
			minute_count = second_count = 0;
		any_busy = FALSE;

		for (list = mailbox_list; list; list = list->next)
			{
			mbox = (Mailbox *) list->data;
			external = mbox->account->mboxtype & MBOX_EXTERNAL;
			switch (mbox->account->mboxtype & MBOX_CHECK_TYPE_MASK)
				{
				case MBOX_CHECK_FETCH:
					if (((Mailproc *)(mail_fetch->private))->read_gstring)
						read_mail_fetch();
					if (remote_check)
						(*mbox->check_func)(mbox);
					break;
					
				case MBOX_CHECK_INLINE:	/* Local mailbox or maildir check */
					if (local_check)
						{
						if (mbox->check_func)
							(*mbox->check_func)(external ? mbox->data : mbox);
						mbox_set_animation_state(mbox);
						}
					break;
#if defined (GKRELLM_THREADS)
				case MBOX_CHECK_THREADED:
					if (remote_check)
						run_mail_check_thread(mbox);
					break;
#endif
				default:	/* Unknown or pseudo mail box type */
					continue;
				}
			/* If a mailbox check is busy (thread or remote fetch program
			|  running), use previous counts until the check is done.
			*/
			if (mbox->busy)
				{
				total_mail_count += mbox->prev_mail_count;
				new_mail_count += mbox->prev_new_mail_count;
				run_animation |= mbox->prev_need_animation;
				any_busy = TRUE;
				}
			else
				{
				total_mail_count += mbox->mail_count;
				new_mail_count += mbox->new_mail_count;
				if (mbox->new_mail_count && cont_animation_mode)
					mbox->need_animation = TRUE;
				run_animation |= mbox->need_animation;
				mbox->prev_mail_count = mbox->mail_count;
				mbox->prev_new_mail_count = mbox->new_mail_count;
				mbox->prev_need_animation = mbox->need_animation;
				if (mbox->warn_msg && !mbox->warned)
					{
					gkrellm_message_window(NULL, mbox->warn_msg, NULL);
					mbox->warned = TRUE;
					}
				}
			}
		force_mail_check = FALSE;

		if ((GK.debug_level & DEBUG_MAIL) && (local_check || remote_check))
			printf(_("**Mail check totals: total=%d new=%d anim=%d\n"),
				total_mail_count, new_mail_count, run_animation);

		if (   prev_new_mail_count != new_mail_count
			|| tooltip->active_tips_data == NULL
		   )
			update_tooltip();

		/* Run the notify (sound) command if the new mail count goes up,
		|  and if the various modes permit.  Ensure a sound command is
		|  issued only once per remote check interval by locking it out
		|  if a new mail count triggers the sound and there are other
		|  remote threads still busy.
		*/
		if (   !sound_inhibit && !mute_mode
			&& new_mail_count > prev_new_mail_count
			&& mail_notify && mail_notify[0] != '\0'
		   )
			{
			snprintf(cmd, sizeof(cmd), "%s &", mail_notify);
			gkrellm_system(cmd);
			if (any_busy)
				sound_inhibit = TRUE;
			}
		if (!any_busy)
			sound_inhibit = FALSE;
		}

	if (mute_mode && (GK.timer_ticks % 15) < 3)	/* Asymmetric blink */
		draw_mail_text_decal(MUTE_FLAG, MUTE_FLAG);
	else
		draw_mail_text_decal(new_mail_count, total_mail_count);

	if (run_animation && (animation_select_mode & ANIMATION_PENGUIN))
		update_krell_animation_frame();
	else
		anim_frame = 0;

	if ((GK.timer_ticks % GK.decal_mail_delay) == 0)
		{
		if (run_animation)
			{
			if (animation_select_mode & ANIMATION_ENVELOPE)
				{
				++decal_frame;
				if (decal_frame >= GK.decal_mail_frames)
					decal_frame = 1;
				}
			else
				decal_frame = 0;
			}
		else
			decal_frame = 1;
		}
	gkrellm_draw_decal_pixmap(mail, DECAL(mail), decal_frame);

	/* All the animation frame drawing is done with the general krell code.
	*/
	KRELL(mail)->previous = 0;
	gkrellm_update_krell(mail, KRELL(mail), anim_frame);
	gkrellm_draw_panel_layers(mail);
	}


static gint
mail_expose_event(GtkWidget *widget, GdkEventExpose *ev)
	{
	if (widget == mail->drawing_area)
		gdk_draw_pixmap(widget->window, GK.draw1_GC, mail->pixmap,
			ev->area.x, ev->area.y, ev->area.x, ev->area.y,
			ev->area.width, ev->area.height);
	return FALSE;
	}

  /* The mua launch button also optionally stops animations and resets
  |  remote mail counts.  So this routine decides if it should be enabled.
  */
static void
set_mua_button_sensitivity(void)
	{
	if (   *(mail_user_agent.command) || reset_remote_mode
		|| (animation_select_mode && !cont_animation_mode)
	   )
		gkrellm_set_button_sensitive(mua_button, TRUE);
	else
		gkrellm_set_button_sensitive(mua_button, FALSE);
	}


  /* Callback for the message count decal button.
  */
static void
cb_mail_button(DecalButton *button)
	{
	GList	*list;
	Mailbox	*mbox;

	if (reset_remote_mode)
		{
		for (list = mailbox_list; list; list = list->next)
			{
			mbox = (Mailbox *) list->data;
			if (   (mbox->account->mboxtype & MBOX_CHECK_THREADED)
				|| (mbox->account->mboxtype & MBOX_CHECK_FETCH)
				|| mbox->account->mboxtype == MBOX_FETCH_TOOLTIP
			   )
				{
				GDK_THREADS_ENTER();
				mbox->mail_count = 0;
				mbox->new_mail_count = 0;
				mbox->old_mail_count = 0;
				mbox->need_animation = FALSE;
				GDK_THREADS_LEAVE();
				}
			}
		}
	if (!cont_animation_mode)
		{
		for (list = mailbox_list; list; list = list->next)
			{
			mbox = (Mailbox *) list->data;
			mbox->need_animation = FALSE;
			}
		run_animation = FALSE;
		}
	if (enable_multimua)
		{
		if (mail_user_agent.command && *mail_user_agent.command != '\0')
			{
			gchar   *cmd;

			cmd = g_strdup_printf("%s &", mail_user_agent.command);
			gkrellm_system(cmd);
			g_free(cmd);
			}
		}
	else if (!mua_is_launched()) 
		pipe_command(&mail_user_agent);
	else
		{
		check_timeout = 0;
		if (GK.debug_level & DEBUG_MAIL)
			printf(_("Mail user agent is already running.\n"));
		}
	}

  /* Callback for any button clicks on the Mail panel.  Must exclude
  |  button 1 clicks on the message decal button.
  */
static gint
cb_panel_press(GtkWidget *widget, GdkEventButton *ev)
	{
	Decal	*d 	= mail_icon_decal;

	if (ev->button == 3)
		gkrellm_open_config_window(mon_mail);

	/* button 2 anywhere in the panel toggles the mute mode.
	*/
	if (ev->button == 2)
		{
		mute_mode = !mute_mode;

		/* If coming out of mute mode and mail checking was inhibited,
		|  force a check.
		*/
		if (   ! mute_mode && super_mute_mode
			&& (! mua_is_launched() || ! mua_inhibit_mode)
		   )
			force_mail_check = TRUE;
		}
	/* Button 1 press on the envelope - must exclude the message count button.
	*/
	if (ev->button == 1 && ev->x >= d->x && ev->x < d->x + d->w)
		force_mail_check = TRUE;
	return TRUE;
	}	

static void
dup_account(MailAccount *dst, MailAccount *src)
	{
	dst->path = g_strdup(src->path);
	dst->server = g_strdup(src->server);
	dst->username = g_strdup(src->username);
	dst->password = g_strdup(src->password);
	dst->imapfolder = g_strdup(src->imapfolder);
	dst->mboxtype = src->mboxtype;
	dst->authtype = src->authtype;
	dst->port = src->port;
	}

static gboolean
get_local_mboxtype(MailAccount *account)
	{
	gchar	*path;

	if (!account->path)
		return FALSE;
	if (*(account->path) == '~')
		{
		path = g_strdup_printf("%s%s", gkrellm_homedir(),
						account->path + 1);
		g_free(account->path);
		account->path = path;
		}
	if (isdir(account->path, "new"))
		account->mboxtype = MBOX_MAILDIR;
	else if (isdir(account->path, NULL))
		account->mboxtype = MBOX_MH_DIR;
	else
		account->mboxtype = MBOX_MBOX;
	return TRUE;
	}

static Mailbox *
add_mailbox(MailAccount *account)
	{
	Mailbox	*mbox;
	gchar	*path;

	if (!account)
		return NULL;
	mbox = g_new0(Mailbox, 1);
	mbox->account = account;
	if (account->path)
		{
		if (*(account->path) == '~')
			{
			path = g_strdup_printf("%s%s", gkrellm_homedir(),
							account->path + 1);
			g_free(account->path);
			account->path = path;
			}
		if (account->mboxtype == MBOX_MAILDIR)
			mbox->check_func = check_maildir;
		else if (account->mboxtype == MBOX_MH_DIR)
			{
			mbox->check_func = check_mh_dir;
			checking_mh_mail = TRUE;
			}
		else
			mbox->check_func = check_mbox;
		}
#if defined(GKRELLM_THREADS)
	else	/* remote mailbox */
		{
		if (account->mboxtype == MBOX_REMOTE)
			{
			if (account->authtype < AUTH_IMAP_PLAIN)
				mbox->check_func = check_pop3;
			else
				mbox->check_func = check_imap;
			}
		}
#endif
	mailbox_list = g_list_append(mailbox_list, mbox);
	return mbox;
	}

  /* Pre 1.2.7  mailbox config parsing.
  */
static Mailbox *
old_add_mailbox(gchar *arg)
	{
	Mailbox		*mbox;
	MailAccount	*account;
	gchar		parms[4][CFG_BUFSIZE];
	gint		n;
	gchar		*p;

	account = g_new0(MailAccount, 1);
	n = sscanf(arg, "%s %s %s %s", parms[0], parms[1], parms[2], parms[3]);
	if (n == 1)
		{
		account->path = g_strdup(parms[0]);
		get_local_mboxtype(account);
		mbox = add_mailbox(account);
		return mbox;
		}
#if defined (GKRELLM_THREADS)
	mbox = g_new0(Mailbox, 1);
	mbox->account = account;
	switch (n)
		{
		case 3:
			account->mboxtype = MBOX_REMOTE;
			if ((p = strchr(parms[0], ':')) != NULL)
				*p++ = '\0';
			account->server = g_strdup(parms[0]);
			account->port = (p && *p) ? atoi(p) : atoi(DEFAULT_POP3_PORT);
			if ((p = strchr(parms[1], '/')) != NULL)
				{
				*p++ = '\0';
				if (strcasecmp(p, "user") == 0)
					account->authtype = AUTH_POP3_PLAIN;
				else if (strcasecmp(p, "apop") == 0)
					account->authtype = AUTH_POP3_APOP;
				else
					{
					g_free(account);
					g_free(mbox);
					return NULL;
					}
				}
			else
				account->authtype = AUTH_POP3_PLAIN;
			account->username = g_strdup(parms[1]);
			account->password = g_strdup(parms[2]);
			mbox->check_func = check_pop3;
			break;

		case 4:
			account->mboxtype = MBOX_REMOTE;
			if ((p = strchr(parms[0], ':')) != NULL)
				*p++ = '\0';
			account->server = g_strdup(parms[0]);
			account->port = (p && *p) ? atoi(p) : atoi(DEFAULT_IMAP_PORT);
			account->imapfolder = g_strdup(parms[1]);
			account->username = g_strdup(parms[2]);
			account->password = g_strdup(parms[3]);
			account->authtype = AUTH_IMAP_PLAIN;
			mbox->check_func = check_imap;
			break;

		default:	/* Invalid mailbox line */
			g_free(account);
			g_free(mbox);
			return NULL;
		}
	mailbox_list = g_list_append(mailbox_list, mbox);
	return mbox;
#else
	g_free(account);
	return NULL;
#endif
	}

static void
create_mail(GtkWidget *vbox, gint first_create)
	{
	Style		*style;
	Margin		*m;
	Panel		*p;
	MailAccount	*account;
	gchar		*s, buf[64];
	gint		x, w;
	static GdkImlibImage	*krell_penguin_image;


	if (first_create)
		mail = gkrellm_panel_new0();
	else
		{
		gkrellm_destroy_decal_list(mail);
		gkrellm_destroy_krell_list(mail);
		}
	p = mail;
	if (! mailbox_list->next)	/* First mailbox is internal fetch */
		{
		if ((s = getenv("MAIL")) == NULL)
			if ((s = getenv("USER")) != NULL)
				{
				if (isdir("/var/mail", NULL))
					snprintf(buf, sizeof(buf), "/var/mail/%s", s);
				else
					snprintf(buf, sizeof(buf), "/var/spool/mail/%s", s);
				s = buf;
				}
		if (s)
			{
			account = g_new0(MailAccount, 1);
			account->path = g_strdup(s);
			account->mboxtype = MBOX_MBOX;
			add_mailbox(account);
			}
		}

	style = gkrellm_meter_style(style_id);
	m = gkrellm_get_style_margins(style);

	if (krell_penguin_image)
		{
		gdk_imlib_kill_image(krell_penguin_image);
		krell_penguin_image = NULL;
		}
	gkrellm_load_image("krell_penguin", NULL, &krell_penguin_image,
			MAIL_STYLE_NAME);
	gkrellm_create_krell(p,
			krell_penguin_image ? krell_penguin_image
								: gkrellm_krell_meter_image(style_id),
			style);
	KRELL(p)->full_scale = style->krell_depth - 1;

	gkrellm_load_image("decal_mail", decal_mail_xpm, &decal_mail_image,
			MAIL_STYLE_NAME);
	gkrellm_render_to_pixmap(decal_mail_image, &decal_mail_pixmap,
			&decal_mail_mask, 0, 0);
	mail_icon_decal = gkrellm_create_decal_pixmap(p, decal_mail_pixmap,
				decal_mail_mask, GK.decal_mail_frames, style, -1, -1);

	/* The new/total mail text needs to be a decal because the text changes
	|  and must be drawn as a layer in update_layers().
	|  Calculate x,w override values. Cannot override w after the create.
	*/
	x = m->left;
	if (style->label_position >= 50)
		x += mail_icon_decal->w;
	w = gkrellm_chart_width() - mail_icon_decal->w - m->left - m->right;

    mail_text_decal = gkrellm_create_decal_text(p, "0",
				gkrellm_meter_textstyle(style_id),
                style, x, -1, w);    /* -1 means use y default */

	gkrellm_panel_configure(p, NULL, style);
	gkrellm_panel_create(vbox, mon_mail, p);

	/* Center the decals with respect to each other.
	*/
	if (mail_icon_decal->h > mail_text_decal->h)
		mail_text_decal->y += (mail_icon_decal->h - mail_text_decal->h) / 2;
	else
		mail_icon_decal->y += (mail_text_decal->h - mail_icon_decal->h) / 2;

	mua_button = gkrellm_put_decal_in_meter_button(p, mail_text_decal,
			cb_mail_button, NULL);
	set_mua_button_sensitivity();

	if(!enable_mail)
		gkrellm_panel_hide(p);

	if (first_create)
		{
		gtk_signal_connect(GTK_OBJECT (p->drawing_area), "expose_event",
				(GtkSignalFunc) mail_expose_event, NULL);
		gtk_signal_connect(GTK_OBJECT(p->drawing_area),"button_press_event",
				(GtkSignalFunc) cb_panel_press, NULL);
		tooltip=gtk_tooltips_new();
		}
	}

static gint
authtype_from_string(gchar *s)
	{
	gint	t;

	for (t = 0; t < N_AUTH_PROTOCOLS; ++t)
		if (!strcmp(s, auth_strings[t]))
			break;
	return t;
	}

static void
save_mail_config(FILE *f)
	{
	Mailbox		*mbox;
	MailAccount	*account;
	Mailproc	*mp		= (Mailproc*) mail_fetch->private;
	GList		*list;
	gchar		*pwd, *qq;

	for (list = mailbox_list; list; list = list->next)
		{
		mbox = (Mailbox *) list->data;
		account = mbox->account;
		switch (account->mboxtype)
			{
			case MBOX_MBOX:
			case MBOX_MAILDIR:
			case MBOX_MH_DIR:
				fprintf(f, "mail mailbox-local %s %s\n",
					mbox_strings[account->mboxtype & 0xf], account->path);
				break;
			case MBOX_REMOTE:
				pwd = account->password;
				if ((qq = strchr(pwd, '"')) != NULL)
					pwd = "password";
				fprintf(f, "mail mailbox-remote %s %s \"%s\" \"%s\" %d",
						auth_strings[account->authtype], account->server,
						account->username, pwd, account->port);
				if (account->authtype >= AUTH_IMAP_PLAIN)
					fprintf(f, " %s", account->imapfolder);
				fprintf(f, "\n");
				if (qq)
					fprintf(f, "mail password %s\n", account->password);
				break;
			default:
				break;
			}
		}
	fprintf(f, "mail mua %s\n", mail_user_agent.command);
	fprintf(f, "mail notify %s\n", mail_notify);
	fprintf(f, "mail fetch_command %s\n", mp->command);
	fprintf(f, "mail fetch_timeout %d\n", mail_fetch_timeout);
	fprintf(f, "mail check_timeout %d\n", mail_check_timeout);
	fprintf(f, "mail msg_count_mode %d\n", msg_count_mode);
	fprintf(f, "mail animation_select_mode %d\n", animation_select_mode);
	fprintf(f, "mail fetch_check_only_mode %d\n", fetch_check_only_mode);
	fprintf(f, "mail reset_remote_mode %d\n", reset_remote_mode);
	fprintf(f, "mail unseen_is_new %d\n", unseen_is_new);
	fprintf(f, "mail enable %d %d %d %d\n", enable_mail, super_mute_mode,
						mua_inhibit_mode, enable_multimua);
	fprintf(f, "mail animation_continuous %d\n", cont_animation_mode);
	fprintf(f, "mail show_tooltip %d\n", show_tooltip);
	fprintf(f, "mail mh_seq_ignore %d\n", mh_seq_ignore);
	}

static void
load_mail_config(gchar *arg)
	{
	static MailAccount	*account_prev;
	MailAccount	*account;
	Mailproc	*mp		= (Mailproc*) mail_fetch->private;
	gchar		mail_config[32], item[CFG_BUFSIZE], path[256];
	gchar		server[256], username[64], password[64], auth[32], folder[64];
	gint		n;

	n = sscanf(arg, "%31s %[^\n]", mail_config, item);
	if (n == 2)
		{
		if (!strcmp(mail_config, "mailbox"))	/* Old config, pre 1.2.7 */
			old_add_mailbox(item);
		else if (!strcmp(mail_config, "mailbox-local"))
			{
			if (sscanf(item, "%*s %256[^\n]", path) == 1)
				{
				account = g_new0(MailAccount, 1);
				account->path = g_strdup(path);
				get_local_mboxtype(account);
				add_mailbox(account);
				}
			}
		else if (!strcmp(mail_config, "mailbox-remote"))
			{
			account = g_new0(MailAccount, 1);
			account_prev = NULL;
			account->mboxtype = MBOX_REMOTE;
			n = sscanf(item, "%32s %256s \"%64[^\"]\" \"%64[^\"]\" %d %64s",
					auth, server, username, password, &account->port, folder);
			if (n < 5)
				n = sscanf(item, "%32s %256s %64s %64s %d %64s",
					auth, server, username, password, &account->port, folder);
			if (n >= 5)
				{
				account->authtype = authtype_from_string(auth);
				account->server = g_strdup(server);
				account->username = g_strdup(username);
				account->password = g_strdup(password);
				if (n == 6 && account->authtype >= AUTH_IMAP_PLAIN)
					account->imapfolder = g_strdup(folder);
				add_mailbox(account);
				account_prev = account;		/* XXX */
				}
			else
				g_free(account);
			}
		else if (account_prev && !strcmp(mail_config, "password"))
			gkrellm_dup_string(&account_prev->password, item);
		else if (strcmp(mail_config, "mua") == 0)
			mail_user_agent.command = g_strdup(item);
		else if (strcmp(mail_config, "notify") == 0)
			mail_notify = g_strdup(item);
		else if (strcmp(mail_config, "fetch_command") == 0)
			mp->command = g_strdup(item);
		else if (strcmp(mail_config, "fetch_timeout") == 0)
			sscanf(item, "%d", &mail_fetch_timeout);
		else if (strcmp(mail_config, "check_timeout") == 0)
			sscanf(item, "%d", &mail_check_timeout);
		else if (strcmp(mail_config, "msg_count_mode") == 0)
			sscanf(item, "%d", &msg_count_mode);
		else if (strcmp(mail_config, "animation_select_mode") == 0)
			sscanf(item, "%d", &animation_select_mode);
		else if (strcmp(mail_config, "fetch_check_only_mode") == 0)
			sscanf(item, "%d", &fetch_check_only_mode);
		else if (strcmp(mail_config, "reset_remote_mode") == 0)
			sscanf(item, "%d", &reset_remote_mode);
		else if (strcmp(mail_config, "unseen_is_new") == 0)
			sscanf(item, "%d", &unseen_is_new);
		else if (strcmp(mail_config, "enable") == 0)
			sscanf(item, "%d %d %d %d", &enable_mail, &super_mute_mode,
						&mua_inhibit_mode, &enable_multimua);
		else if (strcmp(mail_config, "animation_continuous") == 0)
			sscanf(item, "%d", &cont_animation_mode);
		else if (strcmp(mail_config, "show_tooltip") == 0)
			sscanf(item, "%d", &show_tooltip);
		else if (strcmp(mail_config, "mh_seq_ignore") == 0)
			sscanf(item, "%d", &mh_seq_ignore);
		}
	}


/* ---------------------------------------------------------------------*/
static GtkWidget		*account_notebook;
static GtkWidget		*mbox_path_entry;
static GtkWidget		*server_entry,
						*user_entry,
						*password_entry,
						*imapfolder_entry,
						*port_entry,
						*port_button;

static GtkWidget		*local_button,
						*remote_button,
						*remote_option_menu;

static GtkWidget		*mail_user_agent_entry;
static GtkWidget		*enable_multimua_button;
static GtkWidget		*mail_fetch_entry;
static GtkWidget		*mail_notify_entry;
static GtkWidget		*enable_mail_button;
static GtkWidget		*enable_cont_anim_button;
static GtkWidget		*super_mute_button;
static GtkWidget		*mbox_clist;
static GtkWidget		*mail_fetch_spin_button;
static GtkWidget		*mail_check_spin_button;
static GtkWidget		*msg_count_button[3];
static GtkWidget		*animation_select_button[4];
static GtkWidget		*check_only_button;
static GtkWidget		*reset_remote_button;
static GtkWidget		*unseen_is_new_button;
static GtkWidget		*mua_inhibit_button;
static GtkWidget		*show_tooltip_button;
static GtkWidget		*mh_seq_ignore_button;

static GList			*config_mailbox_list;

static gint				mailboxes_modified,
						selected_row,
						optmenu_auth_protocol;


static void
reset_mailbox_entries(void)
	{
	gtk_entry_set_text(GTK_ENTRY(mbox_path_entry), "");

	gtk_entry_set_text(GTK_ENTRY(server_entry), "");
	gtk_entry_set_text(GTK_ENTRY(user_entry), "");
	gtk_entry_set_text(GTK_ENTRY(password_entry), "");
	gtk_entry_set_text(GTK_ENTRY(imapfolder_entry), "");
	gtk_widget_set_sensitive(imapfolder_entry, FALSE);
	gtk_option_menu_set_history(GTK_OPTION_MENU(remote_option_menu), 0);
	optmenu_auth_protocol = 0;

	gtk_entry_set_text(GTK_ENTRY(port_entry), DEFAULT_POP3_PORT);
	gtk_widget_set_sensitive(port_entry, FALSE);
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(port_button), FALSE);
	}

static void
set_mailbox_group(GtkWidget *widget, gpointer data)
	{
	gint	group;

	group = GPOINTER_TO_INT(data);
	gtk_notebook_set_page(GTK_NOTEBOOK(account_notebook), group);
	reset_mailbox_entries();
	if (selected_row >= 0)
		gtk_clist_unselect_row(GTK_CLIST(mbox_clist), selected_row, 0);
	selected_row = -1;
	}

static gboolean
default_port_entry(void)
	{
	gboolean	active;

	active = GTK_TOGGLE_BUTTON(port_button)->active;
	if (!active)
		gtk_entry_set_text(GTK_ENTRY(port_entry),
				(optmenu_auth_protocol < AUTH_IMAP_PLAIN)
				? DEFAULT_POP3_PORT : DEFAULT_IMAP_PORT);
	return active;
	}

static void
cb_specify_port(GtkWidget *widget, gpointer data)
	{
	gtk_widget_set_sensitive(port_entry, default_port_entry());
	}

static void
cb_protocol_selected(GtkMenuItem *menuitem)
	{
	optmenu_auth_protocol =
			GPOINTER_TO_INT(gtk_object_get_user_data(GTK_OBJECT(menuitem)));
	if (optmenu_auth_protocol < AUTH_IMAP_PLAIN)
		{
		gtk_entry_set_text(GTK_ENTRY(imapfolder_entry), "");
		gtk_widget_set_sensitive(imapfolder_entry, FALSE);
		}
	else
		{
		gtk_entry_set_text(GTK_ENTRY(imapfolder_entry), "inbox");
		gtk_widget_set_sensitive(imapfolder_entry, TRUE);
		}
	default_port_entry();
	}

static void
cb_mailbox_selected(GtkWidget *clist, gint row, gint column,
					GdkEventButton *bevent, gpointer data)
	{
	MailAccount	*account;
	gchar		buf[32], *f;
	gint		default_port;
	gboolean	active;

	account = gtk_clist_get_row_data(GTK_CLIST(clist), row);
	if (account->mboxtype == MBOX_REMOTE)
		{
		gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(remote_button), TRUE);
		gtk_entry_set_text(GTK_ENTRY(server_entry), account->server);
		gtk_entry_set_text(GTK_ENTRY(user_entry), account->username);
		gtk_entry_set_text(GTK_ENTRY(password_entry), account->password);
		f = (account->authtype < AUTH_IMAP_PLAIN) ? "" : account->imapfolder;
		optmenu_auth_protocol = account->authtype;
		if (account->authtype < AUTH_IMAP_PLAIN)
			{
			gtk_entry_set_text(GTK_ENTRY(imapfolder_entry), "");
			gtk_widget_set_sensitive(imapfolder_entry, FALSE);
			default_port = atoi(DEFAULT_POP3_PORT);
			}
		else
			{
			gtk_entry_set_text(GTK_ENTRY(imapfolder_entry),
						account->imapfolder);
			gtk_widget_set_sensitive(imapfolder_entry, TRUE);
			default_port = atoi(DEFAULT_IMAP_PORT);
			}
		gtk_option_menu_set_history(GTK_OPTION_MENU(remote_option_menu),
					optmenu_auth_protocol);
		if (account->port < 1)
			account->port = default_port;
		active = (account->port == default_port) ? FALSE : TRUE;
		gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(port_button), active);
		sprintf(buf, "%d", account->port);
		gtk_entry_set_text(GTK_ENTRY(port_entry), buf);		
		}
	else
		{
		gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(local_button), TRUE);
		gtk_entry_set_text(GTK_ENTRY(mbox_path_entry), account->path);
		}
	selected_row = row;
	}

static void
cb_mailbox_unselected(GtkWidget *clist, gint row, gint column,
					GdkEventButton *bevent, gpointer data)
	{
	selected_row = -1;
	reset_mailbox_entries();
	}

static gboolean
dup_entry(gchar **buf, GtkWidget **entry)
	{
	gchar	*s;

	s = gkrellm_entry_get_text(entry);
	if (*s)
		{
		*buf = g_strdup(s);
		return TRUE;
		}
	*buf = NULL;
	return FALSE;
	}

static gchar *
x_out(gchar *s)
	{
	gchar			*p;
	static gchar	xbuf[32];

	strncpy(xbuf, s, sizeof(xbuf));
	xbuf[31] = '\0';
	if ((p = strchr(xbuf, '_')) != NULL)
		*p = ' ';
	return xbuf;
	}

static void
mbox_clist_update(MailAccount *account, gint row)
	{
	gchar	*buf[3], *default_port, abuf[32], pbuf[32];

	if (account->mboxtype == MBOX_REMOTE)
		{
		default_port = (account->authtype < AUTH_IMAP_PLAIN) ?
				DEFAULT_POP3_PORT : DEFAULT_IMAP_PORT;
		sprintf(abuf, "%s", auth_strings[account->authtype]);
		buf[1] = g_strdup(account->server);
		if (account->port != atoi(default_port))
			{
			sprintf(pbuf, ":%d", account->port);
			buf[1] = g_strconcat(buf[1], pbuf, NULL);
			}
		buf[1] = g_strconcat(buf[1], "  ", account->username, NULL);
		if (account->authtype >= AUTH_IMAP_PLAIN)
			buf[1] = g_strconcat(buf[1], "  ", account->imapfolder, NULL);
		}
	else
		{
		sprintf(abuf, "%s", mbox_strings[account->mboxtype & 0xf]);
		buf[1] = g_strdup_printf("%s", account->path);
		}
	buf[0] = x_out(abuf);
	buf[2] = NULL;
	if (row >= 0)
		{
		gtk_clist_set_text(GTK_CLIST(mbox_clist), row, 0, buf[0]);
		gtk_clist_set_text(GTK_CLIST(mbox_clist), row, 1, buf[1]);
		}
	else
		row = gtk_clist_append(GTK_CLIST(mbox_clist), buf);
	gtk_clist_set_row_data(GTK_CLIST(mbox_clist), row, account);
	g_free(buf[1]);
	}

static void
cb_enter_mailbox(void)
	{
	GList		*list;
	MailAccount	*new_account, *account;
	gboolean	remote, valid;
	gint		default_port;

	new_account = g_new0(MailAccount, 1);
	valid = FALSE;
	remote = GTK_TOGGLE_BUTTON(remote_button)->active;
	if (remote)
		{
		if (   dup_entry(&new_account->server, &server_entry)
			&& dup_entry(&new_account->username, &user_entry)
			&& dup_entry(&new_account->password, &password_entry)
		   )
			valid = TRUE;
		if (GTK_TOGGLE_BUTTON(port_button)->active)
			new_account->port = atoi(gkrellm_entry_get_text(&port_entry));
		new_account->mboxtype = MBOX_REMOTE;
		new_account->authtype = optmenu_auth_protocol;
		if (new_account->authtype < AUTH_IMAP_PLAIN)
			default_port = atoi(DEFAULT_POP3_PORT);
		else
			{
			if (valid)
				valid = dup_entry(&new_account->imapfolder, &imapfolder_entry);
			default_port = atoi(DEFAULT_IMAP_PORT);
			}
		if (new_account->port == 0)
			new_account->port = default_port;
		}
	else
		{
		valid = dup_entry(&new_account->path, &mbox_path_entry);
		get_local_mboxtype(new_account);
		}

	if (!valid)
		{
		gkrellm_message_window(_("GKrellM Config Error"),
			_("Incomplete mailbox entries"), NULL);
		free_account(new_account);
		return;
		}
	if (selected_row >= 0)
		{
		account = gtk_clist_get_row_data(GTK_CLIST(mbox_clist), selected_row);
		list = g_list_find(config_mailbox_list, account);
		free_account(account);
		if (list)
			list->data = (gpointer) new_account;
		mbox_clist_update(new_account, selected_row);
		gtk_clist_unselect_row(GTK_CLIST(mbox_clist), selected_row, 0);
		}
	else
		{
		config_mailbox_list = g_list_append(config_mailbox_list, new_account);
		mbox_clist_update(new_account, selected_row);
		}
	reset_mailbox_entries();
	mailboxes_modified = TRUE;
	}


static void
cb_delete_mailbox(GtkWidget *widget, gpointer *data)
	{
	MailAccount	*account;

	if (selected_row >= 0)
		{
		account = gtk_clist_get_row_data(GTK_CLIST(mbox_clist), selected_row);
		config_mailbox_list = g_list_remove(config_mailbox_list, account);
		free_account(account);
		gtk_clist_remove(GTK_CLIST(mbox_clist), selected_row);
		}
	selected_row = -1;
	reset_mailbox_entries();
	mailboxes_modified = TRUE;
	}

#if 0
static void
dump_mailboxes(void)
	{
	Mailbox	*mbox;
	GList	*list;
	gchar	buf[64];

	for (list = mailbox_list->next; list; list = list->next)
		{
		mbox = (Mailbox *) list->data;
		format_remote_mbox_name(mbox, buf, 64);
		printf("%x path=%s remotename=%s\n",
					mbox->account->mboxtype, mbox->account->path, buf);
		}
	printf("\n");
	}
#endif

static void 
apply_mail_config(void)
	{
	Mailbox			*mbox;
	MailAccount		*account, *new_account;
	Mailproc		*mp		= (Mailproc *) mail_fetch->private;
	GList			*list;
	gchar			*s;
	gint			i;
	GtkSpinButton	*spin;

	super_mute_mode = GTK_TOGGLE_BUTTON(super_mute_button)->active;
	enable_multimua = GTK_TOGGLE_BUTTON(enable_multimua_button)->active;
	cont_animation_mode = GTK_TOGGLE_BUTTON(enable_cont_anim_button)->active;
	enable_mail = GTK_TOGGLE_BUTTON(enable_mail_button)->active;
	if (enable_mail)
		gkrellm_panel_show(mail);
	else
		gkrellm_panel_hide(mail);

	for (i = 0; i < 3; ++i)
		if (GTK_TOGGLE_BUTTON(msg_count_button[i])->active)
			msg_count_mode = i;
	for (i = 0; i < 4; ++i)
		if (GTK_TOGGLE_BUTTON(animation_select_button[i])->active)
			animation_select_mode = i;
	mail_text_decal->value += 1000;		/* Force a redraw */
	mua_inhibit_mode = GTK_TOGGLE_BUTTON(mua_inhibit_button)->active;
	fetch_check_only_mode = GTK_TOGGLE_BUTTON(check_only_button)->active;
	reset_remote_mode = GTK_TOGGLE_BUTTON(reset_remote_button)->active;
	i = GTK_TOGGLE_BUTTON(unseen_is_new_button)->active;
	if (unseen_is_new != i)
		force_mail_check = TRUE;
	unseen_is_new = i;
	show_tooltip = GTK_TOGGLE_BUTTON(show_tooltip_button)->active;
	if (mh_seq_ignore_button)
		mh_seq_ignore = GTK_TOGGLE_BUTTON(mh_seq_ignore_button)->active;
	update_tooltip();

	s = gkrellm_entry_get_text(&mail_user_agent_entry);
	gkrellm_dup_string(&mail_user_agent.command, s);
	set_mua_button_sensitivity();
	
	s = gkrellm_entry_get_text(&mail_fetch_entry);
	if (gkrellm_dup_string(&mp->command, s))
		reset_mail_fetch();

	s = gkrellm_entry_get_text(&mail_notify_entry);
	gkrellm_dup_string(&mail_notify, s);

	spin = GTK_SPIN_BUTTON(mail_fetch_spin_button);
	mail_fetch_timeout = gtk_spin_button_get_value_as_int(spin);

	spin = GTK_SPIN_BUTTON(mail_check_spin_button);
	mail_check_timeout = gtk_spin_button_get_value_as_int(spin);

	if (mailboxes_modified)
		{
		/* Destroy MBOX_INTERNAL type mailboxes from the mailbox_list, then
		|  recreate them from the config list.  Skip over the first FETCH mbox.
		*/
		for (list = mailbox_list->next; list;  )
			{
			mbox = (Mailbox *) list->data;
			if (mbox->account->mboxtype & MBOX_INTERNAL)
				{
				GList *tmp;

/*				list = g_list_delete_link(list, list); */ /* glib 1.3 */
				tmp = list;
				list = g_list_remove_link(list, list);
				g_list_free(tmp);
				free_mailbox(mbox);
				}
			else
				list = list->next;
			}
		checking_mh_mail = FALSE;
		for (list = config_mailbox_list; list; list = list->next)
			{
			account = (MailAccount *) list->data;
			new_account = g_new0(MailAccount, 1);
			dup_account(new_account, account);
			add_mailbox(new_account);
			}
		force_mail_check = TRUE;
		}	
	}

static void
config_destroyed(void)
	{
	GList		*list;
	MailAccount	*account;

	for (list = config_mailbox_list; list; list = list->next)
		{
		account = (MailAccount *) list->data;
		free_account(account);
		}
	g_list_free(config_mailbox_list);
	config_mailbox_list = NULL;
	}

static void
copy_mailbox_accounts(void)
	{
	GList		*list;
	MailAccount	*account;
	Mailbox		*mbox;

	for (list = mailbox_list; list; list = list->next)
		{
		mbox = (Mailbox *) list->data;
		if (! (mbox->account->mboxtype & MBOX_INTERNAL))
			continue;
		account = g_new0(MailAccount, 1);
		dup_account(account, mbox->account);
		config_mailbox_list = g_list_append(config_mailbox_list, account);
		}
	}

static void
add_menu_item(GtkWidget *menu, gchar *label, gint type)
	{
	GtkWidget	*menuitem;

	menuitem = gtk_menu_item_new_with_label(label);
	gtk_menu_append(GTK_MENU(menu), menuitem);
	gtk_object_set_user_data(GTK_OBJECT(menuitem), GINT_TO_POINTER(type));
	gtk_signal_connect(GTK_OBJECT(menuitem), "activate",
			GTK_SIGNAL_FUNC(cb_protocol_selected), NULL);
	}

static gchar	*mail_info_text[]	=
{
N_("<b>Mailboxes\n"),

N_("Mailboxes to monitor can be local or remote mailbox types.\n"),
N_("For local mailboxes, the path name may be a mbox file or it may be\n"
"a Maildir or MH mail style directory.\n"),

N_("<b>\n\nSetup: Mail reading program\n"),
N_("If you enter a mail reading program (your mail user agent or MUA)\n"
"it can be launched by clicking on the mail monitor message count button.\n"
"For example you could enter:\n"),
N_("<i>\tsylpheed\n"),
N_("or\n"),
N_("<i>\tpronto\n"),
N_("or\n"),
N_("<i>\txterm -e mutt\n"),

N_("<b>\nSetup: Sound notify program\n"),
N_("If you enter a notify (sound) command, it will be run when new mail\n"
"is detected.  For example, you could enter:\n"),
N_("<i>\tcat /usr/local/sounds/newmail.wav > /dev/dsp\n"),
N_("or\n"),
N_("<i>\tesdplay /usr/local/sounds/newmail.wav\n\n"),

N_("<b>\nSetup: fetch/check Program\n"),
N_("If you want to download remote mail or check for remote mail without\n"
"using the builtin POP3 or IMAP mail checking which is set up in the\n"),
N_("<i>Mailboxes"),
N_(" tab, then do this by entering a mail fetch/check command.\n"
"For example, fetchmail can be run periodically to download mail\n"
"messages from a remote server to your local mailboxes where GKrellM\n"
"can count them.\n\n"

"Or, GKrellM can read the output and report the results from some mail\n"
"checking programs which count remote or local messages.  If you enter\n"
"a command like this that checks mail without downloading, then you\n"
"must tell GKrellM this is the expected behaviour by selecting in the\n"
"Options tab:\n"),
N_("<i>\tFetch/check program checks messages only\n"),
N_("For checking messages on a remote server, GKrellM can read the\n"
"output of the program "),
N_("<i>fetchmail -c"),
N_(" (you must append the -c).\n"),
N_("If you want to check for mail in local MH style mailboxes, then the\n"
"check program should be the nmh "),
N_("<i>flist -all.\n"),
N_("But, do not combine these methods for the same mailbox!  If you enter a\n"
"POP3 mailbox in the "),
N_("<i>Mailboxes"),
N_(" tab, then don't check it again with fetchmail.\n"),

N_("<b>\nMouse Button Actions:\n"),
N_("<b>\tLeft "),
N_("click the mail count button to launch the mail reading program.\n"),
N_("\t\tIf options permit, also stop animations and reset remote counts.\n"),
N_("<b>\tLeft "),
N_("click the envelope decal to force a mail check regardless of\n"
"\t\tany mute mode or inhibit mail checking option settings.\n"),
N_("<b>\tMiddle "),
N_("click the mail panel to toggle a mute mode which inhibits\n"
	"\t\tthe sound notify program and optionally inhibits all mail\n"
	"\t\tchecking.\n"),
};

static gchar	*mbox_title[] = {N_("Protocol"), N_("Mailboxes")};

static void
create_mail_tab(GtkWidget *tab_vbox)
	{
	GtkWidget		*tabs;
	GtkWidget		*table;
	GtkWidget		*menu;
	GtkWidget		*vbox, *vbox1, *hbox;
	GtkWidget		*label;
	GtkWidget		*button;
	GtkWidget		*scrolled;
	GtkWidget		*text;
	GList			*list;
	GSList			*group;
	gint			i;

	mailboxes_modified = FALSE;
	selected_row = -1;
	mh_seq_ignore_button = NULL;

	copy_mailbox_accounts();
	tabs = gtk_notebook_new();
	gtk_notebook_set_tab_pos(GTK_NOTEBOOK(tabs), GTK_POS_TOP);
	gtk_box_pack_start(GTK_BOX(tab_vbox), tabs, TRUE, TRUE, 0);
	gtk_signal_connect(GTK_OBJECT(tabs),"destroy",
				GTK_SIGNAL_FUNC(config_destroyed), NULL);

/* --Setup tab */
	vbox = gkrellm_create_framed_tab(tabs, _("Setup"));

	gkrellm_check_button(vbox, &enable_mail_button, enable_mail,
		FALSE, 10, _("Enable Mailcheck"));

	vbox1 = gkrellm_framed_vbox_end(vbox, NULL, 4, FALSE, 0, 2);
	table = gtk_table_new(5, 2, FALSE /*non-homogeneous*/);
	gtk_table_set_col_spacings(GTK_TABLE(table), 2);
	gtk_box_pack_start(GTK_BOX(vbox1), table, FALSE, FALSE, 2);

	hbox = gtk_hbox_new(TRUE, 0);
	/* Attach left right top bottom */
	gtk_table_attach(GTK_TABLE(table), hbox, 0, 1, 0, 1,
				GTK_SHRINK, GTK_SHRINK, 0, 0);
	label = gtk_label_new(_("Mail reading program:"));
	gtk_misc_set_alignment(GTK_MISC(label), 1.0, 0.5);
	gtk_box_pack_start(GTK_BOX(hbox), label, TRUE, TRUE, 4);
	mail_user_agent_entry = gtk_entry_new_with_max_length(255);
	gtk_table_attach_defaults(GTK_TABLE(table), mail_user_agent_entry,
				1, 2, 0, 1);
	gtk_entry_set_text(GTK_ENTRY(mail_user_agent_entry),
				mail_user_agent.command);

	hbox = gtk_hbox_new(TRUE, 0);
	gtk_table_attach(GTK_TABLE(table), hbox, 0, 1, 1, 2,
				GTK_SHRINK, GTK_SHRINK, 0, 0);
	label = gtk_label_new(_("Notify (sound) program:"));
	gtk_misc_set_alignment(GTK_MISC(label), 1.0, 0.5);
	gtk_box_pack_start(GTK_BOX(hbox), label, TRUE, TRUE, 4);
	mail_notify_entry = gtk_entry_new_with_max_length(255);
	gtk_table_attach_defaults(GTK_TABLE(table), mail_notify_entry, 1, 2, 1, 2);
	gtk_entry_set_text(GTK_ENTRY(mail_notify_entry), mail_notify);

	hbox = gtk_hbox_new(TRUE, 0);
	gtk_table_attach(GTK_TABLE(table), hbox, 0, 1, 2, 3,
				GTK_SHRINK, GTK_SHRINK, 0, 0);
	label = gtk_label_new(_("Mail fetch/check program:"));
	gtk_misc_set_alignment(GTK_MISC(label), 1.0, 0.5);
	gtk_box_pack_start(GTK_BOX(hbox), label, TRUE, TRUE, 4);
	mail_fetch_entry = gtk_entry_new_with_max_length(255);
	gtk_table_attach_defaults(GTK_TABLE(table), mail_fetch_entry, 1, 2, 2, 3);
	gtk_entry_set_text(GTK_ENTRY(mail_fetch_entry),
				((Mailproc *)mail_fetch->private)->command);

	hbox = gtk_hbox_new(TRUE, 0);
	gtk_table_attach(GTK_TABLE(table), hbox, 0, 1, 3, 4,
				GTK_SHRINK, GTK_SHRINK, 0, 0);
	label = gtk_label_new(_("Check local mailboxes every"));
	gtk_misc_set_alignment(GTK_MISC(label), 1.0, 0.5);
	gtk_box_pack_start(GTK_BOX(hbox), label, TRUE, TRUE, 4);
	hbox = gtk_hbox_new(FALSE, 0);
	gtk_table_attach_defaults(GTK_TABLE(table), hbox, 1, 2, 3, 4);
	gkrellm_spin_button(hbox, &mail_check_spin_button,
			(gfloat) mail_check_timeout, 2.0, 100.0, 1.0, 5.0, 0, 60,
			NULL, NULL, FALSE, _("seconds"));

	hbox = gtk_hbox_new(FALSE, 0);
	gtk_table_attach(GTK_TABLE(table), hbox, 0, 1, 4, 5,
				GTK_SHRINK, GTK_SHRINK, 0, 0);
	label = gtk_label_new(_("Do remote checks every"));
	gtk_misc_set_alignment(GTK_MISC(label), 1.0, 0.5);
	gtk_box_pack_start(GTK_BOX(hbox), label, TRUE, TRUE, 4);
	hbox = gtk_hbox_new(FALSE, 0);
	gtk_table_attach_defaults(GTK_TABLE(table), hbox, 1, 2, 4, 5);
	gkrellm_spin_button(hbox, &mail_fetch_spin_button,
			(gfloat) mail_fetch_timeout, 1.0, 60.0, 1.0, 5.0, 0, 60,
			NULL, NULL, FALSE, _("minutes"));

/* --Mailboxes tab */
	vbox = gkrellm_create_framed_tab(tabs, _("Mailboxes"));
	vbox1 = gkrellm_framed_vbox(vbox, NULL, 3, FALSE, 0, 3);
	account_notebook = gtk_notebook_new();
	gtk_box_pack_start(GTK_BOX(vbox1), account_notebook, FALSE, FALSE, 2);
	gtk_notebook_set_show_tabs(GTK_NOTEBOOK(account_notebook), FALSE);

	/* Local mailbox account entry */
	vbox1 = gtk_vbox_new(FALSE, 0);
	gtk_notebook_append_page(GTK_NOTEBOOK(account_notebook), vbox1, NULL);
	hbox = gtk_hbox_new(FALSE, 2);
	gtk_box_pack_start(GTK_BOX(vbox1), hbox, FALSE, FALSE, 0);
	label = gtk_label_new(_("Path name:"));
	gtk_misc_set_alignment(GTK_MISC(label), 1.0, 0.5);
	gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 0);
	mbox_path_entry = gtk_entry_new_with_max_length(255);
	gtk_box_pack_start(GTK_BOX(hbox), mbox_path_entry, TRUE, TRUE, 2);
	gtk_entry_set_text(GTK_ENTRY(mbox_path_entry), "");
	gtk_signal_connect (GTK_OBJECT (mbox_path_entry), "activate",
			(GtkSignalFunc) cb_enter_mailbox, NULL);			

#if defined(GKRELLM_THREADS)
	/* Remote mailbox account entry */
	table = gtk_table_new(3, 4, FALSE /*non-homogeneous*/);
	gtk_table_set_row_spacings(GTK_TABLE(table), 2);
	gtk_table_set_col_spacings(GTK_TABLE(table), 6);
	gtk_table_set_col_spacing(GTK_TABLE(table), 1, 16);
	gtk_notebook_append_page(GTK_NOTEBOOK(account_notebook), table, NULL);

	label = gtk_label_new(_("Server"));
	gtk_misc_set_alignment(GTK_MISC(label), 1.0, 0.5);
	gtk_table_attach_defaults(GTK_TABLE(table), label, 0, 1, 0, 1);
	server_entry = gtk_entry_new_with_max_length(255);
	gtk_table_attach_defaults(GTK_TABLE(table), server_entry, 1, 2, 0, 1);

	label = gtk_label_new(_("User name"));
	gtk_misc_set_alignment(GTK_MISC(label), 1.0, 0.5);
	gtk_table_attach_defaults(GTK_TABLE(table), label, 0, 1, 1, 2);
	user_entry = gtk_entry_new_with_max_length(255);
	gtk_table_attach_defaults(GTK_TABLE(table), user_entry, 1, 2, 1, 2);

	label = gtk_label_new(_("Password"));
	gtk_misc_set_alignment(GTK_MISC(label), 1.0, 0.5);
	gtk_table_attach_defaults(GTK_TABLE(table), label, 0, 1, 2, 3);
	password_entry = gtk_entry_new_with_max_length(255);
	gtk_table_attach_defaults(GTK_TABLE(table), password_entry, 1, 2, 2, 3);
	gtk_entry_set_visibility(GTK_ENTRY(password_entry), FALSE);

	label = gtk_label_new(_("Protocol"));
	gtk_misc_set_alignment(GTK_MISC(label), 1.0, 0.5);
	gtk_table_attach_defaults(GTK_TABLE(table), label, 2, 3, 0, 1);
	remote_option_menu = gtk_option_menu_new();
	gtk_table_attach_defaults(GTK_TABLE(table), remote_option_menu, 3, 4, 0,1);
	menu = gtk_menu_new();
	add_menu_item(menu, x_out(auth_strings[AUTH_POP3_PLAIN]), AUTH_POP3_PLAIN);
	add_menu_item(menu, x_out(auth_strings[AUTH_POP3_APOP]),  AUTH_POP3_APOP);
	add_menu_item(menu, x_out(auth_strings[AUTH_POP3_CRAM]),  AUTH_POP3_CRAM);
	add_menu_item(menu, x_out(auth_strings[AUTH_IMAP_PLAIN]), AUTH_IMAP_PLAIN);
	add_menu_item(menu, x_out(auth_strings[AUTH_IMAP_CRAM]),  AUTH_IMAP_CRAM);
	gtk_widget_show(menu);		/* Gtk bug? make option_menu notice menu size*/
	gtk_option_menu_set_menu(GTK_OPTION_MENU(remote_option_menu), menu);

	label = gtk_label_new(_("IMAP folder"));
	gtk_misc_set_alignment(GTK_MISC(label), 1.0, 0.5);
	gtk_table_attach_defaults(GTK_TABLE(table), label, 2, 3, 1, 2);
	imapfolder_entry = gtk_entry_new_with_max_length(255);
	gtk_table_attach_defaults(GTK_TABLE(table), imapfolder_entry, 3, 4, 1, 2);

	gkrellm_check_button_connected(NULL, &port_button, FALSE, FALSE, FALSE, 0,
				cb_specify_port, NULL, _("Specify port"));
	hbox = gtk_hbox_new(TRUE, 0);
	gtk_table_attach_defaults(GTK_TABLE(table), hbox, 2, 3, 2, 3);
	gtk_box_pack_start(GTK_BOX(hbox), port_button, FALSE, FALSE, 0);
	port_entry = gtk_entry_new_with_max_length(255);
	gtk_table_attach_defaults(GTK_TABLE(table), port_entry, 3, 4, 2, 3);
	gtk_widget_set_sensitive(port_entry, FALSE);
#endif  /* GKRELLM_THREADS */

	hbox = gtk_hbutton_box_new();
	gtk_button_box_set_layout(GTK_BUTTON_BOX(hbox), GTK_BUTTONBOX_END);
	gtk_button_box_set_spacing(GTK_BUTTON_BOX(hbox), 5);
	gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 0);

#if defined(GKRELLM_THREADS)
	button = gtk_radio_button_new_with_label(NULL, _("Local mailbox"));
	local_button = button;
	gtk_box_pack_start(GTK_BOX (hbox), button, TRUE, TRUE, 0);
	gtk_signal_connect(GTK_OBJECT(button), "clicked",
				GTK_SIGNAL_FUNC(set_mailbox_group), GINT_TO_POINTER(0));
	group = gtk_radio_button_group(GTK_RADIO_BUTTON (button));
	button = gtk_radio_button_new_with_label(group, _("Remote mailbox"));
	remote_button = button;
	gtk_box_pack_start(GTK_BOX(hbox), button, TRUE, TRUE, 0);
	gtk_signal_connect(GTK_OBJECT(button), "clicked",
				GTK_SIGNAL_FUNC(set_mailbox_group), GINT_TO_POINTER(1));
#endif

	gkrellm_button_connected(hbox, &button, FALSE, FALSE, 5,
			cb_enter_mailbox, NULL, _("Enter"));

	gkrellm_button_connected(hbox, &button, FALSE, FALSE, 5,
			cb_delete_mailbox, NULL, _("Delete"));

	vbox1 = gkrellm_framed_vbox(vbox, NULL, 6, TRUE, 0, 3);
	scrolled = gtk_scrolled_window_new(NULL, NULL);
	gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrolled),
			GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
	gtk_box_pack_start(GTK_BOX(vbox1), scrolled, TRUE, TRUE, 0);
    mbox_title[0] = _(mbox_title[0]);
    mbox_title[1] = _(mbox_title[1]);
	mbox_clist = gtk_clist_new_with_titles(2, mbox_title);
	gtk_signal_connect (GTK_OBJECT(mbox_clist), "select_row",
		(GtkSignalFunc) cb_mailbox_selected, NULL);
	gtk_signal_connect (GTK_OBJECT(mbox_clist), "unselect_row",
		(GtkSignalFunc) cb_mailbox_unselected, NULL);
	gtk_container_add (GTK_CONTAINER (scrolled), mbox_clist);
	gtk_clist_set_column_width(GTK_CLIST(mbox_clist), 0, 130);

	/* Skip the first internal mailbox.
	*/
	for (list = config_mailbox_list; list; list = list->next)
		{
		MailAccount	*account;

		account = (MailAccount *) list->data;
		mbox_clist_update(account, -1);
		}

/* --Animation tab */
	vbox = gkrellm_create_framed_tab(tabs, _("Animation"));
	vbox1 = gkrellm_framed_vbox(vbox, _("Animation Select"),
			4, FALSE, 0, 2);

	hbox = gtk_hbox_new(FALSE, 0);
	gtk_box_pack_start(GTK_BOX(vbox1), hbox, FALSE, FALSE, 0);

	button = gtk_radio_button_new_with_label(NULL, _("None"));
	gtk_box_pack_start(GTK_BOX(hbox), button, TRUE, TRUE, 0);
	group = gtk_radio_button_group(GTK_RADIO_BUTTON (button));
	animation_select_button[0] = button;
	button = gtk_radio_button_new_with_label(group, _("Envelope"));
	gtk_box_pack_start(GTK_BOX (hbox), button, TRUE, TRUE, 0);
	group = gtk_radio_button_group(GTK_RADIO_BUTTON (button));
	animation_select_button[1] = button;
#ifdef BSD
	button = gtk_radio_button_new_with_label(group, _("Daemon"));
#else
	button = gtk_radio_button_new_with_label(group, _("Penguin"));
#endif
	gtk_box_pack_start(GTK_BOX(hbox), button, TRUE, TRUE, 0);
	group = gtk_radio_button_group(GTK_RADIO_BUTTON (button));
	animation_select_button[2] = button;
	button = gtk_radio_button_new_with_label(group, _("Both"));
	gtk_box_pack_start(GTK_BOX(hbox), button, TRUE, TRUE, 0);
	animation_select_button[3] = button;
	button = animation_select_button[animation_select_mode];
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(button), TRUE);

    gkrellm_check_button(vbox, &enable_cont_anim_button, cont_animation_mode,
		FALSE, 10,
		_("Run animation continuously as long as there is a new mail count"));


/* --Options tab */
	vbox = gkrellm_create_framed_tab(tabs, _("Options"));

	vbox1 = gkrellm_framed_vbox(vbox, _("Message Counting"),
			4, FALSE, 0, 2);
	hbox = gtk_hbox_new (FALSE, 0);
	gtk_box_pack_start(GTK_BOX(vbox1), hbox, FALSE, FALSE, 0);

	button = gtk_radio_button_new_with_label(NULL, _("new/total"));
	gtk_box_pack_start(GTK_BOX (hbox), button, TRUE, TRUE, 0);
	group = gtk_radio_button_group(GTK_RADIO_BUTTON (button));
	msg_count_button[0] = button;
	button = gtk_radio_button_new_with_label(group, _("new"));
	gtk_box_pack_start(GTK_BOX(hbox), button, TRUE, TRUE, 0);
	group = gtk_radio_button_group(GTK_RADIO_BUTTON (button));
	msg_count_button[1] = button;
	button = gtk_radio_button_new_with_label(group, _("do not count"));
	gtk_box_pack_start(GTK_BOX(hbox), button, TRUE, TRUE, 0);
	msg_count_button[2] = button;
	button = msg_count_button[msg_count_mode];
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(button), TRUE);

    gkrellm_check_button(vbox, &check_only_button, fetch_check_only_mode,
		TRUE, 0,
		_("Fetch/check program checks messages only (see Mail Info tab)"));

    gkrellm_check_button(vbox, &reset_remote_button, reset_remote_mode, TRUE,0,
		_("Reset remote message counts when message count button is pressed."));

    gkrellm_check_button(vbox, &unseen_is_new_button, unseen_is_new, TRUE, 0,
	_("Count accessed but unseen mail as new (if this status is available)"));

    gkrellm_check_button(vbox, &super_mute_button, super_mute_mode, TRUE, 0,
   _("Mute mode inhibits all mail checking, not just notify (sound) program"));

    gkrellm_check_button(vbox, &mua_inhibit_button, mua_inhibit_mode, TRUE, 0,
		_("Inhibit all mail checking while the mail reader is running"));

	gkrellm_check_button(vbox, &enable_multimua_button, enable_multimua,
		TRUE, 0, _("Allow multiple launches of the mail reader program"));

    gkrellm_check_button(vbox, &show_tooltip_button, show_tooltip, TRUE, 0,
		_("List mailboxes containing new mail in a tooltip"));

	if (checking_mh_mail)
	    gkrellm_check_button(vbox, &mh_seq_ignore_button, mh_seq_ignore,
			TRUE, 0, _("Ignore .mh_sequences when checking MH mail."));
	
/* --Info tab */
	vbox = gkrellm_create_framed_tab(tabs, _("Info"));
	text = gkrellm_scrolled_text(vbox, NULL,
				GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
	for (i = 0; i < sizeof(mail_info_text)/sizeof(gchar *); ++i)
		gkrellm_add_info_text_string(text, _(mail_info_text[i]));
	}




static Monitor	monitor_mail =
	{
	N_("Mail"),				/* Name, for config tab.	*/
	MON_MAIL,			/* Id,  0 if a plugin		*/
	create_mail,		/* The create function		*/
	update_mail,		/* The update function		*/
	create_mail_tab,	/* The config tab create function	*/
	apply_mail_config,	/* Apply the config function		*/

	save_mail_config,	/* Save user conifg			*/
	load_mail_config,	/* Load user config			*/
	"mail",				/* config keyword			*/

	NULL,				/* Undef 2	*/
	NULL,				/* Undef 1	*/
	NULL,				/* Undef 0	*/

	0,					/* insert_before_id - place plugin before this mon */

	NULL,				/* Handle if a plugin, filled in by GKrellM		*/
	NULL				/* path if a plugin, filled in by GKrellM		*/
	};

Monitor *
init_mail_monitor(void)
	{
	monitor_mail.name = _(monitor_mail.name);
	enable_mail = TRUE;
	enable_multimua = FALSE;
	cont_animation_mode = FALSE;
	super_mute_mode = FALSE;
	mua_inhibit_mode = FALSE;
	GK.decal_mail_frames = 18;
	GK.decal_mail_delay = 1;

	mail_fetch = g_new0(Mailbox, 1);
	mail_fetch->account = g_new0(MailAccount, 1);
	mail_fetch->private = g_new0(Mailproc, 1);
	mail_fetch->account->mboxtype = MBOX_FETCH;
	mail_fetch->check_func = run_fetch_program;
	mailbox_list = g_list_append(mailbox_list, mail_fetch);

	((Mailproc *)(mail_fetch->private))->command = g_strdup("");
	mail_user_agent.command = g_strdup("");
	mail_notify = g_strdup("");

	style_id = gkrellm_add_meter_style(&monitor_mail, MAIL_STYLE_NAME);
	force_mail_check = TRUE;	/* Do a check at startup */
	mon_mail = &monitor_mail;

	return &monitor_mail;
	}


  /* ====================== Exported Mail Functions =====================*/

gboolean
gkrellm_get_mail_mute_mode(void)
	{
	return mute_mode;
	}

  /* ====================================================================*/
  /* Functions to allow for external mailboxes, as in plugins which want
  |  to check for alternate mailbox types but have the mail counts reported
  |  by the builtin mailbox monitor.  A plugin outline would be:
  |
  |	typedef struct
  |		{
  |		gpointer mbox_ptr;
  |		... local plugin stuff ...
  |		} PluginMbox;
  |
  |	PluginMbox	pmbox;
  |
  |	void create_plugin_mbox(GtkWidget *vbox, gint first_create)
  |		{  //See plugin programmers reference for general structure
  |		...
  |		pmbox.mbox_ptr = gkrellm_add_external_mbox(pmbox_check, TRUE, &pmbox);
  |		gkrellm_set_external_mbox_tooltip(pmbox.mbox_ptr, "mbox_name_stuff");
  |		...
  |		}
  |
  |	gint pmbox_check(PluginMbox *pmb)
  |		{ //Collect info from the mail box
  |		gkrellm_set_external_mbox_counts(pmb->mbox_ptr, total_mail, new_mail);
  |		}
  */

  /* External mailbox counts won't show in the tooltip unless you call this.
  |  If a plugin wants only to use the sound/animation feature and not show
  |  up in a tooltip, then do not make this call.
  */
void
gkrellm_set_external_mbox_tooltip(gpointer mbox_ptr, gchar *string)
	{
	Mailbox *mbox = (Mailbox *) mbox_ptr;

	gkrellm_dup_string(&mbox->account->path, string);
	}

  /* Set total and new message counts for an external mailbox so the counts
  |  can appear in a tooltip (must also call above routine), and so the
  |  animation/sound can be triggered.  Since sound and animation is triggered
  |  synchronously with remote and local checks, a plugin should
  |  call this routine from within a check_func that is setup in the
  |  gkrellm_add_external_mbox() routine.
  */
void
gkrellm_set_external_mbox_counts(gpointer mbox_ptr, gint total, gint new)
	{
	Mailbox *mbox = (Mailbox *) mbox_ptr;

	mbox->mail_count = total;
	mbox->new_mail_count = new;
	}

  /* A plugin can have a check_func() called at the mail monitors local
  |  check interval (threaded is FALSE) or at the remote check interval
  |  (threaded is TRUE).  Additionally, if threaded is TRUE, the check_func
  |  will be called as a thread.  The data pointer is a pointer to a plugin
  |  defined structure which specifies a plugins unique mailbox.  This data
  |  pointer will be passed as the argument to check_func(data) when it
  |  is called at the update intervals.  gkrellm_add_external_mbox() returns
  |  the internal Mailbox * which should be treated simply as a gpointer
  |  in the plugin and it must be used as the first argument to the above
  |  gkrellm_set_external_tooltip() and gkrellm_set_external_counts().
  */
gpointer
gkrellm_add_external_mbox(gboolean (*check_func)(), gboolean threaded,
			gpointer data)
	{
	Mailbox	*mbox;

	mbox = g_new0(Mailbox, 1);
	mbox->account = g_new0(MailAccount, 1);
	mbox->account->mboxtype = threaded ? MBOX_REMOTE_PLUGIN :MBOX_LOCAL_PLUGIN;
	mbox->check_func = check_func;
	mbox->data = data;
	mailbox_list = g_list_append(mailbox_list, mbox);
	return (gpointer) mbox;
	}

void
gkrellm_destroy_external_mbox(gpointer mbox_ptr)
	{
	GList	*tmp;

	if ((tmp = g_list_find(mailbox_list, mbox_ptr)) == NULL)
		return;
	mailbox_list = g_list_remove_link(mailbox_list, tmp);
	g_list_free(tmp);
	free_mailbox(mbox_ptr);
	}

  /* =======================================================================*/
