#include <stdio.h>
#include <string.h>
#include <gtk/gtk.h>
#include <libxml/parser.h>
#include <libxml/tree.h>
#include <libmpd/debug_printf.h>
#include <gmpc/plugin.h>
#include <gmpc/gmpc_easy_download.h>
#include <glib.h>
#include <config.h>

static char *__lastfm_art_get_artist_similar(char *nartist);
/* string functions */
static gchar *escape_uri_string (const gchar *string);
static int shrink_string(gchar *string, int start, int end);

static GtkWidget *wp_pref_vbox = NULL;
GMutex *last_fm_second_lock = NULL;

static int lastfm_get_enabled()
{
	return cfg_get_single_value_as_int_with_default(config, "cover-lastfm", "enable", TRUE);
}
static void lastfm_set_enabled(int enabled)
{
	cfg_set_single_value_as_int(config, "cover-lastfm", "enable", enabled);
}

static void lastfm_init(void)
{
	/**
	 * This test shouldn't be needed as gmpc itself does it
	 */
	char *file = gmpc_get_covers_path(NULL);
	if(!g_file_test(file,G_FILE_TEST_EXISTS)) {
		g_mkdir(file, 0755);
	}
	g_free(file);

    last_fm_second_lock = g_mutex_new();
}
static void lastfm_destroy(void)
{
    /* make sure it is unlocked before destroying */
    g_mutex_trylock(last_fm_second_lock);
    g_mutex_unlock(last_fm_second_lock);
    g_mutex_free(last_fm_second_lock);
    last_fm_second_lock = NULL;
}

static int lastfm_fetch_cover_priority(void){
	return cfg_get_single_value_as_int_with_default(config, "cover-lastfm", "priority", 80);
}


static gboolean unlock_lock(gpointer user_data)
{
    g_mutex_unlock(last_fm_second_lock);
    return FALSE;
}
static char * __lastfm_art_process_string(char *name)
{	
	return escape_uri_string(name);
}


static char *__lastfm_art_xml_get_artist_image(char *data,int size)
{
	char *retv = NULL;
	if(size  == 0)
		return NULL;
	if(data[0] != '<')
		return NULL;
	xmlDocPtr doc = xmlParseMemory(data,size);
	if(doc)
	{
		xmlNodePtr root = xmlDocGetRootElement(doc);
		xmlNodePtr cur = root;
		for(;cur;cur = cur->next){
			if(xmlStrEqual(cur->name, (xmlChar *)"similarartists")){
				xmlChar *temp = xmlGetProp(cur, (const xmlChar *) "picture");
				retv = g_strdup(temp);
				xmlFree(temp);
				/* one result is enough */
				xmlFreeDoc(doc);
				xmlCleanupParser();
				return retv;
			}
		}
		xmlFreeDoc(doc);
		xmlCleanupParser();
	}
	return NULL;
}
static char *__lastfm_art_xml_get_album_image(char *data,int size, char *album)
{
	char *retv = NULL;
	int is_album = FALSE;
	if(size  == 0)
		return NULL;
    /* Hack to check for xml file */
	if(data[0] != '<')
		return NULL;
	gchar *url = NULL;
	xmlDocPtr doc = xmlParseMemory(data,size);
	if(doc)
	{
		xmlNodePtr root = xmlDocGetRootElement(doc);
		xmlNodePtr cur = root->xmlChildrenNode;
        is_album = FALSE;
        
        /* loop through all albums */
        for(;cur && !is_album;cur = cur->next){
			if(xmlStrEqual(cur->name, (xmlChar *)"album")){
				xmlNodePtr cur2 = cur->xmlChildrenNode;

				for(;cur2;cur2 = cur2->next)
				{
					if(xmlStrEqual(cur2->name, (xmlChar *)"name"))
					{
						xmlChar *temp = xmlNodeGetContent(cur2);
						if(strcasecmp(temp, album) == 0) 
						{
                            /* found album, set this true, so we will _always_ bail out */
							is_album = TRUE;
						}
                        xmlFree(temp);
                    }
					else if (xmlStrEqual(cur2->name, (xmlChar *)"image"))
					{
						xmlNodePtr cur3 = cur2->xmlChildrenNode;
                        int type = 0;
						for(;cur3 && !url;cur3 = cur3->next)
						{
							if(type < 3 && xmlStrEqual(cur3->name, (xmlChar *)"large"))
							{
								xmlChar *temp = xmlNodeGetContent(cur3);
                                if(url)
                                    g_free(url);
								url = g_strdup(temp);
                                xmlFree(temp);
                                type = 3;
							}
                            else if ( type <2 && xmlStrEqual(cur3->name, (xmlChar *)"medium"))
                            {
								xmlChar *temp = xmlNodeGetContent(cur3);
                                if(url)
                                    g_free(url);
								url = g_strdup(temp);
                                xmlFree(temp);
                                type = 2;                                   
                            }
						}
					}
				}
                /* if we found url, but no album, bail out */
                if(!is_album)
                {
                    if(url)
                        g_free(url);
                    url = NULL;
                }
			}
		}
		xmlFreeDoc(doc);
		xmlCleanupParser();
		if(is_album && url)
		{
			return url;
		}
		if(url)
			g_free(url);
	}
	return NULL;
}


static int __lastfm_art_get_artist_image(char *nartist, char **path)
{

	gmpc_easy_download_struct data= {NULL, 0,-1,NULL, NULL};
	int found = META_DATA_UNAVAILABLE;
	char furl[1024];

	snprintf(furl,1024,"http://ws.audioscrobbler.com/1.0/artist/%s/similar.xml", nartist);

	if(gmpc_easy_download(furl, &data))
	{
		char *url = __lastfm_art_xml_get_artist_image(data.data, data.size);
		gmpc_easy_download_clean(&data);
		if(url && url[0] != '\0' && strstr(url, "noartist_") == NULL) 
		{
			gmpc_easy_download(url, &data);
			if(data.size){
				int i =0;
				FILE *fp = NULL;
				char *imgpath = NULL;
				char *filename = NULL;	
				filename= g_strdup_printf("%s.jpg",nartist);
				imgpath = gmpc_get_covers_path(filename);
				fp = fopen(imgpath, "wb");
				if(fp)
				{
					fwrite(data.data, sizeof(char), data.size,fp);
					fclose(fp);
				}
				g_free(filename);
                *path = imgpath;
				gmpc_easy_download_clean(&data);
				found = META_DATA_AVAILABLE;
			}
		}
		if(url)g_free(url);

	}

	return found;
}
/*
 * Get 20 artists
 */
static char *__lastfm_art_xml_get_artist_similar(char *data,int size)
{
	char *retv = NULL;
	GString *string;
	if(size  == 0)
		return NULL;
	if(data[0] != '<')
		return NULL;
	string = g_string_new("");
	xmlDocPtr doc = xmlParseMemory(data,size);
	if(doc)
	{
		xmlNodePtr root = xmlDocGetRootElement(doc);
		xmlNodePtr cur = root;
		for(;cur;cur = cur->next){
			if(xmlStrEqual(cur->name, (xmlChar *)"similarartists")){
				xmlNodePtr cur2 = cur->xmlChildrenNode;
				for(;cur2;cur2=cur2->next)
				{
					if(xmlStrEqual(cur2->name, (xmlChar *)"artist"))
					{
						xmlNodePtr cur3 = cur2->xmlChildrenNode;
						for(;cur3;cur3=cur3->next)
						{
							if(xmlStrEqual(cur3->name, (xmlChar *)"name"))
							{
								xmlChar *temp = xmlNodeGetContent(cur3);
								g_string_append_printf(string, "%s\n", temp);
								xmlFree(temp);
							}
						}
					}
				}
			}
		}
		xmlFreeDoc(doc);
		xmlCleanupParser();
	}
	if(string->len>0)
		retv = string->str;
	g_string_free(string, FALSE);
	return retv;
}
/*
 * Get's similar list 
 */
gchar *__lastfm_art_get_artist_similar(char *nartist)
{

	gmpc_easy_download_struct data= {NULL, 0,-1,NULL, NULL};
	int found = 0;
	char furl[1024];
	char *artist = __lastfm_art_process_string(nartist);
	snprintf(furl,1024,"http://ws.audioscrobbler.com/1.0/artist/%s/similar.xml", artist);
	g_free(artist);
	if(gmpc_easy_download(furl, &data))
	{
		char *result = __lastfm_art_xml_get_artist_similar(data.data,data.size);
		gmpc_easy_download_clean(&data);
		return result;
	}
	return NULL;
}


static int __lastfm_art_get_album_image(char *nartist, char *nalbum,char *artist, char **path)
{

	gmpc_easy_download_struct data= {NULL, 0,-1,NULL, NULL};
	int found = META_DATA_UNAVAILABLE;
	char furl[1024];

	snprintf(furl,1024,"http://ws.audioscrobbler.com/1.0/artist/%s/topalbums.xml", nartist);
    debug_printf(DEBUG_INFO, "furl: %s\n", furl);
	if(gmpc_easy_download(furl, &data))
	{
		char *url = __lastfm_art_xml_get_album_image(data.data, data.size,nalbum);
		gmpc_easy_download_clean(&data);
		if(url && url[0] != '\0' &&  strstr(url, "noartist_") == NULL && strstr(url, "noimage") == NULL) 
		{
			gmpc_easy_download(url, &data);
			if(data.size){
				int i =0;
				FILE *fp = NULL;
				char *imgpath = NULL;
				char *filename = NULL;
				filename = g_strdup_printf("%s - %s.jpg",nartist,nalbum);
				imgpath = gmpc_get_covers_path(filename);
				fp = fopen(imgpath, "wb");
				if(fp)
				{
					fwrite(data.data, sizeof(char), data.size,fp);
					fclose(fp);
				}
				g_free(filename);
                *path = imgpath;
				gmpc_easy_download_clean(&data);
				found = META_DATA_AVAILABLE;
                debug_printf(DEBUG_INFO, "Found cover arti for: %s %s\n", nartist, nalbum);
			}
		}
		if(url)g_free(url);
	}
	return found;
}


static int lastfm_fetch_cover_album_art(mpd_Song *song, char **path)
{
	int retval;
	char *artist = __lastfm_art_process_string(song->artist);
    /* no need to encode this */
	char *album = song->album; 
    debug_printf(DEBUG_INFO, "Trying to fetch: %s:%s\n", artist, album);
	retval = __lastfm_art_get_album_image(artist,album, song->artist,path);
	g_free(artist);
	return retval;
}

static int lastfm_fetch_cover_art(mpd_Song *song, char **path)
{
	int retval;
	char *artist = __lastfm_art_process_string(song->artist);
	retval = __lastfm_art_get_artist_image(artist,path);
	g_free(artist);
	return retval;
}

static gchar *escape_uri_string (const gchar *string)
{
#define ACCEPTABLE(a) (((a) >= 'a' && (a) <= 'z') || ((a) >= 'A' && (a) <= 'Z') || ((a) >= '0' && (a) <= '9'))
#define LASTFM_SPECIAL(a) ((a) == ';' || (a) == '&' || (a) == '/' || (a) == '+' || (a) == '#')

	const gchar hex[16] = "0123456789ABCDEF";
	const gchar *p;
	gchar *q;
	gchar *result;
	int c;
	gint unacceptable = 0;
	const gchar *tmp_p;
	gchar *new_string;
	int depth = 0;
	int len;
	int i = 0;

	len = strlen(string);

	new_string = g_malloc(len + 1);

	/*	Get count of chars that will need to be converted to %
		and remove ([{}]) and everything between */
	for (p = string; *p != '\0'; p++) {
		c = (guchar) *p;

		if(c == '(' || c == '[' || c == '{') {
			depth++;
		} else if(c == ')' || c == ']' || c == '}') {
			depth--;
			if(depth < 0)
				depth = 0;
		} else if(depth == 0) {
			if (!ACCEPTABLE (c)) 
				unacceptable++;
			if(LASTFM_SPECIAL(c))
				unacceptable++;

			new_string[i] = c;

			i++;
		}
	}

	new_string[i] = '\0';

	len = strlen(new_string);

	/* remove double spaces from the string because removing ([{}])
		tends to create those */
	for(p = new_string + 1; *p != '\0'; p++) {
		c = (guchar) *p;
		if(c == ' ') {
			tmp_p = p - 1;
			if( *tmp_p == ' ') {
				len = shrink_string(new_string,  p - new_string, len);
				p--;
			}
		}
	}
	
	/* make sure first char isn't a space */
	if(new_string[0] == ' ')
		len = shrink_string(new_string, 0, len);

	/* make sure there isn't a trailing space*/
	if(new_string[len - 1] == ' ')
	len--;

	new_string[len] = '\0';

	result = g_malloc (len + unacceptable * 2 + 1);

	/*time to create the escaped string*/
	for (q = result, p = new_string; *p != '\0'; p++)
	{
		c = (guchar) *p;

		if (!ACCEPTABLE (c)) {
			*q++ = '%'; /* means hex coming */
			if(LASTFM_SPECIAL(c)){
				*q++ = '2';
				*q++ = '5';
			}
			*q++ = hex[c >> 4];
			*q++ = hex[c & 15];
		}
		else
			*q++ = *p;
	}

	*q = '\0';

	g_free(new_string);

	return result;
}

static int shrink_string(gchar *string, int start, int end)
{
	int i;
	
	for( i = start; i < end; i++)
		string[i] = string[i + 1];
		
	end--;
	
	return end;
}



static int lastfm_fetch_get_image(mpd_Song *song,MetaDataType type, char **path)
{
	int result = 0;
	if(song->artist == NULL || lastfm_get_enabled() == FALSE)
	{
		return META_DATA_UNAVAILABLE;
	}
	if(type == META_ARTIST_ART)
	{
        /** So we don't do to much queries a second */
        /* this is nicer then waiting one second, because now the result is directly available, only a new query < 1 second is delayed
        */
        g_mutex_lock(last_fm_second_lock);
        result = lastfm_fetch_cover_art(song, path);
        g_timeout_add(1000, (GSourceFunc)unlock_lock, NULL);
        return result;
    }
	else if (type == META_ALBUM_ART && cfg_get_single_value_as_int_with_default(config, "cover-lastfm", "fetch-album", TRUE))
	{
        g_mutex_lock(last_fm_second_lock);
		result = lastfm_fetch_cover_album_art(song, path);
        g_timeout_add(1000, (GSourceFunc)unlock_lock, NULL);
        return result;
	}
	else if (type == META_ARTIST_SIMILAR)
	{
        g_mutex_lock(last_fm_second_lock);
		*path = __lastfm_art_get_artist_similar(song->artist);
        g_timeout_add(1000, (GSourceFunc)unlock_lock, NULL);
        if(*path)
		{
			return META_DATA_AVAILABLE;	
		}
        if(*path) g_free(*path);
    }
	return META_DATA_UNAVAILABLE; 
}	

gmpcMetaDataPlugin lf_cover = {
	.get_priority   = lastfm_fetch_cover_priority,
	.get_image      = lastfm_fetch_get_image
};

int plugin_api_version = PLUGIN_API_VERSION;

gmpcPlugin plugin = {
	.name           = "Last FM Artist Image Fetcher",
	.version        = {PLUGIN_MAJOR_VERSION,PLUGIN_MINOR_VERSION,PLUGIN_MICRO_VERSION},
	.plugin_type    = GMPC_PLUGIN_META_DATA,
	.init           = lastfm_init,
    .destroy        = lastfm_destroy,
	.metadata       = &lf_cover,
	.get_enabled    = lastfm_get_enabled,
	.set_enabled    = lastfm_set_enabled
};
