/*
 * (SLIK) SimpLIstic sKin functions
 * (C) 2002 John Ellis
 *
 * Author: John Ellis
 *
 * This software is released under the GNU General Public License (GNU GPL).
 * Please read the included file COPYING for more information.
 * This software comes with no warranty of any kind, use at your own risk!
 */

#include "ui2_includes.h"
#include "ui2_typedefs.h"
#include "ui2_display.h"

#include "ui2_skin.h"
#include "ui2_util.h"
#include "ui2_widget.h"
#include "ui_pixbuf_ops.h"


/*
 *-----------------------------------------------------------------------------
 * Screen refreshing
 *-----------------------------------------------------------------------------
 */

/* ok, so the freeze is ugly, but does help when first loading a skin */

static void ui_display_render_freeze(UIData *ui)
{
	ui->frozen = TRUE;
}

static void ui_display_render_thaw(UIData *ui)
{
	ui->frozen = FALSE;
}

void ui_display_render_area(UIData *ui, gint x, gint y, gint w, gint h)
{
	GdkPixbuf *pixbuf;
	GdkPixmap *pixmap;

	if (ui->frozen) return;
	if (w < 1 || h < 1) return;

	pixbuf = skin_get_pixbuf(ui->skin);
	pixmap = skin_get_buffer(ui->skin);

	if (!pixbuf || !pixmap || !ui->display->window) return;

	if (debug_mode) printf("render: %d, %d (%d x %d)\n", x, y, w, h);

	gdk_pixbuf_render_to_drawable(pixbuf, pixmap,
				      ui->display->style->fg_gc[GTK_WIDGET_STATE(ui->display)],
				      x, y, x, y, w, h,
				      GDK_RGB_DITHER_NONE, 1, 1); /* FIXME: no dithering */

	gdk_draw_pixmap(ui->display->window, ui->display->style->fg_gc[GTK_WIDGET_STATE(ui->display)],
			pixmap, x, y, x, y, w, h);
}

void ui_display_redraw_area(UIData *ui, gint x, gint y, gint w, gint h)
{
	GList *work;

	pixbuf_copy_area(ui->skin->overlay, x, y,
			 ui->skin->pixbuf, x, y, w, h, FALSE);
	ui_display_render_freeze(ui);

	work = ui->skin->widget_list;
	while(work)
		{
		WidgetData *wd = work->data;
		if (ui_widget_contacts_area(wd, x, y, w, h))
			{
			ui_widget_draw(ui, wd, TRUE, TRUE);
			}
		work = work->next;
		}

	ui_display_render_thaw(ui);
	ui_display_render_area(ui, x, y, w, h);
}

/*
 *-----------------------------------------------------------------------------
 * misc
 *-----------------------------------------------------------------------------
 */

void ui_display_sync_shape(UIData *ui)
{
	if (!ui->skin) return;

	gtk_drawing_area_size(GTK_DRAWING_AREA(ui->display), ui->skin->width, ui->skin->height);
	if (ui->window)
		{
		gtk_widget_shape_combine_mask(ui->window, ui->skin->mask_buffer, 0, 0);
		gtk_widget_set_usize(ui->window, ui->skin->width, ui->skin->height);
		}
	else
		{
		gtk_widget_shape_combine_mask(ui->display, ui->skin->mask_buffer, 0, 0);
		gtk_widget_set_usize(ui->display, ui->skin->width, ui->skin->height);
		}
}

static void ui_display_draw_all(UIData *ui, gint update, gint force)
{
	GList *work;

	work = ui->skin->widget_list;
	while(work)
		{
		ui_widget_draw(ui, work->data, update, force);
		work = work->next;
		}
}

void ui_display_sync(UIData *ui, gint update)
{
	if (!ui->skin) return;

	ui_display_render_freeze(ui);
	ui_display_draw_all(ui, update, TRUE);
	ui_display_render_thaw(ui);

	/* do it all at once */
	ui_display_render_area(ui, 0, 0, ui->skin->width, ui->skin->height);
}

void ui_display_sync_all(UIData *ui)
{
	if (!ui->skin) return;

	skin_check_bounds(ui->skin);

	skin_sync_buffer(ui->skin, ui);
	ui_display_sync_shape(ui);

	ui_display_sync(ui, TRUE);
}

static void ui_display_expose(GtkWidget *widget, GdkEventExpose *event, gpointer data)
{
	UIData *ui = data;
	GdkPixmap *pixmap;

	pixmap = skin_get_buffer(ui->skin);
	if (!pixmap) return;

	gdk_draw_pixmap(ui->display->window, ui->display->style->fg_gc[GTK_WIDGET_STATE(ui->display)],
			pixmap,
			(gint)event->area.x, (gint)event->area.y,
			(gint)event->area.x, (gint)event->area.y,
			(gint)event->area.width, (gint)event->area.height);
}

static gint ui_display_move_idle_cb(gpointer data)
{
	UIData *ui = data;

	skin_sync_back(ui->skin, ui);
	ui_display_sync(ui, FALSE);

	ui->root_win_idle = -1;

	return FALSE;
}

static void ui_display_move(GtkWidget *widget, GdkEventConfigure *event, gpointer data)
{
	UIData *ui = data;

	if (debug_mode) printf("move event\n");

	if (ui->root_win_idle == -1 &&
	    (ui->skin->transparent || slik_transparency_force))
		{
		ui->root_win_idle = gtk_idle_add(ui_display_move_idle_cb, ui);
		}
}

/*
 *-----------------------------------------------------------------------------
 * Button motion / press handling
 *-----------------------------------------------------------------------------
 */

static void ui_display_motion(GtkWidget *w, GdkEventMotion *event, gpointer data)
{
	UIData *ui = data;
	gint x = (gint)event->x;
	gint y = (gint)event->y;
	GList *work;

	if (debug_skin) printf("motion:%d x %d\n", x, y);

	if (ui->active_widget)
		{
		ui_widget_motion(ui, ui->active_widget, x, y);
		return;
		}

	if (ui->in_move)
		{
		GdkModifierType modmask;
		gint px, py;

		if (!ui->window) return;

		gdk_window_get_pointer (NULL, &px, &py, &modmask);
		px -= ui->press_x;
		py -= ui->press_y;
		if (slik_smart_placement)
			{
			gint w = gdk_screen_width();
			gint h = gdk_screen_height();
			if (px < 8 && px > -8) px = 0;
			if (py < 8 && py > -8) py = 0;
			if (px + ui->skin->width > w - 8 && px + ui->skin->width < w + 8) px = w - ui->skin->width;
			if (py + ui->skin->height > h - 8 && py + ui->skin->height < h +8) py = h - ui->skin->height;
			}
		gdk_window_move(ui->window->window, px, py);
		return;
		}

	work = ui->skin->widget_list;
	while(work)
		{
		ui_widget_motion(ui, work->data, x, y);
		work = work->next;
		}
}

static void ui_display_pressed(GtkWidget *w, GdkEventButton *event, gpointer data)
{
	UIData *ui = data;
	GdkModifierType modmask;
	gint x = (gint)event->x;
	gint y = (gint)event->y;
	GList *work;

	if (debug_mode) printf("pressed:%d x %d\n", x, y);

	/* edit uis have no window, and we only want button 1 to work on them */
	if (event->button > 1 && !ui->window) return;

	if (event->button >= 2)
		{
		if (ui->click_func) ui->click_func(ui, event->button, event->time, ui->click_data);
		return;
		}

	if (event->type != GDK_BUTTON_PRESS) return;

	gtk_grab_add(ui->display);

	ui->press_x = x;
	ui->press_y = y;

	work = ui->skin->widget_list;
	while(work)
		{
		if (ui_widget_press(ui, work->data, x, y))
			{
			ui->active_widget = work->data;
			if (debug_mode) printf("pressed widget: \"%s\" (%d)\n", ui->active_widget->key, ui->active_widget->type);
			return;
			}
		work = work->next;
		}

	if (ui->allow_move)
		{
		ui->in_move = TRUE;
		gdk_window_get_pointer (NULL, &ui->press_root_x, &ui->press_root_y, &modmask);
		}
	ui->in_press = TRUE;
}

static void ui_display_released(GtkWidget *w, GdkEventButton *event, gpointer data)
{
	UIData *ui = data;
	gint x = (gint)event->x;
	gint y = (gint)event->y;

	if (debug_mode) printf("released:%d x %d\n", x, y);

	gtk_grab_remove(ui->display);

	if (ui->active_widget)
		{
		if (debug_mode) printf("releasing widget: \"%s\" (%d)\n", ui->active_widget->key, ui->active_widget->type);
		ui_widget_release(ui, ui->active_widget, x, y);
		ui->active_widget = NULL;
		}
	else
		{
		GdkModifierType modmask;
		gint px;
		gint py;
		gdk_window_get_pointer (NULL, &px, &py, &modmask);
		if (px == ui->press_root_x && py == ui->press_root_y && ui->window)
			{
			gdk_window_raise (ui->window->window);
			}
		}

	ui->in_press = FALSE;
	ui->in_move = FALSE;
}

static void ui_display_leave(GtkWidget *widget, GdkEventCrossing *event, gpointer data)
{
	UIData *ui = data;

	if (!ui->active_widget)
		{
		GList *work;

		work = ui->skin->widget_list;
		while(work)
			{
			ui_widget_reset(ui, work->data);
			work = work->next;
			}
		}
}

void ui_display_events_init(UIData *ui)
{
	gtk_widget_set_events(ui->display, GDK_ENTER_NOTIFY_MASK | GDK_LEAVE_NOTIFY_MASK | GDK_POINTER_MOTION_MASK |
			GDK_BUTTON_PRESS_MASK | GDK_BUTTON_RELEASE_MASK);
	gtk_signal_connect(GTK_OBJECT(ui->display),"motion_notify_event", ui_display_motion, ui);
	gtk_signal_connect(GTK_OBJECT(ui->display),"button_press_event", ui_display_pressed, ui);
	gtk_signal_connect(GTK_OBJECT(ui->display),"button_release_event", ui_display_released, ui);
	gtk_signal_connect(GTK_OBJECT(ui->display),"leave_notify_event", ui_display_leave, ui);

	ui->active_widget = NULL;
	ui->in_press = FALSE;
	ui->in_move = FALSE;

	/* the expose event */
	gtk_signal_connect(GTK_OBJECT(ui->display), "expose_event", (GtkSignalFunc) ui_display_expose, ui);
	if (ui->window)
		{
		gtk_signal_connect(GTK_OBJECT(ui->window), "configure_event", (GtkSignalFunc) ui_display_move, ui);
		}
}




