/*
 * GQview
 * (C) 2002 John Ellis
 *
 * Author: John Ellis
 *
 * This software is released under the GNU General Public License (GNU GPL).
 * Please read the included file COPYING for more information.
 * This software comes with no warranty of any kind, use at your own risk!
 */

#include "gqview.h"
#include "cache_maint.h"

#include "cache.h"
#include "ui_fileops.h"
#include "ui_utildlg.h"


typedef struct _CMData CMData;
struct _CMData
{
	GList *list;
	GList *done_list;
	gint idle_id;
	GenericDialog *gd;
	GtkWidget *entry;
	gint clear;
};

#define PURGE_DIALOG_WIDTH 400


/*
 *-------------------------------------------------------------------
 * cache maintenance
 *-------------------------------------------------------------------
 */

static gint extension_truncate(gchar *path, const gchar *ext)
{
	gint l;
	gint el;

	if (!path || !ext) return FALSE;

	l = strlen(path);
	el = strlen(ext);

	if (l < el || strcmp(path + (l - el), ext) != 0) return FALSE;

	path[l - el] = '\0';

	return TRUE;
}

static gchar *extension_find_dot(gchar *path)
{
	gchar *ptr;

	if (!path || *path == '\0') return NULL;

	ptr = path;
	while (*ptr != '\0') ptr++;

	while (ptr > path && *ptr != '.') ptr--;

	if (ptr == path) return NULL;

	return ptr;
}

static gint isempty(const gchar *path)
{
	DIR *dp;
	struct dirent *dir;

	if((dp = opendir(path)) == NULL) return FALSE;

	while ((dir = readdir(dp)) != NULL)
		{
		gchar *name = dir->d_name;

		if (dir->d_ino > 0 &&
		    !(name[0] == '.' && (name[1] == '\0' || (name[1] == '.' && name[2] == '\0'))) )
			{
			closedir(dp);
			return FALSE;
			}
		}

	closedir(dp);
	return TRUE;
}

static void cache_maintain_home_close(CMData *cm)
{
	if (cm->idle_id != -1) gtk_idle_remove(cm->idle_id);
	if (cm->gd) generic_dialog_close(cm->gd);
	path_list_free(cm->list);
	g_list_free(cm->done_list);
	g_free(cm);
}

static gint cache_maintain_home_cb(gpointer data)
{
	CMData *cm = data;
	GList *dlist = NULL;
	GList *list = NULL;
	gchar *path;
	gint just_done = FALSE;
	gint still_have_a_file = TRUE;
	gint base_length;

	base_length = strlen(homedir()) + strlen("/") + strlen(GQVIEW_RC_DIR_THUMBS);

	if (!cm->list)
		{
		if (debug) printf("purge chk done.\n");
		cm->idle_id = -1;
		cache_maintain_home_close(cm);
		return FALSE;
		}

	path = cm->list->data;

	if (debug) printf("purge chk (%d) \"%s\"\n", cm->clear, path);

	if (g_list_find(cm->done_list, path) == NULL)
		{
		cm->done_list = g_list_prepend(cm->done_list, path);

		if (path_list(path, &list, &dlist))
			{
			GList *work;

			just_done = TRUE;
			still_have_a_file = FALSE;
	
			work = list;
			while(work)
				{
				gchar *path_buf = work->data;
				gchar *dot;
	
				dot = extension_find_dot(path_buf);
	
				if (dot) *dot = '\0';
				if (cm->clear ||
				    (strlen(path_buf) > base_length && !isfile(path_buf + base_length)) )
					{
					if (dot) *dot = '.';
					if (unlink(path_buf) < 0) printf("failed to delete:%s\n", path_buf);
					}
				else
					{
					still_have_a_file = TRUE;
					}
				work = work->next;
				}
			}
		}
	path_list_free(list);

	cm->list = g_list_concat(dlist, cm->list);

	if (cm->list && g_list_find(cm->done_list, cm->list->data) != NULL)
		{
		/* check if the dir is empty */
		
		if (cm->list->data == path && just_done)
			{
			if (!still_have_a_file && !dlist && cm->list->next && rmdir(path) < 0)
				{
				printf("Unable to delete dir: %s\n", path);
				}
			}
		else
			{
			/* must re-check for an empty dir */
			path = cm->list->data;
			if (isempty(path) && cm->list->next && rmdir(path) < 0)
				{
				printf("Unable to delete dir: %s\n", path);
				}
			}

		path = cm->list->data;
		cm->done_list = g_list_remove(cm->done_list, path);
		cm->list = g_list_remove(cm->list, path);
		g_free(path);
		}

	if (cm->list)
		{
		const gchar *buf;

		path = cm->list->data;
		if (strlen(path) > base_length)
			{
			buf = path + base_length;
			}
		else
			{
			buf = "...";
			}
		gtk_entry_set_text(GTK_ENTRY(cm->entry), buf);
		}

	return TRUE;
}

static void cache_maintain_home_cancel_cb(GenericDialog *gd, gpointer data)
{
	CMData *cm = data;
	cm->gd = NULL;
	cache_maintain_home_close(cm);
}

/* sorry for complexity (cm->done_list), but need it to remove empty dirs */
void cache_maintain_home(gint clear)
{
	CMData *cm;
	GList *dlist = NULL;
	gchar *base;
	const gchar *msg;

	base = g_strconcat(homedir(), "/", GQVIEW_RC_DIR_THUMBS, NULL);

	if (!path_list(base, NULL, &dlist))
		{
		g_free(base);
		return;
		}

	dlist = g_list_append(dlist, base);

	cm = g_new0(CMData, 1);
	cm->list = dlist;
	cm->done_list = NULL;
	cm->clear = clear;

	if (clear)
		{
		msg = _("Clearing thumbnails...");
		}
	else
		{
		msg = _("Purging old thumbnails...");
		}

	cm->gd = generic_dialog_new(_("Purge thumbnails"), msg,
				    "GQview", "purge_thumbnails", TRUE,
				    cache_maintain_home_cancel_cb, cm);
	gtk_widget_set_usize(cm->gd->dialog, PURGE_DIALOG_WIDTH, -1);
	cm->entry = gtk_entry_new();
	gtk_widget_set_sensitive(cm->entry, FALSE);
	gtk_box_pack_start(GTK_BOX(cm->gd->vbox), cm->entry, FALSE, FALSE, 5);
	gtk_widget_show(cm->entry);
	
	gtk_widget_show(cm->gd->dialog);

	cm->idle_id = gtk_idle_add(cache_maintain_home_cb, cm);
}

/* This checks all files in ~/.gqview/thumbnails and
 * removes them if thay have no source counterpart.
 * (this assumes all cache files have an extension of 4 chars including '.')
 */
gint cache_maintain_home_dir(const gchar *dir, gint recursive, gint clear)
{
	gchar *base;
	gint base_length;
	GList *dlist = NULL;
	GList *flist = NULL;
	gint still_have_a_file = FALSE;

	if (debug) printf("maintainance check: %s\n", dir);

	base_length = strlen(homedir()) + strlen("/") + strlen(GQVIEW_RC_DIR_THUMBS);
	base = g_strconcat(homedir(), "/", GQVIEW_RC_DIR_THUMBS, dir, NULL);

	if (path_list(base, &flist, &dlist))
		{
		GList *work;

		work = dlist;
		while(work)
			{
			gchar *path = work->data;
			if (recursive && strlen(path) > base_length &&
			    !cache_maintain_home_dir(path + base_length, recursive, clear))
				{
				if (debug) printf("Deleting thumb dir: %s\n", path);
				if (rmdir(path) < 0)
					{
					printf("Unable to delete dir: %s\n", path);
					}
				}
			else
				{
				still_have_a_file = TRUE;
				}
			work = work->next;
			}

		work = flist;
		while (work)
			{
			gchar *path = work->data;
			gchar *dot;

			dot = extension_find_dot(path);

			if (dot) *dot = '\0';
			if (clear ||
			    (strlen(path) > base_length && !isfile(path + base_length)) )
				{
				if (dot) *dot = '.';
				if (unlink(path) < 0) printf("failed to delete:%s\n", path);
				}
			else
				{
				still_have_a_file = TRUE;
				}

			work = work->next;
			}
		}

	path_list_free(dlist);
	path_list_free(flist);
	g_free(base);

	return still_have_a_file;
}

/* This checks relative caches in dir/.thumbnails and
 * removes them if they have no source counterpart.
 */
gint cache_maintain_dir(const gchar *dir, gint recursive, gint clear)
{
	GList *list = NULL;
	gchar *cachedir;
	gint still_have_a_file = FALSE;
	GList *work;

	cachedir = g_strconcat(dir, "/", GQVIEW_CACHE_DIR, NULL);

	path_list(cachedir, &list, NULL);
	work = list;

	while (work)
		{
		const gchar *path;
		gchar *source;

		path = work->data;
		work = work->next;

		source = g_strconcat(dir, "/", filename_from_path(path), NULL);

		if (clear ||
		    extension_truncate(source, GQVIEW_CACHE_THUMB_EXT) ||
		    extension_truncate(source, GQVIEW_CACHE_SIMIL_EXT))
			{
			if (!clear && isfile(source))
				{
				still_have_a_file = TRUE;
				}
			else
				{
				if (unlink(path) < 0)
					{
					if (debug) printf("Failed to remove cache file %s\n", path);
					still_have_a_file = TRUE;
					}
				}
			}
		else
			{
			still_have_a_file = TRUE;
			}
		g_free(source);
		}

	path_list_free(list);
	g_free(cachedir);

	if (recursive)
		{
		list = NULL;

		path_list(dir, NULL, &list);
		work = list;
		while (work)
			{
			const gchar *path = work->data;
			work = work->next;

			still_have_a_file |= cache_maintain_dir(path, recursive, clear);
			}

		path_list_free(list);
		}

	return still_have_a_file;
}

static void cache_file_move(const gchar *src, const gchar *dest)
{
	if (!dest || !src || !isfile(src)) return;

	if (!move_file(src, dest))
		{
		if (debug) printf("Failed to move cache file %s\nto %s\n", src, dest);
		/* we remove it anyway - it's stale */
		unlink(src);
		}
}

void cache_maint_moved(const gchar *src, const gchar *dest)
{
	gchar *base;
	mode_t mode = 0755;

	if (!src || !dest) return;

	base = cache_get_location(dest, FALSE, NULL, &mode);
	if (cache_ensure_dir_exists(base, mode))
		{
		gchar *buf;
		gchar *d;

		buf = cache_find_location(src, GQVIEW_CACHE_THUMB_EXT);
		d = cache_get_location(dest, TRUE, GQVIEW_CACHE_THUMB_EXT, NULL);
		cache_file_move(buf, d);
		g_free(d);
		g_free(buf);

		buf = cache_find_location(src, GQVIEW_CACHE_SIMIL_EXT);
		d = cache_get_location(dest, TRUE, GQVIEW_CACHE_SIMIL_EXT, NULL);
		cache_file_move(buf, d);
		g_free(d);
		g_free(buf);
		}
	else
		{
		printf("Failed to create cache dir for move %s\n", base);
		}
	g_free(base);
}

static void cache_file_remove(const gchar *path)
{
	if (path && isfile(path) && unlink(path) < 0)
		{
		if (debug) printf("Failed to remove cache file %s\n", path);
		}
}

void cache_maint_removed(const gchar *source)
{
	gchar *buf;

	buf = cache_find_location(source, GQVIEW_CACHE_THUMB_EXT);
	cache_file_remove(buf);
	g_free(buf);

	buf = cache_find_location(source, GQVIEW_CACHE_SIMIL_EXT);
	cache_file_remove(buf);
	g_free(buf);
}


