/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */

/*
 *  GThumb
 *
 *  Copyright (C) 2001 The Free Software Foundation, Inc.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Street #330, Boston, MA 02111-1307, USA.
 */

#include <config.h>
#include <stdlib.h>
#include <stdio.h>
#include <gtk/gtk.h>
#include <libgnomeui/gnome-file-entry.h>
#include <libgnomeui/gnome-dialog-util.h>
#include <libgnomeui/gnome-color-picker.h>
#include <libgnomeui/gnome-font-picker.h>
#include <libgnomeui/gnome-propertybox.h>
#include <glade/glade.h>
#include "catalog-png-exporter.h"
#include "file-utils.h"
#include "image-list.h"
#include "image-list-utils.h"
#include "main.h"
#include "window.h"


#define GLADE_EXPORTER_FILE "gthumb_png_exporter.glade"


typedef struct {
	GThumbWindow    *window;

	GladeXML  *gui;
	GtkWidget *dialog;

	GtkWidget *dest_fileentry;
	GtkWidget *dest_fileentry_entry;
	GtkWidget *name1_entry, *name2_entry;
	GtkWidget *example1_label, *example2_label, *example3_label;
	GtkWidget *image_map_checkbutton;

	GtkWidget *progress_dialog;
	GtkWidget *progress_progressbar;

	CatalogPngExporter *exporter;
} DialogData;


/* called when the main dialog is closed. */
static void
destroy_cb (GtkWidget *widget, 
	    DialogData *data)
{
        gtk_object_unref (GTK_OBJECT (data->gui));
	if (data->exporter)
		gtk_object_destroy (GTK_OBJECT (data->exporter));
	g_free (data);
}


static void
update_example_labels (DialogData *data)
{
	gchar *first_part, *second_part;
	gchar *example;

	first_part = gtk_entry_get_text (GTK_ENTRY (data->name1_entry));
	second_part = gtk_entry_get_text (GTK_ENTRY (data->name2_entry));

	example = g_strdup_printf ("%s001%s.png", first_part, second_part);
	gtk_label_set_text (GTK_LABEL (data->example1_label), example);
	g_free (example);

	example = g_strdup_printf ("%s002%s.png", first_part, second_part);
	gtk_label_set_text (GTK_LABEL (data->example2_label), example);
	g_free (example);

	example = g_strdup_printf ("%s003%s.png", first_part, second_part);
	gtk_label_set_text (GTK_LABEL (data->example3_label), example);
	g_free (example);
}


static void
update_example_labels_cb (GtkWidget *widget,
			  DialogData *data)
{
	update_example_labels (data);
}


static void
export (GtkWidget *widget,
	DialogData *data)
{
	CatalogPngExporter *exporter = data->exporter;
	char *dir;

	png_exporter_pref.write_imap = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (data->image_map_checkbutton));

	/**/
	dir = remove_ending_separator (gtk_entry_get_text (GTK_ENTRY (data->dest_fileentry_entry)));
	if (! path_is_dir (dir)) {
		g_free (dir);
		g_warning ("Directory does not exist.");
		gtk_widget_destroy (data->dialog);
		return;
	}

	catalog_png_exporter_set_directory (exporter, dir);
	g_free (dir);

	catalog_png_exporter_set_name (exporter, gtk_entry_get_text (GTK_ENTRY (data->name1_entry)), gtk_entry_get_text (GTK_ENTRY (data->name2_entry)));

	/* Page options. */
	if (png_exporter_pref.size_use_row_col)
		catalog_png_exporter_set_page_size_row_col (exporter, png_exporter_pref.page_rows, png_exporter_pref.page_cols);
	else
		catalog_png_exporter_set_page_size (exporter, png_exporter_pref.page_width, png_exporter_pref.page_height);

	catalog_png_exporter_set_page_color (exporter, 
					     png_exporter_pref.bg_color.red,
					     png_exporter_pref.bg_color.green,
					     png_exporter_pref.bg_color.blue);

	catalog_png_exporter_set_sort_method (exporter,
					      png_exporter_pref.sort_method);
	catalog_png_exporter_set_sort_type (exporter,
					    png_exporter_pref.sort_type);

	/* Thumbnails options. */
	catalog_png_exporter_set_caption (exporter, 
				      png_exporter_pref.caption);
	catalog_png_exporter_set_frame_style (exporter, 
					  png_exporter_pref.frame_style);
	catalog_png_exporter_set_frame_color (exporter, 
					  png_exporter_pref.frame_color.red,
					  png_exporter_pref.frame_color.green,
					  png_exporter_pref.frame_color.blue);
	
	catalog_png_exporter_set_thumb_size (exporter, 
					 png_exporter_pref.thumb_size,
					 png_exporter_pref.thumb_size);
	catalog_png_exporter_set_text_color (exporter, 
					 png_exporter_pref.text_color.red,
					 png_exporter_pref.text_color.green,
					 png_exporter_pref.text_color.blue);
	catalog_png_exporter_set_text_font (exporter,
					png_exporter_pref.text_font);

	catalog_png_exporter_write_image_map (exporter,
					  png_exporter_pref.write_imap);

	/* Export. */

	gtk_window_set_transient_for (GTK_WINDOW (data->progress_dialog), GTK_WINDOW (data->window->app));
	gtk_window_set_modal (GTK_WINDOW (data->progress_dialog), TRUE);
	gtk_widget_show_all (data->progress_dialog);

	catalog_png_exporter_export (exporter);
}


static void
export_done (GtkObject *object,
	     DialogData *data)
{
	gtk_widget_destroy (data->progress_dialog);
	gtk_widget_destroy (data->dialog);
}


static void
export_progress (GtkObject *object,
		 float percent,
		 DialogData *data)
{
	gtk_progress_set_percentage (GTK_PROGRESS (data->progress_progressbar),
				     percent);
}


static void dlg_png_exporter_pref (GtkWidget *dialog);


static void
popup_pref_dialog (GtkWidget *widget,
		   DialogData *data)
{
	dlg_png_exporter_pref (data->dialog);
}


/* create the main dialog. */
void
dlg_exporter (GThumbWindow *window)
{
	DialogData *data;
	GtkWidget *btn_ok;
	GtkWidget *btn_cancel;
	GtkWidget *btn_pref;
	ImageList *ilist;
	GList *list;

	data = g_new (DialogData, 1);

	data->window = window;

	ilist = IMAGE_LIST (window->file_list->ilist);
	list = ilist_utils_get_file_list_selection (ilist);
	if (list == NULL) {
		g_warning ("No file selected.");
		g_free (data);
		return;
	}

	data->exporter = catalog_png_exporter_new (list);
	g_list_foreach (list, (GFunc) g_free, NULL);
	g_list_free (list);

	data->gui = glade_xml_new (GTHUMB_GLADEDIR "/" GLADE_EXPORTER_FILE , NULL);
        if (!data->gui) {
                g_warning ("Could not find " GLADE_FILE "\n");
                return;
        }

	/* Get the widgets. */

	data->dialog = glade_xml_get_widget (data->gui, "exporter_save_dialog");
	data->dest_fileentry = glade_xml_get_widget (data->gui, "dest_fileentry");
	data->name1_entry = glade_xml_get_widget (data->gui, "name1_entry");
	data->name2_entry = glade_xml_get_widget (data->gui, "name2_entry");
	data->example1_label = glade_xml_get_widget (data->gui, "example1_label");
	data->example2_label = glade_xml_get_widget (data->gui, "example2_label");
	data->example3_label = glade_xml_get_widget (data->gui, "example3_label");
	
	data->progress_dialog = glade_xml_get_widget (data->gui, "progress_dialog");
	data->progress_progressbar = glade_xml_get_widget (data->gui, "progress_progressbar");
	data->image_map_checkbutton = glade_xml_get_widget (data->gui, "image_map_checkbutton");

        btn_cancel = glade_xml_get_widget (data->gui, "cancel_button");
        btn_ok = glade_xml_get_widget (data->gui, "ok_button");
        btn_pref = glade_xml_get_widget (data->gui, "pref_button");

	data->dest_fileentry_entry = gnome_entry_gtk_entry (GNOME_ENTRY (gnome_file_entry_gnome_entry (GNOME_FILE_ENTRY (data->dest_fileentry))));

	/* Signals. */
	gtk_signal_connect (GTK_OBJECT (data->dialog), "destroy",
			    (GtkSignalFunc) destroy_cb,
			    data);
	gtk_signal_connect_object (GTK_OBJECT (btn_cancel), "clicked",
				   GTK_SIGNAL_FUNC (gtk_widget_destroy),
				   GTK_OBJECT (data->dialog));
	gtk_signal_connect (GTK_OBJECT (btn_ok), "clicked",
			    export,
			    data);
	gtk_signal_connect (GTK_OBJECT (btn_pref), "clicked",
			    popup_pref_dialog,
			    data);

	gtk_signal_connect (GTK_OBJECT (data->exporter), "done",
			    export_done,
			    data);
	gtk_signal_connect (GTK_OBJECT (data->exporter), "progress",
			    (GtkSignalFunc) export_progress,
			    data);

	gtk_signal_connect (GTK_OBJECT (data->name1_entry), "changed",
			    update_example_labels_cb,
			    data);
	gtk_signal_connect (GTK_OBJECT (data->name2_entry), "changed",
			    update_example_labels_cb,
			    data);

	/* Set widgets data. */
	gnome_file_entry_set_default_path (GNOME_FILE_ENTRY (data->dest_fileentry), (window->dir_list->path != NULL) ? window->dir_list->path : g_get_home_dir ());

	gtk_entry_set_text (GTK_ENTRY (data->dest_fileentry_entry),
			    (window->dir_list->path == NULL) ? g_get_home_dir() : window->dir_list->path);

	update_example_labels (data);

	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (data->image_map_checkbutton), png_exporter_pref.write_imap);

	/* run dialog. */
	gtk_window_set_transient_for (GTK_WINDOW (data->dialog), GTK_WINDOW (window->app));
	gtk_window_set_modal (GTK_WINDOW (data->dialog), TRUE);
	gtk_widget_show_all (data->dialog);
}



/* -- Exporter Preferences -- */

typedef struct {
	GladeXML  *gui;
	GtkWidget *dialog;

	GtkWidget *page_bg_colorpicker;
	GtkWidget *sort_method_optionmenu;
	GtkWidget *sort_type_checkbutton;
	GtkWidget *width_entry;
	GtkWidget *height_entry;
	GtkWidget *rows_spinbutton;
	GtkWidget *cols_spinbutton;
	GtkWidget *same_size_checkbutton;
	GtkWidget *pixel_size_radiobutton;
	GtkWidget *row_col_size_radiobutton;
	GtkWidget *rows_cols_table;
	GtkWidget *pixel_table;

	GtkWidget *comment_checkbutton;
	GtkWidget *filepath_checkbutton;
	GtkWidget *filename_checkbutton;
	GtkWidget *filesize_checkbutton;
	GtkWidget *image_dim_checkbutton;

	GtkWidget *frame_style_optionmenu;
	GtkWidget *frame_colorpicker;
	GtkWidget *draw_frame_checkbutton;

	GtkWidget *thumb_size_optionmenu;
	GtkWidget *text_colorpicker;
	GtkWidget *text_fontpicker;
} PrefDialogData;


/* called when the pref dialog is closed. */
static void
pref_destroy_cb (GtkWidget *widget, 
		 PrefDialogData *data)
{
        gtk_object_unref (GTK_OBJECT (data->gui));
	g_free (data);
}


/* get the option menu index from the size value. */
static gint
get_idx_from_size (gint size)
{
	switch (size) {
	case 48: return 0;
	case 64: return 1;
	case 75: return 2;
	case 85: return 3;
	case 95: return 4;
	case 112: return 5;
	case 128: return 6;
	case 164: return 7;
	case 200: return 8;
	case 256: return 9;
	}

	return -1;
}


static gint
get_size_from_idx (gint idx)
{
	switch (idx) {
	case 0: return 48;
	case 1: return 64;
	case 2: return 75;
	case 3: return 85;
	case 4: return 95;
	case 5: return 112;
	case 6: return 128;
	case 7: return 164;
	case 8: return 200;
	case 9: return 256;
	}

	return 0;
}


/* get the option menu index from the frame style. */
static gint
get_idx_from_style (FrameStyle style)
{
	switch (style) {
	case FRAME_STYLE_NONE: return 0;
	case FRAME_STYLE_SIMPLE: return 0;
	case FRAME_STYLE_SLIDE: return 1;
	case FRAME_STYLE_SHADOW: return 2;
	}

	return -1;
}


static FrameStyle
get_style_from_idx (gint idx)
{
	switch (idx) {
	case 0: return FRAME_STYLE_SIMPLE;
	case 1: return FRAME_STYLE_SLIDE;
	case 2: return FRAME_STYLE_SHADOW;
	}

	return FRAME_STYLE_NONE;
}


/* get the option menu index from the sort method. */
static gint
get_idx_from_sort_method (SortMethod style)
{
	switch (style) {
	case SORT_BY_NAME: return 0;
	case SORT_BY_PATH: return 1;
	case SORT_BY_SIZE: return 2;
	case SORT_BY_TIME: return 3;
	default: return -1;
	}

	return -1;
}


static SortMethod
get_sort_method_from_idx (gint idx)
{
	switch (idx) {
	case 0: return SORT_BY_NAME;
	case 1: return SORT_BY_PATH;
	case 2: return SORT_BY_SIZE;
	case 3: return SORT_BY_TIME;
	}

	return SORT_NONE;
}


static guint
opt_menu_get_active_idx (GtkWidget *opt_menu)
{
	GtkWidget *item;
	guint idx;
	GList *scan;
	GtkWidget *menu;

	menu = gtk_option_menu_get_menu (GTK_OPTION_MENU (opt_menu));
	item = gtk_menu_get_active (GTK_MENU (menu));

	idx = 0;
	scan = GTK_MENU_SHELL (menu)->children;
	while (scan && (item != scan->data)) {
		idx++;
		scan = scan->next;
	}

	return idx;
}


/* called when the "ok" button is clicked. */
static void
ok_cb (GtkWidget *widget, 
       PrefDialogData *data)
{
	char *s;

	/* Page */
	s = gtk_entry_get_text (GTK_ENTRY (data->width_entry));
	png_exporter_pref.page_width = atoi (s);
	s = gtk_entry_get_text (GTK_ENTRY (data->height_entry));
	png_exporter_pref.page_height = atoi (s);

	png_exporter_pref.size_use_row_col = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (data->row_col_size_radiobutton));
	png_exporter_pref.all_pages_same_size = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (data->same_size_checkbutton));
	png_exporter_pref.page_rows = gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (data->rows_spinbutton));
	png_exporter_pref.page_cols = gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (data->cols_spinbutton));
	gnome_color_picker_get_i16 (GNOME_COLOR_PICKER (data->page_bg_colorpicker), &png_exporter_pref.bg_color.red, &png_exporter_pref.bg_color.green, &png_exporter_pref.bg_color.blue, NULL);
	if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (data->sort_type_checkbutton)))
		png_exporter_pref.sort_type = GTK_SORT_DESCENDING;
	else
		png_exporter_pref.sort_type = GTK_SORT_ASCENDING;
	png_exporter_pref.sort_method = get_sort_method_from_idx (opt_menu_get_active_idx (data->sort_method_optionmenu));

	/* Thumbnails */

	/* ** Caption */
	png_exporter_pref.caption = 0;

	png_exporter_pref.caption |= gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (data->comment_checkbutton)) ? CAPTION_COMMENT : 0;

	png_exporter_pref.caption |= gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (data->filepath_checkbutton)) ? CAPTION_FILE_PATH : 0;

	png_exporter_pref.caption |= gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (data->filename_checkbutton)) ? CAPTION_FILE_NAME : 0;

	png_exporter_pref.caption |= gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (data->filesize_checkbutton)) ? CAPTION_FILE_SIZE : 0;

	png_exporter_pref.caption |= gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (data->image_dim_checkbutton)) ? CAPTION_IMAGE_DIM : 0;

	/* ** Frame */

	png_exporter_pref.frame_style = get_style_from_idx (opt_menu_get_active_idx (data->frame_style_optionmenu));
	gnome_color_picker_get_i16 (GNOME_COLOR_PICKER (data->frame_colorpicker), &png_exporter_pref.frame_color.red, &png_exporter_pref.frame_color.green, &png_exporter_pref.frame_color.blue, NULL);

	if (! gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (data->draw_frame_checkbutton)))
		png_exporter_pref.frame_style = FRAME_STYLE_NONE;

	/* ** Others */

	png_exporter_pref.thumb_size = get_size_from_idx (opt_menu_get_active_idx ( data->thumb_size_optionmenu));
	gnome_color_picker_get_i16 (GNOME_COLOR_PICKER (data->text_colorpicker), &png_exporter_pref.text_color.red, &png_exporter_pref.text_color.green, &png_exporter_pref.text_color.blue, NULL);
	if (png_exporter_pref.text_font != NULL)
		g_free (png_exporter_pref.text_font);
	png_exporter_pref.text_font = g_strdup (gnome_font_picker_get_font_name (GNOME_FONT_PICKER (data->text_fontpicker)));

	/* Close */

	gtk_widget_destroy (data->dialog);
}


static void
use_pixel_cb (GtkWidget *widget,
	      PrefDialogData *data)
{
	if (! GTK_TOGGLE_BUTTON (widget)->active)
		return;

	gtk_widget_set_sensitive (data->pixel_table, TRUE);
	gtk_widget_set_sensitive (data->rows_cols_table, FALSE);
}


static void
use_row_col_cb (GtkWidget *widget,
		PrefDialogData *data)
{
	if (! GTK_TOGGLE_BUTTON (widget)->active)
		return;

	gtk_widget_set_sensitive (data->rows_cols_table, TRUE);
	gtk_widget_set_sensitive (data->pixel_table, FALSE);
}


/* create the exporter preferences dialog. */
static void dlg_png_exporter_pref (GtkWidget *dialog)
{
	PrefDialogData *data;
	GtkWidget *btn_ok;
	char s[10];

	data = g_new (PrefDialogData, 1);

	data->gui = glade_xml_new (GTHUMB_GLADEDIR "/" GLADE_EXPORTER_FILE , NULL);
        if (!data->gui) {
                g_warning ("Could not find " GLADE_EXPORTER_FILE "\n");
                return;
        }

	/* Get the widgets. */

	data->dialog = glade_xml_get_widget (data->gui, "exporter_propertybox");
	data->page_bg_colorpicker = glade_xml_get_widget (data->gui, "page_bg_colorpicker");
	data->sort_method_optionmenu = glade_xml_get_widget (data->gui, "sort_method_optionmenu");
	data->sort_type_checkbutton = glade_xml_get_widget (data->gui, "sort_type_checkbutton");
	data->width_entry = glade_xml_get_widget (data->gui, "width_entry");
	data->height_entry = glade_xml_get_widget (data->gui, "height_entry");
	data->rows_spinbutton = glade_xml_get_widget (data->gui, "rows_spinbutton");
	data->cols_spinbutton = glade_xml_get_widget (data->gui, "cols_spinbutton");
	data->same_size_checkbutton = glade_xml_get_widget (data->gui, "same_size_checkbutton");
	data->pixel_size_radiobutton = glade_xml_get_widget (data->gui, "pixel_size_radiobutton");
	data->row_col_size_radiobutton = glade_xml_get_widget (data->gui, "row_col_size_radiobutton");
	data->rows_cols_table = glade_xml_get_widget (data->gui, "rows_cols_table");
	data->pixel_table = glade_xml_get_widget (data->gui, "pixel_table");
	data->comment_checkbutton = glade_xml_get_widget (data->gui, "comment_checkbutton");
	data->filepath_checkbutton = glade_xml_get_widget (data->gui, "filepath_checkbutton");
	data->filename_checkbutton = glade_xml_get_widget (data->gui, "filename_checkbutton");
	data->filesize_checkbutton = glade_xml_get_widget (data->gui, "filesize_checkbutton");
	data->image_dim_checkbutton = glade_xml_get_widget (data->gui, "image_dim_checkbutton");

	data->frame_style_optionmenu = glade_xml_get_widget (data->gui, "frame_style_optionmenu");
	data->frame_colorpicker = glade_xml_get_widget (data->gui, "frame_colorpicker");
	data->draw_frame_checkbutton = glade_xml_get_widget (data->gui, "draw_frame_checkbutton");

	data->thumb_size_optionmenu = glade_xml_get_widget (data->gui, "thumb_size_optionmenu");
	data->text_colorpicker = glade_xml_get_widget (data->gui, "text_colorpicker");
	data->text_fontpicker = glade_xml_get_widget (data->gui, "text_fontpicker");
        btn_ok = GNOME_PROPERTY_BOX (data->dialog)->ok_button;

	/* Signals. */
	gtk_signal_connect (GTK_OBJECT (data->dialog), "destroy",
			    (GtkSignalFunc) pref_destroy_cb,
			    data);
	gtk_signal_connect (GTK_OBJECT (btn_ok), "clicked",
                            GTK_SIGNAL_FUNC (ok_cb),
                            data);
	gtk_signal_connect (GTK_OBJECT (data->pixel_size_radiobutton), 
			    "toggled",
			    GTK_SIGNAL_FUNC (use_pixel_cb),
			    data);
	gtk_signal_connect (GTK_OBJECT (data->row_col_size_radiobutton), 
			    "toggled",
			    GTK_SIGNAL_FUNC (use_row_col_cb),
			    data);

	/* Set widget data. */

	/* * Page */

	/* background color */
	gnome_color_picker_set_i16 (GNOME_COLOR_PICKER (data->page_bg_colorpicker),
				    png_exporter_pref.bg_color.red,
				    png_exporter_pref.bg_color.green,
				    png_exporter_pref.bg_color.blue,
				    65535);

	/* sort type */
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (data->sort_type_checkbutton), png_exporter_pref.sort_type == GTK_SORT_DESCENDING);
	gtk_option_menu_set_history (GTK_OPTION_MENU (data->sort_method_optionmenu), get_idx_from_sort_method (png_exporter_pref.sort_method));

	/* page size */

	sprintf (s, "%d", png_exporter_pref.page_width);
	gtk_entry_set_text (GTK_ENTRY (data->width_entry), s);
	sprintf (s, "%d", png_exporter_pref.page_height);
	gtk_entry_set_text (GTK_ENTRY (data->height_entry), s);

	gtk_spin_button_set_value (GTK_SPIN_BUTTON (data->rows_spinbutton), 
				   png_exporter_pref.page_rows);
	gtk_spin_button_set_value (GTK_SPIN_BUTTON (data->cols_spinbutton), 
				   png_exporter_pref.page_cols);

	if (png_exporter_pref.size_use_row_col)
		gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (data->row_col_size_radiobutton), TRUE);
	else
		gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (data->pixel_size_radiobutton), TRUE);
	gtk_widget_set_sensitive (data->rows_cols_table, png_exporter_pref.size_use_row_col);
	gtk_widget_set_sensitive (data->pixel_table, ! png_exporter_pref.size_use_row_col);

	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (data->same_size_checkbutton), png_exporter_pref.all_pages_same_size);

	/* * Thumbnails */

	/* ** Caption */
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (data->comment_checkbutton), png_exporter_pref.caption & CAPTION_COMMENT);

	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (data->filepath_checkbutton), png_exporter_pref.caption & CAPTION_FILE_PATH);

	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (data->filename_checkbutton), png_exporter_pref.caption & CAPTION_FILE_NAME);

	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (data->filesize_checkbutton), png_exporter_pref.caption & CAPTION_FILE_SIZE);

	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (data->image_dim_checkbutton), png_exporter_pref.caption & CAPTION_IMAGE_DIM);

	/* ** Frame */

	gtk_option_menu_set_history (GTK_OPTION_MENU (data->frame_style_optionmenu), get_idx_from_style (png_exporter_pref.frame_style));
	gnome_color_picker_set_i16 (GNOME_COLOR_PICKER (data->frame_colorpicker),
				    png_exporter_pref.frame_color.red,
				    png_exporter_pref.frame_color.green,
				    png_exporter_pref.frame_color.blue,
				    65535);
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (data->draw_frame_checkbutton), png_exporter_pref.frame_style != FRAME_STYLE_NONE);

	/* ** Others */

	gtk_option_menu_set_history (GTK_OPTION_MENU (data->thumb_size_optionmenu), get_idx_from_size (png_exporter_pref.thumb_size));
	gnome_color_picker_set_i16 (GNOME_COLOR_PICKER (data->text_colorpicker),
				    png_exporter_pref.text_color.red,
				    png_exporter_pref.text_color.green,
				    png_exporter_pref.text_color.blue,
				    65535);
	gnome_font_picker_set_font_name (GNOME_FONT_PICKER (data->text_fontpicker), png_exporter_pref.text_font);

	/* run dialog. */
	gtk_window_set_transient_for (GTK_WINDOW (data->dialog), GTK_WINDOW (dialog));
	gtk_window_set_modal (GTK_WINDOW (data->dialog), TRUE);
	gtk_widget_show_all (data->dialog);
}
