/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */

/*
 *  GThumb
 *
 *  Copyright (C) 2001 The Free Software Foundation, Inc.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Street #330, Boston, MA 02111-1307, USA.
 */

#include <config.h>
#include <gnome.h>
#include <libgnomeui/gnome-window-icon.h>
#include <glade/glade.h>
#include <libgnomevfs/gnome-vfs-init.h>

#include "auto-completion.h"
#include "catalog.h"
#include "file-utils.h"
#include "fullscreen.h"
#include "main.h"
#include "menu-callbacks.h"
#include "preferences.h"
#include "image-viewer.h"
#include "image-list.h"
#include "typedefs.h"
#include "window.h"


GList *      window_list = NULL;
FullScreen * fullscreen;
gchar **     file_urls, **dir_urls;
gint         n_file_urls, n_dir_urls;


static gboolean view_comline_catalog = FALSE;


static void prepare_app     ();
static void initialize_data ();
static void release_data    ();


/* -- Main -- */

gint 
main (gint argc, char *argv[])
{
	bindtextdomain (PACKAGE, LOCALEDIR);
	textdomain (PACKAGE);

	if (gnome_init (_("gThumb"), VERSION, argc, argv) != 0) {
		g_warning ("Cannot initialize Gnome.");
		return 1;
	}

	g_thread_init (NULL);

	if (! gnome_vfs_init ()) 
		g_error ("Cannot initialize the Virtual File System.");

	gdk_rgb_init ();
	gtk_widget_set_default_colormap (gdk_rgb_get_cmap ());
        gtk_widget_set_default_visual   (gdk_rgb_get_visual ());

	glade_gnome_init ();

	initialize_data (argc, argv);
	prepare_app ();

	gtk_main ();

	release_data ();

	return 0;
}


static void
ensure_directories_exist ()
{
	gchar *path;

	path = g_strconcat (g_get_home_dir (),
			    "/",
			    RC_THUMBS_DIR,
			    NULL);
	ensure_dir_exists (path, 0700);
	g_free (path);

	path = g_strconcat (g_get_home_dir (),
			    "/",
			    RC_CATALOG_DIR,
			    NULL);
	ensure_dir_exists (path, 0700);
	g_free (path);
}


/* Initialize application data. */
static void 
initialize_data (gint argc, char *argv[])
{
	gchar *current_dir;
	gchar *path;
	gchar *pixmap_file;
	gint   i;

	/* initialize preferences. */
	ensure_directories_exist ();
	preferences_init ();
	preferences_set_default_values ();
	preferences_load ();

	fullscreen = fullscreen_new ();

	pixmap_file = gnome_pixmap_file ("gthumb.png");
	gnome_window_icon_set_default_from_file (pixmap_file);
	g_free (pixmap_file);

	/* Parse command line arguments. */

	if (argc == 1)
		return;

        file_urls = g_new0 (gchar*, argc - 1);
        dir_urls = g_new0 (gchar*, argc - 1);

	n_file_urls = 0;
	n_dir_urls = 0;

	current_dir = g_get_current_dir ();
	for (i = 1; i < argc; i++) {
		gboolean is_dir;

		if (! g_path_is_absolute (argv[i]))
			path = g_strconcat (current_dir,
					    "/",
					    argv[i],
					    NULL);
		else
			path = g_strdup (argv[i]);

		if (path_is_dir (path))
			is_dir = TRUE;
		else if (path_is_file (path))
			is_dir = FALSE;
		else {
			g_free (path);
			continue;
		}

		if (is_dir) {
			dir_urls[n_dir_urls++] = g_strconcat ("file://",
							      path,
							      NULL);
			g_free (path);
		} else
			file_urls[n_file_urls++] = path;
	}

	if (n_file_urls > 0) {
		/* Create a catalog with the command line list. */
		Catalog *catalog;
		gchar *catalog_path;
		gchar *catalog_name;

		catalog = catalog_new ();

		catalog_name = g_strconcat (_("Command Line"),
					    CATALOG_EXT,
					    NULL);
		catalog_path = get_catalog_full_path (catalog_name);
		g_free (catalog_name);

		catalog_set_path (catalog, catalog_path);
		g_free (catalog_path);

		for (i = 0; i < n_file_urls; i++) 
			catalog_add_item (catalog, file_urls[i]);

		catalog_write_to_disk (catalog);
		catalog_free (catalog);

		view_comline_catalog = TRUE;
	}

	g_free (current_dir);
}


/* free application data. */
static void 
release_data ()
{
	fullscreen_close (fullscreen);
	preferences_save ();
	preferences_release ();
	auto_compl_reset ();
}


/* Create the windows. */
static void 
prepare_app ()
{
	GThumbWindow *current_window = NULL;
	gint i;

	if (! view_comline_catalog 
	    && (n_dir_urls == 0) 
	    && (n_file_urls == 0)) {
		current_window = window_new (FALSE);
		gtk_widget_show_all (current_window->app);		
	}

	if (view_comline_catalog) {
		gchar *catalog_path;
		gchar *catalog_name;

		catalog_name = g_strconcat (
			_("Command Line"),
			CATALOG_EXT,
			NULL);
		catalog_path = get_catalog_full_path (catalog_name);
		g_free (catalog_name);

		if (preferences.starting_location)
			g_free (preferences.starting_location);
		preferences.starting_location = g_strconcat (
			"catalog://", 
			catalog_path, 
			NULL);
		g_free (catalog_path);

		current_window = window_new (TRUE);
		gtk_widget_show_all (current_window->app);
	}

	for (i = 0; i < n_dir_urls; i++) {
		/* go to the specified dierectory. */
		if (preferences.starting_location)
			g_free (preferences.starting_location);
		preferences.starting_location = g_strdup (dir_urls[i]);

		current_window = window_new (FALSE);
		gtk_widget_show_all (current_window->app);
	}

	/* free urls. */

	if (n_file_urls > 0) {
		for (i = 0; i < n_file_urls; i++)
			g_free (file_urls[i]);
		g_free (file_urls);
	}

	if (n_dir_urls > 0) {
		for (i = 0; i < n_dir_urls; i++)
			g_free (dir_urls[i]);
		g_free (dir_urls);
	}
}


void 
all_windows_update_file_list ()
{
	g_list_foreach (window_list, (GFunc) window_update_file_list, NULL);
}


void 
all_windows_update_catalog_list ()
{
	g_list_foreach (window_list, (GFunc) window_update_catalog_list, NULL);
}


void 
all_windows_update_bookmark_list ()
{
	g_list_foreach (window_list, (GFunc) window_update_bookmark_list, NULL);
}


void 
all_windows_update_history_list ()
{
	g_list_foreach (window_list, (GFunc) window_update_history_list, NULL);
}


void 
all_windows_update_viewer_options ()
{
	GList *scan;
	gint i = 0;

	for (scan = window_list; scan; scan = scan->next) {
		GThumbWindow *window = scan->data;

		image_viewer_set_check_type (IMAGE_VIEWER (window->viewer),
					     preferences.check_type);
		image_viewer_set_check_size (IMAGE_VIEWER (window->viewer),
					     preferences.check_size);
		image_viewer_set_transp_type (IMAGE_VIEWER (window->viewer),
					      preferences.transp_type);
		image_viewer_set_zoom_quality (IMAGE_VIEWER (window->viewer),
					       preferences.zoom_quality);
		image_viewer_set_zoom_change (IMAGE_VIEWER (window->viewer),
					      preferences.zoom_change);
		image_viewer_update_view (IMAGE_VIEWER (window->viewer));

		/* update menu item state. */
		switch (preferences.transp_type) {
		case TRANSP_TYPE_WHITE:   i = 0; break;
		case TRANSP_TYPE_GRAY:    i = 1; break;
		case TRANSP_TYPE_BLACK:   i = 2; break;
		case TRANSP_TYPE_CHECKED: i = 3; break;
		}
		gtk_signal_handler_block_by_data (GTK_OBJECT (window->mitem_transparenty[i]), window);
		gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM (window->mitem_transparenty[i]), TRUE);
		gtk_signal_handler_unblock_by_data (GTK_OBJECT (window->mitem_transparenty[i]), window);
	}
}


void 
all_windows_update_browser_options ()
{
	GList *scan;

	for (scan = window_list; scan; scan = scan->next) {
		GThumbWindow *window = scan->data;
		ImageListViewMode view_mode;

		window->file_list->enable_thumbs = preferences.enable_thumbnails;
		file_list_set_thumbs_size (window->file_list, preferences.thumb_size);
		if (preferences.show_comments)
			view_mode = IMAGE_LIST_VIEW_ALL;
		else
			view_mode = IMAGE_LIST_VIEW_TEXT;
		image_list_set_view_mode (IMAGE_LIST(window->file_list->ilist),
					  view_mode);
		window_update_file_list (window);

		/* update menu item state. */
		gtk_signal_handler_block_by_data (GTK_OBJECT (window->mitem_thumbnails), window);
		gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM (window->mitem_thumbnails), preferences.enable_thumbnails);
		gtk_signal_handler_unblock_by_data (GTK_OBJECT (window->mitem_thumbnails), window);
	}
}


void 
all_windows_update_editors_menu ()
{
}


void 
all_windows_notify_files_deleted (GList *list)
{
	GList *scan;

	for (scan = window_list; scan; scan = scan->next) {
		GThumbWindow *window = scan->data;
		window_notify_files_deleted (window, list);
	}
}


void 
all_windows_notify_cat_files_deleted (const gchar *catalog_path,
				      GList *list)
{
	GList *scan;

	for (scan = window_list; scan; scan = scan->next) {
		GThumbWindow *window = scan->data;
		window_notify_cat_files_deleted (window, catalog_path, list);
	}
}


void 
all_windows_notify_file_rename (const gchar *old_name,
				const gchar *new_name)
{
	GList *scan;

	for (scan = window_list; scan; scan = scan->next) {
		GThumbWindow *window = scan->data;
		window_notify_file_rename (window, old_name, new_name);
	}
}


void 
all_windows_notify_update_comment (const gchar *filename)
{
	GList *scan;

	for (scan = window_list; scan; scan = scan->next) {
		GThumbWindow *window = scan->data;
		window_notify_update_comment (window, filename);
	}
}


void 
all_windows_notify_update_directory (const gchar *dir_path)
{
	GList *scan;

	for (scan = window_list; scan; scan = scan->next) {
		GThumbWindow *window = scan->data;
		window_notify_update_directory (window, dir_path);
	}
}
