/****************************************************************************
** $Id: filewidget.cpp,v 1.1.1.1 2001/04/15 02:08:04 gis Exp $
**
** Created : 98
**
** Copyright (C) 1998-2001 by Carsten Pfeiffer.  All rights reserved.
**
****************************************************************************/

#include <qkeycode.h>

#include <kapp.h> // KDE_VERSION
#include <kfilereader.h>
#include <kglobal.h>
#include <kglobalsettings.h>
#include <klocale.h>
#include <kpropsdlg.h>
#include <kurlcompletion.h>

#include "filefinder.h"
#include "filewidget.h"
#include "kuickdata.h"
#include "kuickio.h"
#include "kuickshow.h"

#ifdef KeyPress
#undef KeyPress
#endif

FileWidget::FileWidget( const KURL& url, QWidget *parent, const char *name )
    : KDirOperator( url, parent, name ),
      m_validCompletion( false ),
      m_fileFinder( 0L )
{

#if KDE_VERSION >= 220
    setEnableDirHighlighting( true );
#endif

    readConfig( KGlobal::config(), "Filebrowser" );
    setView( KFile::Default );

    // setOnlyDoubleClickSelectsFiles( true );
    reloadConfiguration();

    completionObject()->setCompletionMode( KGlobalSettings::CompletionAuto );
    dirCompletionObject()->setCompletionMode( KGlobalSettings::CompletionAuto);

    slotViewChanged();
    connect( this, SIGNAL( viewChanged( KFileView * )),
	     SLOT( slotViewChanged() ));

    connect( fileReader(), SIGNAL( clear() ), SLOT( slotItemsCleared() ));
    connect( fileReader(), SIGNAL( deleteItem( KFileItem * ) ),
	     SLOT( slotItemDeleted( KFileItem *) ));

    connect( this, SIGNAL( fileHighlighted( const KFileViewItem * )),
	     SLOT( slotHighlighted( const KFileViewItem * )));

    // should actually be KDirOperator's job!
    connect( this, SIGNAL( finishedLoading() ), SLOT( slotFinishedLoading() ));
}

FileWidget::~FileWidget()
{
    delete m_fileFinder;
}

void FileWidget::initActions()
{
    int index = 0;
    KActionCollection *coll = actionCollection();
    KActionSeparator *sep = new KActionSeparator( coll );
    KActionMenu *menu = static_cast<KActionMenu*>( coll->action("popupMenu") );
    menu->insert( coll->action("showInOtherWindow"), index++ );
    menu->insert( coll->action("showInSameWindow"), index++ );
    menu->insert( sep, index++ );

    // those at the bottom
    menu->insert( sep );
    menu->insert( coll->action("delete") );
    menu->insert( sep );
    menu->insert( coll->action("print") );
    menu->insert( sep );
    menu->insert( coll->action("properties") );
}

void FileWidget::reloadConfiguration()
{
    if ( kdata.fileFilter != nameFilter() ) {
	setNameFilter( kdata.fileFilter );
	rereadDir();
    }
}

bool FileWidget::hasFiles() const {
    return (numFiles() > 0);
}

void FileWidget::activatedMenu( const KFileViewItem *item  )
{
    bool image = isImage( item );
    actionCollection()->action("showInSameWindow")->setEnabled( image );
    actionCollection()->action("showInOtherWindow")->setEnabled( image );
    actionCollection()->action("print")->setEnabled( image );
    actionCollection()->action("properties")->setEnabled( item );

    if ( item ) {
	QString text = item->isDir() ? i18n("Delete Directory") :
		                       i18n("Delete File");
	actionCollection()->action("delete")->setText( text );
    }

    KDirOperator::activatedMenu( item );
}

void FileWidget::findCompletion( const QString& text )
{
    if ( text.at(0) == '/' || text.at(0) == '~' ||
	 text.contains('/') ) {
	QString t = m_fileFinder->completionObject()->makeCompletion( text );
	if (m_fileFinder->completionMode() == KGlobalSettings::CompletionPopup)
	    m_fileFinder->setCompletedItems(
			      m_fileFinder->completionObject()->allMatches() );
	else
	    m_fileFinder->setCompletedText( t );

	return;
    }
	
    QString file = makeDirCompletion( text );
    if ( file.isNull() )
	file = makeCompletion( text );

    m_validCompletion = !file.isNull();
    if ( m_validCompletion )
	KDirOperator::setCurrentItem( file );
}

bool FileWidget::eventFilter( QObject *o, QEvent *e )
{
    if ( e->type() == QEvent::KeyPress ) {
	QKeyEvent *k = static_cast<QKeyEvent*>( e );
	
	if ( (k->state() & (ControlButton | AltButton)) == 0 ) {
	    k->accept();

	    int key = k->key();
	    if ( key == Key_Space && !KuickShow::s_viewers.isEmpty() ) {
		topLevelWidget()->hide();
		return true;
	    }
	    else if ( key == Key_Delete ) {
		KFileViewItem *item = getCurrentItem( false );
		if ( item )
		    KuickIO::self( this )->deleteFile( item->url(),
						    k->state() & ShiftButton );
		return true;
	    }
	
	    const QString& text = k->text();
	    if ( !text.isEmpty() && text.unicode()->isPrint() ) {
		if ( !m_fileFinder ) {
		    m_fileFinder = new FileFinder( this, "file finder" );
		    connect( m_fileFinder, SIGNAL( completion(const QString&)),
			     SLOT( findCompletion( const QString& )));
		    connect( m_fileFinder,
			     SIGNAL( enterDir( const QString& ) ),
			     SLOT( slotReturnPressed( const QString& )));
		    m_fileFinder->move( width()  - m_fileFinder->width(),
					height() - m_fileFinder->height() );
		}

		bool first = m_fileFinder->isHidden();

		m_fileFinder->setText( text );
		m_fileFinder->raise();
		m_fileFinder->show();
		m_fileFinder->setFocus();
		if ( first )
		    findCompletion( text );
		
		return true;
	    }
	}
    }
    return KDirOperator::eventFilter( o, e );
}

bool FileWidget::isImage( const KFileItem *item )
{
    return item && !item->isDir();

//     return ( item && item->isReadable() &&
//  	     item->mimetype().startsWith( "image/") );
}
	

KFileViewItem * FileWidget::gotoFirstImage()
{
    KFileViewItemListIterator it( *(fileView()->items()) );

    while ( it.current() ) {
	if ( isImage( it.current() ) ) {
	    setCurrentItem( it.current() );
	    return it.current();
	}
	++it;
    }

    return 0L;
}

KFileViewItem * FileWidget::gotoLastImage()
{
    KFileViewItemListIterator it( *(fileView()->items()) );
    it.toLast();

    while ( it.current() ) {
	if ( isImage( it.current() ) ) {
	    setCurrentItem( it.current() );
	    return it.current();
	}
	--it;
    }

    return 0L;
}

KFileViewItem * FileWidget::getNext( bool go )
{
    KFileViewItem *item = getItem( Next, true );
    if ( item ) {
	if ( go )
	    setCurrentItem( item );
	return item;
    }

    return 0L;
}

KFileViewItem * FileWidget::getPrevious( bool go )
{
    KFileViewItem *item = getItem( Previous, true );
    if ( item ) {
	if ( go )
	    setCurrentItem( item );
	return item;
    }

    return 0L;
}

// returns 0L when there is no previous/next item/image
// this sucks! Use KFileView::currentFileItem() when implemented
KFileViewItem * FileWidget::getItem( WhichItem which, bool onlyImage ) const
{
    KFileViewItemListIterator it( *(fileView()->items()) );

    while ( it.current() ) { // find the iterator to the current item
	if ( it.current()->url() == m_currentURL )
	    break;

	++it;
    }

    if ( it.current() ) {
	switch ( which ) {
	case Previous: {
	    --it;
	    while ( it.current() ) {
		if ( isImage( it.current() ) || !onlyImage )
		    return it.current();
		--it;
	    }
	    return 0L; // no previous item / image
	}

	case Next: {
	    ++it;
	    while ( it.current() ) {
		if ( isImage( it.current() ) || !onlyImage )
		    return it.current();
		++it;
	    }
	    return 0L; // no further item / image
	}
	
	case Current:
	default:
	    return it.current();
	}
    }

    return 0L;
}

void FileWidget::slotViewChanged()
{
    fileView()->widget()->installEventFilter( this );
}

void FileWidget::slotItemsCleared()
{
    m_currentURL = QString::null;
}

void FileWidget::slotItemDeleted( KFileItem *item )
{
    KFileItem *current = getCurrentItem( false );
    if ( item != current ) {
	return; // all ok, we already have a new current item
    }

    KFileItem *next = getNext();
    if ( !next )
	next = getPrevious();

    if ( next )
	m_currentURL = next->url().url();
}

void FileWidget::slotHighlighted( const KFileViewItem *item )
{
    m_currentURL = item->url().url();
}

void FileWidget::slotReturnPressed( const QString& text )
{
    if ( text.at(0) == '/' || text.at(0) == '~' ) {
	QString dir = (static_cast<KURLCompletion*>( m_fileFinder->completionObject() ))->replacedPath( text );
	
	KURL url;
	url.setPath( dir );
	url.setFileName( QString::null );
	setURL( url, true );
    }

    else if ( text.contains('/') ) { // relative path
	QString dir = (static_cast<KURLCompletion*>( m_fileFinder->completionObject() ))->replacedPath( text );
	KURL u = url();
	u.addPath( dir );
	setURL( u, true );
    }

    else if ( m_validCompletion ) {
	KFileViewItem *item = getCurrentItem( true );
	if ( item ) {
	    if ( item->isDir() )
		setURL( item->url(), true );
	    else
		emit fileSelected( item );
	}
    }
}

void FileWidget::setCurrentItem( const KFileViewItem *item )
{
    if ( item ) {
	fileView()->setCurrentItem( QString::null, item );
	fileView()->ensureItemVisible( item );
    }
}

void FileWidget::setInitialItem( const QString& filename )
{
    m_initialName = filename;
}

void FileWidget::slotFinishedLoading()
{
    KFileViewItem *current = getCurrentItem( false );
    if ( !m_initialName.isEmpty() )
	setCurrentItem( m_initialName );
    else if ( !current )
	setCurrentItem( view()->items()->getFirst() );

    m_initialName = QString::null;
    emit finished();
}

void FileWidget::resizeEvent( QResizeEvent *e )
{
    KDirOperator::resizeEvent( e );
    if ( m_fileFinder )
	m_fileFinder->move( width()  - m_fileFinder->width(),
			    height() - m_fileFinder->height() );
}

#include "filewidget.moc"
