/*
  libu - a C++ widget library based on SDL (Simple Direct Layer)
  Copyright (C) 1999  Karsten Laux

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Library General Public
  License as published by the Free Software Foundation; either
  version 2 of the License, or (at your option) any later version.
  
  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Library General Public License for more details.
  
  You should have received a copy of the GNU Library General Public
  License along with this library; if not, write to the
  Free Software Foundation, Inc., 59 Temple Place - Suite 330,
  Boston, MA  02111-1307, SA.
*/
#include "application.h"
#include "multilineedit.h"
#include "pushbutton.h"
#include "togglebutton.h"
#include "painter.h"
#include "listbox.h"
#include "resources.h"
#include "directsurface.h"

using namespace uta;

#include <stdio.h>
#include <stdlib.h>

class LinesApp : public Application
{
  
protected:
  Widget* back ;
  DirectSurface* drawing;

  ListBox* listbox;
  Surface image32;

  int count;
  
public:

  void timer();
  
  //needs to be there, but do not use it, use startupProc instead
  LinesApp(int argc, char** argv) : 
    Application(argc, argv) { };
  
  //needs to be there, but use shutdownProc instead
  virtual ~LinesApp() {};

  //you should at least handle the QUIT event, otherwise
  //your app cannot be closed
  bool handleEvent(const Event* event)
    {
      bool handled = Application::handleEvent(event);
      if(!handled)
	{
	  if(event->type()==Event::QUIT)
	    {
	      quit();
	    }
	}
      return handled;
    }

  int startupProc(void)
    {
      cout << "starting up ..." << endl;
      //do you init here

      rootWindow()->setTitle("LIBUTA Painter Demo");

      //eg fontloading ....
      //(these fonts are used by several gui elements from libu,
      // so you should slways load text_font and button_font when
      // using widgets from libu)
      REG_FONT_RES("text_font","flatbrush.ttf, 18, 0xF0F0F0, 0x101010");
      REG_FONT_RES("button_font","flatbrush.ttf, 18, 0xF0F0F0, 0x101010");

      //register this wav file as resources "press"
      //which is played whenever an button gets pressed
      REG_SAMPLE_RES("press","press.wav");
      //same but for "release"
      REG_SAMPLE_RES("release","release.wav");

      // these resources are used by ListBox:
//       REG_SURFACE_RES("scroller_up","triangle_up.png");
//       REG_SURFACE_RES("scroller_up_pressed","triangle_up_pressed.png");
//       REG_SURFACE_RES("scroller_down","triangle_down.png");
//       REG_SURFACE_RES("scroller_down_pressed","triangle_down_pressed.png");
      REG_SURFACE_RES("button170","button.png");
      REG_SURFACE_RES("button170_pressed","button_pressed.png");

      //create your gui here
      MultiLineEdit* info 
	= new MultiLineEdit(rootWindow(),50,50,400,60);
      info->setText("Painter / DirectSurface Demo\n"
		    "written by Karsten Laux, February 2000");


      //a pushbutton
      PushButton* quit_button 
	= new PushButton(rootWindow(), 
			  rootWindow()->width()-100,
			  rootWindow()->height()-40,
			  90, 30);
      quit_button->setText("Quit");
      //quit_button->setColor(Color(32,32,128));

      listbox 
	= new ListBox(rootWindow(),
		       rootWindow()->width()-230,150, 180,75);

      listbox->setColor(DarkSlateBlue);
      listbox->setSelectedColor(white);

      listbox->addElement("circles");
      listbox->addElement("ellipses");
      listbox->addElement("lines");
     
      uapp->timer.connect(slot(this, &LinesApp::timer));
    
      //connect this button's click event (SIGNAL) with the application's
      //quit action (SLOT)
      quit_button->clicked.connect(slot(this, &LinesApp::quit));

      int w = 350;
      int h = 300;

//       back = new Widget(rootWindow(),50,150,w,h);
      //create the drawing surface:
      drawing = new DirectSurface(50,150,w,h);
      cerr << "... successfully created DirectSurface." << endl;
      drawing->fill(Color(0,16,rand()%64));
      cerr << "... successfully accessed DirectSurface :) " << endl;
      return 0;
    }

  int eventloopProc(void)
    {
      //do your processing here
      return 0;
    }

  int shutdownProc(void)
    {
      cout << "shutting down ..." << endl;
      //do your cleaning up here
      delete drawing;
      
      //all widgets are autodeleted ...
      return 0;
    }
};

void ellipses(Surface* surf)
{
  int w = surf->width();
  int h = surf->height();

  Painter painter(surf);
  Color col;
  Point center;
  int rx, ry;

  col = Color(rand()%255,rand()%255,rand()%255);
  painter.setColor(col);
  
  center = Point(rand()%w, rand()%h);
  rx = rand()%h/3;
  ry = rand()%h/3;
  painter.setFilling(false);
  painter.ellipse(center, rx, ry);
  
  center = Point(rand()%w, rand()%h);
   rx = rand()%h/3;
  ry = rand()%h/3; 
  painter.setFilling(true);
  painter.ellipse(center, rx, ry);
   
}

void circles(Surface* surf)
{
  int w = surf->width();
  int h = surf->height();

  Painter painter(surf);
  Color col;
  Point center;
  int radius;

  col = Color(rand()%255,rand()%255,rand()%255,rand()%255);
  painter.setColor(col);
  
  center = Point(rand()%w, rand()%h);
  radius = rand()%h/2;
  painter.setFilling(false);
  painter.circle(center, radius);
  
  center = Point(rand()%w, rand()%h);
  radius = rand()%h/2;
  painter.setFilling(true);
  painter.circle(center, radius);
   
}

void lines(Surface* surf)
{
  int w = surf->width();
  int h = surf->height();
 
  //create a painter object for this surface

  Painter painter(surf);
  Color col;
  Point start;
  Point end;

 
  start = Point(rand()%w, rand()%h);
  end = Point(rand()%w, rand()%h);
  
  col = Color(rand()%255,rand()%255,rand()%255);
  
  painter.line(start, end, col);
    
}

void LinesApp::timer()
{

  count++;

  drawing->beginModify();

  if(count > 100)
    {
      count = 0;
      drawing->fill(Color(0,16,rand()%64));
    }

  string wish = listbox->getSelection();

  if(wish == "lines")
    lines(drawing);
  else if(wish == "ellipses")
    ellipses(drawing);
  else if(wish == "circles")
    circles(drawing);

  drawing->endModify();

}





  
int main (int argc, char **argv)
{
  DEBUG_INIT(1 << 17);

  LinesApp app(argc, argv);
  
  app.init(640,480,16);
  
  return app.exec();
	
}

