/*
  libu - a C++ widget library based on SDL (Simple Direct Layer)
  Copyright (C) 1999  Karsten Laux

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Library General Public
  License as published by the Free Software Foundation; either
  version 2 of the License, or (at your option) any later version.
  
  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Library General Public License for more details.
  
  You should have received a copy of the GNU Library General Public
  License along with this library; if not, write to the
  Free Software Foundation, Inc., 59 Temple Place - Suite 330,
  Boston, MA  02111-1307, SA.
*/
#include "application.h"
#include "multilineedit.h"
#include "pushbutton.h"
#include "togglebutton.h"
#include "painter.h"
#include "listbox.h"
#include "resources.h"
#include "directsurface.h"

using namespace uta;

#include <stdio.h>
#include <stdlib.h>
#include <math.h>

class Terrain : public Application
{
  
protected:
  
  DirectSurface* drawing;

  
public:

  void timer();
  
  //needs to be there, but do not use it, use startupProc instead
  Terrain(int argc, char** argv) : 
    Application(argc, argv) { };
  
  //needs to be there, but use shutdownProc instead
  virtual ~Terrain() {};

  ///
  void draw();
  ///
  Point w2p(float, float, float);

  //you should at least handle the QUIT event, otherwise
  //your app cannot be closed
  bool handleEvent(const Event* event)
    {
      bool handled = Application::handleEvent(event);
      if(!handled)
	{
	  if(event->type()==Event::QUIT)
	    {
	      quit();
	    }
	}
      return handled;
    }

  int startupProc(void)
    {
      rootWindow()->setTitle("LIBUTA Texture Demo");

      //eg fontloading ....
      //(these fonts are used by several gui elements from libu,
      // so you should slways load text_font and button_font when
      // using widgets from libu)
      REG_FONT_RES("text_font","flatbrush.ttf, 18, 0xF0F0F0, 0x101010");
      REG_FONT_RES("button_font","flatbrush.ttf, 18, 0xF0F0F0, 0x101010");

      REG_SURFACE_RES("button170","button.png");
      REG_SURFACE_RES("button170_pressed","button_pressed.png");
      
      REG_SURFACE_RES("mud","mud_dried.png");
      REG_SURFACE_RES("mud25","mud_dried_shade25.png");
      REG_SURFACE_RES("mud50","mud_dried_shade50.png");
      REG_SURFACE_RES("mud75","mud_dried_shade75.png");

      //a pushbutton
      PushButton* quit_button 
	= new PushButton(rootWindow(), 
			  rootWindow()->width()-100,
			  rootWindow()->height()-40,
			  90, 30);
      quit_button->setText("Quit");
    
      //connect this button's click event (SIGNAL) with the application's
      //quit action (SLOT)
      quit_button->clicked.connect(slot(this, &Terrain::quit));
     
      //create the drawing surface:
      drawing = new DirectSurface(0,0,800,500);

      draw();

      return 0;
    }

  int eventloopProc(void)
    {
      //do your processing here
      return 0;
    }

  int shutdownProc(void)
    {
      //do your cleaning up here
      delete drawing;
      
      //all widgets are autodeleted ...
      return 0;
    }
};

#define SIZE_X 15
#define SIZE_Y 15

Point 
Terrain::w2p(float x, float y, float z)
{ 
  int my = drawing->height();
  int mx = drawing->width() / 2;
  float scale_x = 40.0;
  float scale_y = 20.0; 
  float scale_z = 10.0;

  int _x = mx + (int)((x - y) * scale_x + 0.45);
  int _y = my - (int)((x + y) * scale_y + z * scale_z + 0.45);

  return Point(_x,_y);
}

void Terrain::draw()
{
  Point p1,p2,p3,p4;
  drawing->beginModify();

 
  float grid[SIZE_X][SIZE_Y];
  unsigned x,y;
  for(x = 0; x < SIZE_X; x++)
    {
      for(y = 0; y < SIZE_Y; y ++)
	{
	  grid [x][y] = sin(y/4 + x );
	}
    }

  //clear surface
  drawing->clear();
  
  float z1,z2,z3,z4;

  for(x = 0; x < SIZE_X-1; x++)
    for(y = 0; y < SIZE_Y-1; y ++)
      {
	z1 = grid[x][y+1];
	z2 = grid[x+1][y+1];
	z3 = grid[x][y];
	z4 = grid[x+1][y];

	p1 = w2p(x,y+1,grid[x][y+1]);
	p2 = w2p(x+1,y+1,grid[x+1][y+1]);
	p3 = w2p(x,y,grid[x][y]);
	p4 = w2p(x+1,y,grid[x+1][y]);
	
// 	cerr << p1 << endl;
// 	cerr << p2 << endl;
// 	cerr << p3 << endl;
// 	cerr << p4 << endl;
// 	cerr << endl;

	if(z1 > z2)
	  if(z3 > z4)
	    SURFACE_RES("mud")->textureBlit(drawing,p1,p2,p3,p4);
	  else
	    SURFACE_RES("mud25")->textureBlit(drawing,p1,p2,p3,p4);
	else	  
	  if(z3 > z4)
	    SURFACE_RES("mud50")->textureBlit(drawing,p1,p2,p3,p4);
	  else
	    SURFACE_RES("mud75")->textureBlit(drawing,p1,p2,p3,p4);

	
      }

  drawing->endModify();
}


int main (int argc, char **argv)
{
  DEBUG_INIT(1 << 17);

  Terrain app(argc, argv);
  
  app.init(800,600,0);
  
  return app.exec();
	
}

