/*
  libu - a C++ widget library based on SDL (Simple Direct Layer)
  Copyright (C) 1999  Karsten Laux

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Library General Public
  License as published by the Free Software Foundation; either
  version 2 of the License, or (at your option) any later version.
  
  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Library General Public License for more details.
  
  You should have received a copy of the GNU Library General Public
  License along with this library; if not, write to the
  Free Software Foundation, Inc., 59 Temple Place - Suite 330,
  Boston, MA  02111-1307, SA.
*/

#include "application.h"
#include "multilineedit.h"
#include "pushbutton.h"
#include "togglebutton.h"
#include "painter.h"
#include "listbox.h"
#include "resources.h"
#include "directsurface.h"
#include "blitters.h"

using namespace uta;

#include <stdio.h>
#include <stdlib.h>
#include <math.h>


class Terrain2 : public Application
{
  
protected:
  
  DirectSurface* drawing;
  Surface texture;

public:

  void timer();
  
  //needs to be there, but do not use it, use startupProc instead
  Terrain2(int argc, char** argv) : 
    Application(argc, argv) { };
  
  //needs to be there, but use shutdownProc instead
  virtual ~Terrain2() {};

  ///
  void draw_16();
  ///
  void draw_32();

  //you should at least handle the QUIT event, otherwise
  //your app cannot be closed
  bool handleEvent(const Event* event)
    {
      bool handled = Application::handleEvent(event);
      if(!handled)
	{
	  if(event->type()==Event::QUIT)
	    {
	      quit();
	    }
	}
      return handled;
    }

  int startupProc(void)
    {
      rootWindow()->setTitle("LIBUTA Demo Terrain2");

      //eg fontloading ....
      //(these fonts are used by several gui elements from libu,
      // so you should slways load text_font and button_font when
      // using widgets from libu)
      REG_FONT_RES("text_font","flatbrush.ttf, 18, 0xF0F0F0, 0x101010");
      REG_FONT_RES("button_font","flatbrush.ttf, 18, 0xF0F0F0, 0x101010");

      REG_SURFACE_RES("button170","button.png");
      REG_SURFACE_RES("button170_pressed","button_pressed.png");
      
      texture.readFromFile("mud_dried.png");      
      
      //a pushbutton
      PushButton* quit_button 
	= new PushButton(rootWindow(), 
			  rootWindow()->width()-100,
			  rootWindow()->height()-40,
			  90, 30);
      quit_button->setText("Quit");
    
      //connect this button's click event (SIGNAL) with the application's
      //quit action (SLOT)
      quit_button->clicked.connect(slot(this, &Terrain2::quit));
     
      //create the drawing surface:
      drawing = new DirectSurface(0,0,800,500);

      draw_32();

      draw_16();
     
      return 0;
    }

  int eventloopProc(void)
    {
      //do your processing here
      return 0;
    }

  int shutdownProc(void)
    {
      //do your cleaning up here
      delete drawing;
      
      //all widgets are autodeleted ...
      return 0;
    }
};


// power of two improves performance significantly !!
#define W 256
#define H 256

void Terrain2::draw_16()
{
  
  Surface background(640,384,Pixelformat::RGB565);
  Surface buffer(640,384,Pixelformat::RGB565);

  texture.convert(Pixelformat::RGB565);
  
  /* fill background with the tiled texture */
  Rect src = Rect(0,0,texture.width(), texture.height());
  for(int i=0; i < 10; i++)
    for(int j= 0; j < 6; j++)
      texture.blit(&background, Rect(i*64,j*64,src.width(),src.height()),src);
  
  unsigned start,end;

  GammaFunction gammaFunc;
  gammaFunc.offset_x = 0;
  gammaFunc.offset_y = 0;
  gammaFunc.values = new char[256*256];
  char *lut = gammaFunc.values;

  char value;
  for(unsigned w = 0; w<W/2; w++)
    for(unsigned h= 0; h<H/2; h++)
      {
	value = (char)(127*sin(h*w*3.1415*10));
	lut[w+h*W] = value;
	lut[W-w-1+h*W] = value;
	lut[w+(H-h-1)*W] = value;
	lut[W-w-1+(H-h-1)*W] = value;
      }
  
  cerr << endl
       << "memory to memory blitting:" << endl;

  start = SDL_GetTicks();
  gammablit_RGB565_RGB565(&buffer, &background, (char)0,
			 Rect(0,0,background.width(), background.height()),
			 Point(0,0));
  end = SDL_GetTicks();
  cerr << "gammablit_RGB565_RGB565 with const gamma took " 
       << (end - start) << " ms." << endl;  

  start = SDL_GetTicks();
  gammablit_RGB565_RGB565(&buffer, &background, &gammaFunc,
			 Rect(0,0,background.width(), background.height()),
			 Point(0,0));
  end = SDL_GetTicks();
  cerr << "gammablit_RGB565_RGB565 using MyLUTLightning took " 
       << (end - start) << " ms." << endl << endl;  


  
  cerr << "and now memory to screen blitting: (including screen update)" << endl;

  start = SDL_GetTicks();

  char g;
  int m = 200;
  for(int n= 0; n < m; n++)
    {
      g = (char)(127.0*sin(n/10.0));

      drawing->beginModify();
      gammablit_RGB565_RGB565(drawing, &buffer, g, 
			     Rect(0,0,640,381), Point(10,10));
      drawing->endModify();
    }
  
  end = SDL_GetTicks();
  cerr << "gammablit_RGB565_RGB565 with const gamma took an average time of " 
       << (end - start)/m << " ms." << endl;

  cerr << endl << endl;
  
  delete[] gammaFunc.values;

}

void Terrain2::draw_32()
{
  
  Surface background(640,384,Pixelformat::RGBA8888);
  Surface buffer(640,384,Pixelformat::RGBA8888);

  texture.convert(Pixelformat::RGBA8888);
  
  /* fill background with the tiled texture */
  Rect src = Rect(0,0,texture.width(), texture.height());
  for(int i=0; i < 10; i++)
    for(int j= 0; j < 6; j++)
      texture.blit(&background, Rect(i*64,j*64,src.width(),src.height()),src);
  
  unsigned start,end;
  GammaFunction gammaFunc;
  gammaFunc.offset_x = 0;
  gammaFunc.offset_y = 0;
  gammaFunc.values = new char[256*256];
  char *lut = gammaFunc.values;

  char value;
  for(unsigned w = 0; w<W/2; w++)
    for(unsigned h= 0; h<H/2; h++)
      {
	value = (char)(127*sin(h*w*3.1415*10));
	lut[w+h*W] = value;
	lut[W-w-1+h*W] = value;
	lut[w+(H-h-1)*W] = value;
	lut[W-w-1+(H-h-1)*W] = value;
      }
  
  cerr << endl
       << "memory to memory blitting: " << endl;

  start = SDL_GetTicks();
  gammablit_RGBA8888_RGBA8888(&buffer, &background, (char)0,
			     Rect(0,0,background.width(), background.height()),
			     Point(0,0));
  end = SDL_GetTicks();
  cerr << "gammablit_RGBA8888_RGBA8888 with const gamma took " 
       << (end - start) << " ms." << endl;  
  
   
  start = SDL_GetTicks();
  gammablit_RGBA8888_RGBA8888(&buffer, &background, &gammaFunc,
			     Rect(0,0,background.width(), background.height()),
			     Point(0,0));
  end = SDL_GetTicks();
  cerr << "gammablit_RGBA8888_RGBA8888 using MyLUTLightning took " 
       << (end - start) << " ms." << endl << endl;  
  

  start = SDL_GetTicks();
  gammablit_C(&buffer, &background, (char)0,
			     Rect(0,0,background.width(), background.height()),
			     Point(0,0));
  end = SDL_GetTicks();
  cerr << "gammablit_C with const gamma took " 
       << (end - start) << " ms." << endl;  
  
   
  start = SDL_GetTicks();
  gammablit_C(&buffer, &background, &gammaFunc,
			     Rect(0,0,background.width(), background.height()),
			     Point(0,0));
  end = SDL_GetTicks();
  cerr << "gammablit_C using MyLUTLightning took " 
       << (end - start) << " ms." << endl << endl;  
  

  cerr << "memory to screen blitting: (including screen update)" << endl;

  start = SDL_GetTicks();

  char g;
  int m = 200;
  for(int n= 0; n < m; n++)
    {
      g = (char)(127.0*sin(n/10.0));

      drawing->beginModify();
      gammablit_RGBA8888_RGB565(drawing, &buffer, g, 
			       Rect(0,0,640,381), Point(10,10));
      drawing->endModify();
    }
  
  end = SDL_GetTicks();
  cerr << "gammablit_RGBA8888_RGB565 with const "
       <<" gamma took an average time of "<< (end - start)/m << " ms." << endl;

  cerr << endl << endl;

  delete[] gammaFunc.values;
} 


int main (int argc, char **argv)
{
   DEBUG_INIT(1 << 17);

  bool hasMMX = hasMMXExtension();  

  cerr << "processor features ";
  if(!hasMMX)
    cerr << "no ";
  cerr << "MMX extensions." << endl;

  /* we need MMX extensions here ! */
  assert(hasMMX);

  /* the typical libuta main: */
  Terrain2 app(argc, argv);
  app.init(800,600,16);
  return app.exec();
	
}

