/*
  libuta - a C++ widget library based on SDL (Simple Direct Layer)
  Copyright (C) 1999  Karsten Laux <klaux@student.uni-kl.de>

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Library General Public
  License as published by the Free Software Foundation; either
  version 2 of the License, or (at your option) any later version.
  
  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Library General Public License for more details.
  
  You should have received a copy of the GNU Library General Public
  License along with this library; if not, write to the
  Free Software Foundation, Inc., 59 Temple Place - Suite 330,
  Boston, MA  02111-1307, SA.
*/

#include "application.h"
#include "debug.h"
#include "logo.h"

#include "resources.h"

#include <assert.h>

namespace uta {

///static members

bool Application::hardwareCursor_ = false;
bool Application::sleeping_ = false;
unsigned int Application::idleTime_ = 25;

Application* Application::Instance = NULL;
SigC::Signal1<bool, const KeyEvent*, SigC::FixedMarshal<bool, false> > Application::keystroke;
SigC::Signal1<bool, const MouseEvent*, SigC::FixedMarshal<bool, false> > Application::mousemove;
SigC::Signal1<bool, const ButtonEvent*, SigC::FixedMarshal<bool, false> > Application::mouseclick;

Application::Application(int argc, char **argv) :
  rootWindow_(0),
  mouse_(0),
  startup_(true),
  shutdown_(false),
  running_(false),
  argc_(argc),
  argv_(argv),
  ticks_(0),
  focus_(true),
  iconified_(false),
  T50(50),
  T100(100)
{
  bool noAudio = true;

  if(Application::Instance != NULL)
    {
//      cerr << "only one instance of " << name() << " allowed." << endl;
      assert(false);
    }

  //setting the global app pointer....
  Application::Instance = this;

  /* Initialize SDL step by step*/
  debugN(17,cerr << "trying to initialize SDLvideo ... ";);
  if ( SDL_Init(SDL_INIT_VIDEO | SDL_INIT_NOPARACHUTE) < 0 )
    {
      cerr << " FATAL: Couldn't initialize: " << SDL_GetError() << endl;
      exit(1);
    }
  debugN(17,cerr << " OK." << endl;);
 
  debugN(17,cerr << "trying to initialize SDLaudio ... ";);
  if ( SDL_InitSubSystem(SDL_INIT_AUDIO) < 0 )
    {
      debugN(17,cerr << " Couldn't initialize: " << SDL_GetError() << endl;);
      cerr<<"Audio init failed; will proceed without soundsupport."<<endl;
      noAudio = true;
    }
  else
    {
      noAudio = false;
      debugN(17,cerr << " OK." << endl;);
    }

#if !defined(__WIN32__)
  /* for a clean exit */
  atexit(SDL_Quit);
#endif

  debugN(17,cerr << "Enabling SDL Unicode support." << endl;);
  SDL_EnableUNICODE(1);
  

  mixer_ = new Mixer();
  
  if(!mixer_->audioAvailable())
    {
      cerr<<"Audio init failed; will proceed without sound."<<endl;
    }
  
  soundMapper_ = new SoundMapper();

  SDL_EnableKeyRepeat(SDL_DEFAULT_REPEAT_DELAY, SDL_DEFAULT_REPEAT_INTERVAL);
  
  T50.alarm.connect(timer.slot());
  T50.alarm.connect(slot(this, &Application::updateScreen));
  T50.alarm.connect(slot(this, &Application::processKeys));
  
  T100.alarm.connect(timer100.slot());
  T100.alarm.connect(slot(this, &Application::updateMouse));
  
  
  result = 0;
  running_ = true;
  
  screen_update = false;
  
  lastticks = SDL_GetTicks();

}
  

Application::~Application()
{
  debugN(17,cerr << "Application: removing mouse...";);
  if(mouse_ != 0)
    delete mouse_;

  debugN(17,
	 cerr << "OK." << endl;
	 cerr << "Application: removing widget tree...";
	 );

  if(rootWindow_ != 0)
    delete rootWindow_;

  debugN(17,cerr << "OK." << endl;);

  
  debugN(17,cerr << "Application requesting resource DB to unload all data ..."<<endl;);
  //this is important here, because unloading music when the mixer is already dead  
  //likely leads to segfaults
  Resources::instance->unregisterAll();
  
  if(soundMapper_ != 0)
    delete soundMapper_;

//   why did I remove this !??????????????????????????
//
//   if(mixer_ != 0)
//     {
//       debugN(17,cerr<<"Application shutting down Mixer ...";);	
//       delete mixer_;
//       debugN(17,cerr<<" shutting down SDL_Audio ...";);
//       SDL_QuitSubSystem(SDL_INIT_AUDIO);
//       debugN(17,cerr<<" [OK].";);
//     }

  //the last one turns out the lights :)
}


bool 
Application::modeAvailable(int resX, int resY, int bpp, bool fullscreen, int* true_bpp)
{ 
  Uint32 flags = SDL_ANYFORMAT | SDL_HWSURFACE | SDL_HWPALETTE ;
 
  if(fullscreen)
    flags = flags | SDL_FULLSCREEN  | SDL_HWSURFACE;     

  int h =  SDL_VideoModeOK(resX, resY, bpp, flags);
  
  if(true_bpp)
    *true_bpp = h;

  return (h != 0);
}

void
Application::init(int resX, int resY, int bpp, 
		  bool fullscreen, 
		  const Surface* icon)
{
  if(rootWindow_)
    {
      if(mouse_)
	delete mouse_;
      
      delete rootWindow_;
      
      mouse_ = NULL;
      rootWindow_ = NULL;
    }

  if(icon)
    {  
      debugN(17,cerr<<"setting window icon..."<<endl;);
      //set the given surface as icon for our windows
      SDL_WM_SetIcon(icon->sdlSurface_,NULL);
      debugN(17,cerr<<"ok."<<endl;);
    }
  else
    {
      //otherwise put in a nice libuta logo :)

      debugN(17,cerr<<"creating logo.." << endl;);
      Logo* logo = new Logo();
      debugN(17,cerr<<"success."<<endl;);
      debugN(17,cerr<<"setting window icon..."<<endl;);
      //set the logo as icon for our windows
      SDL_WM_SetIcon(logo->sdlSurface_,NULL);
      //now we don't need the logo anymore
      delete logo;
      debugN(17,cerr<<"ok."<<endl;);
    }

  debugN(17,cerr <<"creating RootWidget...";);
  rootWindow_ = new RootWindow(resX, resY, bpp, fullscreen);
  assert(rootWindow_);
  debugN(17,cerr <<"success." << endl;);


  debugN(17,cerr<<"creating Mouse ...";);
  mouse_ = Mouse::create();
  assert(mouse_);
  debugN(17,cerr<<"success." << endl;);

  //disable/enable SDL cursor... depending if we use a software mousecursor
  useHardwareCursor(hardwareCursor_);
}


int
Application::exec()
{
  if(rootWindow_ == NULL)
    {
      cerr <<"Application has no rootwindow." << endl;
      return -1;
    }
  int returnVal;

  startupProc();
  startup_ = false;

  returnVal = eventloop();
  shutdown_ = true;

  shutdownProc();

  return returnVal;
}


bool 
Application::handleEvent(const Event* event)
{
  bool handled = false;

  if(event->type() == Event::MOUSE)
    {
      if(!hardwareCursor_)
	mouse_->move(((MouseEvent*)event)->position());
      
      if(!Application::sleeping_)
	handled = mousemove.emit((const MouseEvent*)event);
    }
  else if(event->type() == Event::BUTTON)
    {
      if(!Application::sleeping_)
	handled = mouseclick.emit((const ButtonEvent*)event);
    }
  else if(event->type() == Event::KEY)
    {
      if(!Application::sleeping_)
	handled = keystroke.emit((const KeyEvent*)event);
    }
  else if(event->type() == Event::LOSTFOCUS)
    {
      focus_ = false;
    }
  else if(event->type() == Event::GOTFOCUS)
    {
      focus_ = true;
    }    
    else if (event->type() == Event::ICONIFIED)
    {
      iconified_ = true;
    }
  else if (event->type() == Event::RESTORED)
    {
      iconified_ = false;
    }

  return handled;

}



bool
Application::keepAlive()
{

  SDL_Event sdl_event;
  Event *event;

  bool handled=false;
  
  /* release the rootWindow (== screen) during idle time.
  */
  uapp->rootWindow_->access_.release();
  
  //Timer stuff
  uapp->tick = SDL_GetTicks();
  uapp->delta_tick = uapp->tick - uapp->lastticks;
  uapp->lastticks = uapp->tick;
  
  //limit to 40 revolutions per second
  if(uapp->delta_tick < Application::idleTime_)
    {
      SDL_Delay(Application::idleTime_ - uapp->delta_tick);
      uapp->tick = SDL_GetTicks();
	  uapp->delta_tick = uapp->delta_tick + uapp->tick - uapp->lastticks;
	  uapp->lastticks = uapp->tick;	  	  
    }
  uapp->ticks_ += uapp->delta_tick;

  //gain access to rootWindow again 
  uapp->rootWindow_->access_.grab();

  // timers are process all the time, also when sleeping
  Timer::processAllTimers();

  //process SDL events
  while(SDL_PollEvent(&sdl_event))
    {
      event = createUEvent(&sdl_event);
	  
      // do eventhandling in the application
      handled = uapp->handleEvent(event);
      
      // if not already handled let widget tree handle this event
      if(!Application::sleeping_ && !handled)
	uapp->rootWindow_->handleEvent(event);
      
      delete event;
    } // while(SDL_PollEvent()
  
  //derived classes go here
  uapp->result = uapp->eventloopProc();      
    
  // no framebuffer update, when iconified
  if(!Application::isIconified())
    {
      
      // Mouse hide from framebuffer
      if(!uapp->hardwareCursor_ && uapp->focus_)
	uapp->mouse_->remove();

      
      //update the framebuffer
      if(uapp->screen_update)
	uapp->rootWindow_->update();
      
      // Mouse blit to framebuffer and update screen appropriately
      // this is bad, because this screen area may gets 
      // update later again :-(
      // hmmm, the question is whether this really is a performance hit
      if(!uapp->hardwareCursor_ && uapp->focus_)
	uapp->mouse_->paint();

      //blit framebuffer to screen
      if(uapp->screen_update)
	uapp->rootWindow_->updateScreen();
      
      if(uapp->screen_update)
	uapp->screen_update = false;
    }

  return uapp->running_;

}

unsigned int
Application::sleep(unsigned int t)
{
  Application::sleeping_ = true;

  unsigned int loops = t / Application::idleTime_;
  unsigned int start = Application::ticks();
  while(loops && Application::keepAlive())
    loops --;
  
  // sleep for at least t ms
  if(t % Application::idleTime_)
    Application::keepAlive();
  
  Application::sleeping_ = false;

  return Application::ticks() - start;
}

int
Application::eventloop()
{

  // application main loop
  //
  // * timer coordination of the three timers: 50ms, 100ms and 500ms
  // * limit cpu usage by limiting the revolutions to 1000ms/Application::idleTime_ per second
  // * handle event everytime
  // * handle mouse updates everytime, including screen update
  // * update and draw the screen at 20fps
  // * implement key repeat

  while(Application::keepAlive());

  rootWindow_->access_.release();

  return result;
}

void 
Application::updateScreen()
{
  screen_update = true;
}

void 
Application::processKeys()
{

}

void
Application::updateMouse()
{
  //update mouse only when the windows has focus
  if(!hardwareCursor_ && focus_)
    mouse_->tick();
}

void 
Application::quit()
{
  
  debugN(17,cerr << "Application: was requested to quit." << endl;);
  running_ = false;

}

void 
Application::infoMsg(const char* data)
{
  debugN(17,cerr << "Application::Info: " << data << endl;);
}

void 
Application::useHardwareCursor(bool flag)
{
  Application::hardwareCursor_ = flag;
  
  // only if we already have a application instance
  if(uapp)
    {
      // only call SDL if we are already in video mode 
      if(uapp->rootWindow_)
	{
	  //disable/enable SDL cursor... depending if we use a software mousecursor
	  if(uapp->hardwareCursor_)
	    {
	      uapp->mouse()->remove();
	      SDL_ShowCursor(1);
	    }
	  else
	    {
	      uapp->mouse()->paint();
	      SDL_ShowCursor(0);
	    }
	}
    }
}

void
Application::setIdleTime(unsigned int t)
{
  Application::idleTime_ = t;
}

} // namespace uta

