/*
  libuta - a C++ widget library based on SDL (Simple Direct Layer)
  Copyright (C) 1999  Karsten Laux

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Library General Public
  License as published by the Free Software Foundation; either
  version 2 of the License, or (at your option) any later version.
  
  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Library General Public License for more details.
  
  You should have received a copy of the GNU Library General Public
  License along with this library; if not, write to the
  Free Software Foundation, Inc., 59 Temple Place - Suite 330,
  Boston, MA  02111-1307, SA.
*/
// written by Karsten Laux, May 1999  

#include "surface.h"
#include "debug.h"
#include "application.h"

#include <math.h>
#ifndef M_PI
 #define M_PI 3.14159265358979323846 /* stupid, stupid w32 crosscompiler */
#endif

#include "sge_blib.h"
#include "sge_draw.h"

#include <assert.h>

namespace uta {

Surface::Surface():
  transCol_(transparent),
  transparent_(true),
  alpha_(0),
  RLEAccelerated_(false)
{
  sdlSurface_ = 0;
 
  format = Pixelformat::UNKNOWN;
};   

Surface::Surface(unsigned w, unsigned h, int pixelformat):
  transCol_(transparent),
  transparent_(true),
  alpha_(0),
  RLEAccelerated_(false)
{
  sdlSurface_ = 0;
  
  format = pixelformat;

  if(format.valid())
    {
      sdlSurface_ = SDL_AllocSurface(SDL_SWSURFACE |SDL_SRCALPHA ,
				     w,
				     h,
				     format.bpp()*8,
				     format.rMask(),
				     format.gMask(),
				     format.bMask(),
				     format.aMask());

      gatherPalette();
      clear();

    }
}

Surface::Surface(void* pixels, unsigned w, unsigned h, unsigned pitch, int pixelformat):
  transCol_(transparent),
  transparent_(true),
  alpha_(0),
  RLEAccelerated_(false)
{
  sdlSurface_ = 0;
  format = pixelformat;
  if(format.valid())
    {
      
      sdlSurface_ = SDL_CreateRGBSurfaceFrom(pixels,
					     w,h,
					     format.bpp()*8,
					     pitch,
					     format.rMask(),
					     format.gMask(),
					     format.bMask(),
					     format.aMask());
      
      gatherPalette();
    }

}
  

Surface::Surface(const Surface& surf):
  transCol_(transparent),
  transparent_(true),
  alpha_(0),
  RLEAccelerated_(false)
{
  
  sdlSurface_ = SDL_AllocSurface(surf.sdlSurface_->flags,
				 surf.sdlSurface_->w,
				 surf.sdlSurface_->h,
				 surf.sdlSurface_->format->BitsPerPixel,
				 surf.sdlSurface_->format->Rmask,
				 surf.sdlSurface_->format->Gmask,
				 surf.sdlSurface_->format->Bmask,
				 surf.sdlSurface_->format->Amask);

  useRLEAcceleration(surf.RLEAccelerated_);  
  setAlpha(surf.alpha_);
  setTransColor(surf.transCol_);
  setTransparency(surf.transparent_);

  fill(transCol_);

  format = surf.format;

  SDL_BlitSurface(surf.sdlSurface_,0,sdlSurface_,0);

  gatherPalette();

};


  
Surface::~Surface()
{
  clearSurfaceData();
};


Surface*
Surface::createChildSurface(const Rect& rect)
{
  if(!sdlSurface_)
    return NULL;
  Uint8* addr = 
    (Uint8*)(sdlSurface_->pixels) + 
    rect.upperLeft().x * format.bpp() + rect.upperLeft().y * pitch();
  
  return new Surface((void*)addr, rect.width(), rect.height(), pitch(), format());
}

void 
Surface::setPalette(const vector<Color>& pal)
{  
  
  format.palette = pal;
 
  applyPalette();
  
  setAlpha(alpha_);

  if(transparent_)
    {
      setTransColor(transCol_);
      setTransparency(transparent_);
    }
}

void
Surface::applyPalette()
{

    
  unsigned int count = format.palette.size();
  unsigned int n;
  SDL_Color *colors;

  //accept only max 255 entries ...

  if(count > 255)
    count = 255;

  if(count > 0)
    {
      colors = new SDL_Color[count];
      for(n=0; n < count; n++)
	{
	  colors[n].r=format.palette[n].r;
	  colors[n].g=format.palette[n].g;
	  colors[n].b=format.palette[n].b;
	  colors[n].unused=format.palette[n].a;
	}
      
  
      if(sdlSurface_)
	{
	  if(!SDL_SetColors(sdlSurface_,colors,0,count))
	    {
	      debugN(17, cerr << "Surface: unable to set palette properly. "<< endl;);
	    }
	}
  
  
    delete[] colors;
    }
}

void
Surface::gatherPalette()
{

  if(sdlSurface_)
    {
      int n = 0;
      SDL_Palette* pal = sdlSurface_->format->palette;
      
      if(pal)
	{
	  format.palette.clear();
	  for(n = 0; n < pal->ncolors; n++)
	    format.palette.push_back(Color(pal->colors[n].r,
					   pal->colors[n].g,
					   pal->colors[n].b,
					   pal->colors[n].unused));
	}
    }
}


void
Surface::clearSurfaceData()
{  
  if(sdlSurface_ != 0)
    {
      SDL_FreeSurface(sdlSurface_);
    }
}



Surface& 
Surface::operator = (const Surface& surf)
{
  if(this != &surf)
    {
      //clear old surface data
      clearSurfaceData();

      
      //create a new surface
      sdlSurface_ = SDL_AllocSurface(surf.sdlSurface_->flags,
				     surf.sdlSurface_->w,
				     surf.sdlSurface_->h,
				     surf.sdlSurface_->format->BitsPerPixel,
				     surf.sdlSurface_->format->Rmask,
				     surf.sdlSurface_->format->Gmask,
				     surf.sdlSurface_->format->Bmask,
				     surf.sdlSurface_->format->Amask);


      useRLEAcceleration(surf.RLEAccelerated_);        
      setAlpha(surf.alpha_);
      setTransColor(surf.transCol_);
      setTransparency(surf.transparent_);
      
      
      //and try to copy the content
      SDL_BlitSurface(surf.sdlSurface_,0,sdlSurface_,0);
      
      gatherPalette();
    }
  

  
  return *this;
}  

void Surface::setTransparency(bool transp)
{
  transparent_ = transp;

  if(sdlSurface_ == 0)
    return;
  
  if(transp)
    {  
      if(RLEAccelerated_)
	{
	  SDL_SetColorKey(sdlSurface_, 
			  SDL_SRCCOLORKEY | SDL_RLEACCEL,  
			  SDL_MapRGB(sdlSurface_->format,
				     transCol_.r, 
				     transCol_.g, 
				     transCol_.b)); 
	}
      else
	{
	  SDL_SetColorKey(sdlSurface_, 
			  SDL_SRCCOLORKEY,  
			  SDL_MapRGB(sdlSurface_->format,
				     transCol_.r, 
				     transCol_.g, 
				     transCol_.b));
	  
	}
    }
  else
    {
      //a bit faster
      SDL_SetColorKey(sdlSurface_,0,0x00000000);
    }
}
  
void 
Surface::setTransColor(const Color& color)
{
  transCol_ = color;
}

  
void 
Surface::setAlpha(unsigned char alpha)
{
  alpha_ = alpha;

  if(sdlSurface_ == 0)
    return;

  if(alpha > 0)
    SDL_SetAlpha(sdlSurface_, SDL_SRCALPHA,(unsigned char)(alpha*2.55));
  else
    SDL_SetAlpha(sdlSurface_, 0,0);
}


 
void 
Surface::lock() const
{
  /* hmmm.. I am unsure, it seems SDL_MUSTLOCK returns false
     on RLE-encoded surface which is not correct !
     so we lock all RLE encoded surfaces

     this has been fixed in SDL-1.1.4 !
  */

  //lock surface
  if(SDL_MUSTLOCK(sdlSurface_))
     {
      if(SDL_LockSurface(sdlSurface_) < 0)
	{
	  SDL_Delay(10);
	  if(SDL_LockSurface(sdlSurface_) < 0)
	    {
	      cerr << "Surface::lock on surface failed twice." 
		   << endl;
	      cerr << "         no handling here yet :-(" << endl;
	      assert(false);
	    }
	}
     }
}


void 
Surface::unlock() const
{
  //unlock
  if(SDL_MUSTLOCK(sdlSurface_))
    {
      SDL_UnlockSurface(sdlSurface_);
    }
}

void
Surface::writePixel(Uint32 pixeladdr, Uint32 pixel)
{
  Uint8* screen_loc = (Uint8*)pixels()+pixeladdr;

  switch(sdlSurface_->format->BytesPerPixel)
    {
    case 1:
      *((Uint8 *)screen_loc) = pixel;
      break;
    case 2:
      *((Uint16 *)screen_loc) = pixel;
      break;
    case 3:      
      int shift;
      shift = sdlSurface_->format->Rshift;
      *(screen_loc+shift/8) = pixel>>shift;
      shift = sdlSurface_->format->Gshift;
      *(screen_loc+shift/8) = pixel>>shift;
      shift = sdlSurface_->format->Bshift;
      *(screen_loc+shift/8) = pixel>>shift;
      break;
    case 4:
      *((Uint32 *)screen_loc) = pixel;    
      break;
    default:
      break;
    }
}

Uint32 
Surface::readPixel(Uint32 pixeladdr)
{
  Uint8* screen_loc = (Uint8*)pixels()+pixeladdr;
  
  Uint32 pixel=0;

  switch(sdlSurface_->format->BytesPerPixel)
    {
    case 1:
      pixel = *((Uint8 *)screen_loc);
      break;
    case 2:
      pixel = *((Uint16 *)screen_loc);
      break;
    case 3:
      int shift;
      // Im not sure on this, Karsten 01/1999
      shift = sdlSurface_->format->Rshift;
      pixel = *(screen_loc+shift/8) << shift;
      shift = sdlSurface_->format->Gshift;
      pixel |= *(screen_loc+shift/8) << shift;
      shift = sdlSurface_->format->Bshift;
      pixel |= *(screen_loc+shift/8) << shift;
      break;
    case 4:
      pixel = *((Uint32 *)screen_loc);    
      break;
    default:
      break;
    }

  return pixel;
}



Rect
Surface::blit(Surface* destBM) const
{
  Rect src_(0,0,width(),height());

  if(!destBM || !sdlSurface_ || !destBM->sdlSurface_)
    return Rect::invalid;
  
  SDL_BlitSurface(sdlSurface_, 0, destBM->sdlSurface_, 0);
  
  return src_;

}

Rect
Surface::blit(Surface* destBM, const Rect& dest) const
{
  Rect src(0,0,width(),height());

  if(!destBM || !sdlSurface_ || !destBM->sdlSurface_)
    return Rect::invalid;

  //  SDL_Rect* dst = dest;
  static SDL_Rect dst;
  dst.x = dest.upperLeft().x;
  dst.y = dest.upperLeft().y;
  dst.w = dest.width();
  dst.h = dest.height();        

  SDL_BlitSurface(sdlSurface_, 0, destBM->sdlSurface_, &dst);

  //delete dst;

  return src;

}

Rect
Surface::blit(Surface* destBM, const Rect& dest, const Rect& src_) const
{
  if(!destBM || !sdlSurface_  || !destBM->sdlSurface_)
    return Rect::invalid;  
  
 //  SDL_Rect* src = src_;
//   SDL_Rect* dst = dest;
  static SDL_Rect dst;
  dst.x = dest.upperLeft().x;
  dst.y = dest.upperLeft().y;
  dst.w = dest.width();
  dst.h = dest.height();  
  static SDL_Rect src;
  src.x = src_.upperLeft().x;
  src.y = src_.upperLeft().y;
  src.w = src_.width();
  src.h = src_.height();  
                
  //  debugN(17,cerr<<src_<<">>"<< dest<<endl;);

  SDL_BlitSurface(sdlSurface_, &src, destBM->sdlSurface_, &dst);
  
  Rect val(dst.x, dst.y, dst.w, dst.h);

//   delete src;
//   delete dst;

  return val;

}

void
Surface::fill(const Color &color)
{

  if(sdlSurface_ == 0)
    return;

  Uint32 bits = format.mapToPixel(color);

  //  lock();
  SDL_FillRect(sdlSurface_, NULL, bits);
  //  unlock();

}


void
Surface::fill(const Rect &dest_, const Color &color)
{
   if(sdlSurface_ == 0)
     {
       debugN(17, cerr << "Surface.fill() on empty Surface has no effect." << endl;);
       return;
     }

   if(!dest_.isValid())
     {
       debugN(17, cerr << "Surface.fill() was given an invalid rect." << endl;);
       return;
     }
   
   
   Uint32 bits = format.mapToPixel(color);

   //do clipping if neccessary
   Rect dest = dest_.intersect(Rect(0,0,width(),height()));
   static SDL_Rect dst;
   dst.x = dest.upperLeft().x;
   dst.y = dest.upperLeft().y;
   dst.w = dest.width();
   dst.h = dest.height();  

   SDL_FillRect(sdlSurface_, &dst, bits);
     
}

#define CLIP(v) ((v) <= 0 ? 0 : (v) >= 255 ? 255 : (v))   

bool
Surface::convert(const Pixelformat& pixelformat, bool _dither)
{

  debugN(17,cerr << "Surface converting "<< format.asString() 
	<<" to " << pixelformat.asString() << " ... "<< endl;);

  bool success = false;

  // we only dither if target is palettized
  bool dither = false;
  if(pixelformat.bpp() == 1 && _dither)
    dither = true;

  if(pixelformat.valid())
    {
      if(sdlSurface_)
	{   
	  SDL_Surface *tmp = SDL_AllocSurface(SDL_SWSURFACE | SDL_SRCALPHA ,
					      width(),
					      height(),
					      pixelformat.bpp()*8,
					      pixelformat.rMask(),
					      pixelformat.gMask(),
					      pixelformat.bMask(),
					      pixelformat.aMask());
	  
	  if(tmp)
	    {
	      // set palette for target if needed
	      if(!pixelformat.palette.empty())
		{
		  if(tmp->format->palette != 0)
		    free(tmp->format->palette->colors);

		  int n = pixelformat.palette.size();
		  tmp->format->palette->colors=
		    (SDL_Color*)malloc(n*sizeof(SDL_Color));
		  tmp->format->palette->ncolors=n;
		  for(int i=0; i < n; i++)
		    {
		      tmp->format->palette->colors[i].r = pixelformat.palette[i].r;
		      tmp->format->palette->colors[i].g = pixelformat.palette[i].g;
		      tmp->format->palette->colors[i].b = pixelformat.palette[i].b;
		      // really ?
		      tmp->format->palette->colors[i].unused = pixelformat.palette[i].a;
		    }
		}
	      
	      /* converter blit.
		 Handle special case, so that any semitransparencies
		 get full transparent for the target, if target does not
		 has an alpha channel.

	      */
	      
	      SDL_LockSurface(tmp);
	      SDL_LockSurface(sdlSurface_);
	      
	      Uint32 pixel;
	      Uint32 src = 0; 
	      Uint8 *dst = (Uint8*)tmp->pixels;
	      Uint32 src_skip = sdlSurface_->pitch - format.bpp() * width();
	      Uint32 dst_skip = tmp->pitch - pixelformat.bpp()*width();
	      Color col;
	      int ind;
	      int* errors = (int*)calloc(width()+1, sizeof(int)* 3);   
	      
	      for(int y=0; y < height(); y++)
		{               
		  int r, r0, r1, r2;
		  int g, g0, g1, g2;
		  int b, b0, b1, b2;
		  int* e = errors;
		  
		  r = r0 = r1 = 0;
		  g = g0 = g1 = 0;
		  b = b0 = b1 = 0;  
		  
		  for(int x = 0; x < width() ; x++)
		    {      
		      int d2;

		      pixel = readPixel(src);
		      col = format.mapToColor(pixel);
		      
		      // set any semitransparent areas to full transparency
		      if(col.a != SDL_ALPHA_OPAQUE)
			{
			  col = transCol_;
			  transparent_ = true;
			}
		      
		      if(!dither)
			{
			  pixel = pixelformat.mapToPixel(col);
			  
			  switch(pixelformat.bpp())
			    {
			    case 1:
			      *dst = pixel;
			      break;
			    case 2:
			      *((Uint16*)dst) = pixel;
			      break;
			    case 3:
			      int shift;
			      shift = tmp->format->Rshift;
			      *(dst+shift/8) = pixel>>shift;
			      shift = tmp->format->Gshift;
			      *(dst+shift/8) = pixel>>shift;
			      shift = tmp->format->Bshift;
			      *(dst+shift/8) = pixel>>shift;			  
			      break;
			    default:
			      *((Uint32*)dst) = pixel;
			      break;
			    }
			}
		      else
			{
			  r = CLIP(col.r + (r + e[3+0])/16);
			  g = CLIP(col.g + (g + e[3+1])/16);
			  b = CLIP(col.b + (b + e[3+2])/16);
			  
			  // get best matching color
			  ind = pixelformat.mapToPixel(Color(r,g,b));

			  /* write data to target */
			  *dst = ind;
			  
			  r -= (int) pixelformat.palette[ind].r;
			  g -= (int) pixelformat.palette[ind].g;
			  b -= (int) pixelformat.palette[ind].b;
			  
			  /* propagate errors (don't ask ;-)*/
			  r2 = r; d2 = r + r; r += d2; e[0] = r + r0;
			  r += d2; r0 = r + r1; r1 = r2; r += d2;
			  g2 = g; d2 = g + g; g += d2; e[1] = g + g0;
			  g += d2; g0 = g + g1; g1 = g2; g += d2;
			  b2 = b; d2 = b + b; b += d2; e[2] = b + b0;
			  b += d2; b0 = b + b1; b1 = b2; b += d2;
			  
			  e += 3; 
			} // if(!dither ..)
		      
		      src+= format.bpp();
		      dst+= pixelformat.bpp();
		    }  // for (int x= ...
		  
		  e[0] = b0;
		  e[1] = b1;
		  e[2] = b2;  
		  
		  src += src_skip;
		  dst += dst_skip;
		} // for (int y = ...
	      
	      free(errors);
	      
	      SDL_UnlockSurface(sdlSurface_);
	      SDL_UnlockSurface(tmp);
	      

	      //remove old surface
	      SDL_FreeSurface(sdlSurface_);
	      //store new one
	      sdlSurface_ = tmp;
	      //store new surface format
	      format = pixelformat;
	      
	      //retrieve palette from surface
	      gatherPalette();

 	      setAlpha(alpha_);
 	      setTransColor(transCol_);
 	      setTransparency(transparent_);

	      success = true;
	    }
	  else
	    {
	      cerr << "Surface: out of memory." << endl;
	    }
	}

    }
  else
    {
      debugN(17, cerr << "Surface: convert to unknown Pixelformat requested." << endl;);
    }

  debugN(17,cerr << "Surface.convert() done." << endl;);

  return success;
    
}


Rect 
Surface::textureBlit(Surface* dst, 
		     const Point& p1, const Point& p2,
		     const Point& p3, const Point& p4) const
{
  if(sdlSurface_ == 0 || dst == 0 || dst->sdlSurface_ == 0)
    return Rect::invalid;
  return textureBlit(dst,p1,p2,p3,p4,Rect(0,0,width(),height()));
}

Rect
Surface::textureBlit(Surface* dst, 
		     const Point& p1, const Point& p2,
		     const Point& p3, const Point& p4, const Rect& src) const
{
  if(sdlSurface_ == 0 || dst == 0 || dst->sdlSurface_ == 0)
    return Rect::invalid;

  SDL_LockSurface(sdlSurface_);
  
  sge_TexturedRect(dst->sdlSurface_,
		   p1.x,p1.y,
		   p2.x,p2.y,
		   p3.x,p3.y,
		   p4.x,p4.y,
		   sdlSurface_,
		   src.upperLeft().x,src.upperLeft().y,
		   src.lowerRight().x,src.upperLeft().y,
		   src.upperLeft().x,src.lowerRight().y,
		   src.lowerRight().x,src.lowerRight().y);
  SDL_UnlockSurface(sdlSurface_);
  //FIXME ... surely we can be more precise about the changed region...
  return Rect(0,0,dst->width(), dst->height());
}


Rect 
Surface::scaledBlit(Surface* dst) const
{
  if(sdlSurface_ == 0 || dst == 0 || dst->sdlSurface_ == 0)
    return Rect::invalid;
  
  return scaledBlit(dst,
		    Rect(0,0,dst->width(),dst->height()),
		    Rect(0,0,width(),height()));
}

Rect 
Surface::scaledBlit(Surface* dst, const Rect& dest) const
{
  if(sdlSurface_ == 0 || dst == 0 || dst->sdlSurface_ == 0)
    return Rect::invalid;
  return scaledBlit(dst,dest,Rect(0,0,width(),height()));
}

Rect
Surface::scaledBlit(Surface* dst, const Rect& dest, const Rect& src) const
{
  if(sdlSurface_ == 0 || dst == 0 || dst->sdlSurface_ == 0)
    return Rect::invalid;

  if(dest.width() == src.width() && dest.height() == src.height())
    return blit(dst, dest, src);

  SDL_LockSurface(sdlSurface_);
  sge_TexturedRect(dst->sdlSurface_,
		   dest.upperLeft().x,dest.upperLeft().y,
		   dest.lowerRight().x,dest.upperLeft().y,
		   dest.upperLeft().x,dest.lowerRight().y,
		   dest.lowerRight().x,dest.lowerRight().y,
		   sdlSurface_,
		   src.upperLeft().x,src.upperLeft().y,
		   src.lowerRight().x,src.upperLeft().y,
		   src.upperLeft().x,src.lowerRight().y,
		   src.lowerRight().x,src.lowerRight().y);
  SDL_UnlockSurface(sdlSurface_);
  return dest;
}

bool 
Surface::scale(float n)
{
  return scale(int(width()*n), int(height()*n));
}


bool 
Surface::scale(int new_width, int new_height)
{
  if(!sdlSurface_)
    {
      return false;
    }
  
  if(new_width == width() && new_height == height())
    return true;

  SDL_Surface *tmp = SDL_AllocSurface(SDL_SWSURFACE |SDL_SRCALPHA ,
				      new_width,
				      new_height,
				      format.bpp()*8,
				      format.rMask(),
				      format.gMask(),
				      format.bMask(),
				      format.aMask());
  assert(tmp);

  //the source surface may be RLE-encoded, thus we need to lock it ! 
  SDL_LockSurface(sdlSurface_);
  sge_TexturedRect(tmp,0,0,new_width,0,0,new_height,new_width,new_height,
		   sdlSurface_,0,0,width(),0,0,height(),width(),height());
  
  SDL_UnlockSurface(sdlSurface_);

  delete sdlSurface_;
  sdlSurface_ = tmp;
  
  return true;
}

bool
Surface::mirror()
{

  if(!sdlSurface_)
    {
      return false;
    }

  // FIXME !!!!!!!!!!!!!
  //allocate scanline buffer which stores 32bit pixel values
  // this is not the fastest way to go ... this every pixel
  // gets converted force and back ...
  Uint32* scanline = new Uint32[width()];
  
  int x;
  int y;
  int bytes = pixelformat().bpp();

  Uint32 pixeladdr;

  for(y = 0; y < height(); y++)
    {
      pixeladdr = pitch() * y;
      x=0;
      while(x < width())
	{
	  scanline[x] = readPixel(pixeladdr);
	  x++;
	  pixeladdr += bytes;
	}
      pixeladdr = pitch() * y;
      x = width()-1;
      while(x >= 0)
	{
	  writePixel(pixeladdr, scanline[x]);
	  x--;
	  pixeladdr += bytes;
	}
    }
  delete[] scanline;
      
  return true;
} 

void
Surface::clear()
{
  if(!sdlSurface_)
    return;
  
  fill(transCol_);
}


}
