/*
  libuta - a C++ widget library based on SDL (Simple Direct Layer)
  Copyright (C) 1999  Karsten Laux

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Library General Public
  License as published by the Free Software Foundation; either
  version 2 of the License, or (at your option) any later version.
  
  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Library General Public License for more details.
  
  You should have received a copy of the GNU Library General Public
  License along with this library; if not, write to the
  Free Software Foundation, Inc., 59 Temple Place - Suite 330,
  Boston, MA  02111-1307, SA.
*/
#include "terminal.h"
#include "resources.h"

namespace uta {


Terminal::Terminal(Widget* parent, int x, int y, int w, int h,
		     const Font *font, bool readonly):
  MultiLineEdit(parent,x,y,w,h,font,readonly),
  scrollTo_(0),
  currentLine_(0)
{
  if(textFont_->getHeight())
    allLines_ = height()/textFont_->getHeight();  
  else
    allLines_ = 1;

  if(RES_AVAILABLE("terminal_color"))
    Widget::setColor(COLOR_RES("terminal_color"));
  if(RES_AVAILABLE("terminal_surface"))
    Widget::setBackground(SURFACE_RES("terminal_surface"),false);
}

  
Terminal::~Terminal()
{
}

  
  
unsigned 
Terminal::linesCount() const
{
  return allLines_;
}
 
  
unsigned 
Terminal::currentLine() const
{
  return currentLine_;
}
void
Terminal::create()
{
  if(textChanged_)
    {  
      
      unsigned int max_lines = height()/textFont_->getHeight();
      vector<string> all_lines;
      MultiLineEdit::breakText(text_, all_lines, maxLinesToStore_ + max_lines);
      allLines_ = all_lines.size() - max_lines;

      /* now check for offset
	 offset_ = 0, means same behavior like MultiLineEdit. 
      */
      vector<string> lines;
      if(all_lines.size() > max_lines)
	{
	  if(scrollTo_ != currentLine_)
	    {
	      if(scrollTo_ > all_lines.size() - max_lines)
		scrollTo_ = all_lines.size() - max_lines; 
	    }
	  else
	    {
	      scrollTo_ = all_lines.size() - max_lines; 
	    }
	  
	  if(scrollTo_ != currentLine_)
	    {
	      currentLine_ = scrollTo_;
	      scrolled.emit();
	    }

	  for(unsigned n=currentLine_; n < currentLine_+max_lines; n++)
	    lines.push_back(all_lines[n]);
	}
      else
	{
	  if(currentLine_)
	    {
	      currentLine_ = 0;
	    }
	  lines = all_lines;
	}
      
      MultiLineEdit::renderTextLines(lines);
    }

  if(!justCursor_ || textChanged_)
    {  
      Widget::create();

      Rect dest;
      for(unsigned n=0; n < renderedLines_.size(); n++)
	{
	  if(renderedLines_[n])
	    {
	      dest=Rect(0,textFont_->getHeight()*n,
			renderedLines_[n]->width(), 
			renderedLines_[n]->height());
	      if(fontPal_.size() != 0)
		renderedLines_[n]->setPalette(fontPal_);
	      renderedLines_[n]->blit(surface_,dest);
	    }
	}
      
      int x = 0;
      int y = textFont_->getHeight();
      //move the cursor

      if(renderedLines_.size() > 0)
	{
	  if(renderedLines_[renderedLines_.size()-1])
	    x = renderedLines_[renderedLines_.size()-1]->width();
	  y = textFont_->getHeight()*(renderedLines_.size());
	}

      dirtyRects_.push_back(globalCoord(textCursor_));
      textCursor_.warp(Point(x + 1, y-textCursor_.height())) ;

      //reset flags
      justCursor_ = false;
      textChanged_ = false; 
    }
  
  if(!readOnly_)
    {
      if(justCursor_)
	{
	  dirtyRects_.clear();
	  dirtyRects_.push_back(globalCoord(textCursor_));
	  justCursor_ = false;
	}

      //draw the cursor ...
      if(drawCursor_) 
	drawCursor();
      else
	deleteCursor();
    }
}

void Terminal::scrollTo(int line)
{
  if(line < 0)
    return;

  if((unsigned)line != currentLine_)
    {
      scrollTo_ = line;
      textChanged_ = true;
      needsUpdate_ = true;
    }
}

void Terminal::scrollUp(int lines)
{
  if(lines <= 0 || scrollTo_ == 0)
    return;

  if(currentLine_ < (unsigned)lines)
    scrollTo_ = 0;
  else
    scrollTo_ = currentLine_ - lines;

  textChanged_ = true;
  needsUpdate_ = true;
}
  
void Terminal::scrollDown(int lines)
{  
  if(lines <= 0)
    return;
  scrollTo_ = currentLine_ + lines;
  textChanged_ = true;
  needsUpdate_ = true;
}
 
void Terminal::scroll2Top()
{
  scrollTo_ = 0;
  textChanged_ = true;
  needsUpdate_ = true;
}  

void Terminal::scroll2Bottom()
{
  scrollTo_ = 65000;
  textChanged_ = true;
  needsUpdate_ = true;
}
 
}

