#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <stdio.h>
#include <string.h>
#include <sys/stat.h>

#ifdef HAVE_ERRNO_H
#include <errno.h>
#endif
extern int errno;

#include <unistd.h>

#include "icqpacket.h"
#include "icq-defines.h"
#include "translate.h"


//=====Packet===================================================================

ICQOwner *CPacket::m_pcIcqOwner = NULL;

//=====UDP======================================================================
unsigned short CPacketUdp::m_nSpecialSequence = 1;

CPacketUdp::CPacketUdp(unsigned short _nCommand)
{
   m_nVersion = ICQ_VERSION;
   m_nCommand = _nCommand;
   if (_nCommand != ICQ_CMDxSND_ACK && _nCommand != ICQ_CMDxSND_SYSxMSGxDONExACK) 
      m_nSequence = CPacket::getIcqOwner()->getSequence(true);      
   m_nSourceUin = CPacket::getIcqOwner()->getUin();
   
   buffer = NULL;
}


CPacketUdp::~CPacketUdp(void)
{
   if (buffer != NULL) delete buffer;
}


void CPacketUdp::initBuffer(void)
{
   buffer = new CBuffer(getSize());

   buffer->add(m_nVersion);
   buffer->add(m_nCommand);
   buffer->add(m_nSequence);
   buffer->add(m_nSourceUin);
}


//-----Logon--------------------------------------------------------------------
unsigned long CPU_Logon::getSize(void)
{
   return (CPacketUdp::getSize() + 6 + m_nPasswordLength + 25);
}

CPU_Logon::CPU_Logon(unsigned long _nLocalIP, unsigned short _nLogonStatus) 
   : CPacketUdp(ICQ_CMDxSND_LOGON)
{
   m_nLocalPort = CPacket::getIcqOwner()->tcpSocket.LocalPort();
   m_sPassword = strdup(CPacket::getIcqOwner()->getPassword());
   m_nPasswordLength = strlen(m_sPassword) + 1;
   char temp_1[4] = { 0x72, 0x00, 0x04, 0x00 };
   memcpy(m_aUnknown_1, temp_1, sizeof(m_aUnknown_1));
   m_nLocalIP = NetworkIpToPacketIp(_nLocalIP);
   m_aUnknown_2 = 0x04;
   m_nLogonStatus = _nLogonStatus;
   m_nTcpVersion = ICQ_VERSION_TCP;
   char temp_3[12] = { 0, 0, 2, 0, 0, 0, 0, 0, 4, 0, 0x72, 0 };
   memcpy(m_aUnknown_3, temp_3, 14);
   
   initBuffer();

   buffer->add(m_nLocalPort);
   buffer->add(m_nPasswordLength);
   buffer->add(m_sPassword, m_nPasswordLength);
   buffer->add(m_aUnknown_1, 4);
   buffer->add(m_nLocalIP);
   buffer->add(m_aUnknown_2);
   buffer->add(m_nLogonStatus);
   buffer->add(m_nTcpVersion);
   buffer->add(m_aUnknown_3, 12);
}

CPU_Logon::~CPU_Logon(void)
{
   free (m_sPassword);
}


//-----Ack----------------------------------------------------------------------
unsigned long CPU_Ack::getSize(void) 
{ 
   return (CPacketUdp::getSize()); 
}

CPU_Ack::CPU_Ack(unsigned long _nSequence) : CPacketUdp(ICQ_CMDxSND_ACK)
{
   m_nSequence = _nSequence;

   initBuffer();
}


//-----GetUserBasicInfo---------------------------------------------------------
unsigned short CPU_GetUserBasicInfo::s_nInfoSequence = 0x000A;

unsigned long CPU_GetUserBasicInfo::getSize(void) 
{ 
   return (CPacketUdp::getSize() + 6); 
}

CPU_GetUserBasicInfo::CPU_GetUserBasicInfo(unsigned long _nUserUin) 
   : CPacketUdp(ICQ_CMDxSND_USERxGETINFO)
{
   m_nInfoSequence = s_nInfoSequence++;
   m_nUserUin = _nUserUin;

   initBuffer();

   buffer->add(m_nInfoSequence);
   buffer->add(m_nUserUin);
}


//-----GetUserExtInfo-----------------------------------------------------------
unsigned short CPU_GetUserExtInfo::s_nInfoSequence = 0x000A;

unsigned long CPU_GetUserExtInfo::getSize(void) 
{ 
   return (CPacketUdp::getSize() + 6); 
}

CPU_GetUserExtInfo::CPU_GetUserExtInfo(unsigned long _nUserUin) 
   : CPacketUdp(ICQ_CMDxSND_USERxGETDETAILS)
{
   m_nInfoSequence = s_nInfoSequence++;
   m_nUserUin = _nUserUin;

   initBuffer();

   buffer->add(m_nInfoSequence);
   buffer->add(m_nUserUin);
}


//-----AddUser------------------------------------------------------------------
unsigned long CPU_AddUser::getSize(void)  
{ 
   return (CPacketUdp::getSize() + 4); 
}

CPU_AddUser::CPU_AddUser(unsigned long _nAddedUin) 
   : CPacketUdp(ICQ_CMDxSND_USERxADD)
{
   m_nAddedUin = _nAddedUin;

   initBuffer();
   
   buffer->add(m_nAddedUin);
}


//-----Logoff-------------------------------------------------------------------
unsigned long CPU_Logoff::getSize(void)
{
   return (CPacketUdp::getSize() + 24);
}

CPU_Logoff::CPU_Logoff(void) : CPacketUdp(ICQ_CMDxSND_LOGOFF)
{
   char temp_1[2] = { 0x14, 0x00 };
   memcpy(m_aUnknown_1, temp_1, sizeof(m_aUnknown_1));
   strcpy(m_aDisconnect, "B_USER_DISCONNECTED");
   char temp_2[2] = { 0x05, 0x00 };
   memcpy(m_aUnknown_2, temp_2, sizeof(m_aUnknown_2));
   
   initBuffer();
   
   buffer->add(m_aUnknown_1, 2);
   buffer->add(m_aDisconnect, 20);
   buffer->add(m_aUnknown_2, 2);
}


//-----ContactList--------------------------------------------------------------

unsigned long CPU_ContactList::getSize(void)
{
   return(CPacketUdp::getSize() + 1 + (m_vnUins.size() * sizeof(unsigned long)));
}

CPU_ContactList::CPU_ContactList(CUserGroup *_cUsers, unsigned short first, 
                                 unsigned short num) 
   : CPacketUdp(ICQ_CMDxSND_USERxLIST)
{
   // Go through the entire list of users
   for (unsigned short i = first; i < _cUsers->getNumUsers() 
        && i < first + num; i++) 
      m_vnUins.push_back(_cUsers->getUser(i)->getUin());
   m_nNumUsers = m_vnUins.size();      
   
   initBuffer();

   buffer->add(m_nNumUsers);
   for (unsigned short i  = 0; i < m_vnUins.size(); i++)
      buffer->add(m_vnUins[i]);
}


//-----VisibleList--------------------------------------------------------------
unsigned long CPU_VisibleList::getSize(void)
{
   return(CPacketUdp::getSize() + 1 + (m_vnUins.size() * sizeof(unsigned long)));
}

CPU_VisibleList::CPU_VisibleList(CUserGroup *_cUsers) : CPacketUdp(ICQ_CMDxSND_VISIBLExLIST)
{
   // Go through the entire list of users, checking if each one is on 
   // the visible list
   for (unsigned short i = 0; i < _cUsers->getNumUsers(); i++) 
      if (_cUsers->getUser(i)->getVisibleList() )
         m_vnUins.push_back(_cUsers->getUser(i)->getUin());

   m_nNumUsers = m_vnUins.size();      
   
   initBuffer();

   buffer->add(m_nNumUsers);
   for (unsigned short i  = 0; i < m_vnUins.size(); i++)
      buffer->add(m_vnUins[i]);
}


//-----InvisibleList--------------------------------------------------------------
unsigned long CPU_InvisibleList::getSize(void)
{
   return(CPacketUdp::getSize() + 1 + (m_vnUins.size() * sizeof(unsigned long)));
}

CPU_InvisibleList::CPU_InvisibleList(CUserGroup *_cUsers) : CPacketUdp(ICQ_CMDxSND_INVISIBLExLIST)
{
   // Go through the entire list of users, checking if each one is on 
   // the contact list and the invisible list
   for (unsigned short i = 0; i < _cUsers->getNumUsers(); i++) 
      if (_cUsers->getUser(i)->getInvisibleList() )
          //&& _cUsers->getUserOnContactList(_cUsers->getUser(i)))
         m_vnUins.push_back(_cUsers->getUser(i)->getUin());

   m_nNumUsers = m_vnUins.size();      
   
   initBuffer();

   buffer->add(m_nNumUsers);
   for (unsigned short i  = 0; i < m_vnUins.size(); i++)
      buffer->add(m_vnUins[i]);
}



//-----StartSearch--------------------------------------------------------------
unsigned long CPU_StartSearch::getSize(void)
{
   return (CPacketUdp::getSize() + 10 + m_nAliasLength + m_nFirstNameLength + 
           m_nLastNameLength + m_nEmailLength);
}

CPU_StartSearch::CPU_StartSearch(char *_sAlias, char *_sFirstName, 
                                 char *_sLastName, char *_sEmail, 
                                 unsigned short _nSearchSequence)
   : CPacketUdp(ICQ_CMDxSND_SEARCHxSTART)
{
   m_nSearchSequence = _nSearchSequence;
   m_nAliasLength = strlen(_sAlias) + 1;
   m_sAlias = strdup(_sAlias);
   m_nFirstNameLength = strlen(_sFirstName) + 1;
   m_sFirstName = strdup(_sFirstName);
   m_nLastNameLength = strlen(_sLastName) + 1;
   m_sLastName = strdup(_sLastName);
   m_nEmailLength = strlen(_sEmail) + 1;
   m_sEmail = strdup(_sEmail);
   
   initBuffer();
   
   buffer->add(m_nSearchSequence);
   buffer->add(m_nAliasLength);
   buffer->add(m_sAlias, m_nAliasLength);
   buffer->add(m_nFirstNameLength);
   buffer->add(m_sFirstName, m_nFirstNameLength);
   buffer->add(m_nLastNameLength);
   buffer->add(m_sLastName, m_nLastNameLength);
   buffer->add(m_nEmailLength);
   buffer->add(m_sEmail, m_nEmailLength);
}

CPU_StartSearch::~CPU_StartSearch(void)
{
   free (m_sAlias);
   free (m_sFirstName);
   free (m_sLastName);
   free (m_sEmail);   
}


//-----UpdatePersonalInfo-------------------------------------------------------
unsigned long CPU_UpdatePersonalBasicInfo::getSize(void)
{
   return (CPacketUdp::getSize() + 11 + m_nAliasLength + m_nFirstNameLength + 
           m_nLastNameLength + m_nEmailLength);
}

CPU_UpdatePersonalBasicInfo::CPU_UpdatePersonalBasicInfo(const char *_sAlias, 
                                                         const char *_sFirstName, 
                                                         const char *_sLastName, 
                                                         const char *_sEmail, 
                                                         bool _bAuthorization)
   : CPacketUdp(ICQ_CMDxSND_UPDATExBASIC)
{
   m_nUpdateSequence = m_nSpecialSequence++;
   m_nAliasLength = strlen(_sAlias) + 1;
   m_sAlias = strdup(_sAlias);
   m_nFirstNameLength = strlen(_sFirstName) + 1;
   m_sFirstName = strdup(_sFirstName);
   m_nLastNameLength = strlen(_sLastName) + 1;
   m_sLastName = strdup(_sLastName);
   m_nEmailLength = strlen(_sEmail) + 1;
   m_sEmail = strdup(_sEmail);
   m_nAuthorization = (_bAuthorization ? 0 : 1);  // 0 for require authorization

   initBuffer();
   
   buffer->add(m_nUpdateSequence);
   buffer->add(m_nAliasLength);
   buffer->add(m_sAlias, m_nAliasLength);
   buffer->add(m_nFirstNameLength);
   buffer->add(m_sFirstName, m_nFirstNameLength);
   buffer->add(m_nLastNameLength);
   buffer->add(m_sLastName, m_nLastNameLength);
   buffer->add(m_nEmailLength);
   buffer->add(m_sEmail, m_nEmailLength);
   buffer->add(m_nAuthorization);
}


CPU_UpdatePersonalBasicInfo::~CPU_UpdatePersonalBasicInfo(void)
{
   free (m_sAlias);
   free (m_sFirstName);
   free (m_sLastName);
   free (m_sEmail);
}


//-----UpdatePersonalExtInfo-------------------------------------------------------
unsigned long CPU_UpdatePersonalExtInfo::getSize(void)
{
   return (CPacketUdp::getSize() + m_nCityLength + m_nStateLength + 
           m_nPhoneLength + m_nHomepageLength + m_nAboutLength + 18);
}

CPU_UpdatePersonalExtInfo::CPU_UpdatePersonalExtInfo(const char *_sCity, 
                                                     unsigned short _nCountry,
                                                     const char *_sState,
                                                     unsigned short _nAge,
                                                     char _cSex,
                                                     const char *_sPhone,
                                                     const char *_sHomepage,
                                                     const char *_sAbout)
   : CPacketUdp(ICQ_CMDxSND_UPDATExDETAIL)
{
   
   m_nUpdateSequence = m_nSpecialSequence++;
   m_nCityLength = strlen(_sCity) + 1;
   m_sCity = strdup(_sCity);
   m_nCountry = _nCountry;
   m_cCountryStatus = (m_nCountry == 0xFFFF ? 0xFE : 0x10);
   m_nStateLength = strlen(_sState) + 1;
   m_sState = strdup(_sState);
   if (m_nStateLength > 6)  // State is max 5 characters + NULL
   {
      m_sState[5] = '\0';
      m_nStateLength = 6;
   }
   m_nAge = _nAge;
   m_cSex = _cSex;
   m_nPhoneLength = strlen(_sPhone) + 1;
   m_sPhone = strdup(_sPhone);
   m_nHomepageLength = strlen(_sHomepage) + 1;
   m_sHomepage = strdup(_sHomepage);
   m_nAboutLength = strlen(_sAbout) + 1;
   m_sAbout = strdup(_sAbout);
   
   initBuffer();
   
   buffer->add(m_nUpdateSequence);
   buffer->add(m_nCityLength);
   buffer->add(m_sCity, m_nCityLength);
   buffer->add(m_nCountry);
   buffer->add(m_cCountryStatus);
   buffer->add(m_nStateLength);
   buffer->add(m_sState, m_nStateLength);
   buffer->add(m_nAge);
   buffer->add(m_cSex);
   buffer->add(m_nPhoneLength);
   buffer->add(m_sPhone, m_nPhoneLength);
   buffer->add(m_nHomepageLength);
   buffer->add(m_sHomepage, m_nHomepageLength);
   buffer->add(m_nAboutLength);
   buffer->add(m_sAbout, m_nAboutLength);
}


CPU_UpdatePersonalExtInfo::~CPU_UpdatePersonalExtInfo(void)
{
   free (m_sCity);
   free (m_sState);
   free (m_sPhone);
   free (m_sHomepage);
   free (m_sAbout);
}


//-----Ping---------------------------------------------------------------------
unsigned long CPU_Ping::getSize(void)
{
   return (CPacketUdp::getSize());
}

CPU_Ping::CPU_Ping(void) : CPacketUdp(ICQ_CMDxSND_PING)
{
   initBuffer();
}


//-----ThroughServer------------------------------------------------------------
unsigned long CPU_ThroughServer::getSize(void)
{
   return (CPacketUdp::getSize() + 8 + strlen(m_sMessage) + 1);
}

CPU_ThroughServer::CPU_ThroughServer(unsigned long _nSourceUin,
                                     unsigned long _nDestinationUin, 
                                     unsigned short _nSubCommand,
                                     char *_sMessage) 
   : CPacketUdp(ICQ_CMDxSND_THRUxSERVER)
{
   m_nSourceUin = (_nSourceUin == 0 ? CPacket::getIcqOwner()->getUin() : _nSourceUin);
   m_nDestinationUin = _nDestinationUin;
   m_nSubCommand = _nSubCommand;
   m_nMsgLength = strlen(_sMessage) + 1;
   m_sMessage = strdup(_sMessage);

   initBuffer();

   buffer->add(m_nDestinationUin);
   buffer->add(m_nSubCommand);
   buffer->add(m_nMsgLength);
   buffer->add(m_sMessage, m_nMsgLength);
}

CPU_ThroughServer::~CPU_ThroughServer(void)
{
   free (m_sMessage);
}

//-----SetStatus----------------------------------------------------------------
unsigned long CPU_SetStatus::getSize(void)
{
   return (CPacketUdp::getSize() + 4);
}

CPU_SetStatus::CPU_SetStatus(unsigned long _nNewStatus) : CPacketUdp(ICQ_CMDxSND_SETxSTATUS)
{
   m_nNewStatus = _nNewStatus;

   initBuffer();

   buffer->add(m_nNewStatus);
}

//-----Authorize----------------------------------------------------------------
unsigned long CPU_Authorize::getSize(void)
{
   return (CPacketUdp::getSize() + 9);
}

CPU_Authorize::CPU_Authorize(unsigned long _nAuthorizeUin) : CPacketUdp(ICQ_CMDxSND_AUTHORIZE)
{
   m_nAuthorizeUin = _nAuthorizeUin;
   char temp_1[5] = { 0x08, 0x00, 0x01, 0x00, 0x00 };
   memcpy(m_aUnknown_1, temp_1, sizeof(m_aUnknown_1));

   initBuffer();

   buffer->add(m_nAuthorizeUin);
   buffer->add(m_aUnknown_1, 5);
}


//-----RequestSysMsg------------------------------------------------------------
unsigned long CPU_RequestSysMsg::getSize(void)
{
   return(CPacketUdp::getSize());
}

CPU_RequestSysMsg::CPU_RequestSysMsg(void) : CPacketUdp(ICQ_CMDxSND_SYSxMSGxREQ)
{
   initBuffer();
}


//-----SysMsgDoneAck------------------------------------------------------------
unsigned long CPU_SysMsgDoneAck::getSize(void)
{
   return(CPacketUdp::getSize());
}

CPU_SysMsgDoneAck::CPU_SysMsgDoneAck(unsigned long _nSequence) 
   : CPacketUdp(ICQ_CMDxSND_SYSxMSGxDONExACK)
{
   m_nSequence = _nSequence;

   initBuffer();
}


//=====PacketTcp_Handshake======================================================
CPacketTcp_Handshake::~CPacketTcp_Handshake(void)
{
   if (buffer != NULL) delete buffer;
}


CPacketTcp_Handshake::CPacketTcp_Handshake(unsigned long _nLocalPort)
{
   m_cHandshakeCommand = ICQ_CMDxTCP_HANDSHAKE;
   m_nTcpVersion = ICQ_VERSION_TCP;
   m_nLocalPort = _nLocalPort;
   m_nSourceUin = CPacket::getIcqOwner()->getUin();
   m_nLocalHost = LOCALHOST;
   m_aUnknown_2 = 0x04;
   
   initBuffer();
}


void CPacketTcp_Handshake::initBuffer(void)
{
   buffer = new CBuffer(getSize());

   buffer->add(m_cHandshakeCommand);
   buffer->add(m_nTcpVersion);
   buffer->add(m_nLocalPort);
   buffer->add(m_nSourceUin);
   buffer->add(m_nLocalHost);
   buffer->add(m_nLocalHost);
   buffer->add(m_aUnknown_2);
   buffer->add(m_nLocalPort);
}


//=====PacketTcp================================================================
CPacketTcp::CPacketTcp(unsigned long _nSourceUin, unsigned long _nCommand, 
                       unsigned short _nSubCommand, const char *_sMessage, 
                       bool _bAccept, ICQUser *_cUser)
{   
  // Setup the message type and status fields using our online status
  switch(_nCommand)
  {
  case ICQ_CMDxTCP_CANCEL:
  case ICQ_CMDxTCP_START:
    m_nStatus = 0;
    m_nMsgType = ICQ_TCPxMSG_NORMAL;
    switch (CPacket::getIcqOwner()->getStatus()) 
    {
    case ICQ_STATUS_AWAY: m_nMsgType |= ICQ_TCPxMSG_FxAWAY; break;
    case ICQ_STATUS_NA: m_nMsgType |= ICQ_TCPxMSG_FxNA; break;
    case ICQ_STATUS_DND: m_nMsgType |= ICQ_TCPxMSG_FxDND; break;
    case ICQ_STATUS_OCCUPIED: m_nMsgType |= ICQ_TCPxMSG_FxOCCUPIED; break;
    case ICQ_STATUS_ONLINE:
    case ICQ_STATUS_FREEFORCHAT: 
    default: m_nMsgType |= ICQ_TCPxMSG_FxONLINE; break;
    }
    if (CPacket::getIcqOwner()->getStatusInvisible())
      m_nMsgType |= ICQ_TCPxMSG_FxINVISIBLE;
    break;

  case ICQ_CMDxTCP_ACK:
    m_nMsgType = ICQ_TCPxMSG_AUTOxREPLY;
    switch (CPacket::getIcqOwner()->getStatus()) 
    {
    case ICQ_STATUS_AWAY: m_nStatus = ICQ_TCPxACK_AWAY; break;
    case ICQ_STATUS_NA: m_nStatus = ICQ_TCPxACK_NA; break;
    case ICQ_STATUS_DND: //m_nStatus = ICQ_TCPxACK_DND; break;
    case ICQ_STATUS_OCCUPIED: //m_nStatus = ICQ_TCPxACK_OCCUPIED; break;
    case ICQ_STATUS_ONLINE: 
    case ICQ_STATUS_FREEFORCHAT: 
    default: m_nStatus = ICQ_TCPxACK_ONLINE; break;
    }
    if (!_bAccept) m_nStatus = ICQ_TCPxACK_REFUSE;
    break;
  }
  
  m_nSourceUin = (_nSourceUin == 0 ? CPacket::getIcqOwner()->getUin() : _nSourceUin);
  m_nTcpVersion = ICQ_VERSION_TCP;
  m_nCommand = _nCommand;
  m_nSubCommand = _nSubCommand;
  m_nMsgLength = strlen(_sMessage) + 1;
  m_sMessage = strdup(_sMessage);
  m_nLocalIP = NetworkIpToPacketIp(_cUser->tcpSocket.LocalIp());
  m_nLocalHost = LOCALHOST;
  m_nLocalPort = _cUser->tcpSocket.LocalPort();
  m_aUnknown_1 = 0x04;
  //m_nStatus = m_nStatus;
  //m_nMsgType = m_nMsgType;

  m_sLicqTag = 'L';
  m_nLicqVersion = INT_VERSION;

  // don't increment the sequence if this is an ack and cancel packet
  if (m_nCommand == ICQ_CMDxTCP_START) m_nSequence = _cUser->getSequence(true);
  
  buffer = NULL;
}

CPacketTcp::~CPacketTcp(void)
{
   free (m_sMessage);
   if (buffer != NULL) delete buffer;
}

void CPacketTcp::initBuffer(void)
{
   buffer = new CBuffer(getSize());

   buffer->add(m_nSourceUin);
   buffer->add(m_nTcpVersion);
   buffer->add(m_nCommand);
   buffer->add(m_nSourceUin);
   buffer->add(m_nSubCommand);
   buffer->add(m_nMsgLength);
   buffer->add(m_sMessage, m_nMsgLength);
   buffer->add(m_nLocalIP);
   buffer->add(m_nLocalHost);
   buffer->add(m_nLocalPort);
   buffer->add(m_aUnknown_1);
   buffer->add(m_nStatus);
   buffer->add(m_nMsgType);
}

void CPacketTcp::postBuffer(void)
{
   buffer->add(m_nSequence);
   buffer->add(m_sLicqTag);
   buffer->add(m_nLicqVersion);
}

unsigned long CPacketTcp::getSize(void)
{
   return (18 + m_nMsgLength + 24);
}

//-----Message------------------------------------------------------------------
unsigned long CPT_Message::getSize(void)
{
   return (CPacketTcp::getSize());
}

CPT_Message::CPT_Message(unsigned long _nSourceUin, char *_sMessage, 
                         ICQUser *_cUser) 
   : CPacketTcp(_nSourceUin, ICQ_CMDxTCP_START, ICQ_CMDxSUB_MSG, _sMessage, 
                true, _cUser)
{
   initBuffer();
   postBuffer();
}


//-----Url----------------------------------------------------------------------
unsigned long CPT_Url::getSize(void)
{
   return (CPacketTcp::getSize());
}

CPT_Url::CPT_Url(unsigned long _nSourceUin, char *_sMessage, ICQUser *_cUser) 
   : CPacketTcp(_nSourceUin, ICQ_CMDxTCP_START, ICQ_CMDxSUB_URL, _sMessage, 
                true, _cUser)
{
   initBuffer();
   postBuffer();
}


//-----ReadAwayMessage----------------------------------------------------------
unsigned long CPT_ReadAwayMessage::getSize(void)
{
   return (CPacketTcp::getSize());
}

CPT_ReadAwayMessage::CPT_ReadAwayMessage(unsigned long _nSourceUin, ICQUser *_cUser) 
   : CPacketTcp(_nSourceUin, ICQ_CMDxTCP_START, ICQ_CMDxTCP_READxAWAYxMSG, "", 
                true, _cUser)
{
   // Properly set the subcommand to get the correct away message
   switch(_cUser->getStatus())
   {
   case ICQ_STATUS_AWAY: m_nSubCommand = ICQ_CMDxTCP_READxAWAYxMSG; break;
   case ICQ_STATUS_NA: m_nSubCommand = ICQ_CMDxTCP_READxNAxMSG; break;
   case ICQ_STATUS_DND: m_nSubCommand = ICQ_CMDxTCP_READxDNDxMSG; break;
   case ICQ_STATUS_OCCUPIED: m_nSubCommand = ICQ_CMDxTCP_READxOCCUPIEDxMSG; break;
   default: m_nSubCommand = ICQ_CMDxTCP_READxAWAYxMSG; break;
   }
   initBuffer();
   postBuffer();
}


//-----ChatRequest--------------------------------------------------------------
unsigned long CPT_ChatRequest::getSize(void)
{
   return (CPacketTcp::getSize() + 11);
}

CPT_ChatRequest::CPT_ChatRequest(unsigned long _nSourceUin, char *_sMessage, 
                                 ICQUser *_cUser) 
   : CPacketTcp(_nSourceUin, ICQ_CMDxTCP_START, ICQ_CMDxSUB_CHAT, _sMessage, 
                true, _cUser)
{
   char temp_1[11] = { 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0};
   memcpy(m_aUnknown_1, temp_1, 11);

   initBuffer();

   buffer->add(m_aUnknown_1, 11);
   
   postBuffer();
}


//-----FileTransfer--------------------------------------------------------------
unsigned long CPT_FileTransfer::getSize(void)
{
   return (CPacketTcp::getSize() + 14 + m_nFilenameLength);
}

CPT_FileTransfer::CPT_FileTransfer(unsigned long _nSourceUin, const char *_szFilename, 
                                   const char *_szDescription, ICQUser *_cUser)                                 
   : CPacketTcp(_nSourceUin, ICQ_CMDxTCP_START, ICQ_CMDxSUB_FILE, _szDescription, 
                true, _cUser)
{
   m_bValid = true;
   
   // Check file exists and get size
   struct stat buf;
   if (stat(_szFilename, &buf) < 0)
   {
      m_bValid = false;
      return;
   }
   m_nFileSize = buf.st_size;

   // Remove path from filename (if it exists)
   char *pcEndOfPath = strrchr(_szFilename, '/');
   if (pcEndOfPath != NULL) 
      m_szFilename = strdup(pcEndOfPath + 1);
   else
      m_szFilename = strdup(_szFilename);
   
   m_nUnknown_1 = 0;
   m_nFilenameLength = strlen(m_szFilename) + 1;
   // m_szFilename set above
   // m_nFileSize set above
   m_nUnknown_2 = 0;

   initBuffer();

   buffer->add(m_nUnknown_1);
   buffer->add(m_nFilenameLength);
   buffer->add(m_szFilename, m_nFilenameLength);
   buffer->add(m_nFileSize);
   buffer->add(m_nUnknown_2);
   
   postBuffer();
}


//+++++Ack++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
unsigned long CPT_Ack::getSize(void)
{
   return (CPacketTcp::getSize());
}

CPT_Ack::CPT_Ack(unsigned short _nSubCommand, unsigned long _nSequence, 
                 bool _bAccept, ICQUser *_cUser)
   : CPacketTcp(0, ICQ_CMDxTCP_ACK, _nSubCommand, 
                CPacket::getIcqOwner()->getAwayMessage(), _bAccept, _cUser)
{
   m_nSequence = _nSequence;
   gTranslator.ClientToServer(m_sMessage);
}


//-----AckMessage---------------------------------------------------------------
unsigned long CPT_AckMessage::getSize(void)
{
   return (CPT_Ack::getSize());
}

CPT_AckMessage::CPT_AckMessage(unsigned long _nSequence, bool _bAccept, 
                               ICQUser *_cUser)
   : CPT_Ack(ICQ_CMDxSUB_MSG, _nSequence, _bAccept, _cUser)
{
   initBuffer();
   postBuffer();
}


//-----AckReadAwayMessage-------------------------------------------------------
unsigned long CPT_AckReadAwayMessage::getSize(void)
{
   return (CPT_Ack::getSize());
}

CPT_AckReadAwayMessage::CPT_AckReadAwayMessage(unsigned short _nSubCommand, 
                                               unsigned long _nSequence, 
                                               bool _bAccept, ICQUser *_cUser)
   : CPT_Ack(_nSubCommand, _nSequence, _bAccept, _cUser)
{
   initBuffer();
   postBuffer();
}


//-----AckUrl-------------------------------------------------------------------
unsigned long CPT_AckUrl::getSize(void)
{
   return (CPT_Ack::getSize());
}

CPT_AckUrl::CPT_AckUrl(unsigned long _nSequence, bool _bAccept, ICQUser *_cUser)
   : CPT_Ack(ICQ_CMDxSUB_URL, _nSequence, _bAccept, _cUser)
{
   initBuffer();
   postBuffer();
}


//-----AckChatRefuse------------------------------------------------------------
unsigned long CPT_AckChatRefuse::getSize(void)
{
   return (CPT_Ack::getSize() + 11);
}

CPT_AckChatRefuse::CPT_AckChatRefuse(const char *_sReason, 
                                     unsigned long _nSequence, ICQUser *_cUser)
   : CPT_Ack(ICQ_CMDxSUB_CHAT, _nSequence, false, _cUser)
{
   free (m_sMessage);
   m_nMsgLength = strlen(_sReason) + 1;
   m_sMessage = strdup(_sReason);
   char temp_1[11] = { 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0};
   memcpy(m_aUnknown_1, temp_1, 11);

   initBuffer();

   buffer->add(m_aUnknown_1, 11);
   
   postBuffer();
}


//-----AckChatAccept------------------------------------------------------------
unsigned long CPT_AckChatAccept::getSize(void)
{
   return (CPT_Ack::getSize() + 11);
}

CPT_AckChatAccept::CPT_AckChatAccept(unsigned short _nPort, 
                                     unsigned long _nSequence, ICQUser *_cUser)
   : CPT_Ack(ICQ_CMDxSUB_CHAT, _nSequence, true, _cUser)
{
   char temp_1[3] = { 1, 0, 0 };
   memcpy(m_aUnknown_1, temp_1, sizeof(m_aUnknown_1));
   m_nPortReversed = ((_nPort & 0xFF) << 8) + ((_nPort >> 8) & 0xFF);
   m_nPort = _nPort;
   m_nStatus = ICQ_TCPxACK_ONLINE;
   
   initBuffer();

   buffer->add(m_aUnknown_1, sizeof(m_aUnknown_1));
   buffer->add(m_nPortReversed);
   buffer->add(m_nPort);
   
   postBuffer();
}


//-----AckFileRefuse------------------------------------------------------------
unsigned long CPT_AckFileRefuse::getSize(void)
{
   return (CPT_Ack::getSize() + 15);
}

CPT_AckFileRefuse::CPT_AckFileRefuse(const char *_sReason, 
                                     unsigned long _nSequence, ICQUser *_cUser)
   : CPT_Ack(ICQ_CMDxSUB_FILE, _nSequence, false, _cUser)
{
   free (m_sMessage);
   m_nMsgLength = strlen(_sReason) + 1;
   m_sMessage = strdup(_sReason);
   m_nUnknown_1 = 0;
   m_nStrLen = 1;
   m_cEmptyStr = '\0';
   m_nUnknown_2 = 0;
   m_nUnknown_3 = 0;

   initBuffer();

   buffer->add(m_nUnknown_1);
   buffer->add(m_nStrLen);
   buffer->add(m_cEmptyStr);
   buffer->add(m_nUnknown_2);
   buffer->add(m_nUnknown_3);
   
   postBuffer();
}



//-----AckFileAccept------------------------------------------------------------
unsigned long CPT_AckFileAccept::getSize(void)
{
   return (CPT_Ack::getSize() + 15);
}

CPT_AckFileAccept::CPT_AckFileAccept(unsigned short _nPort, 
                                     unsigned long _nSequence, ICQUser *_cUser)
   : CPT_Ack(ICQ_CMDxSUB_FILE, _nSequence, true, _cUser)
{
   m_nPortReversed = ((_nPort & 0xFF) << 8) + ((_nPort >> 8) & 0xFF);
   m_nStrLength = 1;
   m_cEmptyStr = '\0';
   m_nFileSize = 0;  // not used in the ack
   m_nPort = _nPort;
   m_nStatus = ICQ_TCPxACK_ONLINE;
   
   initBuffer();

   buffer->add(m_nPortReversed);
   buffer->add(m_nStrLength);
   buffer->add(m_cEmptyStr);
   buffer->add(m_nFileSize);
   buffer->add(m_nPort);
   
   postBuffer();
}


//+++++Cancel+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
unsigned long CPT_Cancel::getSize(void)
{
   return (CPacketTcp::getSize());
}

CPT_Cancel::CPT_Cancel(unsigned short _nSubCommand, unsigned long _nSequence,
                       ICQUser *_cUser)
   : CPacketTcp(0, ICQ_CMDxTCP_CANCEL, _nSubCommand, "", true, _cUser)
{
   m_nSequence = _nSequence;
}



//-----CancelChat---------------------------------------------------------------
unsigned long CPT_CancelChat::getSize(void)
{
   return (CPT_Cancel::getSize() + 11);
}

CPT_CancelChat::CPT_CancelChat(unsigned long _nSequence, ICQUser *_cUser)
   : CPT_Cancel(ICQ_CMDxSUB_CHAT, _nSequence, _cUser)
{
   char temp_1[11] = { 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0};
   memcpy(m_aUnknown_1, temp_1, 11);

   initBuffer();

   buffer->add(m_aUnknown_1, 11);

   postBuffer();
}


//-----CancelFile---------------------------------------------------------------
unsigned long CPT_CancelFile::getSize(void)
{
   return (CPT_Cancel::getSize() + 15);
}

CPT_CancelFile::CPT_CancelFile(unsigned long _nSequence, ICQUser *_cUser)
   : CPT_Cancel(ICQ_CMDxSUB_FILE, _nSequence, _cUser)
{
   m_nUnknown_1 = 0;
   m_nStrLen = 1;
   m_cEmptyStr = '\0';
   m_nUnknown_2 = 0;
   m_nUnknown_3 = 0;

   initBuffer();

   buffer->add(m_nUnknown_1);
   buffer->add(m_nStrLen);
   buffer->add(m_cEmptyStr);
   buffer->add(m_nUnknown_2);
   buffer->add(m_nUnknown_3);
   
   postBuffer();
}


//=====Chat=====================================================================
void CPacketChat::initBuffer(void)
{
   buffer = new CBuffer(getSize());
}

//-----ChatColor----------------------------------------------------------------
CPChat_Color::CPChat_Color(char *_sLocalName, unsigned short _nLocalPort,
                           unsigned long _nColorForeground,
                           unsigned long _nColorBackground) 
{
   m_nCommand = 0x64;
   char temp_1[4] = { 0xFD, 0xFF, 0xFF, 0xFF };
   memcpy(m_aUnknown_1, temp_1, sizeof(m_aUnknown_1));
   m_nSourceUin = CPacket::getIcqOwner()->getUin();
   m_nLocalNameLength = strlen(_sLocalName) + 1;
   m_sLocalName = strdup(_sLocalName);
   m_aLocalPortReversed[0] = ((char *)(&_nLocalPort))[1];
   m_aLocalPortReversed[1] = ((char *)(&_nLocalPort))[0];
   m_nColorForeground = _nColorForeground;
   m_nColorBackground = _nColorBackground; 
   m_aUnknown_2 = '\0';

   initBuffer();

   buffer->add(m_nCommand);
   buffer->add(m_aUnknown_1, sizeof(m_aUnknown_1));
   buffer->add(m_nSourceUin);
   buffer->add(m_nLocalNameLength);
   buffer->add(m_sLocalName, m_nLocalNameLength);
   buffer->add(m_aLocalPortReversed, sizeof(m_aLocalPortReversed));
   buffer->add(m_nColorForeground);
   buffer->add(m_nColorBackground);
   buffer->add(m_aUnknown_2);
   
}


//-----ChatColorFont----------------------------------------------------------------
CPChat_ColorFont::CPChat_ColorFont(char *_sLocalName, unsigned short _nLocalPort,
                                   unsigned long _nColorForeground,
                                   unsigned long _nColorBackground,                    
                                   unsigned long _nFontSize, 
                                   unsigned long _nFontFace, char *_sFontName)
{
   m_nCommand = 0x64;
   m_nSourceUin = CPacket::getIcqOwner()->getUin();
   m_nLocalNameLength = strlen(_sLocalName) + 1;
   m_sLocalName = strdup(_sLocalName);
   m_nColorForeground = _nColorForeground;
   m_nColorBackground = _nColorBackground; 
   m_nUnknown_1 = 0x00000003;
   m_nLocalPort = _nLocalPort;
   m_nLocalHost = LOCALHOST;
   m_aUnknown_2 = 0x04;
   m_nUnknown_Port = 0x5A75;
   m_nFontSize = _nFontSize;
   m_nFontFace = _nFontFace;
   m_nFontNameLength = strlen(_sFontName) + 1;
   m_sFontName = strdup(_sFontName);
   char temp_3[3] = { 0, 0, 0 };
   memcpy(m_aUnknown_3, temp_3, sizeof(m_aUnknown_3));
   
   initBuffer();

   buffer->add(m_nCommand);
   buffer->add(m_nSourceUin);
   buffer->add(m_nLocalNameLength);
   buffer->add(m_sLocalName, m_nLocalNameLength);
   buffer->add(m_nColorForeground);
   buffer->add(m_nColorBackground);
   buffer->add(m_nUnknown_1);
   buffer->add(m_nLocalPort);
   buffer->add(m_nLocalHost);
   buffer->add(m_nLocalHost);
   buffer->add(m_aUnknown_2);
   buffer->add(m_nUnknown_Port);
   buffer->add(m_nFontSize);
   buffer->add(m_nFontFace);
   buffer->add(m_nFontNameLength);
   buffer->add(m_sFontName, m_nFontNameLength);
   buffer->add(m_aUnknown_3, sizeof(m_aUnknown_3));
}



//-----ChatFont---------------------------------------------------------------------
CPChat_Font::CPChat_Font(unsigned short _nLocalPort, unsigned long _nFontSize, 
                         unsigned long _nFontFace, char *_sFontName)
{
   m_nCommand = 0x03;
   m_nLocalPort = _nLocalPort;
   m_nLocalHost = LOCALHOST;
   m_aUnknown_2 = 0x04;
   m_nUnknown_Port = 0x5A75;
   m_nFontSize = _nFontSize;
   m_nFontFace = _nFontFace;
   m_nFontNameLength = strlen(_sFontName) + 1;
   m_sFontName = strdup(_sFontName);
   char temp_3[3] = { 0, 0, 0 };
   memcpy(m_aUnknown_3, temp_3, sizeof(m_aUnknown_3));
   
   initBuffer();

   buffer->add(m_nCommand);
   buffer->add(m_nLocalPort);
   buffer->add(m_nLocalHost);
   buffer->add(m_nLocalHost);
   buffer->add(m_aUnknown_2);
   buffer->add(m_nUnknown_Port);
   buffer->add(m_nFontSize);
   buffer->add(m_nFontFace);
   buffer->add(m_nFontNameLength);
   buffer->add(m_sFontName, m_nFontNameLength);
   buffer->add(m_aUnknown_3, sizeof(m_aUnknown_3));
}


//-----FileInitClient-----------------------------------------------------------
CPFile_InitClient::CPFile_InitClient(char *_szLocalName,
                                     unsigned long _nNumFiles, 
                                     unsigned long _nTotalSize)
{
   m_cUnknown1 = 0;
   m_nUnknown2 = 0;
   m_nNumFiles = _nNumFiles;
   m_nTotalSize = _nTotalSize;
   m_nUnknown3 = 0x64;   
   m_nLocalNameLength = strlen(_szLocalName) + 1;
   m_szLocalName = strdup(_szLocalName);

   initBuffer();

   buffer->add(m_cUnknown1);
   buffer->add(m_nUnknown2);
   buffer->add(m_nNumFiles);
   buffer->add(m_nTotalSize);
   buffer->add(m_nUnknown3);
   buffer->add(m_nLocalNameLength);
   buffer->add(m_szLocalName, m_nLocalNameLength);
}


//-----FileInitServer-----------------------------------------------------------
CPFile_InitServer::CPFile_InitServer(char *_szLocalName)
{
   m_cUnknown1 = 1;
   m_nUnknown2 = 0x64;
   m_nLocalNameLength = strlen(_szLocalName) + 1;
   m_szLocalName = strdup(_szLocalName);

   initBuffer();

   buffer->add(m_cUnknown1);
   buffer->add(m_nUnknown2);
   buffer->add(m_nLocalNameLength);
   buffer->add(m_szLocalName, m_nLocalNameLength);
}


//-----FileBatch----------------------------------------------------------------
CPFile_Batch::CPFile_Batch(vector <const char *> _vszFileInfo)
{
   m_bValid = true;
   m_nError = 0;
   m_nNumFilesPlusOne = _vszFileInfo.size() + 1;
   
   char *pcNoPath = NULL;
   struct stat buf;
   for (unsigned short i = 0; i < _vszFileInfo.size(); i++) 
   {
      // Remove any path from the filename
      if ( (pcNoPath = strrchr(_vszFileInfo[i], '/')) != NULL) 
         m_vszFileNames.push_back(strdup(pcNoPath + 1));
      else
         m_vszFileNames.push_back(strdup(_vszFileInfo[i]));

      if (stat(_vszFileInfo[i], &buf) < 0)
      {
         m_bValid = false;
         m_nError = errno;
         return;
      }
      m_vnFileSizes.push_back(buf.st_size);
   }
   m_nUnknown1 = 0x64;
   
   initBuffer();

   buffer->add(m_nNumFilesPlusOne);
   
   // Add all the file names
   unsigned short nFilenameLen;
   for (unsigned short i = 0; i < m_vszFileNames.size(); i++) 
   {
      nFilenameLen = strlen(m_vszFileNames[i]) + 1;
      buffer->add(nFilenameLen);
      buffer->add(m_vszFileNames[i], nFilenameLen);
   }
   // Add the empty file name
   buffer->add((unsigned short)1);
   buffer->add((char)0);
   
   //Add the file lengths
   for (unsigned short i = 0; i < m_vnFileSizes.size(); i++) 
      buffer->add(m_vnFileSizes[i]);
   // Add the 0 length file
   buffer->add((unsigned long)0);
   
   buffer->add(m_nUnknown1);
}


unsigned long CPFile_Batch::getSize(void)
{ 
   // Add the length of each file name + 1 for the terminating nulll + 2 for 
   // the short representing the length of the file name + 4 for the file size
   unsigned short n = 0;
   for (unsigned short i = 0; i < m_vszFileNames.size(); i++) 
      n += strlen(m_vszFileNames[i]) + 1 + 2 + 4;
   return (CPacketFile::getSize() + 13 + n); 
}


CPFile_Batch::~CPFile_Batch(void)
{
   for (unsigned short i = 0; i < m_vszFileNames.size(); i++)
      free (m_vszFileNames[i]);
}


//-----FileStart----------------------------------------------------------------
CPFile_Start::CPFile_Start(unsigned long _nFilePos)
{
   m_cUnknown1 = 3;
   m_nFilePos = _nFilePos;
   m_nUnknown2 = 0;
   m_nUnknown3 = 0x64;

   initBuffer();

   buffer->add(m_cUnknown1);
   buffer->add(m_nFilePos);
   buffer->add(m_nUnknown2);
   buffer->add(m_nUnknown3);
}

