#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <stdlib.h>
#include <string.h>
#include <dirent.h>

#ifdef HAVE_MALLOC_H
#include <malloc.h>
#endif

#include <ctype.h>

#ifdef HAVE_ERRNO_H
#include <errno.h>
#else
extern int errno;
#endif

#include "plugin.h"
#include "log.h"

CPluginManager gPluginManager;

//=====CPluginManager===========================================================
int SelectPlugin(const struct dirent *d)
{
  char *pcDot = strrchr(d->d_name, '.');
  if (pcDot == NULL) return (0);
  return (strcmp(pcDot, ".plugin") == 0);
}

CPluginManager::CPluginManager(void)
{
  // does nothing for now
}

unsigned short CPluginManager::LoadPlugins(const char *_szDir)
{
  struct dirent **namelist;
  
  int n = scandir(_szDir, &namelist, SelectPlugin, alphasort);
  if (n < 0)
  {
    gLog.Error("%sError reading plugin directory \"%s\":\n%s%s.\n", L_ERRORxSTR,
              _szDir, L_BLANKxSTR, strerror(errno));
    return (0);
  }

  CPlugin *p;
  for (unsigned short i = 0; i < n; i++)
  {
    char szFile[MAX_FILENAME_LEN];
    sprintf(szFile, "%s%s", _szDir, namelist[i]->d_name);
    free (namelist[i]);
    p = new CPlugin(szFile);
    if (p->Exception())
    {
      gLog.Warn("%sWarning: unable to load plugin \"%s\".\n", L_WARNxSTR, namelist[i]->d_name);
      continue;
    }
    m_vxPlugins.push_back(p);    
  }
  free(namelist);

  return m_vxPlugins.size();
}


//=====CPlugin==================================================================
CPlugin::CPlugin(const char *_szFileName)
{
  // Assumes the given filename is in the form <directory>/<pluginname>.plugin
  bException = false;
  CIniFile fPlugin(INI_FxWARN);
  if (!fPlugin.LoadFile(_szFileName))
  {
    bException = true;
    return;
  }

  fPlugin.SetSection("plugin");
  char szTemp[MAX_LINE_LEN];
  
  // Read in the window
  fPlugin.ReadStr("Window", szTemp, "GUI");
  if (strcmp(szTemp, "GUI") == 0)
    m_eWinType = PluginWinGui;
  else if (strcmp(szTemp, "TERM") == 0)
    m_eWinType = PluginWinTerm;
  else if (strcmp(szTemp, "LICQ") == 0)
    m_eWinType = PluginWinLicq;
  else
  {
    gLog.Warn("%sWarning: Invalid entry in plugin \"%s\":\nWindow = %s\n", 
              L_WARNxSTR, _szFileName, szTemp);
    bException = true;
    return;
  }

  // Read in the command
  if (!fPlugin.ReadStr("Command", szTemp))
  {
    bException = true;
    return;
  }
  m_szCommand = strdup(szTemp);
  m_szFullCommand[0] = '\0';

  // Parse command for %# user fields
  char *pcField = m_szCommand;
  int nField, nCurField = 1;
  while ((pcField = strchr(pcField, '%')) != NULL)
  {
    char cField = *(pcField + 1);
    if (isdigit(cField))
    {
      nField = cField - '0';
      if (nField == 0 || nField > nCurField)
      {
        gLog.Warn("%sWarning: Out-of-order user field id (%d) in plugin \"%s\".\n",
                  L_WARNxSTR, nField, _szFileName);
      }
      else if (nField == nCurField)
      {
        char szTitle[MAX_LINE_LEN], szDefault[MAX_LINE_LEN];
        sprintf(szTemp, "User%d.Title", nField);
        fPlugin.ReadStr(szTemp, szTitle, "User field");
        sprintf(szTemp, "User%d.Default", nField);
        fPlugin.ReadStr(szTemp, szDefault, "");
        m_vxUserField.push_back(new CPluginUserField(szTitle, szDefault));
        nCurField = nField + 1;
      }
    }
    pcField++;
    if (*pcField == '\0') break;
    pcField++;
  }

  strcpy(szTemp, _szFileName);
  // Replace the terminating .plugin by '\0'plugin
  pcField = strrchr(szTemp, '.');
  if (pcField != NULL) *pcField = '\0';
  // Find the beginning of the plugin name
  pcField = strrchr(szTemp, '/');
  if (pcField == NULL)
    pcField = szTemp;
  else
    pcField++;

  m_szName = strdup(pcField);
}


void CPlugin::SetFields(ICQUser *u)
{
  struct UserBasicInfo sUBI;
  u->getBasicInfo(sUBI);
  struct UserExtInfo sUEI;
  u->getExtInfo(sUEI);

  m_szFullCommand[0] = '\0';
  unsigned short nLen = strlen(m_szCommand), i = 0;
  while(i < nLen)
  {
    if (m_szCommand[i] == '%')
    {
      i++;
      if (isdigit(m_szCommand[i]))
      {
        strncat(m_szFullCommand, &m_szCommand[i - 1], 2);
        i++;
        continue;
      }
      switch(m_szCommand[i])
      {
      case 'i':
        strcat(m_szFullCommand, sUBI.ip);
        break;
      case 'p':
        strcat(m_szFullCommand, sUBI.port);
        break;
      case 'e':
        strcat(m_szFullCommand, sUBI.email);
        break;
      case 'n':
        strcat(m_szFullCommand, sUBI.name);
        break;
      case 'f':
        strcat(m_szFullCommand, sUBI.firstname);
        break;
      case 'l':
        strcat(m_szFullCommand, sUBI.lastname);
        break;
      case 'a':
        strcat(m_szFullCommand, sUBI.alias);
        break;
      case 'u':
        strcat(m_szFullCommand, sUBI.uin);
        break;
      case 'w':
        strcat(m_szFullCommand, sUEI.homepage);
        break;
      case 'h':
        strcat(m_szFullCommand, sUEI.phone);
        break;
      default:
        gLog.Warn("%sWarning: Invalid qualifier in command: %%%c.\n", 
                  L_WARNxSTR, m_szCommand[i]);
        break;
      }
      i++;
    }
    else
    {
      strncat(m_szFullCommand, &m_szCommand[i], 1);
      i++;
    }
  }
}


void CPlugin::SetUserFields(vector <const char *> &_vszUserFields)
{
  if ( _vszUserFields.size() != NumUserFields())
  {
    gLog.Warn("%sInternal error: CPlugin::SetUserFields(): incorrect number of data fields (%d/%d).\n", 
              L_WARNxSTR, _vszUserFields.size(), NumUserFields());
    return;
  }
  // Do a quick check to see if there are any users fields at all
  if (NumUserFields() == 0) return;

  char *szTemp = strdup(m_szFullCommand);
  char *pcFieldStart = szTemp, *pcFieldEnd;
  m_szFullCommand[0] = '\0';
  while ((pcFieldEnd = strchr(pcFieldStart, '%')) != NULL)
  {
    *pcFieldEnd = '\0';
    pcFieldEnd++;
    strcat(m_szFullCommand, pcFieldStart);
    // Anything non-digit at this point we just ignore
    if (isdigit(pcFieldEnd[0]))
    {
      unsigned short nField = *pcFieldEnd - '0';
      // We know that any user field numbers are valid from the constructor
      strcat(m_szFullCommand, _vszUserFields[nField - 1]);
    }
    pcFieldStart = pcFieldEnd;
    if (pcFieldStart == '\0') break;
    pcFieldStart++;
  }
  strcat(m_szFullCommand, pcFieldStart);
  strcat(m_szFullCommand, " &");
  free(szTemp);
}
