#include "generate_menu.h"
#include <QDir>
#include<QRegExp>
#include <QTextStream>
#include <QClipboard>

GenerateMenu::GenerateMenu(MainWindow *mainwindow, Operations *operations):QObject(mainwindow)
{
	this->mainwindow=mainwindow;
	this->operations=operations;
}

void GenerateMenu::setPath(QString path)
{
	this->path=path;
}

QString GenerateMenu::menu_path(QString file_path)
{
	if(file_path.startsWith(path))
	{
		QString r(file_path);
		r.remove(0,path.size());
		return r;
	}
	return file_path;
}

QString GenerateMenu::find_icon(QString file_path, QString menu)
{
	QDir dir(file_path);
	if(!dir.exists())
	{
		printf("Error: %s not found\n", path.toLocal8Bit().data() );
		return QString();
	}
	QFileInfoList list=dir.entryInfoList( QDir::NoFilter, QDir::Name );
	for (int i = 0; i < list.size(); ++i)
	{
		QFileInfo fileInfo = list.at(i);
		QString file_name=fileInfo.fileName();
		if(!fileInfo.isDir() && !file_name.startsWith(".") && file_name.endsWith(".png"))
		{
			if(  (menu+QString(".png")) == file_name ) return fileInfo.filePath();
		}
	}
	
	return QString();
}

void GenerateMenu::load_menu()
{
	load_menu(path,NULL);
}

void GenerateMenu::load_menu(QString dir_name, QMenu *parent_menu)
{
	QDir dir(dir_name);
	if(!dir.exists())
	{
		printf("Error: %s not found\n", path.toLocal8Bit().data());
		return;
	}
	QFileInfoList list=dir.entryInfoList( QDir::NoFilter, QDir::Name );
	for (int i = 0; i < list.size(); ++i) {
		QFileInfo fileInfo = list.at(i);
		QString menu_name=fileInfo.fileName();
		//printf("Procesando: %s\n",menu_name.toLocal8Bit().data());
		if(fileInfo.isDir() && !menu_name.startsWith("."))
		{
			//printf("Icono encontrado %s\n",find_icon(dir_name,menu_name).toLocal8Bit().data());
			//printf("Menu path %s\n",menu_path(dir_name+"/"+menu_name).toLocal8Bit().data());
			QMenu *menu=mainwindow->createMenu(menu_path(dir_name+"/"+menu_name),menu_name, find_icon(dir_name,menu_name));
			if(parent_menu==NULL) mainwindow->menubar->addAction(menu->menuAction());
			else parent_menu->addAction(menu->menuAction());
			load_menu(dir_name+"/"+menu_name,menu);
		}
		else if(fileInfo.isExecutable() && !menu_name.startsWith("."))
		{
			QAction *action = mainwindow->createAction(menu_path(dir_name+"/"+menu_name),menu_name, find_icon(dir_name,menu_name));
			mainwindow->menus[menu_path(dir_name)]->addAction(action);
			MenuCallBack *callback=new MenuCallBack(dir_name+"/"+menu_name, mainwindow->octave_connection);
			connect(action, SIGNAL(triggered()), callback, SLOT(actions_callback()));
		}
		else if(!fileInfo.isExecutable()  && !menu_name.startsWith(".") && menu_name.endsWith(".menu"))
		{
			QStringList input_parameters, output_parameters, output_labels, input_labels;
			QString command, _menu_name, help;
			bool accept_blank_parameters;
			
			if( process_menu_file(fileInfo.filePath(), input_labels, input_parameters, output_labels, output_parameters, command, _menu_name, help, accept_blank_parameters) )
			{
				//printf("Icono encontrado %s\n",find_icon(dir_name,menu_name).toLocal8Bit().data());
				//printf("Menu path %s\n",menu_path(dir_name).toLocal8Bit().data());
				//printf("Menu Action path %s\n",menu_path(dir_name+"/"+_menu_name).toLocal8Bit().data());
				QAction *action=mainwindow->createAction(menu_path(dir_name+"/"+_menu_name),_menu_name, find_icon(dir_name,menu_name));
				mainwindow->menus[menu_path(dir_name)]->addAction(action);
				MenuFileCallBack *callback=new MenuFileCallBack(_menu_name, mainwindow->octave_connection, operations,  input_labels, input_parameters, output_labels, output_parameters, command, help, accept_blank_parameters);
				connect(action, SIGNAL(triggered()), callback, SLOT(actions_callback()));
			}
		}
	}
}

enum ParseState {PARAMETER,BLOCK};

bool GenerateMenu::process_menu_file(QString _file,  QStringList &  input_labels,  QStringList &  input_parameters, QStringList & output_labels, QStringList & output_parameters, QString & command, QString & menu_name, QString & help, bool & accept_blank_parameters)
{
	QFile file(_file);
	if (!file.open(QIODevice::ReadOnly | QIODevice::Text)) return false;
	
	accept_blank_parameters=false;
	
	ParseState state=PARAMETER;
	QString block_label;
	
	//Regular expession for parameters
	QRegExp rp("([A-Za-z_]+)[ \\t]*=(.*)\\n");
	//Regular expession for comments
	QRegExp rc("[ \\t]*#.*\\n");
	//Regular expession for blocks -> start
	QRegExp rbb("begin\\{([A-Za-z_]+)\\}\\n");
	//Regular expession for blocks -> end
	QRegExp rbe("end\\{([A-Za-z_]+)\\}\\n");
	
	while (!file.atEnd())
	{
		QString line(file.readLine());
		
		//printf("Estado %d Linea: %s", state, line.toLocal8Bit().data());
		
		switch(state)
		{
			case PARAMETER:
				if(!rp.exactMatch(line) && !rc.exactMatch(line) && !rbb.exactMatch(line) && !rbe.exactMatch(line)) continue;
				
				/*
				{
					int pos;
					pos=rp.indexIn(line);
					QStringList list = rp.capturedTexts();
					for(int i=0;i<list.size();i++)
						printf("%d %s\n", i, list.at(i).toLocal8Bit().data() );
				}
				
				{
					int pos;
					pos=rbb.indexIn(line);
					QStringList list = rbb.capturedTexts();
					for(int i=0;i<list.size();i++)
						printf("%d %s\n", i, list.at(i).toLocal8Bit().data() );
				}
				*/
				
				//Matchs parameter
				if(rp.exactMatch(line))
				{
					QString type = rp.cap(1);
					
					if(type=="input")
					{
						input_parameters << rp.cap(2).trimmed();
					}
					if(type=="input_label")
					{
						input_labels << rp.cap(2).trimmed();
					}
					else if(type=="output")
					{
						output_parameters << rp.cap(2).trimmed();
					}
					if(type=="output_label")
					{
						output_labels << rp.cap(2).trimmed();
					}
					else if(type=="menu_name")
					{
						menu_name = rp.cap(2).trimmed();
					}
					else if(type=="accept_blank_parameters")
					{
						if (rp.cap(2).trimmed()=="true") accept_blank_parameters=true;
					}
				}
				else if(rbb.exactMatch(line)) {block_label=rbb.cap(1);state=BLOCK;}
				else if(rbe.exactMatch(line)) state=PARAMETER;
				//Comments are ignored.
			break;
			
			case BLOCK:
				if(rbe.exactMatch(line) && block_label==rbe.cap(1)) state=PARAMETER;
				else if(block_label=="help") help+=line;
				else if(block_label=="command") command+=line;
			break;
		}
	}
	
	file.close();
	
	if(input_parameters.size()!=input_labels.size())
	{
		printf("Error in %s\n", _file.toLocal8Bit().data());
		printf("input must have same size than input_labels\n");
		return false;
	}
	
	if(output_parameters.size()!=output_labels.size())
	{
		printf("Error in %s\n", _file.toLocal8Bit().data());
		printf("output must have same size than output_parameters\n");
		return false;
	}
	
	return true;
}

MenuFileCallBack::MenuFileCallBack(QString menu_name, OctaveConnection *oc, Operations *operations, QStringList &  input_labels,  QStringList &  input_parameters, QStringList & output_labels, QStringList & output_parameters, QString & command, QString & help, bool & accept_blank_parameters)
{
	this->menu_name=menu_name;
	this->oc=oc;
	this->input_labels=input_labels;
	this->input_parameters=input_parameters;
	this->output_labels=output_labels;
	this->output_parameters=output_parameters;
	this->command=command;
	this->menu_name=menu_name;
	this->help=help;
	this->accept_blank_parameters=accept_blank_parameters;
	this->operations=operations;
}

void MenuFileCallBack::actions_callback()
{
	Operations::MenuResult r=operations->menu_window(menu_name, input_labels, input_parameters, output_labels, output_parameters, help, accept_blank_parameters);
	
	QString _command(command);
	
	if (r!=Operations::CANCEL)
	{
		if(output_labels.size()>0)
		{
			for (int i = 0; i < output_labels.size();i++)
			{
				_command.replace("%o"+QString().setNum(i+1)+"%",output_parameters.at(i));
			}
		}
		
		if(input_labels.size()>0)
		{
			for (int i = 0; i < input_labels.size();i++)
			{
				_command.replace("%i"+QString().setNum(i+1)+"%",input_parameters.at(i));
			}
		}
	}
	else return;
	
	
	if(r==Operations::COPY_CLIPBOARD)
	{
		QClipboard *clipboard = QApplication::clipboard();
		clipboard->setText(_command);
	}
	//r==ACCEPTED
	else
	{
		oc->command_enter(_command);
	}
}

MenuCallBack::MenuCallBack(QString menu_name, OctaveConnection *oc)
{
	this->menu_name=menu_name;
	this->octave_connection=oc;
	connect(&process, SIGNAL(readyReadStandardOutput()), this, SLOT(readyReadStandardOutput()));
	connect(&process, SIGNAL(finished(int, QProcess::ExitStatus)), this, SLOT(finished(int, QProcess::ExitStatus)));
}

void MenuCallBack::readyReadStandardOutput()
{
	output+=process.readAllStandardOutput();
}

void MenuCallBack::finished( int exitCode, QProcess::ExitStatus exitStatus )
{
	output+=process.readAllStandardOutput();
	
	octave_connection->command_enter(output.trimmed());
}

void MenuCallBack::actions_callback()
{
	output="";
	process.start("\""+menu_name+"\"");
	
}

