#include "utility.h"
#include <string>
#include <glibmm.h>
#include <iostream>
#include <gtkmm/messagedialog.h>

/*
 *	subtitle editor
 *
 *	http://kitone.free.fr/subtitleeditor/
 *
 *	Copyright @ 2005-2006, kitone
 *
 *	Contact: kitone at free dot fr
 *
 *	This program is free software; you can redistribute it and/or
 *	modify it under the terms of the GNU General Public
 *	License as published by the Free Software Foundation; either
 *	version 2 of the License, or (at your option) any later version.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	See the GNU
 *	General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public
 *	License along with this program; if not, write to the Free Software
 *	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA	02111-1307	USA
 *
 *	See gpl.txt for more information regarding the GNU General Public License.
 *
 *
 *	\file
 *	\brief 
 *	\author kitone (kitone at free dot fr)
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*
 *
 */
Glib::ustring message(const char *format, ...)
{
	Glib::ustring res;
	
	va_list args;
	char *formatted = NULL;

	va_start(args, format);
	formatted = g_strdup_vprintf(format, args);
	va_end(args);

	res = formatted;
	
	g_free(formatted);

	return res;
}

/*
 *
 */
bool check_end_char(const gchar c)
{
	switch(c)
	{
	case '\0':
	case '\t':
	case '\015':
	//case '\r':
	case '\n':
		return true;
	default:
	 return false;
	}
	return false;
}


/*
 * "\n", "\r\n",... TODO
 */
Glib::ustring get_newline()
{
	return "\n";
}

/*
 *
 */
Glib::ustring check_end_char(const Glib::ustring &str)
{
	Glib::ustring::size_type pos = str.find('\015');
	
	if(pos != Glib::ustring::npos)
		return str.substr(0, pos);
	
	return str;
}


/*
 *
 */
Glib::ustring get_share_dir(const Glib::ustring &file)
{
	Glib::ustring filename;
	
	filename = Glib::build_filename(PACKAGE_SHARE_DIR, file);

	se_debug_message(SE_DEBUG_APP, "%s", filename.c_str());

	if(Glib::file_test(filename, Glib::FILE_TEST_EXISTS))
		return filename;

	// utiliser pour le dev 
	filename = Glib::build_filename("share", file);

	se_debug_message(SE_DEBUG_APP, "%s", filename.c_str());

	if(Glib::file_test(filename, Glib::FILE_TEST_EXISTS))
		return filename;

	se_debug_message(SE_DEBUG_APP, "%s not found!", filename.c_str());
	//
	std::cerr << "get_share_dir not found:" << file << std::endl;
	return "";
}

/*
 *
 */
void find_and_replace(Glib::ustring &text, const Glib::ustring &find, const Glib::ustring &replace)
{
	se_debug_message(SE_DEBUG_SEARCH, "text=<%s> find=<%s> replace=<%s>", 
			text.c_str(), find.c_str(), replace.c_str());

	Glib::ustring::size_type pos = 0;
	
	while((pos = text.find(find, pos)) != Glib::ustring::npos)
	{
		text.replace(pos, find.size(), replace);
	}
}


/*
 *
 */
void dialog_warning(const Glib::ustring &primary_text, const Glib::ustring &secondary_text)
{
	Glib::ustring msg;

	msg += "<span weight=\"bold\" size=\"larger\">";
	msg += primary_text;
	msg += "</span>\n\n";
	msg += secondary_text;

	Gtk::MessageDialog dialog(msg, true, Gtk::MESSAGE_WARNING, Gtk::BUTTONS_OK, true);
	dialog.run();
}

/*
 *
 */
void dialog_error(const Glib::ustring &primary_text, const Glib::ustring &secondary_text)
{
	Glib::ustring msg;

	msg += "<span weight=\"bold\" size=\"larger\">";
	msg += primary_text;
	msg += "</span>\n\n";
	msg += secondary_text;

	Gtk::MessageDialog dialog(msg, true, Gtk::MESSAGE_ERROR, Gtk::BUTTONS_OK, true);
	dialog.run();
}





#ifdef HAVE_ISO_CODES

#include <fstream>
#include <map>

#define ISO_639_DOMAIN	"iso_639"
#define ISO_3166_DOMAIN	"iso_3166"

/*
 * max length of sscanf'ed fields + 1
 */
#define ISO_639_2_TERMINOLOGY_LEN	4
#define ISO_639_2_BIBLIOGRAPHY_LEN	4
#define ISO_639_1_CODE_LEN	3
#define ISO_639_NAME_LEN	256

#define ISO_3166_ALPHA2_LEN	3
#define ISO_3166_NAME_LEN	256

static bool map_iso_639_initialised = false;
static bool map_iso_3166_initialised = false;

std::map<Glib::ustring, Glib::ustring> map_iso_639;
std::map<Glib::ustring, Glib::ustring> map_iso_3166;

/*
 *
 */
void read_iso_639(const Glib::ustring &filename)
{
	std::ifstream file(filename.c_str());
	if(!file)
	{
		std::cerr << "Cannot open file: " << filename << std::endl;
		return;
	}

	std::string line;
	

	gchar *format = g_strdup_printf("%%%is %%%is %%%is %%n%%%ic%%n",
			ISO_639_2_TERMINOLOGY_LEN - 1,
			ISO_639_2_BIBLIOGRAPHY_LEN - 1,
			ISO_639_1_CODE_LEN - 1,
			ISO_639_NAME_LEN - 1);

	char terminology[ISO_639_2_TERMINOLOGY_LEN];
	char bibliography[ISO_639_2_BIBLIOGRAPHY_LEN];
	char code[ISO_639_1_CODE_LEN];
	char name[ISO_639_NAME_LEN];
	int plen, nlen;
	
	while(!file.eof() && std::getline(file, line))
	{
		if(!line.empty() && line[0] != '#')
		{
			if(sscanf(line.c_str(), format, terminology, bibliography, code, &plen, name, &nlen) > 0)
			{
				if(nlen - plen > 0 && std::string(code) != "XX")
				{
					name[nlen - plen] = '\0';
					map_iso_639[code]=name;
				}
			}
		}
	}
	g_free(format);

	file.close();

	map_iso_639_initialised = true;
}


/*
 *
 */
void read_iso_3166(const Glib::ustring &filename)
{
	std::ifstream file(filename.c_str());
	if(!file)
	{
		std::cerr << "Cannot open file: " << filename << std::endl;
		return;
	}

	std::string line;

	gchar *format = g_strdup_printf("%%%is %%n%%%ic%%n",
			ISO_3166_ALPHA2_LEN - 1,
			ISO_3166_NAME_LEN - 1);
	char code[ISO_3166_ALPHA2_LEN];
	char name[ISO_3166_NAME_LEN];
	int plen, nlen;

	while(!file.eof() && std::getline(file, line))
	{
		if(!line.empty() && line[0] != '#')
		{
			if(sscanf(line.c_str(), format, code, &plen, name, &nlen) > 0)
			{
				name[nlen - plen] = '\0';
				map_iso_3166[code]=name;
			}
		}
	}
	g_free(format);

	file.close();

	map_iso_3166_initialised = true;
}


/*
 *
 */
void initialised()
{
	//bindtextdomain (ISO_639_DOMAIN, ISO_CODES_PATH);
	bind_textdomain_codeset (ISO_639_DOMAIN, "UTF-8");

	//bindtextdomain(ISO_3166_DOMAIN, ISO_CODES_PATH);
	bind_textdomain_codeset (ISO_3166_DOMAIN, "UTF-8");

	read_iso_639(Glib::build_filename(ISO_CODES_PATH, "iso_639.tab"));
	read_iso_3166(Glib::build_filename(ISO_CODES_PATH, "iso_3166.tab"));
}

/*
 *
 */
Glib::ustring get_iso_name_for_lang_code(const Glib::ustring &code)
{
	Glib::ustring c1, c2;

	if(!map_iso_639_initialised)
		initialised();

	if(code.size() == 2)
	{
		c1 = code.substr(0, 2);
		
		Glib::ustring s639 = dgettext(ISO_639_DOMAIN, map_iso_639[c1].c_str());

		return s639;
	}
	else if(code.size() == 5)
	{
		c1 = code.substr(0, 2);
		c2 = code.substr(3, 5);

		Glib::ustring s639 = dgettext(ISO_639_DOMAIN, map_iso_639[c1].c_str());
		Glib::ustring s3166 = dgettext(ISO_3166_DOMAIN, map_iso_3166[c2].c_str());
	
		return s639 + " (" + s3166 + ")";
	}

	return code;
}

#else //HAVE_ISO_CODES

Glib::ustring get_iso_name_for_lang_code(const Glib::ustring &code)
{
	return code;
}

#endif //HAVE_ISO_CODES


#include "SubtitleTime.h"

/*
 *
 */
SpinButtonTiming::SpinButtonTiming(BaseObjectType* cobject, const Glib::RefPtr<Gnome::Glade::Xml>& refGlade)
:Gtk::SpinButton(cobject)
{
	set_increments(100, 1);

	set_use_negative(false);

	int width, height;

	create_pango_layout("00:00:00.000")->get_pixel_size(width, height);

	set_size_request(width + 25, -1);
}

/*
 *
 */
int SpinButtonTiming::on_input(double *new_value)
{
	Glib::ustring text = get_text();

	if(SubtitleTime::validate(text))
		*new_value = (double) SubtitleTime(text).totalmsecs;

	return true;
}

/*
 *
 */
bool SpinButtonTiming::on_output()
{
	//bool state = Gtk::SpinButton::on_output();
	//return state;
	
	set_numeric(false);

	set_text(SubtitleTime((long int)get_value()).str());

	set_numeric(true);

	return true;
}

void SpinButtonTiming::set_use_negative(bool use_negative)
{
	if(use_negative)
		set_range(-86399999, 86399999);
	else
		set_range(0, 86399999);
}

