/*******************************************************************************
 * Copyright (c) 2005 IBM Corporation and others. All rights reserved.
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v1.0 which accompanies this distribution,
 * and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: IBM Corporation - initial API and implementation
 ******************************************************************************/
package org.eclipse.wst.rdb.server.internal.ui.wizards;

import java.sql.Connection;
import java.text.MessageFormat;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.wizard.IWizardContainer;
import org.eclipse.jface.wizard.IWizardContainer2;
import org.eclipse.jface.wizard.Wizard;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Display;
import org.eclipse.wst.rdb.connection.internal.ui.wizards.ConnectionWizardProperties;
import org.eclipse.wst.rdb.core.internal.ui.explorer.virtual.IConnectionNode;
import org.eclipse.wst.rdb.internal.core.RDBCorePlugin;
import org.eclipse.wst.rdb.internal.core.connection.ConnectionInfo;
import org.eclipse.wst.rdb.internal.core.rte.ICatalogObject;
import org.eclipse.wst.rdb.internal.core.util.DatabaseProviderHelper;
import org.eclipse.wst.rdb.server.internal.ui.explorer.actions.popup.DisconnectServerAction;
import org.eclipse.wst.rdb.server.internal.ui.util.resources.ResourceLoader;

/**
 * @author ledunnel
 */
public class EditConnectionWizard extends Wizard {
	private EditJDBCConfigurationWizardPage jdbcPage;

	private EditSchemaFilterWizardPage schemaFilterPage;

	private IConnectionNode connectionNode;

	private ConnectionInfo connection;

	private static final String CONFIGURE_JDBC_WIZARD_PAGE_NAME = "org.eclipse.wst.rdb.server.internal.ui.wizards.EditJdbcPage"; //$NON-NLS-1$

	private static final String SCHEMA_FILTER_WIZARD_PAGE_NAME = "org.eclipse.wst.rdb.server.internal.ui.wizards.EditSchemaFilterPage"; //$NON-NLS-1$

	public EditConnectionWizard(IConnectionNode connection) {
		super();
		this.connectionNode = connection;
		setWindowTitle(ResourceLoader.INSTANCE
				.queryString("_UI_DIALOG_EDIT_CONNECTION_TITLE")); //$NON-NLS-1$

	}

	public void addPages() {
		super.addPages();

		jdbcPage = new EditJDBCConfigurationWizardPage(
				CONFIGURE_JDBC_WIZARD_PAGE_NAME,
				getFilteredExistingConnectionNamesList()); //$NON-NLS-1$

		addPage(jdbcPage);

		schemaFilterPage = new EditSchemaFilterWizardPage(
				SCHEMA_FILTER_WIZARD_PAGE_NAME);
		addPage(schemaFilterPage);

	}

	public boolean performFinish() {
		boolean isPerformFinishSuccessful = false;
		if (getDBConnection(true) != null) {
			schemaFilterPage.applyFilter();
			((ICatalogObject) connection.getSharedDatabase()).refresh();
			isPerformFinishSuccessful = true;
		}
		return isPerformFinishSuccessful;
	}

	public void createPageControls(Composite pageContainer) {
		super.createPageControls(pageContainer);
		IWizardContainer container = this.getContainer();
		if (container instanceof IWizardContainer2)
			((IWizardContainer2) container).updateSize();
		ConnectionInfo connInfo = this.connectionNode.getConnectionInfo();
		ConnectionWizardProperties wizProps = new ConnectionWizardProperties();
		wizProps.setConnectionName(connInfo.getName());
		wizProps.setDatabaseName(connInfo.getDatabaseName());
		wizProps.setProduct(connInfo.getDatabaseDefinition()
				.getProductDisplayString());
		wizProps.setVersion(connInfo.getDatabaseDefinition()
				.getVersionDisplayString());

		String driverName = connInfo
				.getCustomProperty(jdbcPage.JDBC_DRIVER_CUSTOM_PROPERTY_NAME);
		if (driverName.equals("")) {
			driverName = org.eclipse.wst.rdb.connection.internal.ui.util.resource.ResourceLoader.INSTANCE
					.queryString("CUI_NEWCW_JDBCDRV_OTHER_VAL_UI_");
		}
		wizProps.setDriverName(driverName);
		wizProps.setDriverClassName(connInfo.getDriverClassName());
		wizProps.setClassLocation(connInfo.getLoadingPath());
		wizProps.setURL(connInfo.getURL());
		wizProps.setUserID(connInfo.getUserName());
		wizProps.setPassword(connInfo.getPassword());
		wizProps.setCustomProperties(connInfo.getCustomProperties());
		jdbcPage.setConnectionProperties(wizProps);
	}

	private List getFilteredExistingConnectionNamesList() {
		ConnectionInfo connInfo = this.connectionNode.getConnectionInfo();
		ConnectionInfo[] infos = RDBCorePlugin.getDefault()
				.getConnectionManager().getAllNamedConnectionInfo();
		List nameList = new ArrayList(infos.length);
		for (int i = 0, n = infos.length; i < n; i++) {
			if ((!nameList.contains(infos[i].getName()))
					&& (!infos[i].getName().equals(connInfo.getName()))) {
				nameList.add(infos[i].getName().toLowerCase());
			}
		}
		String[] names = (String[]) nameList
				.toArray(new String[nameList.size()]);
		return Arrays.asList(names);
	}

	public boolean canFinish() {
		boolean canFinish = false;
		if (jdbcPage.isPageComplete() && schemaFilterPage.isPageComplete()) {
			canFinish = true;
		}
		return canFinish;
	}

	public ConnectionInfo getDBConnection(boolean showServerMismatchWarning) {
		ConnectionInfo originalConnection = this.connectionNode
				.getConnectionInfo();
		if (this.connection == null) {
			if ((jdbcPage.getConnectionName().equals(originalConnection
					.getName()))
					&& (originalConnection.getDatabaseDefinition().getProduct()
							.equals(
									jdbcPage.getCurrentDatabaseDefinition()
											.getProduct()) && originalConnection
							.getDatabaseDefinition().getVersion().equals(
									jdbcPage.getCurrentDatabaseDefinition()
											.getVersion()))) {
				try {
					// disconnect
					if (originalConnection.getSharedConnection() != null) {
						new DisconnectServerAction().run(null);
					}

					// set new properties
					jdbcPage.setConnectionDetails(originalConnection);
					
					// reconnect
					jdbcPage.internalSaveWidgetValues();
					jdbcPage.performTestConnection(false, showServerMismatchWarning);
					if (jdbcPage.isFinalConnection()) {						
						this.connection = originalConnection;						
						displayNewServer(this.connection);
					}
				} catch (Exception e) {
					e.printStackTrace();
				}
			} else {
				jdbcPage.internalSaveWidgetValues();
				jdbcPage
						.performTestConnection(false, showServerMismatchWarning);
				if (jdbcPage.isFinalConnection()) {
					// remove old connection
					RDBCorePlugin.getDefault().getConnectionManager()
							.removeConnectionInfo(originalConnection.getName());
					
					this.connection = jdbcPage.getConnection();
					
					displayNewServer(this.connection);
				}
			}
		}
		return this.connection;
	}

	private boolean displayNewServer(ConnectionInfo connectionInfo) {
		Connection connection = null;
		try {
			connection = connectionInfo.connect();
			connectionInfo.setSharedConnection(connection);
			connectionInfo.saveConnectionInfo();
			new DatabaseProviderHelper().setDatabase(connection,
					connectionInfo, connectionInfo.getDatabaseName());
			return true;
		} catch (Exception e) {
			RDBCorePlugin.getDefault().getConnectionManager()
					.removeConnectionInfo(connectionInfo.getName());
			displayError(e.getMessage());
			return false;
		}
	}

	private void displayError(String message) {
		MessageDialog.openError(Display.getCurrent().getActiveShell(),
				NewConnectionWizard.CONNECTION_NULL, MessageFormat.format(
						NewConnectionWizard.CONNECTION_EXCEPTION,
						new String[] { message }));
	}
}
