/* $Id: util.cpp,v 1.8 2002/01/26 03:14:53 fesnel Exp $ */
/*******************************************************************************
 *   This program is part of a library used by the Archimedes email client     * 
 *                                                                             *
 *   Copyright : (C) 1995-1998 Gennady B. Sorokopud (gena@NetVision.net.il)    *
 *               (C) 1995 Ugen. J. S. Antsilevich (ugen@latte.worldbank.org)   *
 *               (C) 1998-2002 by the Archimedes Project                       *
 *                   http://sourceforge.net/projects/archimedes                *
 *                                                                             *
 *             --------------------------------------------                    *
 *                                                                             *
 *   This program is free software; you can redistribute it and/or modify      *
 *   it under the terms of the GNU Library General Public License as published *
 *   by the Free Software Foundation; either version 2 of the License, or      *
 *   (at your option) any later version.                                       *
 *                                                                             *
 *   This program is distributed in the hope that it will be useful,           *
 *   but WITHOUT ANY WARRANTY, without even the implied warranty of            *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the             *
 *   GNU Library General Public License for more details.                      *
 *                                                                             *
 *   You should have received a copy of the GNU Library General Public License *
 *   along with this program; if not, write to the Free Software               *
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston MA 02111-1307, USA.  *
 *                                                                             *
 ******************************************************************************/


#include <fmail.h>
#include <umail.h>
#ifdef __EMX__
	#include <sys/types.h>
#endif

#define USE_DMALLOC
#include <glib.h>

int do_move(char *, char *);
int fastcopy(char *, char *, struct stat *);

#ifndef HAVE_UTIMES

	#ifdef  HAVE_UTIME_H
		#include <utime.h>
	#endif

int utimes(const char *file, const struct timeval *tvp) {
	struct utimbuf tbuf;

	if(tvp == NULL) {
	#ifdef  HAVE_UTIME_NULL
		return(utime(file, NULL));
	#else
		tbuf.actime = time(NULL);
		tbuf.modtime = time(NULL);
		return(utime(file, &tbuf));
	#endif
	}

	tbuf.actime = tvp[0].tv_sec;
	tbuf.modtime = tvp[1].tv_sec;
	return(utime(file, &tbuf));
}
#endif

int do_move(char *from, char *to) {
	struct stat sb;
	int st;

#ifdef __EMX__          /* Under OS/2 the file will not be deleted during rename() */
	if(access(to, 0) == 0) {
		if(unlink(to) != 0) {
			display_msg(MSG_WARN, "unlink", "delete %s before moving", to);
		}
	}
#endif
	if(!rename(from, to))
		return(0);

	if(errno != EXDEV) {
		display_msg(MSG_WARN, "move", "rename %s to %s", from, to);
		return(1);
	}

	/*
	 * If rename fails because we're trying to cross devices, and
	 * it's a regular file, do the copy internally; otherwise, use
	 * cp and rm.
	 */
	if(stat(from, &sb)) {
		display_msg(MSG_WARN, "move", "%s", from);
		return(1);
	}

	if(!S_ISREG(sb.st_mode)) {
		display_msg(MSG_WARN, "move: not a regular file", "%s", from);
		return 1;
	}

	st = fastcopy(from, to, &sb);

	if(unlink(from)) {
		display_msg(MSG_WARN, "move", "%s: remove", from);
		return(1);
	}

	return st;
}

int fastcopy(char *from, char *to, struct stat *sbp) {
	struct timeval tval[2];
	static u_int blen;
	static char *bp;
	register int nread, from_fd, to_fd;

	if((from_fd = open(from, O_RDONLY, 0)) < 0) {
		display_msg(MSG_WARN, "fastcopy", "%s", from);
		return(1);
	}
	if((to_fd = open(to, O_CREAT | O_TRUNC | O_WRONLY, sbp->st_mode)) < 0) {
		display_msg(MSG_WARN, "fastcopy", "%s", to);
		(void) close(from_fd);
		return(1);
	}
#ifndef __EMX__
	if(!blen && !(bp = (char *) malloc(blen = sbp->st_blksize))) {
#else
	if(!blen && !(bp = (char *) malloc(blen = sbp->st_size))) {
#endif
		display_msg(MSG_WARN, "fastcopy", "");
		return(1);
	}
	while((nread = read(from_fd, bp, blen)) > 0)
		if(write(to_fd, bp, nread) != nread) {
			display_msg(MSG_WARN, "fastcopy", "%s", to);
			goto err;
		}
	if(nread < 0) {
		display_msg(MSG_WARN, "fastcopy", "%s", from);
		err:if(unlink(to))
			display_msg(MSG_WARN, "fastcopy", "%s: remove", to);
		(void) close(from_fd);
		(void) close(to_fd);
		return(1);
	}
	(void) close(from_fd);

	/*
		if (fchown(to_fd, sbp->st_uid, sbp->st_gid))
			display_msg(MSG_WARN, "fastcopy", "%s: set owner/group", to);
	*/
#ifndef __EMX__
	fchown(to_fd, sbp->st_uid, sbp->st_gid);

	if(fchmod(to_fd, sbp->st_mode))
		display_msg(MSG_WARN, "fastcopy", "%s: set mode", to);

	tval[0].tv_sec = sbp->st_atime;
	tval[1].tv_sec = sbp->st_mtime;
	tval[0].tv_usec = tval[1].tv_usec = 0;
	if(utimes(to, tval))
		display_msg(MSG_WARN, "fastcopy", "%s: set times", to);
#endif

	if(close(to_fd)) {
		display_msg(MSG_WARN, "fastcopy", "%s", to);
		return(1);
	}

	return(0);
}

#ifndef HAVE_UNSETENV
void Unsetenv(char *name)
{
	int len;
	char *cp, **p;

	if(!environ || !name)
		return;

	len = strlen(name);
	for(p = environ; (cp = *p) != NULL; ++p) {
		if(strncmp(cp, name, len) || (cp[len] != '='))
			continue;

		for(;; ++p)
			if(!(*p = *(p + 1)))
				break;

		break;
	}
}
#endif

/* strstr() implementation that can ignore case if scase is true */
char *strcasestr(register const char *s, register const char *find,
				 int scase) {
	register size_t len;

	if(!scase)
		return strstr(s, find);

	len = strlen(find);
	while(strncasecmp(s, find, len) != 0) {
		s++;
		if(*s == 0)
			return(NULL);
	}

	return((char *) s);
}

#ifndef HAVE_SNPRINTF

/* Define this as a fall through, HAVE_STDARG_H is probably already set */

/**************************************************************
 * Original:
 * Patrick Powell Tue Apr 11 09:48:21 PDT 1995
 * A bombproof version of doprnt (dopr) included.
 * Sigh.  This sort of thing is always nasty do deal with.  Note that
 * the version here does not include floating point...
 *
 * snprintf() is used instead of sprintf() as it does limit checks
 * for string length.  This covers a nasty loophole.
 *
 * The other functions are there to prevent NULL pointers from
 * causing nast effects.
 *
 * More Recently:
 *  Brandon Long <blong@fiction.net> 9/15/96 for mutt 0.43
 *  This was ugly.  It is still ugly.  I opted out of floating point
 *  numbers, but the formatter understands just about everything
 *  from the normal C string format, at least as far as I can tell from
 *  the Solaris 2.5 printf(3S) man page.
 *
 *  Brandon Long <blong@fiction.net> 10/22/97 for mutt 0.87.1
 *    Ok, added some minimal floating point support, which means this
 *    probably requires libm on most operating systems.  Don't yet
 *    support the exponent (e,E) and sigfig (g,G).  Also, fmtint()
 *    was pretty badly broken, it just wasn't being exercised in ways
 *    which showed it, so that's been fixed.  Also, formated the code
 *    to mutt conventions, and removed dead code left over from the
 *    original.  Also, there is now a builtin-test, just compile with:
 *           gcc -DTEST_SNPRINTF -o snprintf snprintf.c -lm
 *    and run snprintf for results.
 *
 *  Thomas Roessler <roessler@guug.de> 01/27/98 for mutt 0.89i
 *    The PGP code was using unsigned hexadecimal formats.
 *    Unfortunately, unsigned formats simply didn't work.
 *
 **************************************************************/


/* varargs declarations: */

	#if defined(HAVE_STDARG_H)
		#define HAVE_STDARGS        /* let's hope that works everywhere (mj) */
		#define VA_LOCAL_DECL   va_list ap
		#define VA_START(f)     va_start(ap, f)
		#define VA_SHIFT(v,t)  ;    /* no-op for ANSI */
		#define VA_END          va_end(ap)
	#else
		#if defined(HAVE_VARARGS_H)
			#undef HAVE_STDARGS
			#define VA_LOCAL_DECL   va_list ap
			#define VA_START(f)     va_start(ap)    /* f is ignored! */
			#define VA_SHIFT(v,t) v = va_arg(ap,t)
			#define VA_END        va_end(ap)
		#else
/*XX ** NO VARARGS ** XX*/
		#endif
	#endif

int snprintf(char *str, size_t count, const char *fmt, ...);
int vsnprintf(char *str, size_t count, const char *fmt, va_list arg);

static void dopr(char *buffer, size_t maxlen, const char *format,
				 va_list args);
static void fmtstr(char *buffer, size_t * currlen, size_t maxlen,
				   char *value, int flags, int min, int max);
static void fmtint(char *buffer, size_t * currlen, size_t maxlen,
				   long value, int base, int min, int max, int flags);
static void fmtfp(char *buffer, size_t * currlen, size_t maxlen,
				  long double fvalue, int min, int max, int flags);
static void dopr_outch(char *buffer, size_t * currlen, size_t maxlen,
					   char c);

int vsnprintf(char *str, size_t count, const char *fmt, va_list args) {
	str[0] = 0;
	dopr(str, count, fmt, args);
	return(strlen(str));
}

/* VARARGS3 */
	#ifdef HAVE_STDARGS
int snprintf(char *str, size_t count, const char *fmt, ...)
	#else
int snprintf(va_alist)
va_dcl
	#endif
{
	#ifndef HAVE_STDARGS
	char *str;
	size_t count;
	char *fmt;
	#endif
	VA_LOCAL_DECL;

	VA_START(fmt);
	VA_SHIFT(str, char *);
	VA_SHIFT(count, size_t);
	VA_SHIFT(fmt, char *);
	(void) vsnprintf(str, count, fmt, ap);
	VA_END;
	return(strlen(str));
}

/*
 * dopr(): poor man's version of doprintf
 */

/* format read states */
	#define DP_S_DEFAULT 0
	#define DP_S_FLAGS   1
	#define DP_S_MIN     2
	#define DP_S_DOT     3
	#define DP_S_MAX     4
	#define DP_S_MOD     5
	#define DP_S_CONV    6
	#define DP_S_DONE    7

/* format flags - Bits */
	#define DP_F_MINUS  (1 << 0)
	#define DP_F_PLUS   (1 << 1)
	#define DP_F_SPACE  (1 << 2)
	#define DP_F_NUM    (1 << 3)
	#define DP_F_ZERO   (1 << 4)
	#define DP_F_UP     (1 << 5)
	#define DP_F_UNSIGNED   (1 << 6)

/* Conversion Flags */
	#define DP_C_SHORT   1
	#define DP_C_LONG    2
	#define DP_C_LDOUBLE 3

	#define char_to_int(p) (p - '0')
	#define SNMAX(p,q) ((p >= q) ? p : q)

static void dopr(char *buffer, size_t maxlen, const char *format,
				 va_list args) {
	char ch;
	long value;
	long double fvalue;
	char *strvalue;
	int min;
	int max;
	int state;
	int flags;
	int cflags;
	size_t currlen;

	state = DP_S_DEFAULT;
	currlen = flags = cflags = min = 0;
	max = -1;
	ch = *format++;

	while(state != DP_S_DONE) {
		if((ch == '\0') || (currlen >= maxlen))
			state = DP_S_DONE;

		switch(state) {
			case DP_S_DEFAULT:
				if(ch == '%')
					state = DP_S_FLAGS;
				else
					dopr_outch(buffer, &currlen, maxlen, ch);
				ch = *format++;
				break;
			case DP_S_FLAGS:
				switch(ch) {
					case '-':
						flags |= DP_F_MINUS;
						ch = *format++;
						break;
					case '+':
						flags |= DP_F_PLUS;
						ch = *format++;
						break;
					case ' ':
						flags |= DP_F_SPACE;
						ch = *format++;
						break;
					case '#':
						flags |= DP_F_NUM;
						ch = *format++;
						break;
					case '0':
						flags |= DP_F_ZERO;
						ch = *format++;
						break;
					default:
						state = DP_S_MIN;
						break;
				}
				break;
			case DP_S_MIN:
				if(isdigit(ch)) {
					min = 10 * min + char_to_int(ch);
					ch = *format++;
				} else if(ch == '*') {
					min = va_arg(args, int);
					ch = *format++;
					state = DP_S_DOT;
				} else
					state = DP_S_DOT;
				break;
			case DP_S_DOT:
				if(ch == '.') {
					state = DP_S_MAX;
					ch = *format++;
				} else
					state = DP_S_MOD;
				break;
			case DP_S_MAX:
				if(isdigit(ch)) {
					if(max < 0)
						max = 0;
					max = 10 * max + char_to_int(ch);
					ch = *format++;
				} else if(ch == '*') {
					max = va_arg(args, int);
					ch = *format++;
					state = DP_S_MOD;
				} else
					state = DP_S_MOD;
				break;
			case DP_S_MOD:
				/* Currently, we don't support Long Long, bummer */
				switch(ch) {
					case 'h':
						cflags = DP_C_SHORT;
						ch = *format++;
						break;
					case 'l':
						cflags = DP_C_LONG;
						ch = *format++;
						break;
					case 'L':
						cflags = DP_C_LDOUBLE;
						ch = *format++;
						break;
					default:
						break;
				}
				state = DP_S_CONV;
				break;
			case DP_S_CONV:
				switch(ch) {
					case 'd':
					case 'i':
						if(cflags == DP_C_SHORT)
							value = va_arg(args, short int);
						else if(cflags == DP_C_LONG)
							value = va_arg(args, long int);
						else
							value = va_arg(args, int);
						fmtint(buffer, &currlen, maxlen, value, 10, min, max,
							   flags);
						break;
					case 'o':
						flags |= DP_F_UNSIGNED;
						if(cflags == DP_C_SHORT)
							value = va_arg(args, unsigned short int);
						else if(cflags == DP_C_LONG)
							value = va_arg(args, unsigned long int);
						else
							value = va_arg(args, unsigned int);
						fmtint(buffer, &currlen, maxlen, value, 8, min, max,
							   flags);
						break;
					case 'u':
						flags |= DP_F_UNSIGNED;
						if(cflags == DP_C_SHORT)
							value = va_arg(args, unsigned short int);
						else if(cflags == DP_C_LONG)
							value = va_arg(args, unsigned long int);
						else
							value = va_arg(args, unsigned int);
						fmtint(buffer, &currlen, maxlen, value, 10, min, max,
							   flags);
						break;
					case 'X':
						flags |= DP_F_UP;
					case 'x':
						flags |= DP_F_UNSIGNED;
						if(cflags == DP_C_SHORT)
							value = va_arg(args, unsigned short int);
						else if(cflags == DP_C_LONG)
							value = va_arg(args, unsigned long int);
						else
							value = va_arg(args, unsigned int);
						fmtint(buffer, &currlen, maxlen, value, 16, min, max,
							   flags);
						break;
					case 'f':
						if(cflags == DP_C_LDOUBLE)
							fvalue = va_arg(args, long double);
						else
							fvalue = va_arg(args, double);
						/* um, floating point? */
						fmtfp(buffer, &currlen, maxlen, fvalue, min, max, flags);
						break;
					case 'E':
						flags |= DP_F_UP;
					case 'e':
						if(cflags == DP_C_LDOUBLE)
							fvalue = va_arg(args, long double);
						else
							fvalue = va_arg(args, double);
						break;
					case 'G':
						flags |= DP_F_UP;
					case 'g':
						if(cflags == DP_C_LDOUBLE)
							fvalue = va_arg(args, long double);
						else
							fvalue = va_arg(args, double);
						break;
					case 'c':
						dopr_outch(buffer, &currlen, maxlen, va_arg(args, int));
						break;
					case 's':
						strvalue = va_arg(args, char *);
						if(max < 0)
							max = maxlen;   /* ie, no max */
						fmtstr(buffer, &currlen, maxlen, strvalue, flags, min,
							   max);
						break;
					case 'p':
						strvalue = va_arg(args, void *);
						fmtint(buffer, &currlen, maxlen, (long) strvalue, 16, min,
							   max, flags);
						break;
					case 'n':
						if(cflags == DP_C_SHORT) {
							short int *num;
							num = va_arg(args, short int *);
							*num = currlen;
						} else if(cflags == DP_C_LONG) {
							long int *num;
							num = va_arg(args, long int *);
							*num = currlen;
						} else {
							int *num;
							num = va_arg(args, int *);
							*num = currlen;
						}
						break;
					case '%':
						dopr_outch(buffer, &currlen, maxlen, ch);
						break;
					case 'w':
						/* not supported yet, treat as next char */
						ch = *format++;
						break;
					default:
						/* Unknown, skip */
						break;
				}
				ch = *format++;
				state = DP_S_DEFAULT;
				flags = cflags = min = 0;
				max = -1;
				break;
			case DP_S_DONE:
				break;
			default:
				/* hmm? */
				break;      /* some picky compilers need this */
		}
	}
	if(currlen < maxlen - 1)
		buffer[currlen] = '\0';
	else
		buffer[maxlen - 1] = '\0';
}

static void fmtstr(char *buffer, size_t * currlen, size_t maxlen,
				   char *value, int flags, int min, int max) {
	int padlen, strln;      /* amount to pad */
	int cnt = 0;

	if(value == 0) {
		value = "<NULL>";
	}

	for(strln = 0; value[strln]; ++strln); /* strlen */
	padlen = min - strln;
	if(padlen < 0)
		padlen = 0;
	if(flags & DP_F_MINUS)
		padlen = -padlen;   /* Left Justify */

	while((padlen > 0) && (cnt < max)) {
		dopr_outch(buffer, currlen, maxlen, ' ');
		--padlen;
		++cnt;
	}
	while(*value && (cnt < max)) {
		dopr_outch(buffer, currlen, maxlen, *value++);
		++cnt;
	}
	while((padlen < 0) && (cnt < max)) {
		dopr_outch(buffer, currlen, maxlen, ' ');
		++padlen;
		++cnt;
	}
}

/* Have to handle DP_F_NUM (ie 0x and 0 alternates) */

static void fmtint(char *buffer, size_t * currlen, size_t maxlen,
				   long value, int base, int min, int max, int flags) {
	int signvalue = 0;
	unsigned long uvalue;
	char convert[20];
	int place = 0;
	int spadlen = 0;        /* amount to space pad */
	int zpadlen = 0;        /* amount to zero pad */
	int caps = 0;

	if(max < 0)
		max = 0;

	uvalue = value;

	if(!(flags & DP_F_UNSIGNED)) {
		if(value < 0) {
			signvalue = '-';
			uvalue = -value;
		} else if(flags & DP_F_PLUS)   /* Do a sign (+/i) */
			signvalue = '+';
		else if(flags & DP_F_SPACE)
			signvalue = ' ';
	}

	if(flags & DP_F_UP)
		caps = 1;       /* Should characters be upper case? */

	do {
		convert[place++] = (caps ? "0123456789ABCDEF" : "0123456789abcdef")
						   [uvalue % (unsigned) base];
		uvalue = (uvalue / (unsigned) base);
	} while(uvalue && (place < 20));
	if(place == 20)
		place--;
	convert[place] = 0;

	zpadlen = max - place;
	spadlen = min - SNMAX(max, place) - (signvalue ? 1 : 0);
	if(zpadlen < 0)
		zpadlen = 0;
	if(spadlen < 0)
		spadlen = 0;
	if(flags & DP_F_ZERO) {
		zpadlen = SNMAX(zpadlen, spadlen);
		spadlen = 0;
	}
	if(flags & DP_F_MINUS)
		spadlen = -spadlen; /* Left Justifty */

	#ifdef DEBUG_SNPRINTF
	dprint(1,
		   (debugfile, "zpad: %d, spad: %d, min: %d, max: %d, place: %d\n",
			zpadlen, spadlen, min, max, place));
	#endif

	/* Spaces */
	while(spadlen > 0) {
		dopr_outch(buffer, currlen, maxlen, ' ');
		--spadlen;
	}

	/* Sign */
	if(signvalue)
		dopr_outch(buffer, currlen, maxlen, signvalue);

	/* Zeros */
	if(zpadlen > 0) {
		while(zpadlen > 0) {
			dopr_outch(buffer, currlen, maxlen, '0');
			--zpadlen;
		}
	}

	/* Digits */
	while(place > 0)
		dopr_outch(buffer, currlen, maxlen, convert[--place]);

	/* Left Justified spaces */
	while(spadlen < 0) {
		dopr_outch(buffer, currlen, maxlen, ' ');
		++spadlen;
	}
}

static long double abs_val(long double value) {
	long double result = value;

	if(value < 0)
		result = -value;

	return result;
}

static long double pow10(int exp) {
	long double result = 1;

	while(exp) {
		result *= 10;
		exp--;
	}

	return result;
}

static long round(long double value) {
	long intpart;

	intpart = value;
	value = value - intpart;
	if(value >= 0.5)
		intpart++;

	return intpart;
}

static void fmtfp(char *buffer, size_t * currlen, size_t maxlen,
				  long double fvalue, int min, int max, int flags) {
	int signvalue = 0;
	long double ufvalue;
	char iconvert[20];
	char fconvert[20];
	int iplace = 0;
	int fplace = 0;
	int padlen = 0;     /* amount to pad */
	int zpadlen = 0;
	int caps = 0;
	long intpart;
	long fracpart;

	/*
	 * AIX manpage says the default is 0, but Solaris says the default
	 * is 6, and sprintf on AIX defaults to 6
	 */
	if(max < 0)
		max = 6;

	ufvalue = abs_val(fvalue);

	if(fvalue < 0)
		signvalue = '-';
	else if(flags & DP_F_PLUS) /* Do a sign (+/i) */
		signvalue = '+';
	else if(flags & DP_F_SPACE)
		signvalue = ' ';

	#if 0
	if(flags & DP_F_UP)
		caps = 1;       /* Should characters be upper case? */
	#endif

	intpart = ufvalue;

	/*
	 * Sorry, we only support 9 digits past the decimal because of our
	 * conversion method
	 */
	if(max > 9)
		max = 9;

	/* We "cheat" by converting the fractional part to integer by
	 * multiplying by a factor of 10
	 */
	fracpart = round((pow10(max)) * (ufvalue - intpart));

	if(fracpart >= pow10(max)) {
		intpart++;
		fracpart -= pow10(max);
	}
	#ifdef DEBUG_SNPRINTF
	dprint(1,
		   (debugfile, "fmtfp: %f =? %d.%d\n", fvalue, intpart, fracpart));
	#endif

	/* Convert integer part */
	do {
		iconvert[iplace++] =
		(caps ? "0123456789ABCDEF" : "0123456789abcdef")[intpart % 10];
		intpart = (intpart / 10);
	} while(intpart && (iplace < 20));
	if(iplace == 20)
		iplace--;
	iconvert[iplace] = 0;

	/* Convert fractional part */
	do {
		fconvert[fplace++] =
		(caps ? "0123456789ABCDEF" : "0123456789abcdef")[fracpart %
														 10];
		fracpart = (fracpart / 10);
	} while(fracpart && (fplace < 20));
	if(fplace == 20)
		fplace--;
	fconvert[fplace] = 0;

	/* -1 for decimal point, another -1 if we are printing a sign */
	padlen = min - iplace - max - 1 - ((signvalue) ? 1 : 0);
	zpadlen = max - fplace;
	if(zpadlen < 0)
		zpadlen = 0;
	if(padlen < 0)
		padlen = 0;
	if(flags & DP_F_MINUS)
		padlen = -padlen;   /* Left Justifty */

	if((flags & DP_F_ZERO) && (padlen > 0)) {
		if(signvalue) {
			dopr_outch(buffer, currlen, maxlen, signvalue);
			--padlen;
			signvalue = 0;
		}
		while(padlen > 0) {
			dopr_outch(buffer, currlen, maxlen, '0');
			--padlen;
		}
	}
	while(padlen > 0) {
		dopr_outch(buffer, currlen, maxlen, ' ');
		--padlen;
	}
	if(signvalue)
		dopr_outch(buffer, currlen, maxlen, signvalue);

	while(iplace > 0)
		dopr_outch(buffer, currlen, maxlen, iconvert[--iplace]);

	/*
	 * Decimal point.  This should probably use locale to find the correct
	 * char to print out.
	 */
	dopr_outch(buffer, currlen, maxlen, '.');

	while(fplace > 0)
		dopr_outch(buffer, currlen, maxlen, fconvert[--fplace]);

	while(zpadlen > 0) {
		dopr_outch(buffer, currlen, maxlen, '0');
		--zpadlen;
	}

	while(padlen < 0) {
		dopr_outch(buffer, currlen, maxlen, ' ');
		++padlen;
	}
}

static void dopr_outch(char *buffer, size_t * currlen, size_t maxlen,
					   char c) {
	if(*currlen < maxlen)
		buffer[(*currlen)++] = c;
}

#endif /* !HAVE_SNPRINTF */
