/* $Id: addr_book.cpp,v 1.12 2002/07/27 20:07:24 cfreeze Exp $ */
/*******************************************************************************
 *   This program is part of the XFMail email client.                          *
 *                                                                             *
 *   Copyright : (C) 1995-1998 Gennady B. Sorokopud (gena@NetVision.net.il)    *
 *               (C) 1995 Ugen. J. S. Antsilevich (ugen@latte.worldbank.org)   *
 *               (C) 1998-2002 by the Archimedes Project                       *
 *                   http://sourceforge.net/projects/archimedes                *
 *                                                                             *
 *             --------------------------------------------                    *
 *                                                                             *
 *   This program is free software; you can redistribute it and/or modify      *
 *   it under the terms of the GNU General Public License as published by      *
 *   the Free Software Foundation; either version 2 of the License, or         *
 *   (at your option) any later version.                                       *
 *                                                                             *
 *   This program is distributed in the hope that it will be useful,           *
 *   but WITHOUT ANY WARRANTY, without even the implied warranty of            *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the             *
 *   GNU General Public License for more details.                              *
 *                                                                             *
 *   You should have received a copy of the GNU General Public License         *
 *   along with this program; if not, write to the Free Software               *
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston MA 02111-1307, USA.  *
 *                                                                             *
 *   Additional Permission granted:                                            *
 *                                                                             *
 *   This program is designed to use the XForms library, so we consider        *
 *   permission to link to that non-GPL-compatible library is implicit.        *
 *   However, in case this is not considered so, we explicitly state:          *
 *                                                                             *
 *   "As a special exception, the Archimedes Project, with the permission      *
 *    of all earlier copyright holders, formally gives permission to link      *
 *    this program with the XForms library, and distribute the resulting       *
 *    executable without the source code for XForms in the source              *
 *    distribution".                                                           *
 *                                                                             *
 ******************************************************************************/


#include <glib.h>

#include <fmail.h>
#include <umail.h>
#include <choose_folder.h>
#include <cfgfile.h>

void Address_Browse_Dbl_Call(FL_OBJECT * obj, long param);
int Address_Browse_Handler(FL_OBJECT * obj, int event, FL_Coord mx,
						   FL_Coord my, int key, void *raw_ev);
void AB_Alias_Call(FL_OBJECT * obj, long param);
void display_book();
void display_books();
int load_book(char *name);
int save_book();
int convert_book();
void free_book();
void sort_book();
void edit_entry();
int get_entry_idx(struct _abook_entry *entry);
struct _abook_entry *get_entry_by_idx(int index);
struct _abook_entry *add_book(struct _mail_addr *addr, int ask);

extern cfgfile Config;

static int ready = 0;
static int modready = 0;
static int already = 0;
static int type = TO_TYPE;
static int changed = 0;
static FD_Address_Book *addr_book_obj;
static FD_addrmod *amod_obj;
static FD_aliased *alias_obj;

static struct _abook *book = NULL;
static struct _abook_entry *cur_entry = NULL;
static addr_dbl_cb dbl_callback;
static int dbl_callback_data;
static int abookx = -1, abooky = -1;

void add_msg_addr(struct _mail_msg *msg) {
	struct _mail_addr *addr, *addr1;

	if(ready)
		return;

	if(!book || strcmp(book->name, "default")) {
		if(load_book(NULL) == -1)
			return;
	}

	if(!msg) {
		save_book();
		return;
	}

	if(!msg->header)
		return;

	addr = msg->header->From;
	while(addr) {
		addr1 = addr->next_addr;
		addr->next_addr = NULL;
		addr->pgpid = NULL;
		add_book(addr, 0);
		addr->next_addr = addr1;
		addr = addr1;
	}

	addr = msg->header->To;
	while(addr) {
		addr1 = addr->next_addr;
		addr->next_addr = NULL;
		addr->pgpid = NULL;
		add_book(addr, 0);
		addr->next_addr = addr1;
		addr = addr1;
	}

	addr = msg->header->Cc;
	while(addr) {
		addr1 = addr->next_addr;
		addr->next_addr = NULL;
		addr->pgpid = NULL;
		add_book(addr, 0);
		addr->next_addr = addr1;
		addr = addr1;
	}

	addr = msg->header->Bcc;
	while(addr) {
		addr1 = addr->next_addr;
		addr->next_addr = NULL;
		addr->pgpid = NULL;
		add_book(addr, 0);
		addr->next_addr = addr1;
		addr = addr1;
	}

	return;
}

struct _abook_entry *find_addr(struct _mail_addr *addr) {
	struct _abook_entry *entry;
	struct _mail_addr *addr1;

	if(!addr)
		return NULL;

	if(!book)
		if(load_book(NULL) == -1)
			return NULL;

	entry = book->entries;
	while(entry) {
		addr1 = entry->addr;
		while(addr1) {
			if(!strcasecmp(addr1->addr, addr->addr))
				return entry;

			addr1 = addr1->next_addr;
		}
		entry = entry->next;
	}

	return NULL;
}

struct _abook_entry *add_book(struct _mail_addr *addr, int ask) {
	struct _abook_entry *entry;
	struct _mail_addr *addr1;
	int num;

	if(addr == NULL)
		return NULL;

	if(book == NULL) {
		if(load_book(NULL) == -1)
			return NULL;
	}

	num = 0;
	addr1 = addr;
	while(addr1) {
		num++;
		addr1 = addr1->next_addr;
	}

	if((num == 1) && find_addr(addr)) {
		if(!ask)
			return NULL;
		if(ask
		   && !display_msg(MSG_QUEST | MSG_DEFNO, NULL,
						   "\"%s\"\nalready in address book, still add?",
						   addr->addr)) return NULL;
	}

	if(
	  (entry =
	   (struct _abook_entry *) malloc(sizeof(struct _abook_entry))) ==
	  NULL) {
		display_msg(MSG_FATAL, "add to book", "Malloc failed");
		return NULL;
	}

	entry->addr = copy_address_chain(addr);
	entry->description = NULL;
	entry->type = (num == 1) ? ADDR_SINGLE : ADDR_ALIAS;
	entry->next = book->entries;
	book->entries = entry;

	return entry;
}

void free_book() {
	struct _abook_entry *entry, *entry1;

	if(book == NULL)
		return;

	entry = book->entries;
	while(entry) {
		if(entry->description)
			free(entry->description);
		discard_address(entry->addr);
		entry1 = entry;
		entry = entry->next;
		free(entry1);
	}

	free(book);
	book = NULL;
	cur_entry = NULL;
}

int load_book(char *name) {
	struct _abook_entry *entry, *lentry;
	struct _mail_addr *addr = NULL, *laddr;
	char buf[255], *p;
	int num, eenum, fline, pgpid_ok_here;
	FILE *afd;

	if(name && (strlen(name) > 15)) {
		display_msg(MSG_WARN, "address book",
					"Address book name too long");
		return -1;
	}

	if(changed) {
		if(display_msg
		   (MSG_QUEST, "address book",
			"Do you want to save changes into the address book?"))
			save_book();
	}

	free_book();
	changed = 0;

	if(name)
		snprintf(buf, sizeof(buf), "%s/.xfbook.%s", configdir, name);
	else
		snprintf(buf, sizeof(buf), "%s/.xfbook", configdir);

	if((afd = fopen(buf, "r")) == NULL) {
		if((afd = fopen(buf, "w")) == NULL) {
			display_msg(MSG_WARN, "address book",
						"Failed to create address book \"%s\"",
						name ? name : "default");
			return -1;
		}
		fclose(afd);
		afd = fopen(buf, "r");
		display_msg(MSG_MSG, "created address book", "%s", buf);
	}

	lentry = NULL;
	if((book = (struct _abook *) malloc(sizeof(struct _abook))) == NULL) {
		display_msg(MSG_FATAL, "address book", "malloc failed\n");
		fclose(afd);
		return -1;
	}

	book->entries = NULL;
	strcpy(book->name, name ? name : "default");
	book->flags = ABOOK_LOCAL;
	eenum = 0;
	fline = 0;

	while(fgets(buf, 255, afd)) {
		if(strncmp(buf, "@ ", 2)) {
			if(!name && (fline == 0)) {
				if(display_msg
				   (MSG_QUEST, NULL,
					"Your address book is in old format, convert?")) {
					fclose(afd);
					if(convert_book() == -1) {
						display_msg(MSG_WARN, "conver",
									"Failed to convert address book");
						return -1;
					}
					return load_book(NULL);
				} else
					return -1;
			}
			fline = 1;
			continue;
		}
		fline = 1;

		newent:
		strip_newline(buf);

		/*if(eenum >= MAX_ADDR_IN_BOOK) {
			display_msg(MSG_WARN, "addr book",
						"Address book too big\nignoring");
			break;
		}*/

		if(
		  (entry =
		   (struct _abook_entry *) malloc(sizeof(struct _abook_entry)))
		  == NULL) {
			display_msg(MSG_FATAL, "address book", "malloc failed\n");
			fclose(afd);
			return -1;
		}

		eenum++;
		entry->next = NULL;

		p = buf + 2;
		p = rem_tr_space(p);
		if(p && strlen(p))
			entry->description = strdup(p);
		else
			entry->description = NULL;

		entry->addr = NULL;
		num = 0;
		buf[0] = '\0';
		laddr = NULL;
		pgpid_ok_here = 0;

		while(fgets(buf, 255, afd)) {
			if(buf[0] != ' ')
				break;

			strip_newline(buf);

			p = rem_tr_space(buf);
			if(!strlen(p))
				continue;

			if(num >= MAX_ADDR_IN_ALIAS)
				break;

			if(pgpid_ok_here && (strncmp(p, "PGPId:", 6) == 0)) {
				p += 6;
				while((*p == ' ') || (*p == 0x09))
					p++;
				if(strncmp(p, "0x", 2))
					display_msg(MSG_WARN, "addr book",
								"Invalid PGP Id: %s", p);
				else {
					if(addr)
						addr->pgpid = strdup(p);
				}
				pgpid_ok_here = 0;
				continue;
			}

			if((addr = get_address(p, ADDR_IGNORE_COMMAS)) == NULL)
				continue;

			num++;
			addr->next_addr = NULL;
			if(laddr)
				laddr->next_addr = addr;
			else
				entry->addr = addr;
			laddr = addr;
			pgpid_ok_here = 1;
		}

		if(num == 0) {
			if(entry->description)
				free(entry->description);
			free(entry);
		} else {
			if(lentry)
				lentry->next = entry;
			else
				book->entries = entry;
			lentry = entry;

			if(entry->description)
				entry->type = ADDR_ALIAS;
			else
				entry->type = ADDR_SINGLE;
		}

		if(!strncmp(buf, "@ ", 2))
			goto newent;
	}

	fclose(afd);
	cur_entry = NULL;
	return 0;
}

int save_book() {
	FILE *afd;
	char buf[255], buftmp[255];
	struct _abook_entry *entry;
	struct _mail_addr *addr;

	if((book == NULL) || readonly) {
		changed = 0;
		return 0;
	}

	snprintf(buftmp, sizeof(buftmp), "%s/.__save_xfbook", configdir);
	if(book->name && strlen(book->name) && strcmp(book->name, "default"))
		snprintf(buf, sizeof(buf), "%s/.xfbook.%s", configdir, book->name);
	else
		snprintf(buf, sizeof(buf), "%s/.xfbook", configdir);

	if((afd = fopen(buftmp, "w")) == NULL) {
		display_msg(MSG_WARN, "save book", "Can not open\n%s", buftmp);
		return -1;
	}

	entry = book->entries;
	while(entry) {
		fprintf(afd, "@ %s\n",
				entry->description ? entry->description : "");
		addr = entry->addr;
		while(addr) {
			fprintf(afd, " %s\n", get_full_addr_line(addr));
			if(addr->pgpid && *addr->pgpid)
				fprintf(afd, " PGPId:%s\n", addr->pgpid);
			addr = addr->next_addr;
		}
		entry = entry->next;
		if(ferror(afd)) {
			if(errno == ENOSPC)
				display_msg(MSG_WARN, "save book", "DISK FULL!");
			else
				display_msg(MSG_WARN, "save book", "Failed to write %s",
							buftmp);
			fclose(afd);
			unlink(buftmp);
			return -1;
		}
	}

	if(fclose(afd) == EOF) {
		if(errno == ENOSPC)
			display_msg(MSG_WARN, "save book", "DISK FULL!");
		else
			display_msg(MSG_WARN, "save book", "Failed to write %s",
						buftmp);
		unlink(buftmp);
		return -1;
	}
#ifdef __EMX__          /* Under OS/2 the file will not be deleted during rename() */
	if(access(buf, 0) == 0) {
		if(unlink(buf) != 0) {
			display_msg(MSG_WARN, "unlink", "delete %s before moving",
						buf);
		}
	}
#endif
	if(rename(buftmp, buf) == -1) {
		display_msg(MSG_WARN, "save book", "rename failed");
		unlink(buftmp);
		return -1;
	}

	changed = 0;
	return 0;
}

struct _abook_entry *get_entry_by_idx(int index) {
	struct _abook_entry *entry;

	if(!book || (index <= 0))
		return NULL;

	entry = book->entries;
	while(entry && --index)
		entry = entry->next;

	return entry;
}

int get_entry_idx(struct _abook_entry *entry) {
	int index;
	struct _abook_entry *entry1;

	if(entry == NULL)
		return -1;

	entry1 = book->entries;
	index = 1;
	while(entry1 && (entry1 != entry)) {
		entry1 = entry1->next;
		index++;
	}

	return entry1 ? index : -1;
}

int convert_book() {
	FILE *afd, *nafd;
	char buf[255], buf1[255], buf2[255];

	snprintf(buf, sizeof(buf), "%s/.xfbook", configdir);
	snprintf(buf1, sizeof(buf1), "%s_tmp", buf);

	if((afd = fopen(buf, "r")) == NULL) {
		display_msg(MSG_WARN, "address book",
					"Can not open address book %s", buf);
		return -1;
	}

	if((nafd = fopen(buf1, "w")) == NULL) {
		display_msg(MSG_WARN, "address book",
					"Can not open address book %s", buf1);
		return -1;
	}


	while(fgets(buf2, 255, afd))
		fprintf(nafd, "@ \n %s", buf2);

	fclose(afd);
	fclose(nafd);

#ifdef __EMX__          /* Under OS/2 the file will not be deleted during rename() */
	if(access(buf, 0) == 0) {
		if(unlink(buf) != 0) {
			display_msg(MSG_WARN, "unlink", "delete %s before moving",
						buf);
		}
	}
#endif
	if(rename(buf1, buf) == -1) {
		display_msg(MSG_WARN, "convert", "failed to rename %s\nto%s", buf1,
					buf);
		return -1;
	}

	return 0;
}

void display_books() {
	const FL_Dirlist *dir;
	int i, num, lbooknum;
	//int books = 1;
	char *p, lbook[MAX_ABOOK_NAMELEN + 1];

	dir = fl_get_dirlist(configdir, ".xfbook.*", &num, 1);
	fl_freeze_form(addr_book_obj->Address_Book);
	fl_clear_browser(addr_book_obj->Abooks_Browse);
	fl_add_browser_line(addr_book_obj->Abooks_Browse, "default");
	strcpy(lbook, "default");
	lbooknum = 1;
	if(dir == NULL)
		goto extbook;
	for(i = 0; i < num; i++) {
		//if(books > MAX_BOOKS)
			//break;
		if(!dir[i].name || !strlen(dir[i].name) ||
		   ((p = strstr(dir[i].name, ".xfbook.")) != dir[i].name) ||
		   (dir[i].type != FT_FILE))
			continue;

		p = dir[i].name + 8;
		if(strlen(p) >= MAX_ABOOK_NAMELEN)
			continue;
		while(*p != '\0') {
			if(!isalpha(*p) && !isdigit(*p))
				break;
			p++;
		}

		if(*p != '\0')
			continue;

		p = dir[i].name + 8;
		fl_add_browser_line(addr_book_obj->Abooks_Browse, p);
		//books++;
		if(book && !strcmp(p, book->name)) {
			lbooknum++;
			strcpy(lbook, p);
		}
	}
	fl_free_dirlist((FL_Dirlist *) dir);

	extbook:
	fl_select_browser_line(addr_book_obj->Abooks_Browse, lbooknum);
	fl_unfreeze_form(addr_book_obj->Address_Book);
	if((book == NULL) || !strcmp(book->name, "default"))
		load_book(NULL);
	else
		load_book(lbook);

	sort_book();
	display_book();
}

int
compare_aentries(struct _abook_entry **entry1,
				 struct _abook_entry **entry2) {
	struct _abook_entry *e1 = *entry1, *e2 = *entry2;
	char *et1, *et2;

	if(e1->type & ADDR_ALIAS)
		et1 = e1->description ? e1->description : e1->addr->addr;
	else
		et1 = e1->addr->addr;

	if(e2->type & ADDR_ALIAS)
		et2 = e2->description ? e2->description : e2->addr->addr;
	else
		et2 = e2->addr->addr;

	if((et1 == NULL) || (et2 == NULL))
		return 0;

	return strcasecmp(et1, et2);
}

void sort_book() {
	struct _abook_entry *entry, **earr;
	int i, total;

	if(!book || !book->entries)
		return;

	total = 0;
	entry = book->entries;
	while(entry) {
		total++;
		entry = entry->next;
	}

	if(
	  (earr =
	   (struct _abook_entry **) malloc(sizeof(struct _abook_entry *) *
									   total)) == NULL) {
		display_msg(MSG_FATAL, "addr_book", "malloc failed");
		return;
	}

	for(entry = book->entries, i = 0; entry; entry = entry->next, i++)
		earr[i] = entry;

	qsort(earr, total, sizeof(struct _abook_entry *),
		  (int (*)(const void *, const void *)) compare_aentries);

	total--;
	for(i = 0; i < total; i++)
		earr[i]->next = earr[i + 1];

	earr[total]->next = NULL;
	book->entries = earr[0];

	free(earr);

	return;
}

void display_book() {
	struct _abook_entry *entry;
	char buf[255], tl = 1, i = 1;

	if(!book)
		return;

	if(cur_entry)
		tl = fl_get_browser_topline(addr_book_obj->Address_Browse);

	fl_freeze_form(addr_book_obj->Address_Book);
	fl_clear_browser(addr_book_obj->Address_Browse);
	entry = book->entries;
	while(entry) {
		switch(entry->type) {
			case ADDR_SINGLE:
				fl_add_browser_line(addr_book_obj->Address_Browse,
									get_full_addr_line(entry->addr));
				break;

			case ADDR_ALIAS:
				if(entry->description)
					snprintf(buf, sizeof(buf), "[%s] %s", entry->description,
							 get_full_addr_line(entry->addr));
				else
					snprintf(buf, sizeof(buf), "[%s]",
							 get_full_addr_line(entry->addr));
				fl_add_browser_line(addr_book_obj->Address_Browse, buf);
				break;
		}

		if(entry == cur_entry) {
			fl_set_input(addr_book_obj->AB_Input,
						 entry->description ? entry->
						 description : get_full_addr_line(entry->addr));
			fl_set_input(addr_book_obj->AB_PGP,
						 (entry->addr
						  && entry->addr->pgpid) ? entry->addr->
						 pgpid : "");
			fl_select_browser_line(addr_book_obj->Address_Browse, i);
			if((i < tl) ||
			   (i >=
				(tl +
				 fl_get_browser_screenlines(addr_book_obj->
											Address_Browse)))) tl = i;
		}
		entry = entry->next;
		i++;
	}

	fl_set_browser_topline(addr_book_obj->Address_Browse, tl);
	if(!book->entries)
		fl_set_input(addr_book_obj->AB_Input, "");
	fl_set_input_selected(addr_book_obj->AB_Input, 1);
	fl_set_input_cursorpos(addr_book_obj->AB_Input, 0, 0);
	fl_set_input_selected(addr_book_obj->AB_PGP, 1);
	fl_set_input_cursorpos(addr_book_obj->AB_PGP, 0, 0);

	fl_unfreeze_form(addr_book_obj->Address_Book);

}

int close_abook(FL_FORM * form, void *data) {
	if(addr_book_obj)
		fl_trigger_object(addr_book_obj->AB_Ok);

	return FL_IGNORE;
}

void addr_book_set_placement(int x, int y) {
	abookx = x;
	abooky = y;
}

struct _mail_addr *addr_book(int *addr_type, addr_dbl_cb callback,
							 int data) {
	XWMHints *xwmh;
	int w, h;
	char geom[16];

	if(ready) {
		XRaiseWindow(fl_display, addr_book_obj->Address_Book->window);
		display_msg(MSG_WARN, "addrbook",
					"Address book is currently in use by another window, close it and try again");
		return NULL;
	}

	if(!book) {
		if(load_book(NULL) == -1)
			return NULL;
	}

	ready = 1;
	changed = 0;
	addr_book_obj = create_form_Address_Book();

	fl_set_browser_fontsize(addr_book_obj->Address_Browse,
							Config.getInt("AddrBookFSize",
										  FL_NORMAL_SIZE));
	fl_set_browser_fontstyle(addr_book_obj->Address_Browse,
							 Config.getInt("AddrBookFStyle",
										   FL_NORMAL_STYLE));
	fl_set_object_color(addr_book_obj->Address_Browse,
						Config.getInt("AddrBookBgCol",
									  FL_TOP_BCOL),
						Config.getInt("AddrBookFgCol",
									  FL_YELLOW));
	fl_set_browser_dblclick_callback(addr_book_obj->Address_Browse,
									 Address_Browse_Dbl_Call, 0);
	fl_set_object_posthandler(addr_book_obj->Address_Browse,
							  Address_Browse_Handler);

	fl_set_browser_fontsize(addr_book_obj->Abooks_Browse,
							Config.getInt("AddrBookFSize",
										  FL_NORMAL_SIZE));
	fl_set_browser_fontstyle(addr_book_obj->Abooks_Browse,
							 Config.getInt("AddrBookFStyle",
										   FL_NORMAL_STYLE));
	fl_set_object_color(addr_book_obj->Abooks_Browse,
						Config.getInt("AddrBookBgCol",
									  FL_TOP_BCOL),
						Config.getInt("AddrBookFgCol",
									  FL_YELLOW));
	fl_set_object_lsize(addr_book_obj->AB_Input,
						Config.getInt("AddrBookFSize",
									  FL_NORMAL_SIZE));
	fl_set_object_lstyle(addr_book_obj->AB_Input,
						 Config.getInt("AddrBookFStyle",
									   FL_NORMAL_STYLE));
	fl_set_input_color(addr_book_obj->AB_Input, FL_BLACK,
					   Config.getInt("AddrBookFgCol",
									 FL_BLUE));
	fl_set_object_color(addr_book_obj->AB_Input,
						Config.getInt("AddrBookBgCol",
									  FL_BLACK),
						Config.getInt("AddrBookBgCol",
									  FL_BLUE));
	fl_set_input_return(addr_book_obj->AB_Input, FL_RETURN_ALWAYS);

	fl_set_object_lsize(addr_book_obj->AB_PGP,
						Config.getInt("AddrBookFSize",
									  FL_NORMAL_SIZE));
	fl_set_object_lstyle(addr_book_obj->AB_PGP,
						 Config.getInt("AddrBookFStyle",
									   FL_NORMAL_STYLE));
	fl_set_input_color(addr_book_obj->AB_PGP, FL_BLACK,
					   Config.getInt("AddrBookFgCol",
									 FL_BLUE));
	fl_set_object_color(addr_book_obj->AB_PGP,
						Config.getInt("AddrBookBgCol",
									  FL_BLACK),
						Config.getInt("AddrBookBgCol",
									  FL_BLUE));
	fl_set_form_minsize(addr_book_obj->Address_Book, 700, 280);
	fl_set_form_maxsize(addr_book_obj->Address_Book, 900, 600);
	fl_set_form_atclose(addr_book_obj->Address_Book, close_abook, NULL);
	Config.setFlags("abookgeom", CF_NOTCHANGED);
	w = 700;
	h = 280;
	sscanf(Config.getCString("abookgeom", ""), "%d %d", &w, &h);
	if(Config.getInt("wplace", 1))
		fl_set_form_geometry(addr_book_obj->Address_Book,
							 (abookx > 0)
							 || !main_form ? abookx : main_form->x,
							 (abooky > 0)
							 || !main_form ? abooky : main_form->y, w, h);

	dbl_callback = callback;
	dbl_callback_data = data;

	fl_show_form(addr_book_obj->Address_Book, FL_PLACE_FREE, FL_FULLBORDER,
				 "Address Book");

	if((xwmh = XAllocWMHints()) != NULL) {
		xwmh->icon_pixmap = icon_ABook;
		xwmh->icon_mask = icon_ABook_sh;
		xwmh->flags = IconPixmapHint | IconMaskHint;
		XSetWMHints(fl_display, addr_book_obj->Address_Book->window, xwmh);
		XFree(xwmh);
	}

	display_books();

	if(addr_type) {
		type = *addr_type;
		switch(*addr_type) {
			case TO_TYPE:
				fl_set_button(addr_book_obj->AB_To, 1);
				break;

			case CC_TYPE:
				fl_set_button(addr_book_obj->AB_Cc, 1);
				break;

			case BCC_TYPE:
				fl_set_button(addr_book_obj->AB_Bcc, 1);
				break;

			default:
				fl_set_button(addr_book_obj->AB_To, 1);
				break;
		}
	} else {
		fl_set_button(addr_book_obj->AB_To, 1);
		type = TO_TYPE;
	}

	fl_do_only_forms();

	if(changed && !readonly) {
		if(display_msg
		   (MSG_QUEST, "address book",
			"Do you want to save changes into the address book?"))
			save_book();
	}
	changed = 0;

	sprintf(geom, "%d %d", addr_book_obj->Address_Book->w,
			addr_book_obj->Address_Book->h);
	Config.set("abookgeom", geom);
	fl_hide_form(addr_book_obj->Address_Book);
	fl_free_form(addr_book_obj->Address_Book);
	free(addr_book_obj);
	addr_book_obj = NULL;

	if(addr_type)
		*addr_type = type;

	ready = 0;
	return cur_entry ? cur_entry->addr : NULL;
}

void AB_Add_Call(FL_OBJECT * obj, long param) {
	char *addr = (char *) fl_get_input(addr_book_obj->AB_Input);
	struct _mail_addr *ma;

	if(strlen(addr) < 2)
		return;

	if(!(ma = get_address(addr, ADDR_IGNORE_COMMAS))) {
		display_msg(MSG_WARN, "add", "Invalid address:\n%s", addr);
		return;
	}

	addr = (char *) fl_get_input(addr_book_obj->AB_PGP);
	if(addr && *addr && strncmp(addr, "0x", 2)) {
		display_msg(MSG_WARN, "add", "Invalid PGP Id: %s", addr);
		return;
	}

	discard_address(ma->next_addr);
	ma->next_addr = NULL;
	ma->pgpid = (addr && *addr) ? strdup(addr) : NULL;

	if((cur_entry = add_book(ma, 1)) != NULL)
		changed = 1;
	discard_address(ma);
	fl_set_input(addr_book_obj->AB_Input, "");
	fl_set_input(addr_book_obj->AB_PGP, "");
	sort_book();
	display_book();
}

void AB_Delete_Call(FL_OBJECT * obj, long param) {
	int i;
	struct _abook_entry *entry, *entry1;

	for(i = 1; i <= fl_get_browser_maxline(addr_book_obj->Address_Browse);
	   i++) {
		if(fl_isselected_browser_line(addr_book_obj->Address_Browse, i)) {
			if((entry = get_entry_by_idx(i)) == NULL)
				continue;

			if(i == 1)
				book->entries = entry->next;
			else {
				if((entry1 = get_entry_by_idx(i - 1)) == NULL)
					continue;
				entry1->next = entry->next;
			}

			discard_address(entry->addr);
			if(entry->description)
				free(entry->description);
			free(entry);
			changed = 1;
			break;
		}
	}

	cur_entry = get_entry_by_idx(i);
	sort_book();
	display_book();
	if(i <= fl_get_browser_maxline(addr_book_obj->Address_Browse)) {
		fl_select_browser_line(addr_book_obj->Address_Browse, i);
		fl_show_browser_line(addr_book_obj->Address_Browse, i);
	}

	return;
}

void AB_Copy(char *name) {
	int i, k = 0;
	struct _abook_entry *entry;
	struct _mail_addr *addr;
	FILE *afd;
	char buf[255];

	if(!strcmp(book->name, name ? name : "default"))
		return;

	for(i = 1; i <= fl_get_browser_maxline(addr_book_obj->Address_Browse);
	   i++) {
		if(fl_isselected_browser_line(addr_book_obj->Address_Browse, i)) {
			if((entry = get_entry_by_idx(i)) == NULL)
				continue;

			if(name && strcmp(name, "default"))
				snprintf(buf, sizeof(buf), "%s/.xfbook.%s", configdir,
						 name);
			else
				snprintf(buf, sizeof(buf), "%s/.xfbook", configdir);

			if((afd = fopen(buf, "a")) == NULL) {
				display_msg(MSG_WARN, "addr copy", "Can not append to\n%s",
							buf);
				return;
			}

			fprintf(afd, "@ %s\n",
					entry->description ? entry->description : "");
			addr = entry->addr;
			while(addr) {
				fprintf(afd, " %s\n", get_full_addr_line(addr));
				if(addr->pgpid && *addr->pgpid)
					fprintf(afd, " PGPId:%s\n", addr->pgpid);
				k++;
				addr = addr->next_addr;
			}

			display_msg(MSG_MSG, "copy",
						"%d address(es) copied to address book %s", k,
						name ? name : "default");

			fclose(afd);
			return;
		}
	}

	return;
}

void AB_To_Call(FL_OBJECT * obj, long param) {
	type = TO_TYPE;
}

void AB_Cc_Call(FL_OBJECT * obj, long param) {
	type = CC_TYPE;
}

void AB_Bcc_Call(FL_OBJECT * obj, long param) {
	type = BCC_TYPE;
}

void AB_Save_Call(FL_OBJECT * obj, long param) {
	save_book();
}

void AB_Import_Book_Call(FL_OBJECT * obj, long param) {
	int imp_pup;

	imp_pup = fl_defpup(obj->form->window, "Import from%t|.mailrc|Pine|Text");

	switch(fl_dopup(imp_pup)) {
		case 1:
			mailrc_import_aliases(NULL);
			break;

		case 2:
			pine_import_aliases(NULL);
			break;
		case 3:
			text_import_aliases(NULL);
			break;
	}

	fl_freepup(imp_pup);
}

void Address_Browse_Dbl_Call(FL_OBJECT * obj, long param) {
	if(dbl_callback && cur_entry)
		dbl_callback(cur_entry->addr, type, dbl_callback_data);
}

void Address_Browse_Call(FL_OBJECT * obj, long param) {
}

int
Address_Browse_Handler(FL_OBJECT * obj, int event,
					   FL_Coord mx, FL_Coord my, int key, void *raw_ev) {
	int num, k;
	struct _abook_entry *entry;
	struct _mail_addr *addr, *addr1;
	char *h;

	if(!book)
		return 0;

	switch(event) {
		case FL_PUSH:
			num = fl_get_browser(obj);

			if((entry = get_entry_by_idx(num)) == NULL)
				break;

			if(key == 3) {
				if(entry->type != ADDR_ALIAS) {
					display_msg(MSG_WARN, "copy address",
								"Can copy only to aliases and address books");
					return 0;
				}
				for(addr = cur_entry->addr, k = 0; addr;
				   addr = addr->next_addr, k++) {
				}

				addr = copy_address_chain(cur_entry->addr);
				if(entry->addr) {
					addr1 = entry->addr;
					while(addr1->next_addr)
						addr1 = addr1->next_addr;
					addr1->next_addr = addr;
				} else
					entry->addr = addr;
				display_msg(MSG_MSG, "copy address",
							"%d address(es) appended to alias %s", k,
							entry->description ? entry->
							description : "noname");
				changed = 1;
			} else {
				h = entry->addr->pgpid;
				if(entry->type == ADDR_SINGLE)
					fl_set_input(addr_book_obj->AB_Input,
								 get_full_addr_line(entry->addr));
				else
					fl_set_input(addr_book_obj->AB_Input,
								 entry->description ? entry->
								 description : get_full_addr_line(entry->
																  addr));
				fl_set_input(addr_book_obj->AB_PGP, h ? h : "");
				fl_set_input_selected(addr_book_obj->AB_Input, 1);
				fl_set_input_cursorpos(addr_book_obj->AB_Input, 0, 0);
				fl_set_input_selected(addr_book_obj->AB_PGP, 1);
				fl_set_input_cursorpos(addr_book_obj->AB_PGP, 0, 0);
				fl_set_focus_object(addr_book_obj->Address_Book,
									addr_book_obj->AB_Input);
			}

			cur_entry = entry;

			if(key == 2) {
				if(dbl_callback && cur_entry)
					dbl_callback(cur_entry->addr, type, dbl_callback_data);
			}
			break;
	}

	return 0;
}

void AB_Input_Call(FL_OBJECT * obj, long param) {
	int i, sline, eline;
	const char *ln;
	const char *inp = fl_get_input(addr_book_obj->AB_Input);

	if(book == NULL)
		return;

	if(strlen(inp) < 1)
		return;

	eline = fl_get_browser_maxline(addr_book_obj->Address_Browse);
	if(eline <= 1)
		return;

	if(param == 1)
		sline = fl_get_browser(addr_book_obj->Address_Browse) + 1;
	else
		sline = 1;

	srch:
	
	for(i = sline; i <= eline; i++) {
		ln = fl_get_browser_line(addr_book_obj->Address_Browse, i);
		if(strlen(ln) < strlen(inp))
			continue;

		if(strcasestr((char *) ln, (char *) inp, 1)) {
			fl_select_browser_line(addr_book_obj->Address_Browse, i);
			fl_show_browser_line(addr_book_obj->Address_Browse, i);
			//here?
			cur_entry = get_entry_by_idx(i);
			fl_set_input(addr_book_obj->AB_PGP,
						 cur_entry->addr->pgpid ? cur_entry->addr->
						 pgpid : "");
			return;
		}
	}

	if(param == 1) {
		param = 0;
		eline = sline;
		sline = 1;
		goto srch;
	}

	return;
}

void AB_Modify_Call(FL_OBJECT * obj, long param) {
	int i;
	struct _mail_addr *ma = NULL;
	struct _abook_entry *entry;
	char *p, buf[255];
	char *addr = (char *) fl_get_input(addr_book_obj->AB_Input);

	if(addr && *addr) {
		if(!(ma = get_address(addr, ADDR_IGNORE_COMMAS))) {
			display_msg(MSG_WARN, "modify", "Invalid address\n\"%s\"",
						addr);
			return;
		}

		discard_address(ma->next_addr);
		ma->next_addr = NULL;
		p = (char *) fl_get_input(addr_book_obj->AB_PGP);
		if(p && *p)
			ma->pgpid = strdup(p);
	}

	if(ma && ma->pgpid && *ma->pgpid && strncmp(ma->pgpid, "0x", 2)) {
		display_msg(MSG_WARN, "modify", "Invalid PGP Id: %s", ma->pgpid);
		discard_address(ma);
		return;
	}

	for(i = 1; i <= fl_get_browser_maxline(addr_book_obj->Address_Browse);
	   i++) {
		if(fl_isselected_browser_line(addr_book_obj->Address_Browse, i)) {
			if((entry = get_entry_by_idx(i)) == NULL)
				break;

			if(entry->type == ADDR_SINGLE) {
				if(!addr || !*addr) {
					display_msg(MSG_WARN, "modify",
								"Empty input, can not modify");
					discard_address(ma);
					return;
				}

				discard_address(entry->addr);
				entry->addr = ma;
			} else {
				if(ma)
					discard_address(ma);
				if(!addr || !*addr) {
					if(!display_msg
					   (MSG_QUEST | MSG_DEFNO, NULL,
						"Convert to single address?\n(all addresses except the first will be lost!!)"))
						return;
					entry->type = ADDR_SINGLE;
					if(entry->description)
						free(entry->description);
					entry->description = NULL;
					discard_address(entry->addr->next_addr);
					entry->addr->next_addr = NULL;
				} else {
					p = addr;
					if(strlen(p) > 16) {
						display_msg(MSG_WARN, "modify",
									"alias name too long");
						return;
					}

					while(*p != '\0') {
						if(!isalpha(*p) && !isdigit(*p) &&
						   (*p != '-') && (*p != '_')) {
							display_msg(MSG_WARN, "modify",
										"invalid character in alias name");
							return;
						}
						p++;
					}

					if(entry->description)
						free(entry->description);
					entry->description = strdup(addr);
				}
			}

			if(entry == NULL)
				return;

			changed = 1;
			if(entry->type == ADDR_ALIAS) {
				if(entry->description)
					snprintf(buf, sizeof(buf), "[%s] %s",
							 cur_entry->description,
							 get_full_addr_line(entry->addr));
				else
					snprintf(buf, sizeof(buf), "[%s]",
							 get_full_addr_line(entry->addr));
			} else
				snprintf(buf, sizeof(buf), "%s",
						 get_full_addr_line(entry->addr));
			fl_replace_browser_line(addr_book_obj->Address_Browse,
									get_entry_idx(entry), buf);

			return;
		}
	}

	discard_address(ma);
	display_book();

	return;
}

void Abook_Browse_Call(FL_OBJECT * obj, long param) {
	int num;
	char *p;

	num = fl_get_browser(obj);
	if(num == 0)
		return;

	p = (char *) fl_get_browser_line(obj, num);
	if(!p)
		return;

	if(fl_mouse_button() == FL_RIGHT_MOUSE) {
		AB_Copy(p);
		return;
	}

	if(num == 1) {
		load_book(NULL);
		sort_book();
		display_book();
		return;
	}

	if(load_book(p) == -1) {
		display_msg(MSG_WARN, "address book",
					"Can not open address book %s", p);
		load_book(NULL);
	}

	sort_book();
	display_book();

	return;
}

void AB_Memb_Call(FL_OBJECT * obj, long param) {
	if(!cur_entry)
		return;

	if(cur_entry->type != ADDR_ALIAS) {
		if(!display_msg(MSG_QUEST, NULL, "convert to alias?"))
			return;
		AB_Alias_Call(NULL, 1);
		return;
	}

	fl_deactivate_form(addr_book_obj->Address_Book);
	edit_entry();
	fl_activate_form(addr_book_obj->Address_Book);
}

void AB_Alias_Call(FL_OBJECT * obj, long param) {
	char *aname, *p;
	struct _abook_entry *entry;
	char buf[255];

	if(param && !cur_entry)
		return;

	aname = (char *) fl_show_input("Alias name:", "");
	if(!aname || !*aname)
		return;

	if(strlen(aname) > 16) {
		display_msg(MSG_WARN, "alias", "alias name too long");
		return;
	}

	p = aname;
	while(*p != '\0') {
		if(!isalpha(*p) && !isdigit(*p) && (*p != '-') && (*p != '_')) {
			display_msg(MSG_WARN, "alias",
						"invalid character in alias name");
			return;
		}
		p++;
	}

	entry = book->entries;
	while(entry) {
		if(entry->description && !strcasecmp(aname, entry->description)) {
			display_msg(MSG_WARN, "alias",
						"alias %s already exists in this address book",
						aname);
			return;
		}
		entry = entry->next;
	}

	if(param == 0) {
		if(
		  (entry =
		   (struct _abook_entry *) malloc(sizeof(struct _abook_entry)))
		  == NULL) {
			display_msg(MSG_FATAL, "add alias to book", "Malloc failed");
			return;
		}

		entry->addr = NULL;
		entry->next = book->entries;
		entry->description = strdup(aname);
		entry->type = ADDR_ALIAS;
		book->entries = entry;
		cur_entry = entry;
		display_book();
	} else {
		entry = cur_entry;
		if(entry->description)
			free(entry->description);
		entry->description = strdup(aname);
		entry->type = ADDR_ALIAS;
	}

	ee:
	if(cur_entry->description)
		snprintf(buf, sizeof(buf), "[%s] %s", cur_entry->description,
				 get_full_addr_line(cur_entry->addr));
	else
		snprintf(buf, sizeof(buf), "[%s]",
				 get_full_addr_line(cur_entry->addr));
	fl_replace_browser_line(addr_book_obj->Address_Browse,
							get_entry_idx(cur_entry), buf);

	fl_deactivate_form(addr_book_obj->Address_Book);
	edit_entry();
	fl_activate_form(addr_book_obj->Address_Book);
	if(!cur_entry->addr) {
		if(display_msg
		   (MSG_QUEST | MSG_DEFNO, "alias", "Alias is empty, delete?"))
			AB_Delete_Call(obj, param);
		else
			goto ee;
	}

	changed = 1;
	return;
}

void AB_Refresh_Book_Call(FL_OBJECT * obj, long param) {
	sort_book();
	display_books();
}

void AB_New_Book_Call(FL_OBJECT * obj, long param) {
	char *p, *p1, buf[255];
	FILE *afd;

	/*if(fl_get_browser_maxline(addr_book_obj->Abooks_Browse) >= MAX_BOOKS) {
		display_msg(MSG_WARN, "address book",
					"Can not have more then %d address books", MAX_BOOKS);
		return;
	}*/

	p = (char *) fl_show_input("Enter address book name:", "");
	if(!p || !*p)
		return;

	if(strlen(p) >= MAX_ABOOK_NAMELEN) {
		display_msg(MSG_WARN, "address book",
					"address book name can not be longer\nthen %d characters",
					MAX_ABOOK_NAMELEN);
		return;
	}

	for(p1 = p; *p1 != '\0'; p1++) {
		if(!isalpha(*p1) && !isdigit(*p)) {
			display_msg(MSG_WARN, "address book",
						"invalid character [%c] in address book name",
						*p1);
			return;
		}
	}

	snprintf(buf, sizeof(buf), "%s/.xfbook.%s", configdir, p);


	if((afd = fopen(buf, "w")) == NULL) {
		display_msg(MSG_WARN, "address book", "Can not open\n%s", buf);
		return;
	}

	fclose(afd);

	fl_add_browser_line(addr_book_obj->Abooks_Browse, p);
	return;
}

void AB_Del_Book_Call(FL_OBJECT * obj, long param) {
	int num;
	char *p, buf[255];

	num = fl_get_browser(addr_book_obj->Abooks_Browse);
	if(num <= 1) {
		display_msg(MSG_WARN, "address book",
					"You can not delete default address book");
		return;
	}

	p = (char *) fl_get_browser_line(addr_book_obj->Abooks_Browse, num);
	if(!p || !strlen(p))
		return;

	if(!display_msg
	   (MSG_QUEST | MSG_DEFNO, NULL,
		"Address book %s will be deleted\nAre you sure?", p))
		return;

	snprintf(buf, sizeof(buf), "%s/.xfbook.%s", configdir, p);
	if(unlink(buf) == -1)
		display_msg(MSG_WARN, "address book",
					"Failed to delete address book %s", p);
	else {
		fl_delete_browser_line(addr_book_obj->Abooks_Browse, num);
		load_book(NULL);
		fl_select_browser_line(addr_book_obj->Abooks_Browse, 1);
		display_book();
	}

	return;
}

void AB_PGP_Call(FL_OBJECT * obj, long param) {
	char *pgpid;

	if(!cur_entry || !cur_entry->addr)
		return;

	pgpid = (char *) fl_get_input(addr_book_obj->AB_PGP);
	if(strlen(pgpid))
		if((cur_entry->addr->pgpid)
		   && !(strcmp(cur_entry->addr->pgpid, pgpid))) {
			free(cur_entry->addr->pgpid);
			cur_entry->addr->pgpid = strdup(pgpid);
		}

	fl_set_focus_object(addr_book_obj->Address_Book,
						addr_book_obj->AB_Input);
	fl_set_input_selected(addr_book_obj->AB_PGP, 1);
	fl_set_input_cursorpos(addr_book_obj->AB_PGP, 0, 0);
}

int mod_addr(struct _mail_addr *oldadr, int *type) {
	struct _mail_addr *newadr;
	int ftype;
	char *p;

	if(modready) {
		XRaiseWindow(fl_display, amod_obj->addrmod->window);
		display_msg(MSG_WARN, "modify address",
					"This dialog is currently in use by another window, close it and try again");
		return -1;
	}

	modready = 1;
	newadr = NULL;
	amod_obj = create_form_addrmod();

	ftype = -1;
	fl_set_input(amod_obj->AMOD_Input,
				 get_charset_addr_line(NULL, oldadr, &ftype));
	if(oldadr->pgpid && *oldadr->pgpid)
		fl_set_input(amod_obj->AMOD_PGP_Input, oldadr->pgpid);
	if(ftype >= 0) {
		fl_set_object_lstyle(amod_obj->AMOD_Input,
							 supp_charsets[ftype].font_style);
		fl_set_object_lsize(amod_obj->AMOD_Input,
							supp_charsets[ftype].font_size);
		fl_set_object_lstyle(amod_obj->AMOD_PGP_Input,
							 supp_charsets[ftype].font_style);
		fl_set_object_lsize(amod_obj->AMOD_PGP_Input,
							supp_charsets[ftype].font_size);
	} else {
		fl_set_object_lstyle(amod_obj->AMOD_Input,
							 supp_charsets[def_charset].font_style);
		fl_set_object_lsize(amod_obj->AMOD_Input,
							supp_charsets[def_charset].font_size);
		fl_set_object_lstyle(amod_obj->AMOD_PGP_Input,
							 supp_charsets[def_charset].font_style);
		fl_set_object_lsize(amod_obj->AMOD_PGP_Input,
							supp_charsets[def_charset].font_size);
	}


	if(type) {
		switch(*type) {
			case TO_TYPE:
				fl_set_button(amod_obj->AMOD_To, 1);
				break;

			case CC_TYPE:
				fl_set_button(amod_obj->AMOD_Cc, 1);
				break;

			case BCC_TYPE:
				fl_set_button(amod_obj->AMOD_Bcc, 1);
				break;

			default:
				fl_set_button(amod_obj->AMOD_To, 1);
				break;
		}
	} else
		fl_set_button(amod_obj->AMOD_To, 1);

	fl_set_form_minsize(amod_obj->addrmod, 470, 120);
	fl_set_form_maxsize(amod_obj->addrmod, 600, 135);
	fl_show_form(amod_obj->addrmod, FL_PLACE_FREE, FL_TRANSIENT,
				 "Modify Address");
	adrwait:
	fl_do_only_forms();
	p = (char *) fl_get_input(amod_obj->AMOD_Input);
	if(p && strlen(p)) {
		if((newadr = get_address(p, ADDR_IGNORE_COMMAS)) == NULL) {
			display_msg(MSG_WARN, "modify",
						"Invalid address, please reenter");
			goto adrwait;
		}
	} else {
		display_msg(MSG_WARN, "modify",
					"Use \"Delete\" button if you want to delete this address");

		fl_hide_form(amod_obj->addrmod);
		fl_free_form(amod_obj->addrmod);
		free(amod_obj);
		amod_obj = NULL;
		modready = 0;

		return -1;
	}

	if(oldadr->addr)
		free(oldadr->addr);
	oldadr->addr = newadr->addr;

	if(oldadr->name)
		free(oldadr->name);
	oldadr->name = newadr->name;

	if(oldadr->comment)
		free(oldadr->comment);
	oldadr->comment = newadr->comment;

	if(oldadr->pgpid)
		free(oldadr->pgpid);
	oldadr->pgpid = strdup(fl_get_input(amod_obj->AMOD_PGP_Input));

	free(newadr);

	if(type) {
		if(fl_get_button(amod_obj->AMOD_To))
			*type = TO_TYPE;
		else if(fl_get_button(amod_obj->AMOD_Cc))
			*type = CC_TYPE;
		else if(fl_get_button(amod_obj->AMOD_Bcc))
			*type = BCC_TYPE;
		else
			*type = TO_TYPE;
	}

	fl_hide_form(amod_obj->addrmod);
	fl_free_form(amod_obj->addrmod);
	free(amod_obj);
	amod_obj = NULL;
	modready = 0;

	return 0;
}

void AMOD_Call(FL_OBJECT * obj, long param) {
}

void display_entry() {
	struct _mail_addr *addr;

	if(!cur_entry)
		return;

	fl_freeze_form(alias_obj->aliased);
	fl_set_input(alias_obj->AL_Input,
				 cur_entry->addr ? get_full_addr_line(cur_entry->
													  addr) : "");
	fl_set_input(alias_obj->AL_InputPGP,
				 cur_entry->addr ? (cur_entry->addr->pgpid ? cur_entry->
									addr->pgpid : "") : "");
	fl_set_input_selected(alias_obj->AL_Input, 1);
	fl_set_input_cursorpos(alias_obj->AL_Input, 0, 0);
	fl_set_input_selected(alias_obj->AL_InputPGP, 1);
	fl_set_input_cursorpos(alias_obj->AL_InputPGP, 0, 0);
	fl_clear_browser(alias_obj->AL_Browse);
	addr = cur_entry->addr;
	while(addr) {
		fl_add_browser_line(alias_obj->AL_Browse,
							get_full_addr_line(addr));
		addr = addr->next_addr;
	}
	fl_select_browser_line(alias_obj->AL_Browse, 1);
	fl_unfreeze_form(alias_obj->aliased);
}

void edit_entry() {
	int w, h;
	char geom[16];

	if(!cur_entry)
		return;

	if(already) {
		XRaiseWindow(fl_display, alias_obj->aliased->window);
		display_msg(MSG_WARN, "alias",
					"This dialog is currently in use by another window, close it and try again");
		return;
	}

	already = 1;
	alias_obj = create_form_aliased();

	fl_set_browser_fontsize(alias_obj->AL_Browse,
							Config.getInt("AddrBookFSize",
										  FL_NORMAL_SIZE));
	fl_set_browser_fontstyle(alias_obj->AL_Browse,
							 Config.getInt("AddrBookFStyle",
										   FL_NORMAL_STYLE));
	fl_set_object_color(alias_obj->AL_Browse,
						Config.getInt("AddrBookBgCol",
									  FL_TOP_BCOL),
						Config.getInt("AddrBookFgCol",
									  FL_YELLOW));
	fl_set_object_lsize(alias_obj->AL_Input,
						Config.getInt("AddrBookFSize",
									  FL_NORMAL_SIZE));
	fl_set_object_lstyle(alias_obj->AL_Input,
						 Config.getInt("AddrBookFStyle",
									   FL_NORMAL_STYLE));
	fl_set_input_color(alias_obj->AL_Input, FL_BLACK,
					   Config.getInt("AddrBookFgCol",
									 FL_BLUE));
	fl_set_object_color(alias_obj->AL_Input,
						Config.getInt("AddrBookBgCol",
									  FL_BLACK),
						Config.getInt("AddrBookBgCol",
									  FL_BLUE));
	fl_set_object_lsize(alias_obj->AL_InputPGP,
						Config.getInt("AddrBookFSize",
									  FL_NORMAL_SIZE));
	fl_set_object_lstyle(alias_obj->AL_InputPGP,
						 Config.getInt("AddrBookFStyle",
									   FL_NORMAL_STYLE));
	fl_set_input_color(alias_obj->AL_InputPGP, FL_BLACK,
					   Config.getInt("AddrBookFgCol",
									 FL_BLUE));
	fl_set_object_color(alias_obj->AL_InputPGP,
						Config.getInt("AddrBookBgCol",
									  FL_BLACK),
						Config.getInt("AddrBookBgCol",
									  FL_BLUE));
	Config.setFlags("aliasgeom", CF_NOTCHANGED);
	w = 410;
	h = 240;
	sscanf(Config.getCString("aliasgeom", ""), "%d %d", &w, &h);
	display_entry();
	fl_set_form_minsize(alias_obj->aliased, 410, 240);
	fl_set_form_maxsize(alias_obj->aliased, 600, 400);
	fl_set_form_size(alias_obj->aliased, w, h);
	fl_show_form(alias_obj->aliased, FL_PLACE_FREE, FL_FULLBORDER,
				 "Alias Members");
	fl_do_only_forms();

	sprintf(geom, "%d %d", alias_obj->aliased->w, alias_obj->aliased->h);
	Config.set("aliasgeom", geom);
	fl_hide_form(alias_obj->aliased);
	fl_free_form(alias_obj->aliased);
	free(alias_obj);
	alias_obj = NULL;
	already = 0;

	return;
}

void AL_Add_Call(FL_OBJECT * obj, long param) {
	char *addr = (char *) fl_get_input(alias_obj->AL_Input);
	char buf[255];
	struct _mail_addr *ma, *ma1;

	if(strlen(addr) < 2)
		return;

	if(!(ma = get_address(addr, ADDR_IGNORE_COMMAS))) {
		display_msg(MSG_WARN, "add", "Invalid address:\n%s", addr);
		return;
	}
	discard_address(ma->next_addr);
	ma->pgpid = strdup((char *) fl_get_input(alias_obj->AL_InputPGP));
	ma->next_addr = NULL;

	cur_entry->type = ADDR_ALIAS;

	if(cur_entry->addr) {
		ma1 = cur_entry->addr;
		while(ma1->next_addr)
			ma1 = ma1->next_addr;

		ma1->next_addr = ma;
	} else
		cur_entry->addr = ma;

	changed = 1;
	display_entry();
	if(cur_entry->description)
		snprintf(buf, sizeof(buf), "[%s] %s", cur_entry->description,
				 get_full_addr_line(cur_entry->addr));
	else
		snprintf(buf, sizeof(buf), "[%s]",
				 get_full_addr_line(cur_entry->addr));
	fl_replace_browser_line(addr_book_obj->Address_Browse,
							get_entry_idx(cur_entry), buf);

	return;
}

void AL_Delete_Call(FL_OBJECT * obj, long param) {
	struct _mail_addr *ma, *ma1;
	char buf[255];
	int num, i;

	num = fl_get_browser(alias_obj->AL_Browse);
	if(num < 1)
		return;
	if(num == 1) {
		if(cur_entry->addr->next_addr == NULL) {
			display_msg(MSG_WARN, "delete alias",
						"Alias should have at least one e-mail address in it");
			return;
		}
	}

	i = num - 1;
	ma = NULL;

	if(i > 0) {
		ma = cur_entry->addr;
		while(ma && --i)
			ma = ma->next_addr;

		if((ma == NULL) || (ma->next_addr == NULL)) {
			display_msg(MSG_WARN, "delete alias", "delete failed");
			return;
		}
	}

	if(ma == NULL) {
		ma1 = cur_entry->addr;
		cur_entry->addr = cur_entry->addr->next_addr;
	} else {
		ma1 = ma->next_addr;
		ma->next_addr = ma1->next_addr;
	}

	ma1->next_addr = NULL;
	discard_address(ma1);
	changed = 1;
	display_entry();

	if(cur_entry->addr && cur_entry->addr->next_addr) {
		cur_entry->type = ADDR_ALIAS;
		if(cur_entry->description)
			snprintf(buf, sizeof(buf), "[%s] %s", cur_entry->description,
					 get_full_addr_line(cur_entry->addr));
		else
			snprintf(buf, sizeof(buf), "[%s]",
					 get_full_addr_line(cur_entry->addr));
	} else
		snprintf(buf, sizeof(buf), "%s",
				 get_full_addr_line(cur_entry->addr));

	fl_replace_browser_line(addr_book_obj->Address_Browse,
							get_entry_idx(cur_entry), buf);
	return;
}

void AL_Modify_Call(FL_OBJECT * obj, long param) {
	char *addr = (char *) fl_get_input(alias_obj->AL_Input);
	struct _mail_addr *ma, *ma1;
	char buf[255];
	int num, i;

	num = fl_get_browser(alias_obj->AL_Browse);
	if(num < 1)
		return;

	i = num - 1;
	ma = cur_entry->addr;

	while(ma && i--)
		ma = ma->next_addr;

	if(!ma)
		return;

	if(!(ma1 = get_address(addr, ADDR_IGNORE_COMMAS))) {
		display_msg(MSG_WARN, "modify", "Invalid address:\n%s", addr);
		return;
	}

	discard_address(ma1->next_addr);
	ma1->next_addr = NULL;
	ma1->pgpid = strdup((char *) fl_get_input(alias_obj->AL_InputPGP));

	if(ma->addr)
		free(ma->addr);
	ma->addr = ma1->addr;

	if(ma->name)
		free(ma->name);
	ma->name = ma1->name;

	if(ma->comment)
		free(ma->comment);
	ma->comment = ma1->comment;

	if(ma->pgpid)
		free(ma->pgpid);
	ma->pgpid = ma1->pgpid;

	ma1->addr = NULL;
	ma1->name = NULL;
	ma1->comment = NULL;
	ma1->pgpid = NULL;
	free(ma1);

	changed = 1;
	display_entry();

	if(cur_entry->addr) {
		cur_entry->type = ADDR_ALIAS;
		if(cur_entry->description)
			snprintf(buf, sizeof(buf), "[%s] %s", cur_entry->description,
					 get_full_addr_line(cur_entry->addr));
		else
			snprintf(buf, sizeof(buf), "[%s]",
					 get_full_addr_line(cur_entry->addr));
	} else
		snprintf(buf, sizeof(buf), "%s",
				 get_full_addr_line(cur_entry->addr));

	fl_replace_browser_line(addr_book_obj->Address_Browse,
							get_entry_idx(cur_entry), buf);
	return;
}

void AL_Call(FL_OBJECT * obj, long param) {
}

void AL_CallPGP(FL_OBJECT * obj, long param) {
}

void AL_Browse_Call(FL_OBJECT * obj, long param) {
	struct _mail_addr *ma = cur_entry->addr;
	int line = fl_get_browser(obj);
	fl_set_input(alias_obj->AL_Input, fl_get_browser_line(obj, line));
	while(line > 1 && ma->next_addr)
		ma = ma->next_addr;
	fl_set_input(alias_obj->AL_InputPGP, ma->pgpid ? ma->pgpid : "");
	fl_set_input_selected(alias_obj->AL_Input, 1);
	fl_set_input_cursorpos(alias_obj->AL_Input, 0, 0);
	fl_set_input_selected(alias_obj->AL_InputPGP, 1);
	fl_set_input_cursorpos(alias_obj->AL_InputPGP, 0, 0);
}

struct _mail_addr *find_alias(char *name) {
	struct _abook_entry *entry;

	if(!book || strcmp(book->name, "default")) {
		if(load_book(NULL) == -1)
			return NULL;
	}

	entry = book->entries;
	while(entry) {
		if(entry->description && !strcasecmp(entry->description, name))
			return entry->addr;
		entry = entry->next;
	}

	return NULL;
}

void mailrc_import_aliases(char *afile) {
	FILE *afd;
	int aerrs = 0;
	char buf[255], *p, *p1;
	struct _abook_entry *entry;
	struct _mail_addr *addr;

	if(!book)
		return;

	if(!afile) {
		fl_set_fselector_title("Select .mailrc file");
		if(!
		   (afile =
			(char *) fl_show_file_selector("Mailrc file", homedir, ".*",
										   ".mailrc")))
			return;
	}

	if((afd = fopen(afile, "r")) == NULL) {
		display_msg(MSG_WARN, "import aliases", "Can not open %s", afile);
		return;
	}

	while(fgets(buf, 255, afd)) {
		strip_newline(buf);
		if(!strncmp(buf, "a ", 2))
			p = buf + 2;
		else if(!strncmp(buf, "alias ", 6))
			p = buf + 6;
		else
			continue;

		while((*p == ' ') || (*p == 0x09))
			p++;

		if((p1 = strchr(p, ' ')) == NULL) {
			display_msg(MSG_LOG, "import alias", "null alias body %-32s",
						p);
			aerrs++;
			continue;
		}

		*p1 = '\0';

		if(strlen(p) > 16)
			p[16] = '\0';

		if(
		  (entry =
		   (struct _abook_entry *) malloc(sizeof(struct _abook_entry)))
		  == NULL) {
			display_msg(MSG_FATAL, "import alias", "Malloc failed");
			fclose(afd);
			return;
		}

		entry->description = strdup(p);
		entry->addr = NULL;
		entry->type = ADDR_ALIAS;

		p = entry->description;
		while(*p != '\0') {
			if(!isgraph(*p))
				*p = '#';
			p++;
		}

		do {
			p = p1 + 1;

			while((*p == ' ') || (*p == 0x09))
				p++;

			if(*p == '\0')
				break;

			if(((*p == '\'') || (*p == '"')) &&
			   ((p1 = strchr(p + 1, *p)) != NULL)) {
				p++;
				*p1 = '\0';
			} else if((p1 = strchr(p, ' ')) != NULL)
				*p1 = '\0';

			if((addr = get_address(p, ADDR_IGNORE_COMMAS)) == NULL) {
				display_msg(MSG_LOG, "import alias",
							"invalid address %-32s", p);
				aerrs++;
				continue;
			}

			addr->next_addr = entry->addr;
			entry->addr = addr;
		} while(p1 != NULL);

		entry->next = book->entries;
		book->entries = entry;
	}

	fclose(afd);

	if(aerrs)
		display_msg(MSG_WARN, "import alias",
					"there were errors during import\ncheck log for more details");

	changed = 1;
	sort_book();
	display_book();

	return;
}

void pine_import_aliases(char *afile) {
	FILE *afd;
	int aerrs = 0;
	char buf[255], buf1[255], *p, *p1, *comm;
	struct _abook_entry *entry;

	if(!afile) {
		fl_set_fselector_title("Select pine .addressbook file");
		if(!
		   (afile =
			(char *) fl_show_file_selector("Pine address book file",
										   homedir, ".*",
										   ".addressbook"))) return;
	}

	if((afd = fopen(afile, "r")) == NULL) {
		display_msg(MSG_WARN, "import aliases", "Can not open %s", afile);
		return;
	}

	buf1[0] = '\0';
	buf[0] = '\0';
	if(!fgets(buf, 255, afd)) {
		display_msg(MSG_WARN, "import aliases", "Empty address book");
		fclose(afd);
		return;
	}

	while(1) {
		if(*buf1 != '\0')
			strcpy(buf, buf1);
		else if(*buf == '\0')
			break;

		if((*buf == '#') || (*buf == ' ')) {
			buf1[0] = '\0';
			if(!fgets(buf, 255, afd))
				break;
			continue;
		}

		strip_newline(buf);

		buf1[0] = '\0';

		while(fgets(buf1, 255, afd)) {
			if(*buf1 == '#')
				continue;

			if(*buf1 != ' ')
				break;

			strip_newline(buf1);
			if((strlen(buf1) + strlen(buf) + 2) >= sizeof(buf)) {
				display_msg(MSG_LOG, "import alias",
							"input buffer too long, truncating");
				aerrs++;
				break;
			}

			p = buf1;
			while(*p == ' ')
				p++;
			strcat(buf, " ");
			strcat(buf, p);
			buf1[0] = '\0';
		}

		if((p = strchr(buf, '\t')) == NULL) {
			aerrs++;
			if(strlen(buf) > 32)
				buf[32] = '\0';
			display_msg(MSG_LOG, "import alias",
						"invalid entry in address book: %s", buf);
			buf[0] = '\0';
			continue;
		}

		*p++ = '\0';

		if(
		  (entry =
		   (struct _abook_entry *) malloc(sizeof(struct _abook_entry)))
		  == NULL) {
			display_msg(MSG_FATAL, "import alias", "Malloc failed");
			fclose(afd);
			return;
		}

		if(strlen(buf) > 16)
			buf[16] = '\0';

		entry->description = strdup(buf);
		entry->addr = NULL;
		entry->type = ADDR_ALIAS;

		p1 = entry->description;
		while(*p1 != '\0') {
			if(!isgraph(*p1))
				*p1 = '#';
			p1++;
		}

		comm = p;
		if((p1 = strchr(p, '\t')) == NULL) {
			aerrs++;
			if(strlen(p) > 32)
				p[32] = '\0';
			display_msg(MSG_LOG, "import alias",
						"invalid entry in address book: %s", p);
			buf[0] = '\0';
			free(entry->description);
			free(entry);
			continue;
		}

		*p1 = '\0';
		p = ++p1;
		if(*p == '(')
			p++;

		if((p1 = strchr(p, '\t')) == NULL)
			p1 = p + strlen(p);
		else
			*p1-- = '\0';

		if(*p1 == ')')
			*p1 = '\0';

		if((entry->addr = get_address(p, 0)) == NULL) {
			aerrs++;
			if(strlen(p) > 32)
				p[32] = '\0';
			display_msg(MSG_LOG, "import alias",
						"invalid address entry in address book: %s", p);
			buf[0] = '\0';
			free(entry->description);
			free(entry);
			continue;
		}

		if((entry->addr->num == 1) && comm) {
			if(entry->addr->name == NULL)
				entry->addr->name = strdup(comm);
			else if(entry->addr->comment == NULL)
				entry->addr->comment = strdup(comm);
		}

		entry->next = book->entries;
		book->entries = entry;
		buf[0] = '\0';
	}

	fclose(afd);

	if(aerrs)
		display_msg(MSG_WARN, "import alias",
					"there were errors during import\ncheck log for more details");

	changed = 1;
	sort_book();
	display_book();

	return;
}

/* expand address list by parsing aliases, macros and eliminating duplicates */
struct _mail_addr *expand_addr_list(struct _mail_msg *msg,
									struct _mail_addr *addr) {
	struct _mail_addr *expanded = NULL, *addr1, *addr2, *aprev, *alias;
	int do_expand, total = 0;

	if(!addr)
		return NULL;

	addr1 = addr;
	aprev = NULL;
	while(addr1) {
		if(total++ >= MAX_RECP_NUM)
			break;
		do_expand = 0;
		alias = NULL;
		if(msg && (addr1->name == NULL) && (addr1->comment == NULL)) {
			if((!strcmp(addr1->addr, "$from") ||
				!strcmp(addr1->addr, "$f")) && msg->header->From) {
				alias = copy_address(msg->header->From);
				alias->next_addr = addr1->next_addr;
				addr1->next_addr = NULL;
				discard_address(addr1);
				if(aprev)
					aprev->next_addr = alias;
				else
					addr = alias;
				addr1 = alias;
				do_expand = -1;
			} else
				if((!strcmp(addr1->addr, "$sender") ||
					!strcmp(addr1->addr, "$s")) && msg->header->Sender) {
				alias = copy_address(msg->header->Sender);
				alias->next_addr = addr1->next_addr;
				addr1->next_addr = NULL;
				discard_address(addr1);
				if(aprev)
					aprev->next_addr = alias;
				else
					addr = alias;
				addr1 = alias;
				do_expand = -1;
			}
		}

		if((do_expand >= 0) &&
		   (addr1->name == NULL) &&
		   (addr1->comment == NULL) &&
		   (strchr(addr1->addr, '@') == NULL)) {
		   if((alias = find_alias(addr1->addr)) != NULL) {
			   do_expand = 1;
	   } 
#ifdef HAVE_LDAP
	   else if ((alias = find_ldap_expansion(addr1->addr)) != NULL) {
		   do_expand = 2;
	   }
#endif
	}

		if(do_expand > 0) {
			addr2 = expanded;
			while(addr2) {
				if(!strcasecmp(addr2->addr, addr1->addr)) {
					do_expand = 0;
					if(aprev)
						aprev->next_addr = addr1->next_addr;
					else
						addr = addr1->next_addr;
					addr1->next_addr = NULL;
					discard_address(addr1);
					addr1 = aprev ? aprev->next_addr : addr;
					break;
				}
				addr2 = addr2->next_addr;
			}

			if(do_expand > 0) {
				alias = copy_address_chain(alias);
				addr2 = alias;
				while(addr2->next_addr && total++)
					addr2 = addr2->next_addr;
				addr2->next_addr = addr1->next_addr;
				addr1->next_addr = expanded;
				expanded = addr1;
				if(aprev)
					aprev->next_addr = alias;
				else
					addr = alias;
				addr1 = alias;
			}
			continue;
		}

		aprev = addr1;
		addr1 = addr1->next_addr;
	}

	discard_address(expanded);

	addr1 = addr;
	aprev = NULL;
	while(addr1) {
		addr2 = addr1->next_addr;

		if(is_newsgroup_addr(addr1, 1)) {
			if(aprev)
				aprev->next_addr = addr1->next_addr;
			else
				addr = addr1->next_addr;
			addr2 = addr1->next_addr;
			addr1->next_addr = NULL;
			discard_address(addr1);
			addr1 = addr2;
			if(addr1 == NULL)
				break;
			continue;
		}

		while((addr2 != NULL) && (addr2->addr != NULL)) {
			if(!strcasecmp(addr1->addr, addr2->addr)) {
				if(aprev)
					aprev->next_addr = addr1->next_addr;
				else
					addr = addr1->next_addr;
				addr2 = addr1->next_addr;
				addr1->next_addr = NULL;
				discard_address(addr1);
				addr1 = addr2;
			}
			if(addr1 == NULL)
				break;
			addr2 = addr2->next_addr;
		}

		if(addr1 == NULL)
			break;
		aprev = addr1;
		addr1 = addr1->next_addr;
	}

	return addr;
}

void AB_Lookup_Call(FL_OBJECT * obj, long data) {
	int keyid;
	char buf[12];
	my_deactivate();
	keyid = PGP_Lookup_PGPId(fl_get_input(addr_book_obj->AB_Input),
							 addr_book_obj->Address_Book);
	my_activate();
	if(keyid) {
		snprintf(buf, sizeof(buf), "0x%X", keyid);
		fl_set_input(addr_book_obj->AB_PGP, buf);
	}
}

void AMOD_Lookup_Call(FL_OBJECT * obj, long data) {
	int keyid;
	char buf[12];
	keyid =
	PGP_Lookup_PGPId(fl_get_input(amod_obj->AMOD_Input),
					 amod_obj->addrmod);
	if(keyid) {
		snprintf(buf, sizeof(buf), "0x%X", keyid);
		fl_set_input(amod_obj->AMOD_PGP_Input, buf);
	}
}

void AMOD_PGP_Call(FL_OBJECT * obj, long data) {
}

void AL_Lookup_Call(FL_OBJECT * obj, long data) {
	int keyid;
	char buf[12];
	keyid =
	PGP_Lookup_PGPId(fl_get_input(alias_obj->AL_Input),
					 alias_obj->aliased);
	if(keyid) {
		snprintf(buf, sizeof(buf), "0x%X", keyid);
		fl_set_input(alias_obj->AL_InputPGP, buf);
	}
}

void text_import_aliases(char *afile) {
	FILE *afd;
	int aerrs = 0;
	char buf[255], *p, *p1, *comm;
	struct _abook_entry *entry;
	struct _mail_addr *addr;
	bool bDone = false;
	char descript[255];
	int iNumAddress = 0;

	display_msg(MSG_MSG, "Text Import", "The first line should be the alias, then each address on a separate line.");

	if(!afile) {
		fl_set_fselector_title("Select text list file");
		if(!(afile = (char *) fl_show_file_selector("Text address book file",
										   homedir, ".*",
										   "list.txt"))) return;
	}

	if((afd = fopen(afile, "r")) == NULL) {
		display_msg(MSG_WARN, "Text Import", "Can not open %s", afile);
		return;
	}

	if(!fgets(descript, 255, afd)) {
		display_msg(MSG_WARN, "Text Import", "Empty address book");
		return;
	}

	if((entry = (struct _abook_entry *) malloc(sizeof(struct _abook_entry))) == NULL) {
		display_msg(MSG_FATAL, "Text Import", "Malloc failed");
		bDone = true;
		return;
	}

	descript[254] = '\0';

	strip_newline(descript);

	entry->description = strdup(descript);
	entry->addr =  NULL;
	entry->type = ADDR_ALIAS;

	display_msg(MSG_LOG, "Text Import", "Importing -> List Description / Alias: \"%s\"", descript);
	while(!bDone) {

		if(!fgets(buf, 255, afd)) {
			bDone = true;
			continue;
		}

		buf[254] = '\0';

		strip_newline(buf);

		addr = get_address(buf,0);

		if(addr == NULL) {
			aerrs++;
			display_msg(MSG_LOG, "Text Import",
						"invalid address entry in address book: %s", p);
			continue;
		}
		display_msg(MSG_LOG, "Text Import", "Found %s (%s)", buf, addr->addr);

		addr->next_addr = entry->addr;
		entry->addr = addr;
		iNumAddress++;
	}

	entry->next = book->entries; //tack onto back of address books
	book->entries = entry; //set new head on address book list

	fclose(afd);

	if(aerrs)
		display_msg(MSG_WARN, "Text Import",
					"there were errors during import\ncheck log for more details");

	display_msg(MSG_MSG, "Text Import", "Imported %i addresses", iNumAddress);
	display_msg(MSG_LOG, "Text Import", "Imported %i addresses", iNumAddress);

	changed = 1;
	sort_book();
	display_book();

	return;
}
