/*
 *   This file is part of AkariXB
 *   Copyright 2015-2018  JanKusanagi JRR <jancoding@gmx.com>
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the
 *   Free Software Foundation, Inc.,
 *   51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA .
 */

#include "activityhandler.h"


ActivityHandler::ActivityHandler(GlobalObject *globalObject,
                                 QObject *parent) : QObject(parent)
{
    this->globalObj = globalObject;

    this->xmppClient = this->globalObj->getXmppClient();
    this->mucManager = this->globalObj->getMucManager();
    this->varParser = this->globalObj->getVariableParser();


    // Start main timer, ticking once a minute, to check if an activity needs to be started
    this->mainTimer = new QTimer(this);
    mainTimer->setInterval(60000); // 60 seconds
    connect(mainTimer, SIGNAL(timeout()),
            this, SLOT(onMainTimerTick()));
    mainTimer->start();


    // Activity-specific timer; signals the end of the activity
    this->currentActivityTimer = new QTimer(this);
    currentActivityTimer->setSingleShot(true);
    connect(currentActivityTimer, SIGNAL(timeout()),
            this, SLOT(activityFinished()));

    qDebug() << "ActivityHandler created";
}

ActivityHandler::~ActivityHandler()
{
    qDebug() << "ActivityHandler destroyed";
}



void ActivityHandler::setActivityData(QVariantList dataList)
{
    this->activityDataList = dataList;
}


QXmppPresence::AvailableStatusType ActivityHandler::presenceFromIndex(int presenceType)
{
    QXmppPresence::AvailableStatusType statusType;

    switch (presenceType)
    {
    case 1:
        statusType = QXmppPresence::Chat;
        break;

    case 2:
        statusType = QXmppPresence::Away;
        break;

    case 3:
        statusType = QXmppPresence::XA;
        break;

    case 4:
        statusType = QXmppPresence::DND;
        break;

    default:
        statusType = QXmppPresence::Online;
    }

    return statusType;
}


//////////////////////////////////////////////////////////////////////////////
///////////////////////////////////// SLOTS //////////////////////////////////
//////////////////////////////////////////////////////////////////////////////


void ActivityHandler::onMainTimerTick()
{
    QTime timeNow = QTime::currentTime();
    QDate dateNow = QDate::currentDate();

    if (timeNow.hour() == 0 && timeNow.minute() == 0) // FIXME: ensure it's not skipped
    {
        qDebug() << "--AH DEBUG-- Midnight! Resetting DoneToday values";
        this->doneToday.clear();
    }

    if (!activityInProgress.isEmpty())
    {
        return;
    }


    foreach (QVariant mapVariant, this->activityDataList)
    {
        QVariantMap map = mapVariant.toMap();

        QString activityName = map.value("name").toString();

        bool shouldDoIt = true; // Until proven guilty

        int timesDoneToday = this->doneToday.value(activityName).toInt();
        if (timesDoneToday >= map.value("timesPerDay").toInt())
        {
            shouldDoIt = false;
        }

        if (shouldDoIt)
        {
            // FIXME: handle cases such as 22:00 to 02:00
            if (map.value("minTime").toTime() > timeNow
             || map.value("maxTime").toTime() < timeNow)
            {
                shouldDoIt = false;
            }

            // TODO: Check also dateNow
        }

        if (shouldDoIt)
        {
            shouldDoIt = Helpers::probability(map.value("probability").toInt());
        }



        if (shouldDoIt) // It passed all the trials, and survived!
        {
            this->activityInProgress = activityName;
            this->doneToday.insert(activityName, timesDoneToday + 1);

            int duration = Helpers::randomBetween(map.value("minDuration").toInt(),
                                                  map.value("maxDuration").toInt());

            QTime durationTime;
            durationTime.setHMS(0, 0, 0); // Qt 5.x requires this initialization
            durationTime = durationTime.addSecs(duration * 60);

            QString durationString;
            if (durationTime.hour() > 0)
            {
                durationString.append(tr("%Ln hour(s)", "",
                                         durationTime.hour()));
            }

            if (durationTime.minute() > 0)
            {
                if (!durationString.isEmpty())
                {
                    durationString.append(", ");
                }

                durationString.append(tr("%Ln minute(s)", "",
                                         durationTime.minute()));
            }


            this->currentActivityTimer->start(duration * 60000); // mins to msecs


            QTime endTime = timeNow.addSecs(duration * 60);

            this->globalObj->addToLog(tr("Starting activity %1, which "
                                         "will end at %2.")
                                      .arg("'" + activityInProgress + "'")
                                      .arg(endTime.toString(Qt::SystemLocaleShortDate))
                                      + " [" + durationString + "]");


            QString timeRange = timeNow.toString(Qt::SystemLocaleShortDate)
                                + " ~ "
                                + endTime.toString(Qt::SystemLocaleShortDate);

            this->globalObj->notifyActivityChange(activityInProgress, timeRange);


            // Status type and message
            QXmppPresence presence = this->xmppClient->clientPresence();
            this->oldStatusType = presence.availableStatusType();
            this->oldStatusMessage = presence.statusText();

            presence.setAvailableStatusType(this->presenceFromIndex(map.value("statusType")
                                                                        .toInt()));

            QString statusMsg = Helpers::randomString(map.value("statusMessages")
                                                         .toStringList());
            statusMsg = this->varParser->getParsed(statusMsg);
            presence.setStatusText(statusMsg);


            this->messagesAfter = map.value("msgAfter").toStringList();


            if (this->globalObj->connectedToServer())
            {
                // Send the new presence
                this->xmppClient->setClientPresence(presence);


                // Send messages to configured recipients about the activity
                m_allRecipientJids.clear();

                const int msgToRooms = map.value("msgToRooms").toInt();
                if (msgToRooms == 0)       // All rooms
                {
                    m_allRecipientJids = this->globalObj->getJoinedRooms();
                }
                else if (msgToRooms == 1)  // Several rooms
                {
                    m_allRecipientJids = Helpers::someStrings(globalObj->getJoinedRooms(),
                                                              50);
                }
                else if (msgToRooms == 2)  // A few rooms
                {
                    m_allRecipientJids = Helpers::someStrings(globalObj->getJoinedRooms(),
                                                              20);
                }


                // TODO: add JIDs of active private chats


                // Additional JIDs can be added regardless of the other options
                m_allRecipientJids += map.value("specificJids")
                                         .toString()
                                         .split(QStringLiteral(" "));

                QStringList messagesBefore = map.value("msgBefore")
                                                .toStringList();
                QString msgBefore;
                foreach (QString recipient, m_allRecipientJids)
                {
                    msgBefore = Helpers::randomString(messagesBefore);
                    msgBefore = this->varParser->getParsed(msgBefore);

                    if (msgBefore != "*") // TMP FIXME?
                    {
                        this->globalObj->sendMessageToJid(recipient, msgBefore);
                    }
                }
            }

            this->globalObj->notifyStatusChange(presence);


            break;
        }
    }
}


void ActivityHandler::activityFinished()
{
    this->globalObj->addToLog(tr("Activity ended: %1.").arg(activityInProgress));

    activityInProgress.clear();
    this->globalObj->notifyActivityChange("[ " + tr("None",
                                                    "Regarding an activity")
                                          + " ]",
                                          ""); // No duration

    QXmppPresence presence = this->xmppClient->clientPresence();
    presence.setAvailableStatusType(this->oldStatusType);
    presence.setStatusText(this->oldStatusMessage);

    if (this->globalObj->connectedToServer())
    {
        this->xmppClient->setClientPresence(presence);

        // Send messages to configured recipients, about the end of the activity
        QString msgAfter;
        foreach (QString recipient, m_allRecipientJids)
        {
            msgAfter = Helpers::randomString(this->messagesAfter);
            msgAfter = this->varParser->getParsed(msgAfter);

            if (msgAfter != "*")  // TMP FIXME?
            {
                this->globalObj->sendMessageToJid(recipient, msgAfter);
            }
        }
    }

    this->globalObj->notifyStatusChange(presence);

    // Cleanup
    m_allRecipientJids.clear();
    this->messagesAfter.clear();
}

