/*
 *   This file is part of AkariXB
 *   Copyright 2015-2019  JanKusanagi JRR <jancoding@gmx.com>
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the
 *   Free Software Foundation, Inc.,
 *   51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA .
 */

#include "stringlistwidget.h"


StringListWidget::StringListWidget(bool forKeyValues,
                                   QWidget *parent) : QWidget(parent)
{
    m_forKeyValues = forKeyValues;
    m_lowercaseItems = false;
    // By default, don't allow duplicates when it's a key-value kind of widget
    m_duplicatesAllowed = !m_forKeyValues;


    m_listWidget = new QListWidget(this);
    m_listWidget->setDragDropMode(QAbstractItemView::InternalMove); // Movable items
    connect(m_listWidget, &QListWidget::itemDoubleClicked,
            this, &StringListWidget::editItem);

    m_addButton = new QToolButton(this);
    m_addButton->setIcon(QIcon::fromTheme("list-add",
                                          QIcon(":/images/list-add.png")));
    m_addButton->setToolTip("<b></b>" // HTMLize to get wordwrap
                            + tr("Add an item to the list"));
    connect(m_addButton, &QAbstractButton::clicked,
            this, &StringListWidget::addToList);


    m_editButton = new QToolButton(this);
    m_editButton->setIcon(QIcon::fromTheme("document-edit",
                                           QIcon(":/images/button-edit.png")));
    m_editButton->setToolTip("<b></b>"
                             + tr("Edit the selected item (you can also "
                                  "double-click it)"));
    connect(m_editButton, &QAbstractButton::clicked,
            this, &StringListWidget::editItem);


    m_removeButton = new QToolButton(this);
    m_removeButton->setIcon(QIcon::fromTheme("list-remove",
                                             QIcon(":/images/list-remove.png")));
    m_removeButton->setToolTip("<b></b>"
                               + tr("Remove the selected item"));
    connect(m_removeButton, &QAbstractButton::clicked,
            this, &StringListWidget::removeFromList);


    if (m_forKeyValues)
    {
        m_itemTextEdit = new QTextEdit(this);
        connect(m_listWidget, &QListWidget::currentRowChanged,
                this, &StringListWidget::onRowChanged);

        m_updateButton = new QPushButton(QIcon::fromTheme("dialog-ok-apply",
                                                          QIcon(":/images/button-apply.png")),
                                         QString(),
                                         this);
        m_updateButton->setToolTip("<b></b>"
                                   + tr("Update the definition of the item"));
        connect(m_updateButton, &QAbstractButton::clicked,
                this, &StringListWidget::updateItemText);
    }


    // Layout
    m_buttonsLayout = new QVBoxLayout();
    m_buttonsLayout->addWidget(m_addButton);
    m_buttonsLayout->addWidget(m_editButton);
    m_buttonsLayout->addSpacing(4);
    m_buttonsLayout->addWidget(m_removeButton);
    m_buttonsLayout->addStretch();

    m_mainLayout = new QHBoxLayout();
    m_mainLayout->setContentsMargins(0, 0, 0, 0);
    m_mainLayout->addWidget(m_listWidget,       1);
    m_mainLayout->addLayout(m_buttonsLayout);
    if (m_forKeyValues)
    {
        m_mainLayout->addWidget(m_itemTextEdit, 3);
        m_mainLayout->addWidget(m_updateButton);
    }

    this->setLayout(m_mainLayout);

    enableRelevantButtons();

    qDebug() << "StringListWidget created";
}


StringListWidget::~StringListWidget()
{
    qDebug() << "StringListWidget destroyed";
}


void StringListWidget::setLowercaseItems(bool lowercaseItems)
{
    m_lowercaseItems = lowercaseItems;
}

void StringListWidget::setDuplicatesAllowed(bool duplicatesAllowed)
{
    m_duplicatesAllowed = duplicatesAllowed;
}


void StringListWidget::clearList()
{
    for (int counter = 0; counter < m_listWidget->count(); ++counter)
    {
        delete m_listWidget->item(counter);
    }

    m_listWidget->clear();

    if (m_forKeyValues)
    {
        m_itemTextEdit->clear();
    }

    enableRelevantButtons();
}


void StringListWidget::addStringsToList(QStringList stringList)
{
    m_listWidget->addItems(stringList); // FIXME? Check duplicates if needed

    // If nothing's selected, select the first one, so all buttons make sense
    if (m_listWidget->currentRow() < 0)
    {
        m_listWidget->setCurrentRow(0);
    }

    enableRelevantButtons();
}


QStringList StringListWidget::getStringsFromList()
{
    QStringList stringList;

    for (int counter = 0; counter < m_listWidget->count(); ++counter)
    {
        stringList.append(m_listWidget->item(counter)->text());
    }

    return stringList;
}


void StringListWidget::addItemsToList(QVariantMap stringMap)
{
    foreach (QString key, stringMap.keys())
    {
        key = key.trimmed();

        if (m_lowercaseItems)
        {
            key = key.toLower();
        }

        // FIXME? Check for duplicates? Data comes from a file, should be OK

        QListWidgetItem *item = new QListWidgetItem(key);
        item->setData(Qt::UserRole, stringMap.value(key));

        m_listWidget->addItem(item);
    }

    enableRelevantButtons();

    m_listWidget->setCurrentRow(0);
}

QVariantMap StringListWidget::getItemsFromList()
{
    QVariantMap stringMap;

    for (int counter = 0; counter < m_listWidget->count(); ++counter)
    {
        stringMap.insert(m_listWidget->item(counter)->text(),
                         m_listWidget->item(counter)->data(Qt::UserRole)
                                                    .toString());
    }

    return stringMap;
}


bool StringListWidget::stringExists(QString text)
{
    for (int counter = 0; counter < m_listWidget->count(); ++counter)
    {
        if (m_listWidget->item(counter)->text() == text)
        {
            return true;
        }
    }

    return false;
}


void StringListWidget::enableRelevantButtons()
{
    const bool buttonEnabled = (m_listWidget->count() > 0);

    m_editButton->setEnabled(buttonEnabled);
    m_removeButton->setEnabled(buttonEnabled);

    if (m_forKeyValues)
    {
        m_itemTextEdit->setEnabled(buttonEnabled);
        m_updateButton->setEnabled(buttonEnabled);
    }
}


//////////////////////////////////////////////////////////////////////////////
///////////////////////////////////// SLOTS //////////////////////////////////
//////////////////////////////////////////////////////////////////////////////


void StringListWidget::addToList()
{
    QString newString = QInputDialog::getText(this,
                                              tr("Enter text") + " - AkariXB",
                                              tr("Enter a new line of text "
                                                 "for the list") // kinda TMP
                                              + "\n"
                                                "\t\t\t\t\t\t\t\t\t\t" // Ensure extra width
                                                "\n");
    newString = newString.trimmed();

    if (!newString.isEmpty())
    {
        if (m_lowercaseItems)
        {
            newString = newString.toLower();
        }

        if (!m_duplicatesAllowed && this->stringExists(newString))
        {
            QMessageBox::warning(this,
                                 tr("The item already exists") + " - AkariXB",
                                 tr("Error: There is already an item in "
                                    "the list with the text %1, and this "
                                    "list does not allow duplicates.")
                                 .arg("<b>'" + newString + "'</b>"));
            return;
        }

        m_listWidget->addItem(newString);

        const int lastRow = m_listWidget->count() - 1;
        m_listWidget->setCurrentRow(lastRow);

        enableRelevantButtons();

        if (m_forKeyValues)
        {
            m_itemTextEdit->setFocus();
        }
    }
}


void StringListWidget::editItem()
{
    const int row = m_listWidget->currentRow();

    if (row != -1)
    {
        QListWidgetItem *item = m_listWidget->item(row);

        // FIXME: this widget should have the option to be set to use
        // QInputDialog::getMultiLineText() for certain uses
        QString newString = QInputDialog::getText(this,
                                                  tr("Enter text") + " - AkariXB",
                                                  tr("Update the text for "
                                                     "this item") // kinda TMP
                                                  + "\n"
                                                    "\t\t\t\t\t\t\t\t\t\t" // Ensure extra width
                                                    "\n",
                                                  QLineEdit::Normal,
                                                  item->text());
        newString = newString.trimmed();

        if (!newString.isEmpty())
        {
            if (m_lowercaseItems)
            {
                newString = newString.toLower();
            }

            item->setText(newString);
        }
    }
}


void StringListWidget::removeFromList()
{
    const int row = m_listWidget->currentRow();

    // FIXME: ask for confirmation? Could be optional upon widget creation
    if (row != -1)
    {
        QListWidgetItem *item = m_listWidget->takeItem(row);
        delete item;
    }

    if (m_listWidget->count() == 0) // No items left
    {
        enableRelevantButtons();
    }
}


void StringListWidget::onRowChanged(int row)
{
    QString itemText;

    if (row != -1)
    {
        itemText = m_listWidget->item(row)->data(Qt::UserRole).toString();
    }

    m_itemTextEdit->setText(itemText);
    m_itemTextEdit->setEnabled(true);
    m_updateButton->setEnabled(true);
}


void StringListWidget::updateItemText()
{
    const int row = m_listWidget->currentRow();

    if (row != -1)
    {
        m_listWidget->item(row)->setData(Qt::UserRole,
                                         m_itemTextEdit->toPlainText());
    }
}

