/*
  AntiRight
  (c) 2002-2007 Jeffrey Bedard
  antiright@gmail.com

  This file is part of AntiRight.

  AntiRight is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  AntiRight is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with AntiRight; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
*/

#include "gtkshell.h"

static void
add_piped_label(GSH * gsh, const gchar * output)
{
	$(gsh, add.label, output);
}

static void
add_piped_text(GSH * gsh, const gchar * output)
{
	GtkWidget * text_area;

	text_area=gsh_text_area(gsh);
	(void)gsh_text_area_set_text(text_area, output);
}

#define GENERIC_FILE_CONTENTS(gsh, add, input)\
{\
	add(gsh, input);\
	g_free(input);\
}

#define ADD_PIPED_GENERIC(gsh, add)\
	GENERIC_FILE_CONTENTS(gsh, add, antiright_read_file(stdin))

#define GENERIC_COMMANDED_ADD(gsh, command, add)\
	GENERIC_FILE_CONTENTS(gsh, add, antiright_pipe_read(command))

static void
add_new_window(GSH * gsh)
{
	$(gsh, add.window);
	gsh->rows->current=1;
	$(gsh, init);
	GSH_SET(GSH_IN_SECONDARY_WIN);
}

static void
add_separator(GSH * gsh)
{
	GtkWidget * separator;

	/* Create a separator widget.  If GSH has one row per column, 
	   use a vertical separator, else use a horizontal separator.  
	   This allows the separator to be used for ACE's Deskbar.  */
	separator = (gsh->rows->total == 1) ? 
		gtk_vseparator_new() : gtk_hseparator_new();
	/* Show and manage the separator.  */
	$(gsh, manage, separator);
}

#define GCA_WIDGET(func) \
	GENERIC_COMMANDED_ADD(gsh, argv[*counter], func)

static void
handle_commanded_add_arguments(GSH * gsh, const gint argc, const gchar **argv, 
	gint *counter)
{
#ifndef DEBUG
	UNUSED(argc);
#endif /* DEBUG */
	if(!argv[*counter] || !argv[(*counter)+1])
		return;
	GSH_COUNT(argc, counter);
	switch(argv[(*counter)-1][3])
	{
	case 'b':
		GCA_WIDGET(gsh->add.button);
		break;
	case 'l':
		GCA_WIDGET(gsh->add.label);
		break;
	}
}

#ifdef HAVE_VTE
static void
add_shell(GSH * gsh, const gchar * command)
{
	/* Create a terminal widget that is running COMMAND.  */
	const GtkWidget * terminal = $(gsh->terminal, add, command);

	/* Let GSH manage the terminal widget.  */
	$(gsh, manage, (GtkWidget *)terminal);
}
#endif /* HAVE_VTE */

static void
add_command_shell(GSH * gsh, const gchar * command)
{
	GSHCommandShell * shell; 

	shell = ARNEW(gsh, GSHCommandShell);
	$(gsh, manage, $(shell, get));
	if(command)
		$(shell, run, command);
	/* The command instance is not freed here,
	 * as it is accessed by callbacks.  */
}

static void
handle_applet_arguments(GSH * gsh, const gint argc, const gchar ** argv,
	gint * counter)
{
	const gchar * current_arg = argv[*counter];
	g_assert(strlen(current_arg) > 2);
	switch(current_arg[3])
	{
	case 'c':  /* Add clock.  */
		$(gsh, manage, gsh_clock_new(gsh));
		break;
	default:
		ARWARN("invalid applet argument");
	}
}

void
gsh_handle_add_arguments(GSH * gsh, const gint argc, const gchar **argv, 
	gint *counter)
{
	g_assert(strlen(argv[(*counter)]) > 1);
	switch (argv[(*counter)][2])
	{
	case 'A': /* Add an internally updated applet.  */
		handle_applet_arguments(gsh, argc, argv, counter);
		break;
	case 'b':
		GSH_COUNT(argc, counter);
		gsh_setup_hpaned(gsh, argv[*counter]);
		break;

	case 'C':
		add_command_shell(gsh, NULL);
		break;

	case 'c':
		handle_commanded_add_arguments(gsh, argc, argv, counter);
		break;

	case 'e':
		GSH_COUNT(argc, counter);
		gsh_text_editor(gsh, argv[*counter]);
		break;

	case 'f': /* Add a form with check boxes or radio buttons.  */
		GSH_COUNT(argc, counter);
		gsh_add_form(gsh, argv[(*counter)]);
		break;

	case 'L': /* Add a list.  */
		GSH_COUNT(argc, counter);
		$(gsh, manage, gsh_tree_view_new(gsh, argv[*counter]));
		break;

	case 'l': /* Add a label.  */
		GSH_COUNT(argc, counter);
		$(gsh, add.label, argv[(*counter)]);
		break;

	case 'P':
		ADD_PIPED_GENERIC(gsh, add_piped_text);
		break;

	case 'p':
		ADD_PIPED_GENERIC(gsh, add_piped_label);
		break;

	case 'r': /* Force creation of a new row.  */
		$(gsh->rows, new);
		break;

	case 'S': /* Add a separator.  */
		add_separator(gsh);
		break;

	case 's': /* Add a shell.  */
		GSH_COUNT(argc, counter);
#ifdef HAVE_VTE
		add_shell(gsh, argv[*counter]);
#else /* not HAVE_VTE */
		add_command_shell(gsh, argv[*counter]);
#endif /* HAVE_VTE */
		break;

	case 't':
		if (argc != *counter)
			GSH_COUNT(argc, counter);

		gsh_setup_command_entry(gsh, argv[(*counter)]);

		break;
	case 'T':
		gsh_text_area(gsh);
		break;

	case 'u':
		gsh_handle_updated_arguments(gsh, argc, argv, counter);
		break;

	case 'w':
		add_new_window(gsh);
		break;

	default:
		ARWARN("invalid add option");
	}
}
