/*
  AntiRight
  (c) 2002-2007 Jeffrey Bedard
  antiright@gmail.com

  This file is part of AntiRight.

  AntiRight is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  AntiRight is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with AntiRight; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
*/

#include "gtkshell.h"

void
gsh_send_signal(GtkWidget * widget, const gchar * signal_name)
{
	g_signal_emit_by_name(widget, signal_name);
}

static void
gsh_delete_GSHCBData(struct GSHCBData * cb)
{
	g_free(cb->data);
	g_free(cb);
}

static void
terminal_feed(gchar * command, GtkWidget * terminal)
{
	gchar ** split_command;
	gchar * feed_command;

	if(!terminal)
	{
		ARWARN("terminal command requires a terminal" 
			"to have been created");
		return;
	}	

	command+=6;
	split_command=g_strsplit(command, "#", 2);
	ar_asprintf(&feed_command, "%s\n", split_command[0]);
	g_strfreev(split_command);
#ifdef HAVE_VTE
	vte_terminal_feed_child(VTE_TERMINAL(terminal), 
		feed_command, strlen(feed_command));
#endif /* HAVE_VTE */
	g_free(feed_command);
}


#define MATCH(str) !strncmp(command, str, strlen(str))

static gboolean
test_special_text_editor_command(struct GSH * gsh, const gchar * command)
{
	if(MATCH("Save_As...___"))
		gsh_editor_save_as(gsh);
	else if(MATCH("Save___"))
		gsh_editor_save(gsh);
	else if(MATCH("Open...___"))
		gsh_editor_open(gsh);
	else if(MATCH("Cut___"))
		gsh_send_signal(gsh->editor.widget, "cut-clipboard");
	else if(MATCH("Copy___"))
		gsh_send_signal(gsh->editor.widget, "copy-clipboard");
	else if(MATCH("Paste___"))
		gsh_send_signal(gsh->editor.widget, "paste-clipboard");
	else
		return FALSE;

	/* A special command match was found.  */
	return TRUE;
}

static gboolean
test_special_command(struct GSHCBData * cb)
{
	const gchar * command = cb->data;
	struct GSH * gsh = cb->gsh;


	if(MATCH("Exit__"))
	{	
		/* GSH is cleaned up when this causes 
		 * a return back to main().  */
		gtk_main_quit();
	}
	else if(MATCH("TERM__"))
	{
		terminal_feed((gchar *)command, gsh->terminal->widget);
	}
	else 
	{
		/* Test other special command(s).  */
		if(!test_special_text_editor_command(gsh, command))
		{
			/* No special command found.  */
			return FALSE;
		}
	}

	/* Special command handled.  */
	return TRUE;
}

static void
gsh_system_cb(GtkWidget *widget, gpointer data)
{
#ifdef DEBUG
	ARBUG("system_cb");
#endif	/* DEBUG */
	UNUSED(widget);

	ARWIFNP(data)
	{
		struct GSHCBData *cb = data;
		if(!test_special_command(cb))
			(void) antiright_system(cb->data);
		GSH_TEST_EXIT(cb->gsh);
	}
#ifdef LIBGC
	GC_gcollect();
#endif /* LIBGC */
}

static void
exec_and_test_print(struct GSHCBData * cb, const gchar * input)
{
	struct GSH * gsh;

	gsh=cb->gsh;

	if(GSH_FLAG(GSH_PROMPT_ECHOES))
		g_print("%s\n", input);
	else
		sysprintf("%s %s", (char *) cb->data, 
			 (char *) input);
}

static void
gsh_entry_cb(GtkWidget *widget, gpointer data)
{
	struct GSHCBData *cb;

#ifdef DEBUG
	ARPASSERT(widget);
	ARPASSERT(data);
#endif /* DEBUG */

	/* Convert to dereferencable form.  */
	cb = (struct GSHCBData *) data;

	{
		GtkWidget * combo_box;
		/* INPUT must not be freed, as 
		   it is internal to the widget.   */
		const gchar *input;	
		/* Get widget from which to retrieve text.  */
		combo_box=gtk_widget_get_ancestor(widget, GTK_TYPE_COMBO_BOX);
		/* Get the text from the widget.  */
		input = gtk_combo_box_get_active_text(
				GTK_COMBO_BOX(combo_box));
#ifdef DEBUG
		/* Check return value INPUT.  */
		ARPASSERT(input);
#endif /* DEBUG */
		exec_and_test_print(cb, input);
		/* Determine whether or not to exit 
		   because of this callback.  */
		GSH_TEST_EXIT(cb->gsh);
		/* Add retrieved text to combo box popup list.  */
		gtk_combo_box_prepend_text(GTK_COMBO_BOX(combo_box),
			(gchar *)input);
	}
}

/* Assign the function pointers for GSHCBData's methods.  */
static void
assign_GSHCBData_methods(struct GSHCBData * cb)
{
	cb->delete=&gsh_delete_GSHCBData;
	cb->entry=&gsh_entry_cb;
	cb->system=&gsh_system_cb;
}

/* GSHCBData Constructor */
struct GSHCBData *
gsh_new_GSHCBData(struct GSH * gsh, gpointer data)
{
	struct GSHCBData * cb;

	/* Allocate.  */
	cb = xmalloc(sizeof(struct GSHCBData));
	/* Store constructor arguments.  */
	cb->gsh=gsh;
	cb->data=data;
	/* Set up methods.  */
	assign_GSHCBData_methods(cb);

	return cb;
}

