/*  Begin callback.cpp  */

/*  OpenGL (GLUT) callbacks functions  */

/*
  Copyright (C) 2003  Jocelyn Frchot

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; version 2 of the License.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/


/****************  includes  ****************/


#include "callback.h"

#include "camera.h"
#include "draw.h"
#include "fps.h"
#include "help_phillips.h"
#include "scene_phill_ugauss.h"
#include "sun.h"
#include "transform.h"

/*  config  */
#include "config.h"

/*  graphic lib  */
extern "C"
{
#include <GL/gl.h>
#include <GL/glu.h>
#include <GL/glut.h>
}

/*  C++ lib  */
#include <string>


/****************  namespaces  ****************/


using namespace std;

namespace
{

  /****************  static classes  ****************/


  class Camera *camera;
  class Fps *fps;
  class Help_Phillips *help;
  class Scene_Phill_Ugauss *scene;
  class Sun *sun;


  /****************  static variables  ****************/


  /*  used to synchronize everything  */
  float time_current;  /*  miliseconds  */
  /*  time elapsed during pause  */
  float time_pause;    /*  miliseconds  */

  /*  strings to display  */
  string *string_fps;
  string *string_scene;
  bool is_string_help_display;
  bool is_string_fps_display;
  bool is_string_scene_display;

  /*  pause  */
  bool is_pause;

  /*  mouse  */
  bool mouse_is_moving;
  int mouse_start_x, mouse_start_y;


  /**************** static functions prototypes ****************/


  void reset_camera(class Camera *camera, class Scene *scene);
  /*  GLUT callback functions  */
  void idle(void);
  void display(void);
  void reshape(int w, int h);
  void mouse(int button, int state, int x, int y);
  void motion(int x, int y);
  void keyboard(unsigned char c, int x, int y);
  void special(int key, int x, int y);


}  /*  namespace  */


/**************** functions ****************/


void
callback_init(void)
{
  /*  classes  */

  /*  scene  */
  scene = new class Scene_Phill_Ugauss(Config_scene_points_x_index,
				       Config_scene_points_z_index,
				       Config_scene_size_x_index,
				       Config_scene_size_z_index,
				       Config_scene_depth_index,
				       Config_scene_displ_fact_index,
				       Config_scene_spect_fact_index,
				       Config_scene_small_wave_index,
				       Config_scene_wind_speed_index,
				       Config_scene_wind_angle_index,
				       Config_scene_surface_alpha_index,
				       Config_scene_is_wired,
				       Config_scene_is_normals_draw,
				       Config_scene_is_tiled,
				       Config_scene_is_stone_draw);
  /*  camera  */
  camera = new class Camera(Config_camera_view_mode);
  reset_camera(camera, scene);
  /*  fps  */
  fps = new class Fps(Config_fps_frames_max, glutGet(GLUT_ELAPSED_TIME));
  /*  sun  */
  sun = new class Sun(GL_LIGHT0,
		      Config_gluperspective_zfar * 0.9,
		      Config_sun_day_length,
		      Config_sun_latitude,
		      Config_sun_day_of_year);

  /*  variables  */

  /*  time  */
  time_current = 0.0;
  time_pause = 0.0;
  /*  strings  */
  help = new class Help_Phillips();
  string_fps = new string;
  string_scene = new string;
  is_string_help_display = Config_is_string_help_display;
  is_string_fps_display = Config_is_string_fps_display;
  is_string_scene_display = Config_is_string_scene_display;
  /*  pause  */
  is_pause = Config_is_pause;
  /*  mouse  */
  mouse_is_moving = false;
  mouse_start_x = 0;
  mouse_start_y = 0;

  /*  screen clear color  */
  glClearColor(0.4, 0.7, 1.0, 0.0);

  /*  hints  */
  glHint(GL_PERSPECTIVE_CORRECTION_HINT, Config_glhint_perspective);
  glHint(GL_POINT_SMOOTH_HINT, Config_glhint_point);
  glHint(GL_LINE_SMOOTH_HINT, Config_glhint_line);
  glHint(GL_POLYGON_SMOOTH_HINT, Config_glhint_polygon);

  /*  light  */
  /*  for sun reflection on water surface  */
  glLightModeli(GL_LIGHT_MODEL_LOCAL_VIEWER, GL_TRUE);
  glLightModeli(GL_LIGHT_MODEL_TWO_SIDE, GL_FALSE);
  glShadeModel(GL_SMOOTH);
  glEnable(GL_LIGHTING);

  /*  cull face  */
  glCullFace(GL_FRONT);
  glEnable(GL_CULL_FACE);

  /*  depth  */
  glEnable(GL_DEPTH_TEST);

  /*  set GLUT functions  */
  glutIdleFunc(idle);
  glutDisplayFunc(display);
  glutReshapeFunc(reshape);
  glutMouseFunc(mouse);
  glutMotionFunc(motion);
  glutKeyboardFunc(keyboard);
  glutSpecialFunc(special);
}


/**************** static functions ****************/


namespace
{


void
reset_camera(class Camera *camera, class Scene *scene)
{
  float position_z;

  position_z = scene->get_whole_size_z() / 2;
  camera->reset(0.0, 5.0, position_z);
}


/****  GLUT callback functions  ****/

void
idle(void)
{
  float lib_time;

  lib_time = 0.0;

  if (!is_pause)
    {
      time_current = glutGet(GLUT_ELAPSED_TIME) - time_pause;
      /*  sets scene time  */
      if (is_string_fps_display)
	{
	  lib_time = glutGet(GLUT_ELAPSED_TIME);
	}
      scene->set_time(time_current / 1000.0);
      if (is_string_fps_display)
	{
	  lib_time = glutGet(GLUT_ELAPSED_TIME) - lib_time;
	}
    }

  /*  sets camera if swimming */
  if (!is_pause && (camera->view_mode == Swimming))
    {
      int position_x, position_z;

      position_x = scene->get_points_x() / 2;
      position_z = 0;  /*  or scene->get_points_z(), which is the same  */
      /*  sets camera one meter above the surface  */
      camera->set_position_y(1.0 + scene->get_surface_height(position_x,
							     position_z));
    }

  glutPostRedisplay();

  /*  computes FPS  */
  if (is_string_fps_display)
    {
      glFinish();  /*  waits until all GL commands are realized  */
      fps->set_elapsed_time(glutGet(GLUT_ELAPSED_TIME), lib_time);
    }
}


void
display(void)
{
  /*  clears screen  */
  glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);

  glMatrixMode(GL_MODELVIEW);
  glPushMatrix();
  {
    glLoadIdentity();

    /*  scene  */
    transform_scene_push(scene, camera);
    {
      /*  sets light, must be done after transform  */
      sun->set_position(time_current / 1000.0);
      draw(scene);
    }
    transform_scene_pop();

    /*  sun  */
    /*  drawn after scene because of transparency  */
    transform_sun_push(sun, camera);
    {
      draw(sun);
    }
    transform_sun_pop();

    /*  strings  */
    if (is_string_fps_display)
      {
	fps->get_string(string_fps);
	transform_string_push(true);
	{
	  draw_string(string_fps, 10, 20);
	}
	transform_string_pop();
      }
    if (is_string_help_display)
      {
	transform_string_push(true);
	{
	  draw_string(help->get_message(), 10, 40);
	}
	transform_string_pop();
      }
    if (is_string_scene_display)
      {
	scene->get_string(string_scene);
	/*  displays from bottom  */
	transform_string_push(false);
	{
	  draw_string(string_scene, 10, 20);
	}
	transform_string_pop();
      }
  }
  glMatrixMode(GL_MODELVIEW);
  glPopMatrix();

  glutSwapBuffers();
}


void
reshape(int w, int h)
{
  glViewport(0, 0, w, h);
  glMatrixMode(GL_PROJECTION);
  {
    glLoadIdentity();
    gluPerspective(Config_gluperspective_fovy,
		   static_cast<float>(w) / static_cast<float>(h),
		   Config_gluperspective_znear,
		   Config_gluperspective_zfar);
  }
  glMatrixMode(GL_MODELVIEW);
  glLoadIdentity();
}


void
mouse(int button, int state, int x, int y)
{
  /*  rotate the scene with the left mouse button  */
  if (button == GLUT_LEFT_BUTTON)
    {
      if (state == GLUT_DOWN)
	{
	  mouse_is_moving = true;
	  mouse_start_x = x;
	  mouse_start_y = y;
	}

      if (state == GLUT_UP)
	{
	  mouse_is_moving = false;
	}
    }
}


void
motion(int x, int y)
{
  if (mouse_is_moving)
    {
      camera->turn(y - mouse_start_y, x - mouse_start_x);
      mouse_start_x = x;
      mouse_start_y = y;

      glutPostRedisplay();
    }
}


void
keyboard(unsigned char c, int x, int y)
{
  /*  avoids compiler warnings for unused parameters  */
  static_cast<void>(x);
  static_cast<void>(y);

  /****  general  ****/
  
  /*  help (keys list)  */
  if (c == help->Key_help)
    {
      is_string_help_display = !is_string_help_display;
    }

  /*  quit  */
  else if (c == help->Key_quit || c == 27)  /*  escape key  */
    {
      exit(EXIT_SUCCESS);
    }

  /*  pause  */
  else if (c == help->Key_pause)
    {
      is_pause = !is_pause;
      if (!is_pause)
	{
	  time_pause = glutGet(GLUT_ELAPSED_TIME) - time_current;
	}
    }

  /*  display info  */
  else if (c == help->Key_info)
    {
      is_string_scene_display = !is_string_scene_display;
    }

  /*  display FPS  */
  else if (c == help->Key_fps)
    {
      is_string_fps_display = !is_string_fps_display;
      if (is_string_fps_display)
	{
	  fps->reset(time_current);
	}
    }

  /*  draw graduated stone  */
  else if (c == help->Key_stone)
    {
      scene->is_stone_draw_toggle();
    }


  /****  camera  ****/

  /*  switch view mode  */
  else if (c == help->Key_view)
    {
      camera->view_mode_switch();
      reset_camera(camera, scene);
    }


  /****  surface look  ****/

  /*  switch surface display color/wire  */
  else if (c == help->Key_wire)
    {
      scene->is_wired_toggle();
    }

  /*  show surface normals (wire mode)  */
  else if (c == help->Key_normals)
    {
      scene->is_normals_draw_toggle();
    }

  /*  tiled surface  */
  else if (c == help->Key_tiled)
    {
      scene->is_tiled_toggle();
      /*  whole size of the scene has changed  */
      reset_camera(camera, scene);
    }

  /*  surface alpha (opacity)  */
  else if (c == help->Key_surface_alpha_decrease)
    {
      scene->surface_alpha_decrease();
    }
  else if (c == help->Key_surface_alpha_increase)
    {
      scene->surface_alpha_increase();
    }


  /****  surface physics  ****/

  /*  number of points  */
  else if (c == help->Key_points_decrease)
    {
      scene->points_decrease();
    }
  else if (c == help->Key_points_increase)
    {
      scene->points_increase();
    }
  /*  only X direction  */
  else if (c == help->Key_points_x_decrease)
    {
      scene->points_x_decrease();
    }
  else if (c == help->Key_points_x_increase)
    {
      scene->points_x_increase();
    }
  /*  only Z direction  */
  else if (c == help->Key_points_z_decrease)
    {
      scene->points_z_decrease();
    }
  else if (c == help->Key_points_z_increase)
    {
      scene->points_z_increase();
    }

  /*  surface size  */
  /*  change whole size of the scene so we call "reset_camera()"  */
  else if (c == help->Key_size_decrease)
    {
      scene->size_decrease();
      reset_camera(camera, scene);
    }
  else if (c == help->Key_size_increase)
    {
      scene->size_increase();
      reset_camera(camera, scene);
    }
  /*  only X direction  */
  else if (c == help->Key_size_x_decrease)
    {
      scene->size_x_decrease();
      reset_camera(camera, scene);
    }
  else if (c == help->Key_size_x_increase)
    {
      scene->size_x_increase();
      reset_camera(camera, scene);
    }
  /*  only Z direction  */
  else if (c == help->Key_size_z_decrease)
    {
      scene->size_z_decrease();
      reset_camera(camera, scene);
    }
  else if (c == help->Key_size_z_increase)
    {
      scene->size_z_increase();
      reset_camera(camera, scene);
    }

  /*  displacement factor  */
  else if (c == help->Key_displacement_factor_decrease)
    {
      scene->displacement_factor_decrease();
    }
  else if (c == help->Key_displacement_factor_increase)
    {
      scene->displacement_factor_increase();
    }

  /*  floor depth  */
  else if (c == help->Key_depth_decrease)
    {
      scene->depth_decrease();
    }
  else if (c == help->Key_depth_increase)
    {
      scene->depth_increase();
    }


  /****  Phillips spectrum  ****/

  /*  spectrum factor  */
  else if (c == help->Key_spectrum_factor_decrease)
    {
      scene->spectrum_factor_decrease();
    }
  else if (c == help->Key_spectrum_factor_increase)
    {
      scene->spectrum_factor_increase();
    }

  /*  smallest wave length  */
  else if (c == help->Key_smallest_wave_decrease)
    {
      scene->smallest_wave_decrease();
    }
  else if (c == help->Key_smallest_wave_increase)
    {
      scene->smallest_wave_increase();
    }

  /*  wind speed  */
  else if (c == help->Key_wind_speed_decrease)
    {
      scene->wind_speed_decrease();
    }
  else if (c == help->Key_wind_speed_increase)
    {
      scene->wind_speed_increase();
    }

  /*  wind angle  */
  else if (c == help->Key_wind_angle_decrease)
    {
      scene->wind_angle_decrease();
    }
  else if (c == help->Key_wind_angle_increase)
    {
      scene->wind_angle_increase();
    }
}


void
special(int key, int x, int y)
{
  /*  avoids compiler warnings for unused parameters  */
  static_cast<void>(x);
  static_cast<void>(y);

  switch (key)
    {
      /*  forward  */
    case GLUT_KEY_UP:
      camera->move_forward();
      break;

      /*  backward  */
    case GLUT_KEY_DOWN:
      camera->move_backward();
      break;

      /*  left  */
    case GLUT_KEY_LEFT:
      camera->move_left();
      break;

      /*  right  */
    case GLUT_KEY_RIGHT:
      camera->move_right();
      break;

      /*  up  */
    case GLUT_KEY_PAGE_UP:
      camera->move_up();
      break;

      /*  down  */
    case GLUT_KEY_PAGE_DOWN:
      camera->move_down();
      break;
    }
}


}  /*  namespace  */


/*  End callback.cpp  */
