/*  Begin camera.cpp  */

/*
  Copyright (C) 2003  Jocelyn Frchot

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; version 2 of the License.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/


/****************  includes  ****************/


#include "camera.h"

/*  local includes  */
#include "include/constant.h"

/*  C lib  */
extern "C"
{
#include <math.h>
}


/****************  public functions  ****************/


/*  creates a camera at the origin, looking down the Z axis  */
Camera::Camera(enum camera_view_mode view_mode)
{
  this->view_mode = view_mode;

  this->reset(0.0, 0.0, 0.0);
}


/*  virtual destructor  */
Camera::~Camera(void)
{
}


/****  set  ****/

/*  Observing -> Flying -> Swimming  */
void
Camera::view_mode_switch(void)
{
  switch (this->view_mode)
    {
    case Observing:
      this->view_mode = Flying;
      break;
    case Flying:
      this->view_mode = Swimming;
      break;
    case Swimming:
      this->view_mode = Observing;
      break;
    }
}


/*  sets camera position (in meters), looking down the Z axis  */
void
Camera::reset(float position_x, float position_y, float position_z)
{
  this->position_x = position_x;
  this->position_z = position_z;
  this->position_y = position_y;
  this->angle_x = 0.0;
  this->angle_y = 0.0;
  this->angle_y_rad = 0.0;
}


void
Camera::set_position_y(float position_y)
{
  this->position_y = position_y;
}


/****  move  ****/

void
Camera::move_forward(void)
{
  switch (this->view_mode)
    {
    case Observing:
      this->position_z -= 1;
      break;

    case Flying:
      this->position_x += sinf(this->angle_y_rad);
      this->position_z -= cosf(this->angle_y_rad);
      break;

    case Swimming:
      break;
    }
}


void
Camera::move_backward(void)
{
  switch (this->view_mode)
    {
    case Observing:
      this->position_z += 1;
      break;

    case Flying:
      this->position_x -= sinf(this->angle_y_rad);
      this->position_z += cosf(this->angle_y_rad);
      break;

    case Swimming:
      break;
    }
}


void
Camera::move_left(void)
{
  switch (this->view_mode)
    {
    case Observing:
      this->position_x -= 1;
      break;

    case Flying:
      this->position_x -= cosf(this->angle_y_rad);
      this->position_z -= sinf(this->angle_y_rad);
      break;

    case Swimming:
      break;
    }
}


void
Camera::move_right(void)
{
  switch (this->view_mode)
    {
    case Observing:
      this->position_x += 1;
      break;

    case Flying:
      this->position_x += cosf(this->angle_y_rad);
      this->position_z += sinf(this->angle_y_rad);
      break;

    case Swimming:
      break;
    }
}


void
Camera::move_up(void)
{
  switch (this->view_mode)
    {
    case Observing:
    case Flying:
      this->position_y += 1;
      break;

    case Swimming:
      break;
    }
}


void
Camera::move_down(void)
{
  switch (this->view_mode)
    {
    case Observing:
    case Flying:
      this->position_y -= 1;
      break;

    case Swimming:
      break;
    }
}


/*  angles in degrees  */
void
Camera::turn(float angle_x, float angle_y)
{
  this->angle_x += angle_x;
  this->angle_y += angle_y;
  this->angle_y_rad += angle_y * Constant_deg_to_rad;
}


/*
  fills three elements "position" vector with "opposite" camera position
  (i.e. suitable for OpenGL transforms)
*/
void
Camera::get_opposite_position(float *position) const
{
  position[0] = - this->position_x;
  position[1] = - this->position_y;
  position[2] = - this->position_z;
}


/*  gets angles in degrees  */
void
Camera::get_angle(float &angle_x, float &angle_y) const
{
  angle_x = this->angle_x;
  angle_y = this->angle_y;
}


/*  End camera.cpp  */
