/*  Begin draw_surface.cpp  */

/*  OpenGL drawing functions for surface  */

/*
  Copyright (C) 2003  Jocelyn Frchot

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; version 2 of the License.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/


/****************  includes  ****************/


#include "draw_surface.h"

/*  local include  */
#include "include/memory.h"

/*  graphic lib  */
extern "C"
{
#include <GL/gl.h>
}


/****************  static functions prototypes  ****************/


namespace
{
  /****  color  ****/
  void draw_color_squares(int number_of_surfaces_width,
			  int points_x,
			  int points_z,
			  float size_x,
			  int step,
			  int z_dir_start,
			  float ***surface,
			  float ***normals);
  void draw_color_squares_with_junction(int number_of_surfaces_width,
					int points_x,
					int points_z,
					float size_x,
					int step,
					float ***surface,
					float ***normals);
  /****  wire  ****/
  void draw_wire_lines(int number_of_surfaces_width,
		       int points_x,
		       float size_x,
		       int z_end_z_dir,
		       int z_end_x_dir,
		       int step,
		       float ***surface);
  void draw_wire_normals(int number_of_surfaces_width,
			 float scale_normals,
			 int points_x,
			 float size_x,
			 int z_end,
			 int step,
			 float ***surface,
			 float ***normals);
  void draw_wire_one_normal(int index_x,
			    int index_z,
			    float shift_x,
			    float scale_normals,
			    float ***surface,
			    float *normal);

  /****  non draw  ****/
  /*  copys three elements from normals to vector at given indexe  */
  void normals_copy(int index_x,
		    int index_z,
		    float ***normals,
		    float *vector);
  void junction_surface_compute(float ***surface, int points_x, int step);
}


/****************  functions  ****************/


void
draw_surface_color(int points_x,
		   int points_z,
		   float size_x,
		   float size_z,
		   float alpha,
		   bool is_tiled,
		   float ***surface,
		   float ***normals)
{
  int number_of_surfaces_width;
  int step;

  GLfloat mat_shininess[] = { 120.0 };
  GLfloat mat_specular[] = { 0.8, 0.8, 0.8, 1.0 };
  GLfloat mat_diffuse[] = { 0.3, 0.5, 0.75, alpha };
  GLfloat mat_ambient[] = { 0.15, 0.5, 0.35, 1.0 };

  glPushAttrib(GL_ALL_ATTRIB_BITS);
  {
    glMaterialfv(GL_FRONT, GL_SHININESS, mat_shininess);
    glMaterialfv(GL_FRONT, GL_SPECULAR, mat_specular);
    glMaterialfv(GL_FRONT, GL_DIFFUSE, mat_diffuse);
    glMaterialfv(GL_FRONT, GL_AMBIENT, mat_ambient);
    /*  transparency  */
    glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
    glEnable(GL_BLEND);

    if (!is_tiled)
      {
	number_of_surfaces_width = 1;
	step = 1;
	draw_color_squares(number_of_surfaces_width,
			   points_x,
			   points_z,
			   size_x,
			   step,
			   0,
			   surface,
			   normals);
      }
    else
      {
	number_of_surfaces_width = 3;

	/*  draws at normal resolution  */
	glPushMatrix();
	{
	  glTranslatef(0.0, 0.0, size_z * 2.0);

	  step = 1;
	  draw_color_squares_with_junction(number_of_surfaces_width,
					   points_x,
					   points_z,
					   size_x,
					   step,
					   surface,
					   normals);
	}
	glPopMatrix();
	/*  draws at resolution / 4  */
	glPushMatrix();
	{
	  glTranslatef(0.0, 0.0, size_z);

	  step = 2;
	  draw_color_squares_with_junction(number_of_surfaces_width,
					   points_x,
					   points_z,
					   size_x,
					   step,
					   surface,
					   normals);
	}
	glPopMatrix();

	/*  draws at resolution / 16  */
	step = 4;
	draw_color_squares(number_of_surfaces_width,
			   points_x,
			   points_z,
			   size_x,
			   step,
			   0,
			   surface,
			   normals);
      }
  }
  glPopAttrib();
}


void
draw_surface_wire(int points_x,
		  int points_z,
		  float size_x,
		  float size_z,
		  float resolution_x,
		  float resolution_z,
		  bool is_normals_draw,
		  bool is_tiled,
		  float ***surface,
		  float ***normals)
{
  int number_of_surfaces_width;
  float scale_normals;
  int z_end_z_dir, z_end_x_dir, z_end_normals, step;

  scale_normals = resolution_x + resolution_z / 2.0;

  glPushAttrib(GL_ALL_ATTRIB_BITS);
  {
    glDisable(GL_LIGHTING);
    glShadeModel(GL_FLAT);

    if (!is_tiled)
      {
	number_of_surfaces_width = 1;
	z_end_z_dir = points_z + 1;
	z_end_x_dir = points_z + 1;
	step = 1;

	draw_wire_lines(number_of_surfaces_width,
			points_x,
			size_x,
			z_end_z_dir,
			z_end_x_dir,
			step,
			surface);
	if (is_normals_draw)
	  {
	    z_end_normals = points_z + 1;
	    draw_wire_normals(number_of_surfaces_width,
			      scale_normals,
			      points_x,
			      size_x,
			      z_end_normals,
			      step,
			      surface,
			      normals);
	  }
      }
    else
      {
	number_of_surfaces_width = 3;
	/*  draws at normal resolution  */
	glPushMatrix();
	{
	  glTranslatef(0.0, 0.0, size_z * 2.0);

	  z_end_z_dir = points_z + 1;
	  z_end_x_dir = points_z + 1;
	  step = 1;
	  draw_wire_lines(number_of_surfaces_width,
			  points_x,
			  size_x,
			  z_end_z_dir,
			  z_end_x_dir,
			  step,
			  surface);
	  if (is_normals_draw)
	    {
	      z_end_normals = points_z + 1;
	      draw_wire_normals(number_of_surfaces_width,
				scale_normals,
				points_x,
				size_x,
				z_end_normals,
				step,
				surface,
				normals);
	    }
	}
	glPopMatrix();
	/*  draws at resolution / 4  */
	glPushMatrix();
	{
	  glTranslatef(0.0, 0.0, size_z);

	  z_end_z_dir = points_z + 1;
	  z_end_x_dir = points_z;
	  step = 2;
	  draw_wire_lines(number_of_surfaces_width,
			  points_x,
			  size_x,
			  z_end_z_dir,
			  z_end_x_dir,
			  step,
			  surface);
	  if (is_normals_draw)
	    {
	      z_end_normals = points_z;
	      draw_wire_normals(number_of_surfaces_width,
				scale_normals,
				points_x,
				size_x,
				z_end_normals,
				step,
				surface,
				normals);
	    }
	}
	glPopMatrix();
	/*  draws at resolution / 16  */
	z_end_z_dir = points_z + 1;
	z_end_x_dir = points_z;
	step = 4;
	draw_wire_lines(number_of_surfaces_width,
			points_x,
			size_x,
			z_end_z_dir,
			z_end_x_dir,
			step,
			surface);
	if (is_normals_draw)
	  {
	    z_end_normals = points_z;
	    draw_wire_normals(number_of_surfaces_width,
			      scale_normals,
			      points_x,
			      size_x,
			      z_end_normals,
			      step,
			      surface,
			      normals);
	  }
      }
  }
  glPopAttrib();
}


/****************  static functions  ****************/


namespace
{


/****  color  ****/
void
draw_color_squares(int number_of_surfaces_width,
		   int points_x,
		   int points_z,
		   float size_x,
		   int step,
		   int z_dir_start,
		   float ***surface,
		   float ***normals)
{
  float temp_normals[3];
  float shift_x;
  int index_x;
  int i, j, k, l;

  for (l = 0; l < number_of_surfaces_width; l++)
    {
      shift_x = size_x * l;
      for (i = 0; i < points_x; i += step)
	{
	  glBegin(GL_TRIANGLE_STRIP);
	  {
	    for (j = z_dir_start; j < points_z + 1; j += step)
	      {
		for (k = 0; k < step * 2; k += step)
		  {
		    index_x = i + k;
		    normals_copy(index_x, j, normals, temp_normals);
		    glNormal3fv(temp_normals);
		    glVertex3f(surface[0][index_x][j] + shift_x,
			       surface[1][index_x][j],
			       surface[2][index_x][j]);
		  }
	      }
	  }
	  glEnd();
	}
    }
}


void
draw_color_squares_with_junction(int number_of_surfaces_width,
				 int points_x,
				 int points_z,
				 float size_x,
				 int step,
				 float ***surface,
				 float ***normals)
{
  float ***temp_junction_surface;

  int k, i;


  temp_junction_surface = memory_alloc_3d<float>(3, points_x + 1, step);

  for (k = 0; k < 3; k++)
    {
      for (i = 0; i < points_x + 1; i++)
	{
	  temp_junction_surface[k][i][0] = surface[k][i][0];
	  temp_junction_surface[k][i][step] = surface[k][i][step];
	}
    }
  junction_surface_compute(temp_junction_surface, points_x, step);

  /*  draws junction  */
  draw_color_squares(number_of_surfaces_width,
		     points_x,
		     step,
		     size_x,
		     step,
		     0,
		     temp_junction_surface,
		     normals);
  /*  draws the rest of the surface  */
  draw_color_squares(number_of_surfaces_width,
		     points_x,
		     points_z,
		     size_x,
		     step,
		     step,
		     surface,
		     normals);

  memory_free_3d(3, points_x + 1, temp_junction_surface);
}


/****  wire  ****/

void
draw_wire_lines(int number_of_surfaces_width,
		int points_x,
		float size_x,
		int z_end_z_dir,
		int z_end_x_dir,
		int step,
		float ***surface)
{
  float shift_x;
  int i, j, k;

  float color[3] = { 0.0, 0.0, 1.0 };

  glPushAttrib(GL_ALL_ATTRIB_BITS);
  {
    glColor3fv(color);

    /*  draws lines in z direction  */
    for (i = 0; i < points_x; i += step)
      {
	for (k = 0; k < number_of_surfaces_width; k++)
	  {
	    shift_x = size_x * k;
	    glBegin(GL_LINE_STRIP);
	    {
	      for (j = 0; j < z_end_z_dir; j += step)
		{
		  glVertex3f(surface[0][i][j] + shift_x,
			     surface[1][i][j],
			     surface[2][i][j]);
		}
	    }
	    glEnd();
	  }
      }
    glBegin(GL_LINE_STRIP);
    {
      for (j = 0; j < z_end_z_dir; j += step)
	{
	  glVertex3f(surface[0][i][j] + size_x * (number_of_surfaces_width - 1),
		     surface[1][i][j],
		     surface[2][i][j]);
	}
    }
    glEnd();

    /*  draws lines in x direction  */
    for (j = 0; j < z_end_x_dir; j += step)
      {
	glBegin(GL_LINE_STRIP);
	{
	  for (k = 0; k < number_of_surfaces_width; k++)
	    {
	      shift_x = size_x * k;
	      for (i = 0; i < points_x; i += step)
		{
		  glVertex3f(surface[0][i][j] + shift_x,
			     surface[1][i][j],
			     surface[2][i][j]);
		}
	    }
	  glVertex3f(surface[0][i][j] + size_x * (number_of_surfaces_width - 1),
		     surface[1][i][j],
		     surface[2][i][j]);
	}
	glEnd();
      }
  }
  glPopAttrib();
}


void
draw_wire_normals(int number_of_surfaces_width,
		  float scale_normals,
		  int points_x,
		  float size_x,
		  int z_end,
		  int step,
		  float ***surface,
		  float ***normals)
{
  float temp_normal[3];
  float shift_x;
  int i, j, k;
  float color[3] = { 1.0, 0.0, 0.0 };

  glPushAttrib(GL_ALL_ATTRIB_BITS);
  {
    glColor3fv(color);

    for (i = 0; i < points_x; i += step)
      {
	for (j = 0; j < z_end; j += step)
	  {
	    normals_copy(i, j, normals, temp_normal);
	    for (k = 0; k < number_of_surfaces_width; k++)
	      {
		shift_x = size_x * k;
		draw_wire_one_normal(i,
				     j,
				     shift_x,
				     scale_normals,
				     surface,
				     temp_normal);
	      }
	  }
      }
    for (j = 0; j < z_end; j += step)
      {
	normals_copy(points_x, j, normals, temp_normal);
	shift_x = size_x * (number_of_surfaces_width - 1);
	draw_wire_one_normal(points_x,
			     j,
			     shift_x,
			     scale_normals,
			     surface,
			     temp_normal);
      }
  }
  glPopAttrib();
}


void
draw_wire_one_normal(int index_x,
		     int index_z,
		     float shift_x,
		     float scale_normals,
		     float ***surface,
		     float *normal)
{
  float base_x, base_y, base_z;

  glBegin(GL_LINE_STRIP);
  {
    base_x = surface[0][index_x][index_z] + shift_x;
    base_y = surface[1][index_x][index_z];
    base_z = surface[2][index_x][index_z];
    glVertex3f(base_x, base_y, base_z);
    glVertex3f(base_x + normal[0] * scale_normals,
	       base_y + normal[1] * scale_normals,
	       base_z + normal[2] * scale_normals);
  }
  glEnd();
}


/*  copys three elements from normals to vector at given indexe  */
void
normals_copy(int index_x, int index_z, float ***normals, float *vector)
{
  int i;

  for (i = 0; i < 3; i++)
    {
      vector[i] = normals[i][index_x][index_z];
    }
}


void
junction_surface_compute(float ***surface, int points_x, int step)
{
  int k, i;

  for (k = 0; k < 3; k++)
    {
      for (i = step; i < points_x + 1; i += step * 2)
	{
	  surface[k][i][0] =
	    (surface[k][i - step][0] + surface[k][i + step][0]) / 2.0;
	}
    }
}


}  /*  namespace  */


/*  End draw_surface.cpp  */
