/*  Begin transform.cpp  */

/*  OpenGL viewing and projection transformations  */

/*
  Copyright (C) 2003  Jocelyn Frchot

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; version 2 of the License.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/


/****************  includes  ****************/


#include "transform.h"

#include "camera.h"
#include "scene.h"
#include "sun.h"

/*  graphic lib  */
extern "C"
{
#include <GL/gl.h>
#include <GL/glu.h>
#include <GL/glut.h>
}


/**************** static functions prototypes ****************/


namespace
{
  void move_rotate_around_origin(float *position,
				 float angle_x,
				 float angle_y);
  void move_rotate_around_observer(float *position,
				   float angle_x,
				   float angle_y);
  void rotate(float angle_x, float angle_y);

  /*  modelview matrix  */
  void modelview_push(void);
  void modelview_pop(void);

  /*  projection matrix  */
  void projection_push(void);
  void projection_pop(void);
}


/****************  functions  ****************/


/*  viewing transformation for the scene  */

void
transform_scene_push(class Scene *scene, class Camera *camera)
{
  float position[3];
  float angle_x, angle_y;

  
  camera->get_opposite_position(position);
  camera->get_angle(angle_x, angle_y);

  modelview_push();

  switch(camera->view_mode)
    {
    case Observing:
      move_rotate_around_origin(position, angle_x, angle_y);
      break;

    case Flying:
    case Swimming:
      move_rotate_around_observer(position, angle_x, angle_y);
      break;
    }
  /*  sets the origin in the middle of the scene  */
  glTranslatef(- scene->get_whole_size_x() / 2.0,
	       0.0,
	       - scene->get_whole_size_z() / 2.0);
}


void
transform_scene_pop(void)
{
  modelview_pop();
}


/*  viewing transformation for the sun  */

void
transform_sun_push(class Sun *sun, class Camera *camera)
{
  /*
    sun must be "infinitely" far,
    i.e. it must not react to the camera translations (but to the rotations)
  */
  float angle_x, angle_y;
  float position[3];

  camera->get_angle(angle_x, angle_y);
  sun->get_position(position);

  modelview_push();

  move_rotate_around_observer(position, angle_x, angle_y);
}


void
transform_sun_pop(void)
{
  modelview_pop();
}


/*  projection transformation for string  */

void
transform_string_push(bool draw_from_top)
{
  int window_width;
  int window_height;

  window_width = glutGet(GLUT_WINDOW_WIDTH);
  window_height = glutGet(GLUT_WINDOW_HEIGHT);

  /*  sets orthographic projection  */
  projection_push();
  glLoadIdentity();
  gluOrtho2D(0, window_width, 0, window_height);

  if (draw_from_top)
    {
      glScalef(1, -1, 1);
      glTranslatef(0, -window_height, 0);
    }
}


void
transform_string_pop(void)
{
  projection_pop();
}


/**************** static functions  ****************/


namespace
{


void
move_rotate_around_origin(float *position, float angle_x, float angle_y)
{
  /*  the rotation center of the camera is on the origin  */
  glTranslatef(position[0], position[1], position[2]);
  rotate(angle_x, angle_y);
}


void
move_rotate_around_observer(float *position, float angle_x, float angle_y)
{
  /*  the rotation center of the camera is on the observer  */
  rotate(angle_x, angle_y);
  glTranslatef(position[0], position[1], position[2]);
}


void
rotate(float angle_x, float angle_y)
{
  glRotatef(angle_x, 1.0, 0.0, 0.0);
  glRotatef(angle_y, 0.0, 1.0, 0.0);
}


/*  modelview matrix  */

void
modelview_push(void)
{
  glMatrixMode(GL_MODELVIEW);
  glPushMatrix();
}


void
modelview_pop(void)
{
  glMatrixMode(GL_MODELVIEW);
  glPopMatrix();
}


/*  projection matrix  */

void
projection_push(void)
{
  glMatrixMode(GL_PROJECTION);
  glPushMatrix();
}


void
projection_pop(void)
{
  glMatrixMode(GL_PROJECTION);
  glPopMatrix();
}


}  /*  namespace  */


/*  End transform.cpp  */
