/*  Begin aqua_surface_phillips.cpp  */

/*  "Aqua_Surface" with Phillips spectrum  */

/*
  Copyright (C) 2003  Jocelyn Frchot

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; version 2 of the License.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/


/****************  includes  ****************/


#include "aqua_surface_phillips.h"

#include "aqua_rng.h"
#include "aqua_rng_ugaussian.h"
#include "aqua_wave_phillips.h"
#include "aqua_waves_set.h"


/****************  public functions  ****************/


/*  creates a surface and sets it at time 0.0  */
Aqua_Surface_Phillips::
Aqua_Surface_Phillips(int points_x,        /* numbers of discrete sample...   */
		      int points_z,        /* ...points, must be power of two */
		      float size_x,               /*  meters  */
		      float size_z,               /*  meters  */
		      float depth,     /* of water, meters, 0.0 means infinite*/
		      float displacement_factor,  /*  horizontal  */
		      float spectrum_factor,
		      float smallest_wave,
		      float wind_speed,
		      float wind_angle,
		      class Aqua_Rng *rng,
		      float loop_time)       /*  seconds (0.0 means no loop)  */
  : Aqua_Surface(points_x,
		 points_z,
		 size_x,
		 size_z,
		 depth,
		 displacement_factor,
		 this->wave_factory = new class Aqua_Wave_Factory_Phillips(),
		 this->spectrum_context =
		 new class Aqua_Wave_Spectrum_Context_Phillips(spectrum_factor,
							       smallest_wave,
							       wind_speed,
							       wind_angle),
		 rng,
		 loop_time)
{
  delete this->wave_factory;
}


Aqua_Surface_Phillips::~Aqua_Surface_Phillips(void)
{
  delete this->spectrum_context;
}


/****  set  ****/

void
Aqua_Surface_Phillips::set_spectrum_factor(float spectrum_factor)
{
  this->spectrum_context->spectrum_factor = spectrum_factor;
  this->waves_set->set_spectrum(spectrum_context);
  /*  recomputes Fourier amplitudes  */
  this->waves_set->get_fourier_amplitude(this->time,
					 this->field_fourier_amplitudes);
}


void
Aqua_Surface_Phillips::set_smallest_wave(float smallest_wave)
{
  this->spectrum_context->smallest_wave = smallest_wave;
  this->waves_set->set_spectrum(spectrum_context);
  /*  recomputes Fourier amplitudes  */
  this->waves_set->get_fourier_amplitude(this->time,
					 this->field_fourier_amplitudes);
}


void
Aqua_Surface_Phillips::set_wind_speed(float wind_speed)
{
  this->spectrum_context->set_wind_speed(wind_speed);
  this->waves_set->set_spectrum(spectrum_context);
  /*  recomputes Fourier amplitudes  */
  this->waves_set->get_fourier_amplitude(this->time,
					 this->field_fourier_amplitudes);
}


void
Aqua_Surface_Phillips::set_wind_angle(float wind_angle)
{
  this->spectrum_context->set_wind_angle(wind_angle);
  this->waves_set->set_spectrum(spectrum_context);
  /*  recomputes Fourier amplitudes  */
  this->waves_set->get_fourier_amplitude(this->time,
					 this->field_fourier_amplitudes);
}


/****  get  ****/

float
Aqua_Surface_Phillips::get_spectrum_factor(void) const
{
  return this->spectrum_context->spectrum_factor;
}


float
Aqua_Surface_Phillips::get_smallest_wave(void) const
{
  return this->spectrum_context->smallest_wave;
}


float
Aqua_Surface_Phillips::get_wind_speed(void) const
{
  return this->spectrum_context->get_wind_speed();
}


float
Aqua_Surface_Phillips::get_largest_wave(void) const
{
  return this->spectrum_context->get_largest_wave();
}


float
Aqua_Surface_Phillips::get_wind_angle(void) const
{
  return this->spectrum_context->get_wind_angle();
}


/*  fills two elements vector "wind_vector"  */
void
Aqua_Surface_Phillips::get_wind_vector(float *wind_vector) const
{
  this->spectrum_context->get_wind_vector(wind_vector);
}


/*  End aqua_surface_phillips.cpp  */
