/*  Begin aqua_wave.cpp  */

/*
  Copyright (C) 2003  Jocelyn Frchot

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; version 2 of the License.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/


/****************  includes  ****************/


#include "aqua_wave.h"

#include "vector.h"

/*  local includes  */
#include "include/constant.h"

/*  C lib  */
extern "C"
{
#include <math.h>
}


/*****************************************************/
/****************  Aqua_Wave_Factory  ****************/
/*****************************************************/


/****************  public functions  ****************/


/*  virtual destructor  */
Aqua_Wave_Factory::~Aqua_Wave_Factory(void)
{
}


/**************************************************************/
/****************  Aqua_Wave_Spectrum_Context  ****************/
/**************************************************************/


/****************  public functions  ****************/


/*  virtual destructor  */
Aqua_Wave_Spectrum_Context::~Aqua_Wave_Spectrum_Context(void)
{
}


/*********************************************/
/****************  Aqua_Wave  ****************/
/*********************************************/


/****************  public functions  ****************/


/*  virtual destructor  */
Aqua_Wave::~Aqua_Wave(void)
{
}


/*  inits the wave and sets magnitude_[xz] and unit_vector_[xz]  */
void
Aqua_Wave::init(float &magnitude_x,      /*  vector magnitude  */
		float &magnitude_z,
		float &unit_vector_x,    /*  normalized vector magnitude  */
		float &unit_vector_z,
		int vector_x,         /*  vector X coordinate, [ -N/2, N/2 [  */
		int vector_z,         /*  vector Z coordinate, [ -M/2, M/2 [  */
		float size_x,          /* size_x of the whole surface, meters */
		float size_z,          /* size_z of the whole surface, meters */
		float random_real,       /*  random number  */
		float random_imaginary,  /*  random number  */
		float depth,    /*  water depth, meters (0.0 means infinite)  */
		float basic_frequency,   /*  s-1, 0.0 means no loop  */
		class Aqua_Wave_Spectrum_Context *spectrum_context)
{
  /*  avoids compiler warnings for unused parameters  */
  static_cast<void>(spectrum_context);

  float vector[2];

  magnitude_x = static_cast<float>(vector_x) * Constant_2_pi / size_x;
  magnitude_z = static_cast<float>(vector_z) * Constant_2_pi / size_z;

  this->magnitude = hypotf(magnitude_x, magnitude_z);

  vector[0] = magnitude_x;
  vector[1] = magnitude_z;
  vector_normalize_2d(vector);
  unit_vector_x = vector[0];
  unit_vector_z = vector[1];

  if (isnormal(this->magnitude) == 0)
    {
      this->length = 0.0;
    }
  else
    {
      this->length = Constant_2_pi / this->magnitude;
    }

  this->random_real = random_real;
  this->random_imaginary = random_imaginary;

  this->set_frequency(depth, basic_frequency);
}


/****  set  ****/

/*
  if depth == 0.0 (infinite depth) or basic_frequency == 0.0 (no loop),
  doesn't take them into account
*/
void
Aqua_Wave::set_frequency(float depth, float basic_frequency)
{
  float frequency;

  frequency = Constant_g * this->magnitude;

  if (depth != 0.0)
    {
      frequency *= tanhf(this->magnitude * depth);
    }

  frequency = sqrtf(frequency);

  if (basic_frequency != 0.0)
    {
      frequency = floorf(frequency / basic_frequency) * basic_frequency;
    }

  this->frequency = frequency;
}


void
Aqua_Wave::set_spectrum(class Aqua_Wave_Spectrum_Context *context)
{
  float spectrum;

  spectrum = this->compute_spectrum(context);

  this->spectrum = sqrtf(2.0 * spectrum);
}


/****  get  ****/

/*  returns the fourier_amplitude of the wave at time "time" (in second)  */
float
Aqua_Wave::get_fourier_amplitude(float time) const
{
  float temp, fourier_amplitude;

  temp = this->frequency * time;
  fourier_amplitude = this->spectrum * (this->random_real * cosf(temp)
					- this->random_imaginary * sinf(temp));

  return fourier_amplitude;
}


/*  End aqua_wave.cpp  */
