/*  Begin aqua_waves_set.cpp  */

/*
  Copyright (C) 2003  Jocelyn Frchot

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; version 2 of the License.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/


/****************  includes  ****************/


#include "aqua_waves_set.h"

#include "aqua_rng.h"
#include "aqua_wave.h"


/****************  public functions  ****************/


/*
  creates a set of "points_x" * "points_z" Aqua_Waves
  and fills "field_waves_magnitudes_[xz]" and "field_waves_unit_vector_[xz]"
*/
Aqua_Waves_Set::
Aqua_Waves_Set(int points_x,           /* numbers of discrete sample... */
	       int points_z,           /*...points, must be powers of two*/
	       float size_x,           /*  meters  */
	       float size_z,           /*  meters  */
	       float depth,            /*  meters  */
	       float basic_frequency,  /*  s-1  */
	       class Aqua_Wave_Factory *wave_factory,
	       class Aqua_Wave_Spectrum_Context *spectrum_context,
	       class Aqua_Rng *rng,
	       float **field_waves_magnitudes_x,
	       float **field_waves_magnitudes_z,
	       float **field_waves_unit_vector_x,
	       float **field_waves_unit_vector_z)
  : points_x(points_x), points_z(points_z), waves_set_size(points_x * points_z)
{
  int vector_x, vector_z;  /*  [ -points / 2, points / 2 [  */

  int i, j;


  this->waves_set_allocate(wave_factory, this->waves_set_size);

  for (i = 0; i < points_x; i++)
    {
      vector_x = i - points_x / 2;
      for (j = 0; j < points_z; j++)
	{
	  vector_z = j - points_z / 2;
	  this->get_waves_set(i, j)->init(field_waves_magnitudes_x[i][j],
					  field_waves_magnitudes_z[i][j],
					  field_waves_unit_vector_x[i][j],
					  field_waves_unit_vector_z[i][j],
					  vector_x,
					  vector_z,
					  size_x,
					  size_z,
					  rng->get_number(),
					  rng->get_number(),
					  depth,
					  basic_frequency,
					  spectrum_context);
	}
    }
}


Aqua_Waves_Set::~Aqua_Waves_Set(void)
{
  this->waves_set_free(this->waves_set_size);
}


/****  set  ****/

/*  depth in meters, basic_frequency in s-1  */
void
Aqua_Waves_Set::set_frequency(float depth, float basic_frequency)
{
  int i;

  for (i = 0; i < this->waves_set_size; i++)
    {
      this->waves_set[i]->set_frequency(depth, basic_frequency);
    }
}


void
Aqua_Waves_Set::
set_spectrum(struct Aqua_Wave_Spectrum_Context *spectrum_context)
{
  int i;

  for (i = 0; i < this->waves_set_size; i++)
    {
      this->waves_set[i]->set_spectrum(spectrum_context);
    }
}


/****  get  ****/

/*
  fills "amplitudes_array" of size "points_x" by "points_z",
  at time "time" (in seconds)
*/
void
Aqua_Waves_Set::get_fourier_amplitude(float time,
				      float **amplitudes_array) const
{
  int i, j;

  for (i = 0; i < this->points_x; i++)
    {
      for (j = 0; j < this->points_z; j++)
	{
	  amplitudes_array[i][j] =
	    this->get_waves_set(i, j)->get_fourier_amplitude(time);
	}
    }
}


/****************  protected functions  ****************/


void
Aqua_Waves_Set::waves_set_allocate(class Aqua_Wave_Factory *wave_factory,
				   int waves_set_size)
{
  int i;

  this->waves_set = new class Aqua_Wave* [waves_set_size];
  for (i = 0; i < waves_set_size ; i++)
    {
      this->waves_set[i] = wave_factory->create();
    }
}


void
Aqua_Waves_Set::waves_set_free(int waves_set_size)
{
  int i;

  for (i = 0; i < waves_set_size ; i++)
    {
      delete this->waves_set[i];
    }
  delete[] this->waves_set;
}


class Aqua_Wave *
Aqua_Waves_Set::get_waves_set(int index_x, int index_z) const
{
  return this->waves_set[index_x * this->points_z + index_z];
}


/*  End aqua_waves_set.cpp  */
