/* Create and register an archive

   Copyright (C) 2001, 2002 Tom Lord
   Copyright (C) 2002, 2003 Walter Landry and the Regents of the
                                  University of California
   Copyright (C) 2004 Walter Landry

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License as
   published by the Free Software Foundation; either version 2 of the
   License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
   USA */

#include "parse_common_options.hpp"
#include "parse_unknown_options.hpp"
#include "check_extra_args.hpp"
#include "command_initializer.hpp"
#include "create_archive_registration.hpp"
#include "boost/filesystem/operations.hpp"
#include "boost/filesystem/exception.hpp"
#include "arx_error.hpp"
#include "gvfs.hpp"
#include <algorithm>
#include "Component_Regexes.hpp"
#include "add_key_to_archive.hpp"
#include "get_config_option.hpp"
#include "signed_archive.hpp"
#include "registered_archive.hpp"

using namespace std;
using namespace boost;
namespace fs=boost::filesystem;
using fs::path;

int make_archive(list<string> &argument_list, const command &cmd);

static command_initializer make_archive_init(command("make-archive",
"Create and register an archive",
"usage: make-archive [options] name directory",
" --mirror                Make the location a mirror\n\
 --key KEY               Put KEY in the archive\n\
\n\
NAME is the global name for the archive.  You should choose something\n\
unique, such as your email address followed by something descriptive.\n\
\n\
An archive with that name must not already be registered.  It will create\n\
the directory if it does not already exist.\n\
\n\
With the --mirror option, the new location will be made a mirror of the\n\
master location.  To actually populate it, see \"arx mirror --help\".\n\
\n\
With the --key option, the new archive will have a copy of the public\n\
gpg key for KEY.  This may also be set with the gpg-key parameter in param,\n\
in which case you can make an unsigned archive by giving an empty argument\n\
to --key.\n\
\n\
See \"arx sig --help\" for details on how signed archives work.",
make_archive,"Basic",true));

int make_archive(list<string> &argument_list, const command &cmd)
{
  Command_Info info(cmd);
  const string ArX_archive_version("ArX archive, format version 1");
  string gpg_key(get_config_option("gpg-key"));
  bool set_key(false), mirror(false);

  while(!argument_list.empty())
    if(!parse_common_options(argument_list,info))
      {
        if(*(argument_list.begin())=="--mirror")
          {
            mirror=true;
            argument_list.pop_front();
          }
        else if(*(argument_list.begin())=="--key")
          {
            set_key=true;
            argument_list.pop_front();
            if(argument_list.empty())
              {
                throw arx_error("Need an argument for --key");
              }
            gpg_key=*(argument_list.begin());
            argument_list.pop_front();
          }
        else
          {
            parse_unknown_options(argument_list);
            break;
          }
      }

  if(mirror && set_key)
    throw arx_error("You may not specify both --mirror and --key");

  if(argument_list.size()<2)
    {
      throw arx_error ("Not enough arguments");
    }
  string name(*(argument_list.begin()));
  argument_list.pop_front();
  if(!regex_match(name,Component_Regexes().archive))
    throw arx_error("Invalid archive name: " + name);

  string directory(*(argument_list.begin()));
  argument_list.pop_front();
  check_extra_args(argument_list,info);

  if(!mirror && registered_archive(name+"/"))
    throw arx_error("You already have an archive registered with the same name.  You must either\nchoose a different name, or use the --mirror option.\n\t"
                    + name);

  gvfs::Init();
  
  /* Create directories if needed */
  gvfs::uri dir_path(directory);
  
  if(!gvfs::exists(dir_path))
    {
      gvfs::make_directory(dir_path);
    }
  else if(!gvfs::is_directory(dir_path))
    {
      throw arx_error("The argument: " + directory
                      + "\n is not a directory");
    }
  
  gvfs::uri meta_path(dir_path / ",meta-info");
  if(!exists(meta_path))
    {
      gvfs::make_directory(meta_path);
    }
  else if(!is_directory(meta_path))
    {
      throw arx_error("Corrupt archive\n" + meta_path.string()
                      + "\n is not a directory");
    }
  
  /* Enter the characteristics of the archive. */
  
  gvfs::write_archive(meta_path / "name",name);
  gvfs::write_archive(dir_path / ",archive_version",ArX_archive_version);
  if(mirror)
    gvfs::write_archive(meta_path / "mirror",name);

  /* Copy over public keys if this is a mirror and the master archive
     is signed. */

  if(mirror)
    {
      if(signed_archive(name + "/"))
        gvfs::copy(key_location(name + "/").native_file_string(),
                   meta_path / "public_keys");
    }
  /* Otherwise, add the gpg_key if necessary. */
  else if(!gpg_key.empty())
    {
      add_key_to_archive(dir_path.string() + "/",gpg_key);
      gvfs::copy(meta_path / "public_keys",
                 key_location(name + "/").native_file_string());
      
    }
  
  /* Register the new archive. */
  create_archive_registration(name + "/",dir_path.string());

  return 0;
}
