/* Apply a patch to a source tree.  There are two versions, one for
   exact patching and one for inexact patching.

  Copyright (C) 2001, 2002 Tom Lord
  Copyright (C) 2002, 2003 Walter Landry and the Regents
                           of the University of California
  Copyright (C) 2004 Walter Landry
  
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; version 2 dated June, 1991.

  This program is distributed in the hope that it will be useful, but
  WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
  USA */

#include "boost/filesystem/operations.hpp"
#include "boost/filesystem/exception.hpp"
#include "boost/date_time/posix_time/posix_time.hpp"
#include "file_attributes.hpp"
#include "arx_error.hpp"
#include "Inventory_Flags.hpp"
#include "Inventory.hpp"
#include "inventory_directory.hpp"
#include <list>
#include <utility>
#include "Moved_Path.hpp"
#include "Current_Path.hpp"
#include "read_and_sort_index.hpp"
#include "read_checksums.hpp"
#include "edit_path.hpp"
#include "Temp_Path.hpp"

using namespace std;
using namespace boost;
using namespace boost::posix_time;
namespace fs=boost::filesystem;
using fs::path;

extern void output_patching_errors(list<pair<string,string> > &move_conflicts,
                                   list<pair<string,string> > &patch_conflicts,
                                   list<Moved_Path> &missing_moves,
                                   list<string> &missing_patches,
                                   list<pair<pair<string,file_attributes>,string> >
                                   &add_conflicts);

extern void apply_patches(list<pair<pair<string,string>,string> > patched[2],
                          const Inventory &source_inventory,
                          const bool &reverse,
                          const path &patch, const path &destination,
                          list<string> &missing_patches,
                          list<pair<string,string> > &patch_conflicts);

extern void move_to_destination(list<Moved_Path> &moved,
                                const path &deleted_path,
                                const path &source,
                                list<pair<string,string> > &move_conflicts,
                                const Checksums &checksums);

extern void set_aside_paths(const path &deleted_path, const path &source,
                            list<Moved_Path> &moved,
                            list<Moved_Path> &missing_moves,
                            Inventory &source_inventory);

extern void categorize_files(const list<pair<string,string> > original[2],
                             const list<pair<string,string> > modified[2],
                             list<pair<pair<string,string>,string> > patched[2],
                             list<Moved_Path> &moved,
                             list<Moved_Path> &mod_only,
                             const bool &reversed);

extern void add_files_and_directories(list<Moved_Path> &new_paths,
                                      const path &patch, const path &source,
                                      const path &temp_path,
                                      list<Moved_Path> &moved,
                                      const string &new_files_string,
                                      const string &mod,
                                      const Inventory &source_inventory,
                                      list<pair<pair<string,file_attributes>,string> >
                                      &add_conflicts);

/* This version is the general version for inexact patching. */

bool do_patch(const path &patch, const path &source, const bool &reverse,
              const bool &delete_removed)
{
  bool applied_ok(false);
  
  if(!exists(patch) || !is_directory(patch))
    {
      throw arx_error("Patch directory does not exist or is not a directory: "
                      + patch.native_file_string());
    }
  if(!exists(source) || !is_directory(source))
    {
      throw arx_error("Source directory does not exist or is not a directory: "
                      + source.native_file_string());
    }

  /* Strings to parametrize whether a patch is to be reversed. */
  string orig("orig"), mod("mod"), new_files_string("added");
  if(reverse)
    {
      orig="mod";
      mod="orig";
      new_files_string="deleted";
    }
          
  /* Inventory the source tree. */

  Inventory source_inventory;
  Checksums source_checksums;

  /* If the source directory is empty, don't inventory it.  This
     is useful for applying the very first patch. */

  if(fs::directory_iterator(source)!=fs::directory_iterator())
    {
      Current_Path current_path(source);
      Inventory_Flags flags(fs::current_path());
      flags.source=flags.control=true;
      flags.files=flags.directories=true;

      read_checksums(fs::current_path(),source_checksums);
      Checksums checksums(source_checksums);
      inventory_directory("", source_inventory, flags, checksums);
    }
  else
    {
      /* We have to create an _arx directory so that tree_root
         will work. */
      create_directory(source/"_arx");
    }

  /* Sort by ids to allow efficient searching later. */
  for(int m=0;m<2;++m)
    for(int l=0;l<num_inventory_types;++l)
      {
        source_inventory(l,m).sort(inventory_id_cmp);
      }

  /* Read the patch indices and categorize files. */

  list<pair<string,string> > original[2], modified[2];

  read_and_sort_index(patch/(orig + "-files-index"),original[arx_file]);
  read_and_sort_index(patch/(orig + "-dirs-index"),original[arx_dir]);
  read_and_sort_index(patch/(mod + "-files-index"),modified[arx_file]);
  read_and_sort_index(patch/(mod + "-dirs-index"),modified[arx_dir]);

  list<pair<pair<string,string>,string> > patched[2];
  list<Moved_Path> mod_only, moved;
  categorize_files(original,modified,patched,moved,mod_only,reverse);

  /* Do the real meat of deleting, renaming, adding and patching.
     We do the set asides because, among other things, a file may
     be renamed into a directory that has just been created or
     renamed.  */

  list<pair<string,string> > move_conflicts, patch_conflicts;
  list<pair<pair<string,file_attributes>,string> > add_conflicts;
  list<Moved_Path> missing_moves;
  list<string> missing_patches;

  /* Create a directory where I can put deleted things and other
     things temporarily. */

  path deleted_path;
  deleted_path=Temp_Path(",,deleted");
  if(lexists(source/deleted_path))
    {
      throw arx_error("Deleted items directory already exists\n\t"
                      + (source/deleted_path).native_file_string());
    }
  else
    {
      fs::create_directory(source/deleted_path);
    }

  set_aside_paths(deleted_path,source,moved,missing_moves,
                  source_inventory);
  add_files_and_directories(mod_only,patch,source,deleted_path,moved,
                            new_files_string,mod,source_inventory,
                            add_conflicts);
  move_to_destination(moved,deleted_path,source,move_conflicts,
                      source_checksums);
  apply_patches(patched,source_inventory,reverse,patch/"patches",source,
                missing_patches,patch_conflicts);
  output_patching_errors(move_conflicts,patch_conflicts,missing_moves,
                         missing_patches,add_conflicts);

  applied_ok=move_conflicts.empty() && patch_conflicts.empty()
    && missing_moves.empty() && missing_patches.empty();
  /* Remove the deleted paths if the option is set or the
     directory is empty. */

  if(delete_removed || fs::directory_iterator(source/deleted_path)
     ==fs::directory_iterator())
    remove_all(source/deleted_path);

  /* If we are patching a non-edit tree, we update the ++edit file
     with the patched files. */

  if(fs::exists(source/"_arx/++edit"))
    {
      for(list<pair<pair<string,string>,string> >::iterator
            i=patched[arx_file].begin(); i!=patched[arx_file].end(); ++i)
        edit_path(source,i->second);
    }

  return applied_ok;
}


/* Versions of functions for exact patching. */

extern void apply_patches(list<pair<pair<string,string>,string> > patched[2],
                          const bool &reverse,
                          const path &patch, const path &destination);

extern void move_to_destination(list<Moved_Path> &moved,
                                const path &deleted_path,
                                const path &source);

extern void set_aside_paths(const path &deleted_path, const path &source,
                            list<Moved_Path> &moved);

extern void add_files_and_directories(list<Moved_Path> &new_paths,
                                      const path &patch, const path &source,
                                      const path &temp_path,
                                      list<Moved_Path> &moved,
                                      const string &new_files_string,
                                      const string &mod,
                                      const bool exact_patching=true);

/* Do exact patching.  If the patch is not exact, this will barf in
   unexpected ways. */

void do_patch(const path &patch, const path &source, const bool &reverse)
{
  /* Strings to parametrize whether a patch is to be reversed. */
  string orig("orig"), mod("mod"), new_files_string("added");
  if(reverse)
    {
      orig="mod";
      mod="orig";
      new_files_string="deleted";
    }

  /* If the source directory is empty, create an _arx directory so
     that tree_root will work. */

  if(fs::directory_iterator(source)==fs::directory_iterator())
    {
      create_directory(source/"_arx");
    }

  /* Read the patch indices and categorize files. */

  list<pair<string,string> > original[2], modified[2];

  read_and_sort_index(patch/(orig + "-files-index"),original[arx_file]);
  read_and_sort_index(patch/(orig + "-dirs-index"),original[arx_dir]);
  read_and_sort_index(patch/(mod + "-files-index"),modified[arx_file]);
  read_and_sort_index(patch/(mod + "-dirs-index"),modified[arx_dir]);

  list<pair<pair<string,string>,string> > patched[2];
  list<Moved_Path> mod_only, moved;
  categorize_files(original,modified,patched,moved,mod_only,reverse);

  /* Do the real meat of deleting, renaming, adding and patching.
     We do the set asides because, among other things, a file may
     be renamed into a directory that has just been created or
     renamed.  */

  /* Create a directory where I can put deleted things and other
     things temporarily. */

  path deleted_path;
  deleted_path=Temp_Path(",,deleted");
  if(lexists(source/deleted_path))
    {
      throw arx_error("Deleted items directory already exists\n\t"
                      + (source/deleted_path).native_file_string());
    }
  else
    {
      fs::create_directory(source/deleted_path);
    }

  set_aside_paths(deleted_path,source,moved);
  add_files_and_directories(mod_only,patch,source,deleted_path,moved,
                            new_files_string,mod);
  move_to_destination(moved,deleted_path,source);
  apply_patches(patched,reverse,patch/"patches",source);

  /* Remove the deleted paths */

  remove_all(source/deleted_path);
}
