/* Perform a merge of a branch with its sibling

  Copyright (C) 2001, 2002, 2003 Tom Lord
  Copyright (C) 2002, 2003 Walter Landry
                           and the Regents of the University of California
  Copyright (C) 2004, 2005 Walter Landry
   
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; version 2 dated June, 1991.

  This program is distributed in the hope that it will be useful, but
  WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
  USA */

#include "boost/filesystem/operations.hpp"
#include <list>
#include "patch_level.hpp"
#include "get_revision.hpp"
#include "Revision_List.hpp"
#include "Parsed_Name.hpp"
#include "Temp_Directory.hpp"
#include "make_patch.hpp"
#include "do_patch.hpp"
#include "copy_tree.hpp"
#include "valid_package_name.hpp"
#include "compute_ancestor.hpp"
#include <iostream>
#include "latest_archive_revision.hpp"

using namespace std;
using namespace boost;
namespace fs=boost::filesystem;
using fs::path;

void merge_branches(const Parsed_Name &tree, const Parsed_Name &sibling,
                    const path &tree_path, const bool &delete_removed)
{
  valid_package_name(tree,Revision);
  valid_package_name(sibling,Revision);

  Temp_Directory merge_dir(",,merge_branches");

  path sibling_path(merge_dir.path/"sibling");

  /* If tree and sibling are the same, then we don't have to do anything. */
  if(tree==sibling)
    {
      if(Command_Info::verbosity>=default_output)
        cout << "No merge done.  Sibling is the same as the project tree\n\t"
             << sibling.complete_revision() << endl;
      return;
    }

  get_revision(sibling,sibling_path,tree_path);

  list<Parsed_Name> ancestors(compute_ancestor(sibling,sibling_path,tree,
                                               tree_path));
  if(ancestors.empty())
    throw arx_error("Could not find a common ancestor between the two revisions:\n\t" + tree.complete_revision() + "\n\t" + sibling.complete_revision());
  
  Parsed_Name ancestor(*ancestors.begin());

  if(Command_Info::verbosity>=report)
    cout << "Using as the common ancestor: " << ancestor.complete_revision()
         << endl;

  get_revision(ancestor,merge_dir.path/"ancestor",tree_path);
  make_patch(merge_dir.path/"ancestor",sibling_path,merge_dir.path/"patch",
             Path_List());

  do_patch(merge_dir.path/"patch",tree_path,false,delete_removed);
}

