/*  BSD-License:

Copyright (c) 2007 by Nils Springob, nicai-systems, Germany
Copyright (c) 2010 by Matthias Bunte, Germany

All rights reserved.

Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

  * Redistributions of source code must retain the above copyright notice,
    this list of conditions and the following disclaimer.
  * Redistributions in binary form must reproduce the above copyright notice,
    this list of conditions and the following disclaimer in the documentation
    and/or other materials provided with the distribution.
  * Neither the names of the authors the name nicai-systems nor
    the names of its contributors may be used to endorse or promote products
    derived from this software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
"AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR
CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

/*! @file    hal_motorctrl.h
 *  @brief   HAL for ATmega16: Controlling the motor speed by pwm
 *  @author  Matthias Bunte (m_bunte@arcor.de)
 *  @date    2010-04-02
 */

#ifndef _BEELIB_HAL_MOTORCTRL_H_
#define _BEELIB_HAL_MOTORCTRL_H_

#include <avr/io.h>

#ifdef __cplusplus
extern "C" {
#endif

/*****************************************************************************/
/*  HAL - for motor control                                                  */
/*****************************************************************************/

/*
 * TIMER1 setup:
 *
 * System clock   = 15 Mhz
 * Prescaler      = 1
 * //Frequency      = 14.677 kHz
 * Frequency      = 20 kHz
 * Frequency      = 10 kHz
 * Frequency      = 5 kHz
 * Frequency      = 1 kHz
 *
 * Timer1_TOP = System clock / (2 * Prescaler * PWM-Frequency)
 *            = 15 MHz / (2 * 36 kHz) = 208
 *            = 15 MHz / (2 * 25 kHz) = 300
 *            = 15 MHz / (2 * 20 kHz) = 375
 *            = 15 MHz / (2 * 10 kHz) = 750
 *            = 15 MHz / (2 * 5 kHz)  = 1500
 *            = 15 MHz / (2 * 1 kHz)  = 7500
 *
 * Timer1_TOP has to be loaded in ICR1-Register to set "TOP"-Value.
 *
 * PWM on TIMER1 setup: 
 *
 * Mode "PWM, Phase Correct, TOP = ICR1"
 * => WGM10:WGM13 = 10 (0x0a)
 *
 * Output set to inverting on OC1A / OC1B
 * OC1A and OC1B both drive the 74HC139 decoder chip which again inverts the signal
 * => COM1A0:COM1A1 = 3 (0x03)
 * => COM1B0:COM1B1 = 3 (0x03)
 *
 */

#define MOTORCTRL_PWM_TIMER1_TOP  (300)
 
#define MOTORCTRL_CHANGE_DIR_IRQ_MASK   (TIMSK)
#define MOTORCTRL_CHANGE_L_DIR_IRQ      _BV(OCIE1A)
#define MOTORCTRL_CHANGE_R_DIR_IRQ      _BV(OCIE1B)

#define MOTORCTRL_HAL_PWM_REG_L   (OCR1A)

#define MOTORCTRL_HAL_PWM_REG_R   (OCR1B)

#define MOTORCTRL_PWM_TIMER_INIT()  do { \
   TCCR1A = _BV(COM1A1) | _BV(COM1A0) | _BV(COM1B1) | _BV(COM1B0) | _BV(WGM11); \
   TCCR1B = _BV(CS10) | _BV(WGM13); \
   ICR1   = MOTORCTRL_PWM_TIMER1_TOP; \
   } while (0)

/*****************************************************************************/
/*  End of HAL - for the motor control                                       */
/*****************************************************************************/


#ifdef __cplusplus
} // extern "C"
#endif


#endif // _BEELIB_HAL_MOTORCTRL_H_

