-- Copyright (C) 2008 Papavasileiou Dimitris                             
--                                                                      
-- This program is free software: you can redistribute it and/or modify 
-- it under the terms of the GNU General Public License as published by 
-- the Free Software Foundation, either version 3 of the License, or    
-- (at your option) any later version.                                  
--                                                                      
-- This program is distributed in the hope that it will be useful,      
-- but WITHOUT ANY WARRANTY; without even the implied warranty of       
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        
-- GNU General Public License for more details.                         
--                                                                      
-- You should have received a copy of the GNU General Public License    
-- along with this program.  If not, see <http://www.gnu.org/licenses/>.


local spring = joints.slider {}
local rows, shots = {}, {}
local tape

local recorder = frames.timer {
   period = 0.02,

   link = function ()
	     local v = cue.velocity

	     -- Rewind the tape.

	     tape.frames = 0
	     tape.duration = 0
	     tape.deltas[0] = 0

	     -- Save the cue position and orientation
	     -- to animate the cue during replay.
	     
	     tape.cueposition = cue.position
	     tape.cueorientation = cue.orientation
	     tape.cuespeed = math.pow(v[1]^2 + v[2]^2 + v[3]^2, 0.25)

	     -- Finally save the initial ball state.

	     for i, ball in ipairs (balls) do
		tape.positions[i] = {[0] = ball.position}
		tape.orientations[i] = {[0] = ball.orientation}
	     end
	  end,

   transform = function (n, delta, elapsed)
		  -- Record a frame.

		  if simulator.timescale > 0 then
		     tape.deltas[n] = tape.deltas[n - 1] +
			delta * simulator.timescale

		     for i, ball in ipairs (balls) do
			tape.positions[i][n] = ball.position
			tape.orientations[i][n] = ball.orientation
		     end

		     tape.frames = n
		     tape.duration = tape.deltas[n]
		  end
	       end
}

local replayer = frames.timer {
   period = 0,

   link = function ()
	     tape.current = 0
	  end,

   prepare = function (n, delta, elapsed)
		tape.current = tape.current + delta * simulator.timescale
		
		if tape.current < tape.duration then
		   local frame = 1
		   local delta

		   -- Interpolate the frames.
		
		   while tape.current >= tape.deltas[frame] do
		      frame = frame + 1
		   end

		   delta = (tape.current - tape.deltas[frame - 1]) /
		           (tape.deltas[frame] - tape.deltas[frame - 1])

		   for i, ball in ipairs (balls) do
		      local r_0 = tape.positions[i][frame - 1]
		      local r = tape.positions[i][frame]

		      local R_0 = tape.orientations[i][frame - 1]
		      local R = tape.orientations[i][frame]

		      ball.position = {
			 r_0[1] + (r[1] - r_0[1]) * delta,
			 r_0[2] + (r[2] - r_0[2]) * delta,
			 r_0[3] + (r[3] - r_0[3]) * delta,
		      }

		      ball.orientation = {
			 R_0[1] + (R[1] - R_0[1]) * delta,
			 R_0[2] + (R[2] - R_0[2]) * delta,
			 R_0[3] + (R[3] - R_0[3]) * delta,
			 R_0[4] + (R[4] - R_0[4]) * delta,
			 R_0[5] + (R[5] - R_0[5]) * delta,
			 R_0[6] + (R[6] - R_0[6]) * delta,
			 R_0[7] + (R[7] - R_0[7]) * delta,
			 R_0[8] + (R[8] - R_0[8]) * delta,
			 R_0[9] + (R[9] - R_0[9]) * delta
		      }

		      ball.velocity = {0, 0, 0}
		      ball.spin = {0, 0, 0}
		   end
		else
		   graph.replayer = nil
		end
	     end,

   unlink = function ()
	       finished = true
	       replaying = false

	       for i, ball in ipairs (balls) do
		  ball.position = ball.oldposition
		  ball.orientation = ball.oldorientation
	       end
	    end
}

local restriker = frames.timer {
   period = 0,

   link = function ()
	     cue.swings = math.random (2, 4)

	     -- Set up the scene.

	     for i, ball in ipairs (balls) do
		ball.oldposition = ball.position
		ball.oldorientation = ball.orientation

		ball.position = tape.positions[i][0]
		ball.orientation = tape.orientations[i][0]
		ball.velocity = {0, 0, 0}
		ball.spin = {0, 0, 0}
	     end

 	     hand.bodies = {nil, nil}

	     cue.position = tape.cueposition
	     cue.orientation = tape.cueorientation

	     spring.axis = transforms.fromnode (cue, {0, 0, 1})
 	     spring.bodies = {cue, nil}

	     -- We don't want the cue to actually collide
	     -- with anything (everything is prerecorded,
	     -- no need to simulate).

	     cue.oldclassification = cue.classification
	     cue.classification = nil

	     graph.observer.gear.cue = cue
	     
	     finished = false
	     replaying = true
	  end,

   transform = function (n, delta, elapsed)
		  local l_0, l, k_d, v

		  if cue.swings >= 0 then
		     -- These are empirical values for stroke limits,
		     -- viscosity, and pull-back velocity.

		     l_0 = 0.02 * tape.cuespeed
		     l = 0.005 * cue.swings
		     k_d = 50 * math.sqrt(cue.swings)
		     v = 0.05 * 1.5 ^ cue.swings
		     
		     -- When the cue comes to a stop or touches the
		     -- cueball, count another swing and start pulling
		     -- away.

		     if (math.abs(spring.state[2]) < 0.01 or
		         spring.state[1] <= 0) and spring.motor[1] <= 0 then
			spring.motor = {v, billiards.cueforce}
			spring.stops = {{-1, 1}, {0, 0}, 0}

			cue.swings = cue.swings - 1
		     end

		     -- Let go once we've pulled far enough.

		     if spring.state[1] > l_0 and spring.motor[1] > 0 then
			spring.motor = {0, 0}
			spring.stops = {{l, l}, physics.spring (750, k_d), 0}
		     end

		     -- When we're done swinging start replaying.
		  else
		     graph.replayer = replayer
		  end
	       end,

   unlink = function ()
	       cue.classification = cue.oldclassification

	       spring.bodies = {nil, nil}
	       hand.bodies = {cue, nil}
	       
	       graph.observer.gear.cue = nil	       
	    end
}

local function callhooks (hooks)
   if type (hooks) == "table" then
      for _, hook in pairs(hooks) do
	 hook ()
      end
   elseif type (hooks) == "function" then
      hook ()
   end
end

local function replayshot (n)
   local shot = shots[n]

   if shot and finished then
      tape = shot.tape
      graph.replayer = restriker
   end
end

local function backtrackto (n)
   local undone

   -- Remove the last row(s) from the table.
   
   for i = n, #shots do
      undone = table.remove(shots)
   end

   -- Backtrack.

   if undone then
      for i = 1, #rows - #shots do
	 table.remove (rows)
      end
	 
      cueball = undone.cueball
      objectball = undone.objectball
	 
      for i = 1, #balls do
	 balls[i].counts = undone.counts[i]
	 balls[i].games = undone.games[i]
	 balls[i].position = undone.positions[i]
	 balls[i].velocity = {0, 0, 0}
	 balls[i].spin = {0, 0, 0}
      end

      dofile "scripts/billiards.lua"
   end
end

local function drawtable()
   local content
   local w, w_2 = billiards.tablewidth, 0.5 * billiards.tablewidth
   local h, h_2 = billiards.tableheight, 0.5 * billiards.tableheight

   content = string.format([[

  <defs>

  <g id="diamond" transform="translate(0 -0.021) rotate(45)">
    <rect width="0.03" height="0.03"
	  fill="white" stroke="none"/>
  </g>

  <g id="ball">
    <circle r="0.0305" stroke-width="0.003"/>
  </g>

  </defs>

  <rect x="%f" y="%f" rx="0.09" width="%f" height="%f" fill="rgb(120, 80, 50)" 
	stroke="rgb(225, 185, 135)" stroke-width="0.015"/>
  <rect x="%f" y="%f" width="%f" height="%f" fill="rgb(85, 130, 160)" 
	stroke="rgb(50, 50, 50)" stroke-width="0.003"/>
  <rect x="%f" y="%f" width="%f" height="%f" fill="rgb(125, 165, 200)" 
        stroke="rgb(70, 90, 125)" stroke-width="0.008"/>
			      
  <circle cx="%f" cy="0" r="0.01"
          stroke="none" fill="rgb(85, 130, 160)"/>
  <circle cx="%f" cy="0" r="0.01"
          stroke="none" fill="rgb(85, 130, 160)"/>
  <circle cx="%f" cy="-0.1524" r="0.01"
	  stroke="none" fill="rgb(85, 130, 160)"/>
			]],
			-w_2 - 0.19, -h_2 - 0.19, w + 0.38, h + 0.38,
			-w_2 - 0.04, -h_2 - 0.04, w + 0.08, h + 0.08,
			-w_2, -h_2, w, h,
			0.5 * w_2, -0.5 * w_2, -0.5 * w_2)

   for i = -3, 3 do
      content = content .. string.format([[
  <use x="%f" y="%f" xlink:href="#diamond"/>
  <use x="%f" y="%f" xlink:href="#diamond"/>
				      ]],
				      0.25 * i * w_2, h_2 + 0.11,
				      0.25 * i * w_2, -h_2 - 0.11)
   end

   for i = -1, 1 do
      content = content .. string.format([[
  <use x="%f" y="%f" xlink:href="#diamond"/>
  <use x="%f" y="%f" xlink:href="#diamond"/>
				      ]],
				      w_2 + 0.11, 0.5 * i * h_2,
				     -w_2 - 0.11, 0.5 * i * h_2)
   end

   return content
end

local function drawshot()
   local content
   local shot = shots[#shots]
   local w, w_2 = billiards.tablewidth, 0.5 * billiards.tablewidth
   local h, h_2 = billiards.tableheight, 0.5 * billiards.tableheight

   content = string.format([[
<?xml version="1.0" standalone="no"?>
<!DOCTYPE svg PUBLIC "-//W3C//DTD SVG 1.1//EN" 
  "http://www.w3.org/Graphics/SVG/1.1/DTD/svg11.dtd">

<svg viewBox="%f %f %f %f"
     xmlns="http://www.w3.org/2000/svg" version="1.1" 
     xmlns:xlink="http://www.w3.org/1999/xlink">
			   ]],
			   -w_2 - 0.19, -h_2 - 0.19, w + 0.38, h + 0.38,
			   -w_2 - 0.04, -h_2 - 0.04, w + 0.08, h + 0.08,
			   -w_2, -h_2, w, h)

   content = content .. drawtable()
   content = content .. string.format([[
  <circle cx="%f" cy="0" r="0.01"
          stroke="none" fill="darkslategrey"/>
  <circle cx="%f" cy="0" r="0.01"
          stroke="none" fill="darkslategrey"/>
  <circle cx="%f" cy="-0.1524" r="0.01"
	  stroke="none" fill="darkslategrey"/>
				   ]],
				   0.5 * w_2, -0.5 * w_2, -0.5 * w_2)
   
   if tape.frames > 0 then
      for i, ball in ipairs(balls) do 
	 content = content .. string.format([[
	   <path d="M %f %f    
					 ]],
					 tape.positions[i][0][1],
					 tape.positions[i][0][2])

	 for i, position in ipairs(tape.positions[i]) do
	    content = content .. string.format([[
             L %f %f
					       ]],
					       position[1], position[2])
	 end

	 content = content .. string.format([[
     " fill="none" stroke="rgb(%d, %d, %d)" stroke-width="0.005" />
                                            ]],
					    ball.surface.diffuse[1] * 255,
					    ball.surface.diffuse[2] * 255,
					    ball.surface.diffuse[3] * 255)
      end

      for i, ball in ipairs(balls) do
	 content = content .. string.format([[
      <use x="%f" y="%f" stroke="rgb(%d, %d, %d)"
           fill="none" xlink:href="#ball"/>
					 ]],
					 shot.positions[i][1],
					 shot.positions[i][2],
					 ball.surface.diffuse[1] * 255,
					 ball.surface.diffuse[2] * 255,
					 ball.surface.diffuse[3] * 255)
      end
   end

   for i, ball in ipairs(balls) do
      content = content .. string.format([[
      <use x="%f" y="%f" fill="rgb(%d, %d, %d)"
           stroke="rgb(75, 75, 75)" xlink:href="#ball"/>
				      ]],
				      ball.position[1], ball.position[2],
				      ball.surface.diffuse[1] * 255,
				      ball.surface.diffuse[2] * 255,
				      ball.surface.diffuse[3] * 255)
   end

   content = content .. [[
</svg>
      ]]

   return content
end

local function drawshots(page, arguments)
   local content, from, to

   if arguments.setup then
      -- Remove the current shot.

      table.remove (rows)
      
      for i = 1, #shots - #rows do
	 table.remove (shots)
      end

      -- Set up the balls.

      for i, ball in ipairs(balls) do
	 ball.position =  {tonumber(arguments["x" .. i]),
			   tonumber(arguments["y" .. i]),
			   billiards.ballradius}

	 balls[i].velocity = {0, 0, 0}
	 balls[i].spin = {0, 0, 0}
      end

      cueball = balls[tonumber(arguments.cueball)]
      objectball = balls[3 - tonumber(arguments.cueball)]

      -- And start a new one.

      dofile "scripts/billiards.lua"
   end   

   if arguments.undo then
      backtrackto(tonumber(arguments.undo))
   end

   if arguments.replay then
      replayshot(tonumber(arguments.replay))
   end

   from = tonumber(arguments.from) or #rows - 4
   if from <= 0 then from = 1 end

   to = tonumber(arguments.to) or from + 4
   if to > #rows then to = #rows end

   content = [[
<link rel="stylesheet" href="stylesheet" type="text/css">
<head>

<title>Shots</title>
<style>

table {
   width : 100% ;
   border-collapse: separate ;
   border-spacing: 0px 0px ;
   border-bottom : solid ;
   border-width : thin ;
   border-color : #a0a0a0 ;
   empty-cells: show
}

tr {
   border-top : solid ;
   border-width : thin ;
   border-color : #a0a0a0 ;
}

td {
   text-align: center ; 
   vertical-align : middle ;
   font-size : 3em ;
   font-family : serif ;
   padding-left : 25pt ;
}
    
</style>

<body>
<hr> 
<h1><iframe style="width : 8em ; height : 3em"
            frameborder="0" src="logo">Billiards
</iframe></h1>
   ]]

   if from > 1 then
      local newfrom = from - 5 > 0 and from - 5 or 1

      content = content .. string.format([[
<a href = "/shots?from=%d&to=%d">previous</a>
&middot;
				      ]], newfrom, newfrom + 4)
   else
      content = content .. [[
previous
&middot;
      ]]
   end

   if to < #rows then
      local newto = to + 5 <= #rows and to + 5 or #rows

      content = content .. string.format([[
<a href = "/shots?from=%d&to=%d">next</a>
&middot;
					 ]], newto - 4, newto)
   else
      content = content .. [[
next
&middot;
      ]]
   end

   content = content .. [[
<a href = "/setupshot">setup</a>
&middot;
<a href = "/shots">refresh</a>
&middot;
<a href = "/">back</a>
<hr> 
<p>
<table>
      ]]
   
   for i = from, to do
      content = content .. rows[i]
   end

   return content .. [[
</table>
</p>
</body>
      ]]
end

local function drawsetup()
   local content
   local w, w_2 = billiards.tablewidth, 0.5 * billiards.tablewidth
   local h, h_2 = billiards.tableheight, 0.5 * billiards.tableheight

   content = string.format([=[
<?xml version="1.0" standalone="no"?>
<!DOCTYPE svg PUBLIC "-//W3C//DTD SVG 1.1//EN" 
  "http://www.w3.org/Graphics/SVG/1.1/DTD/svg11.dtd">

<svg onmousemove="mousemove(evt)"
     onmouseup="mouseup(evt)"
     viewBox="%f %f %f %f"
     xmlns="http://www.w3.org/2000/svg" version="1.1" 
     xmlns:xlink="http://www.w3.org/1999/xlink">
   
  <script type="text/ecmascript"> <![CDATA[
    var selected, ball;
    
    top.getLatitude = getLatitude;
    top.getLongitude = getLongitude;
    top.getCueball = getCueball;

    function getLatitude(id) {
	var ball = document.getElementById(id);
	return ball.getAttribute("x")
    }

    function getLongitude(id) {
	var ball = document.getElementById(id);
	return ball.getAttribute("y")
    }

    function getCueball() {
	return selected.getAttribute("id")
    }

    function mousedown(evt) {
        ball = evt.target;

        ball.setAttribute("fill-opacity", "0.5");
        ball.setAttribute("pointer-events", "none");

	if (ball.getAttribute("id") <= "%d") {
            selected = evt.target;
	    mousemove(evt);
	}
    }

    function mousemove(evt) {
        if (ball) {
	    var selection = document.getElementById("selection");
            var p = document.documentElement.createSVGPoint();
	    var M = ball.getCTM();
      
	    p.x = evt.clientX;
	    p.y = evt.clientY;

	    p = p.matrixTransform(M.inverse());
	  
	    if (p.x > 1.3895) {
	       p.x = 1.3895;
	    }

	    if (p.x < -1.3895) {
	       p.x = -1.3895;
	    }
	  
	    if (p.y > 0.6795) {
	       p.y = 0.6795;
	    }

	    if (p.y < -0.6795) {
	       p.y = -0.6795;
	    }

	    ball.setAttribute("x", p.x);
	    ball.setAttribute("y", p.y);

	    if (ball.getAttribute("id") <= "%d") {
	        selection.setAttribute("cx", p.x);
	        selection.setAttribute("cy", p.y);
	    }
	 }
    }
 
    function mouseup(evt) {
        ball.setAttribute("fill-opacity", "1.0");
        ball.setAttribute("pointer-events", "all")

	ball = false;
    }
  ]]>
  </script> 
                           ]=],
			   -w_2 - 0.2, -h_2 - 0.2, w + 0.4, h + 0.4,
			   billiards.players, billiards.players)

   content = content .. drawtable() 
   content = content .. string.format([[
  <circle id="selection" cx="%f" cy="%f" r="0.04" stroke="red"
          fill="none"  stroke-width="0.01"/>
				   ]],
				   cueball.position[1], cueball.position[2])

   for i, ball in ipairs(balls) do
      content = content .. string.format([=[
      <use id="%s" x="%f" y="%f" fill="rgb(%d, %d, %d)"
	   onmousedown="mousedown(evt)" cursor="move"
           xlink:href="#ball"/>
				      ]=],
				      tostring(i),
				      ball.position[1], ball.position[2],
				      ball.surface.diffuse[1] * 255,
				      ball.surface.diffuse[2] * 255,
				      ball.surface.diffuse[3] * 255)
   end

   content = content .. [=[
</svg>
   ]=]

   return content
end

function setupshot(uri, arguments)
   local content

   content = [=[
<head>

<title>Billiards</title>
<link rel="stylesheet" href="stylesheet" type="text/css">

</head>

<script type="text/javascript">
  function submit ()
  {
     document.forms[0].elements[0].value = getLatitude("1")
     document.forms[0].elements[1].value = getLongitude("1")

     document.forms[0].elements[2].value = getLatitude("2")
     document.forms[0].elements[3].value = getLongitude("2")

     document.forms[0].elements[4].value = getLatitude("3")
     document.forms[0].elements[5].value = getLongitude("3")

     document.forms[0].elements[6].value = getCueball()

     document.forms[0].submit()
  }
</script> 

<body>
<hr> 
<h2><iframe width="337" height="125" frameborder="0" src="logo">Billiards</iframe></h2>
<a href="javascript:submit()">done</a>&middot;
<a href="shots">cancel</a>
<hr> 

<p>
Move the balls around by dragging them with the left mouse button and click <q>done</q> do set up the current shot according to the diagram.  The highlighted ball will be the cue ball.
</p>

<iframe style="width : 100% ; height : 100%"
        frameborder="0" src="/drawsetup">
Frames aren't enabled on your broswer.
</iframe>

<form action="/shots" method="get">
<input type="hidden" name="x1">
<input type="hidden" name="y1">
<input type="hidden" name="x2">
<input type="hidden" name="y2">
<input type="hidden" name="x3">
<input type="hidden" name="y3">
<input type="hidden" name="cueball">
<input type="hidden" name="setup" value="yes">
</form>

</body>
   ]=]

   return content
end

function saveshot(page, arguments)
   local shot, href, content

   shot = shots[tonumber(arguments.shot)]
   
   if shot then
      href = string.format ("/shots?setup=yes&x1=%f&y1=%f&x2=%f&y2=%f&x3=%f&y3=%f&cueball=%d",
			    shot.positions[1][1], shot.positions[1][2],
			    shot.positions[2][1], shot.positions[2][2],
			    shot.positions[3][1], shot.positions[3][2],
			    shot.cueball == balls[1] and 1 or 2)
   else
      href = string.format ("/shots?setup=yes&x1=%f&y1=%f&x2=%f&y2=%f&x3=%f&y3=%f&cueball=%d",
			    balls[1].position[1], balls[1].position[2],
			    balls[2].position[1], balls[2].position[2],
			    balls[3].position[1], balls[3].position[2],
			    cueball == balls[1] and 1 or 2)
   end

   content = [[
<link rel="stylesheet" href="stylesheet" type="text/css">
<head>

<title>Save shot</title>

<body>
<p>
Bookmark <a href="]] .. href .. [[">this</a> link.  You can then use it to restore the current shot.  Remember that using this bookmark will be equivalent to following the <q>setup</q> link and then moving the balls to their current location.  This means:
<ul>
<li> The bookmark will only setup the current shot, that is, it will only affect ball positions, not the the score, names etc.
<li> You cannot restore when a shot is in progress.
   </ul>
You can now either follow this <a href="/shots">link</a> to the <q>shots</q> page or use your browser's back button.
</p>
</body
   ]]
   
   return content
end

function billiards.waiting.replayer ()
   graph.recorder = recorder

   -- Save the new shot.

   table.insert (shots, {
		    tape = tape,
		    cueball = cueball,
		    objectball = objectball,
		    counts = {balls[1].counts, balls[2].counts},
		    games = {balls[1].games, balls[2].games},
		    positions = {balls[1].position,
				 balls[2].position,
				 balls[3].position}
		 })
end

function billiards.finished.replayer ()
   -- Stop recording or replaying.

   graph.recorder = nil
   graph.replayer = nil

   -- Save a snapshot of the shot.

    network.pages[string.format("/drawshot%d", #rows)] = drawshot()   
end

function billiards.newshot.replayer ()
   local row

   -- Stop recording.

   graph.recorder = nil

   -- Save a snapshot.

   network.mime[string.format("/drawshot%d", #rows + 1)] = "image/svg+xml"
   network.pages[string.format("/drawshot%d", #rows + 1)] = drawshot   

   -- Write the html.

   row = string.format([[
<tr style="background : rgb(%d, %d, %d)">
<td>%d
<td style="text-align : center ; font-size : medium">
<iframe frameborder="0" src="/drawshot%d"></iframe>
<a href="/drawshot%s">enlarge</a>
<a href="/shots?undo=%d">backtrack</a>
<a href="/shots?replay=%d">replay</a>
<a href="/saveshot?shot=%d">save</a>
<td style="width : 100%%"">
		       ]],
		    cueball.surface.diffuse[1] * 255,
		    cueball.surface.diffuse[2] * 255,
		    cueball.surface.diffuse[3] * 255,
		    #rows + 1, #rows + 1, #rows + 1,
		    #rows + 1, #rows + 1, #rows + 1)

   if billiards.players == 1 then
      row = row .. string.format([[
<div style="font-size : large ; vertical-align : 50%%">%s</div> 
<div>%d</div>
				 ]],
			      balls[1].name, balls[1].counts)
   else
      row = row .. string.format([[
<span style="font-size : large ; vertical-align : 50%%">%s</span> 
%d - %d
<span style="font-size : large ; vertical-align : 50%%">%s</span> 
				 ]],
			      balls[1].name,
			      balls[1].counts, balls[2].counts,
			      balls[2].name)
   end

   table.insert(rows, row)

   -- Erase the tape.
   
   tape = {
      frames = 0,
      duration = 0,
      current = 0,

      deltas = {},
      positions = {},
      orientations = {}
   }
end

graphics.keypress.shots = function (key)
   if key == billiards.replay then
      replayshot(#shots)
   elseif key == billiards.undo then
      backtrackto(#shots)
   end
end

network.mime["/drawshot"] = "image/svg+xml"
network.pages["/drawshot"] = drawshot

network.mime["/drawsetup"] = "image/svg+xml"
network.pages["/drawsetup"] = drawsetup

network.mime["/saveshot"] = "text/html"
network.pages["/saveshot"] = saveshot

network.mime["/setupshot"] = "text/html"
network.pages["/setupshot"] = setupshot

network.mime["/shots"] = "text/html"
network.pages["/shots"] = drawshots
